/*
 * chgutil.c - native operating system interface for cHg server
 *
 * Copyright 2014 Yuya Nishihara <yuya@tcha.org>
 *
 * This software may be used and distributed according to the terms of the
 * GNU General Public License version 2 or any later version.
 */

#include <Python.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/types.h>

/*
 * recvfds() simply does not release GIL during blocking io operation because
 * Mercurial command server is known to be single-threaded.
 */

static ssize_t recvfdstobuf(int sockfd, int **rfds, void *cbuf, size_t cbufsize)
{
	char dummy[1];
	struct iovec iov = {dummy, sizeof(dummy)};
	struct msghdr msgh = {0};
	struct cmsghdr *cmsg;

	msgh.msg_iov = &iov;
	msgh.msg_iovlen = 1;
	msgh.msg_control = cbuf;
	msgh.msg_controllen = (socklen_t)cbufsize;
	if (recvmsg(sockfd, &msgh, 0) < 0)
		return -1;

	for (cmsg = CMSG_FIRSTHDR(&msgh); cmsg;
	     cmsg = CMSG_NXTHDR(&msgh, cmsg)) {
		if (cmsg->cmsg_level != SOL_SOCKET ||
		    cmsg->cmsg_type != SCM_RIGHTS)
			continue;
		*rfds = (int *)CMSG_DATA(cmsg);
		return (cmsg->cmsg_len - CMSG_LEN(0)) / sizeof(int);
	}

	*rfds = cbuf;
	return 0;
}

static PyObject *recvfds(PyObject *self, PyObject *args)
{
	int sockfd;
	int *rfds = NULL;
	ssize_t rfdscount, i;
	char cbuf[256];
	PyObject *rfdslist = NULL;

	if (!PyArg_ParseTuple(args, "i", &sockfd))
		return NULL;

	rfdscount = recvfdstobuf(sockfd, &rfds, cbuf, sizeof(cbuf));
	if (rfdscount < 0)
		return PyErr_SetFromErrno(PyExc_OSError);

	rfdslist = PyList_New(rfdscount);
	if (!rfdslist)
		goto error;
	for (i = 0; i < rfdscount; i++) {
		PyObject *obj = PyInt_FromLong(rfds[i]);
		if (!obj)
			goto error;
		PyList_SET_ITEM(rfdslist, i, obj);
	}
	return rfdslist;

error:
	Py_XDECREF(rfdslist);
	return NULL;
}

static PyMethodDef methods[] = {
	{"recvfds", recvfds, METH_VARARGS,
	 "receive list of file descriptors via socket"},
	{NULL, NULL, 0, NULL},
};

PyMODINIT_FUNC init_chgutil(void)
{
	Py_InitModule("_chgutil", methods);
}
