/*
 * Copyright 2016 Facebook, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <gtest/gtest.h>

#include <folly/futures/Future.h>
#include <folly/futures/Promise.h>
#include <folly/Baton.h>

using namespace folly;

TEST(Interrupt, raise) {
  std::runtime_error eggs("eggs");
  Promise<Unit> p;
  p.setInterruptHandler([&](const exception_wrapper& e) {
    EXPECT_THROW(e.throwException(), decltype(eggs));
  });
  p.getFuture().raise(eggs);
}

TEST(Interrupt, cancel) {
  Promise<Unit> p;
  p.setInterruptHandler([&](const exception_wrapper& e) {
    EXPECT_THROW(e.throwException(), FutureCancellation);
  });
  p.getFuture().cancel();
}

TEST(Interrupt, handleThenInterrupt) {
  Promise<int> p;
  bool flag = false;
  p.setInterruptHandler([&](const exception_wrapper& /* e */) { flag = true; });
  p.getFuture().cancel();
  EXPECT_TRUE(flag);
}

TEST(Interrupt, interruptThenHandle) {
  Promise<int> p;
  bool flag = false;
  p.getFuture().cancel();
  p.setInterruptHandler([&](const exception_wrapper& /* e */) { flag = true; });
  EXPECT_TRUE(flag);
}

TEST(Interrupt, interruptAfterFulfilNoop) {
  Promise<Unit> p;
  bool flag = false;
  p.setInterruptHandler([&](const exception_wrapper& /* e */) { flag = true; });
  p.setValue();
  p.getFuture().cancel();
  EXPECT_FALSE(flag);
}

TEST(Interrupt, secondInterruptNoop) {
  Promise<Unit> p;
  int count = 0;
  p.setInterruptHandler([&](const exception_wrapper& /* e */) { count++; });
  auto f = p.getFuture();
  f.cancel();
  f.cancel();
  EXPECT_EQ(1, count);
}

TEST(Interrupt, withinTimedOut) {
  Promise<int> p;
  Baton<> done;
  p.setInterruptHandler([&](const exception_wrapper& /* e */) { done.post(); });
  p.getFuture().within(std::chrono::milliseconds(1));
  // Give it 100ms to time out and call the interrupt handler
  auto t = std::chrono::steady_clock::now() + std::chrono::milliseconds(100);
  EXPECT_TRUE(done.timed_wait(t));
}
