/*
 * Copyright 2018-present Facebook, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#pragma once

#include <exception>

#include <folly/CPortability.h>
#include <folly/CppAttributes.h>

namespace folly {

/// throw_exception
///
/// Throw an exception if exceptions are enabled, or terminate if compiled with
/// -fno-exceptions.
template <typename Ex>
[[noreturn]] FOLLY_NOINLINE FOLLY_COLD void throw_exception(Ex&& ex) {
#if (__GNUC__ && !__EXCEPTIONS)
  std::terminate();
#else
  throw static_cast<Ex&&>(ex);
#endif
}

// clang-format off
namespace detail {
template <typename T>
FOLLY_ALWAYS_INLINE FOLLY_ATTR_VISIBILITY_HIDDEN T&& to_exception_arg_(T&& t) {
  return static_cast<T&&>(t);
}
template <std::size_t N>
FOLLY_ALWAYS_INLINE FOLLY_ATTR_VISIBILITY_HIDDEN char const* to_exception_arg_(
    char const (&array)[N]) {
  return static_cast<char const*>(array);
}
template <typename Ex, typename... Args>
[[noreturn]] FOLLY_NOINLINE FOLLY_COLD void throw_exception_(Args&&... args) {
  throw_exception(Ex(static_cast<Args&&>(args)...));
}
} // namespace detail
// clang-format on

/// throw_exception
///
/// Construct and throw an exception if exceptions are enabled, or terminate if
/// compiled with -fno-exceptions.
///
/// Converts any arguments of type `char const[N]` to `char const*`.
template <typename Ex, typename... Args>
[[noreturn]] FOLLY_ALWAYS_INLINE FOLLY_ATTR_VISIBILITY_HIDDEN void
throw_exception(Args&&... args) {
  detail::throw_exception_<Ex>(
      detail::to_exception_arg_(static_cast<Args&&>(args))...);
}

} // namespace folly
