module CDs where

import HsShellScript
import Arg
import Posix
import Bits
import Char
import Parsec
import List
import Monad
import System
import Maybe

hashliste = "/home/v/.cdhashliste"

mount echo mountp =
   run_ee True echo "mount" [mountp]

umount echo mountp =
   run_ee True echo "umount" [mountp]


-- Name eingelegter CD bestimmen. Aus den in der Hashliste gespeicherten
-- bekannten Namen und ggF. einem in der Kommandozeile angegebenen wird der
-- Name eingelegten CD bestimmt, und ggF. die Hashliste erweitert. Wenn die
-- CD unbekannt ist, und auch in der Kommandozeile nichts angegeben, wird
-- das Programm abgebrochen. Es wird auf jeden Fall versucht, zu mounten
-- und einen angegebenen CD-Name zu registrieren. --echt wird dabei
-- ignoriert.
-- Die CD mu ungemountet sein, und wird cd cdname gemountet und wieder
-- entmountet.
cdname :: Bool          -- ob Kommando anzeigen (--echo)
       -> String        -- Mountpunkt
       -> Maybe String  -- CD-Name in der Kommandozeile
       -> IO String     -- Name der CD
cdname echo mountp name = do
   mount echo mountp
   cdhash <- cd_hash mountp
   hashliste <- lies_hashliste
   umount echo mountp

   -- CD-Name bestimmen - angegeben oder aus Hashwert des CD-Hauptverzeichnisses
   case name of
      (Just kzname) -> do
          -- CD-Name auf der Kommandozeile angegeben
          schreibe_hashliste (setze hashliste (cdhash, kzname))
          return kzname
      Nothing ->
          -- CD-Name nicht auf der Kommandozeile angegeben
          case lookup cdhash hashliste of
              (Just name) -> do
                  putStrLn ("Name der CD: " ++ quote name)
                  return name
              Nothing -> do
                  failIO "CD-Name nicht bekannt. Mit --name angeben."


-- einfache Hashfunktion fr Strings
hash :: String -> Integer
hash str =
   hash' str 0 0
   where
      hash' "" h _ = h
      hash' (x:xs) h sh =
          let s   = sh + 7
              sh' = if s > 100 then s-100 else s
          in hash' xs ((toInteger (ord x) `shift` sh) `xor` h) sh'

-- Wert in Namensliste setzen
setze :: (Eq a, Eq b) => [(a,b)] -> (a,b) -> [(a,b)]
setze ((a,b):r) (a',b') =
   if a == a' || b == b' then (a',b') : r
                         else (a,b) : setze r (a',b')
setze [] (a,b) = [(a,b)]

-- Hash ber Hauptverzeichnis der CD
cd_hash :: String       -- Mountpunkt
        -> IO Integer   -- Hashwert
cd_hash mountp = do
   c <- getDirectoryContents mountp
   let alles = concat (intersperse "\0" c)
   return (hash alles)

-- Hashliste aus der Hashlistendatei lesen
lies_hashliste :: IO [(Integer, String)]
lies_hashliste = do
    txt <- readFile hashliste
    case reads txt of
       [(l, "")] -> return l
       ergebnis  -> failIO $ "Parsfehler bei der Hashliste " ++ hashliste ++ "\nergebnis=" ++ quote (show ergebnis)

-- Hashliste in die Hashlistendatei schreiben
schreibe_hashliste :: [(Integer, String)] -> IO ()
schreibe_hashliste hl =
    writeFile hashliste (show hl)





-- Behlter in den Statisch-Verzeichnissen suchen.
behaelter_suchen :: String                 -- Serie
                 -> [String]               -- Behlterverzeichnis ("statisch"-Verzeichnis)
                 -> Bool                   -- ob neue Serie, falls es sie noch nicht gibt
                 -> IO ( String               -- Behlterverzeichnis der Serie
                       , [(String, Int)]      -- Behlterliste von (Pfad,Nummer) dort
                       )
behaelter_suchen serie verzl neu = do
   -- Behlter suchen, d.h. Verzeichnisinhalte parsen
   -- bll: Liste von (Behlterverzeichnis, Behlterliste von (Pfad,Nummer))
   -- bll': nur mit den Behlterverzeichnissen, die Behlter enthalten
   bll <- fmap (zip verzl) $ mapM (parse_verzeichnis serie) verzl
   let bll' = filter (\(_,l) -> l /= []) bll
   when (bll' == [] && not neu) $
      failIO ("Eine Serie \"" ++ serie ++ "\" gibt es nicht. GgF. --neu verwenden")

   -- verz: Behlterverzeichnis der Serie
   -- bl:   Behlterliste von (Pfad,Nummer) dort
   (verz, bl) <- case (bll',verzl) of
      ([], [verz]) ->
          return (verz, [])
      ([(verz,bl)], _) ->
          -- Die Serie wurde in genau einem Behlterverzeichnis gefunden.
          return (verz, bl)
      (_, _) ->
          failIO ("Die Serie " ++ quote serie ++ " gibt es in mehreren Verzeichnissen.")

   return (verz, bl)



-- Ein Verzeichnis Behltern (offen oder geschlossen) durchsuchen.
parse_verzeichnis :: String             -- Serie
                  -> String             -- Verzeichnis
                  -> IO [ ( String      -- voller Pfad
                          , Int         -- Behlternummer
                          )
                        ]
parse_verzeichnis serie verz =
    do alle_beh <- getDirectoryContents verz
       let nummern = map fromJust (filter isJust (map (nr_parser serie) alle_beh))
           pfade   = map (behpfad verz serie) nummern
           erg     = sortBy (\(_,n1) (_,n2) -> if n1 == n2 then EQ else if n1 < n2 then LT else GT) 
                            (zip pfade nummern)
       return erg



---------------------------------------------------------------------------
-- Parser fr einen Behltername

-- Verzeichnisname (ohne Pfad) parsen. Ergebnis (Just Behlternummer) oder Nothing.
nr_parser serie str =
   either (const Nothing) Just
          (parse (ablage_nr serie) "Verzeichnisname" str)

ablage_nr :: String -> Parser Int
ablage_nr serie = do
   string serie
   nr <- option 0 (do char ' '
                      nr <- number
                      return nr)
   eof
   return nr

number  :: Parser Int
number  = do sgn <- ( (char '-' >> return (-1))
                      <|> return 1
                    )
             ds <- many1 digit
             return (sgn * read ds)
        <?> "number"
---------------------------------------------------------------------------



-- voller Pfad zu einem Behlter
behpfad behverz serie nr = behverz ++ "/" ++ behname serie nr

-- Name des Behlters
behname serie nr = if nr == 0 then serie else serie ++ " " ++ show nr



---------------------------------------------------------------------------


-- Prfen, ob ein Behlter offen ist. Prft u+x Bit.
beh_offen pfad = do
   fs <- getFileStatus pfad
   let fm = fileMode fs
   if (intersectFileModes fm ownerWriteMode == nullFileMode) then return False
                                                             else return True

-- Prfen, ob ein Behlter semioffen ist. Prft u+s Bit.
beh_semioffen pfad = do
   fs <- getFileStatus pfad
   let fm = fileMode fs
   if (intersectFileModes fm setUserIDMode == nullFileMode) then return False
                                                            else return True

-- Behlter ffnen. Macht ihn schreibbar.
beh_semioeffnen echt echo verz = do
   run_ee echt echo "chmod" ["-R","u=rwX,go=rX",verz]
   run_ee echt echo "chattr" ["-R","+d",verz]


---------------------------------------------------------------------------

-- Behlterverzeichnis aus Pfad in einen Behlter.
behverz_aus_abspfad abspfad = 
    let norm = normalise_path abspfad
    in if isPrefixOf "/usr/local/share/statisch/" abspfad 
         then ( "/usr/local/share/statisch"
              , head (slice_path (drop 25 abspfad))
              )
         else if isPrefixOf "/home/v/lib/statisch/" abspfad 
            then 
              ( "/home/v/lib/statisch"
              , head (slice_path (drop 20 abspfad))
              )
            else ("", "")
