//
//  HookOperation.m
//  iHook
//
//  Created by Andrew Mortensen on 7/14/08.
//  Copyright 2008 University of Michigan, The. All rights reserved.
//

#import "HookOperation.h"
#import "LHController.h"
#import "AMAuthorization.h"
#import "NSArray(CreateArgv).h"

#include <sys/param.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

extern char		**environ;
extern pid_t		childpgid;
extern int		input_queue_count;

@implementation HookOperation

@synthesize	    controller;
@synthesize	    executable;
@synthesize	    arguments;
@synthesize	    inputQueue;

- ( id )initWithController: ( id )aController
	    executable: ( NSString * )anExecutable
	    arguments: ( NSArray * )args
	    inputQueue: ( NSMutableArray * )queue
{
    self = [ super init ];
    if ( self ) {
	hookTask = HOOK_OPERATION_EXECUTE;
	self.controller = aController;
	self.executable = anExecutable;
	self.arguments = args;
	self.inputQueue = queue;
    }
    
    return( self );
}

- ( id )initAuthorizedCancellationOfTaskWithPID: ( pid_t )pid
	    controller: ( id )aController
{
    self = [ super init ];
    if ( self ) {
	hookTask = HOOK_OPERATION_CANCEL;
	hookPid = pid;
	self.controller = aController;
	self.executable = nil;
	self.arguments = nil;
	self.inputQueue = nil;
    }
    
    return( self );
}

- ( void )executeHook
{
    AMAuthorization		*auth;
    AuthorizationExternalForm	externalForm;
    NSArray			*av;
    NSAutoreleasePool		*pool = nil;
    NSString			*error = nil;
    NSString			*authtool = nil;
    NSString			*input = nil;
    NSLock			*lock = nil;
    int				ofd[ 2 ], efd[ 2 ], ifd[ 2 ];
    int				i, ac = 0, rr = 0, status = 0;
    int				count = 1;
    FILE			*of;
    pid_t			pid;
    char			**execargs = NULL;
    char			line[ MAXPATHLEN ];
    fd_set			readmask;

    av = [ NSArray array ];
    
    if ( [[ self.executable pathExtension ] isEqualToString: @"shook" ] ) {
	auth = [ AMAuthorization authorizationWithName: @"edu.umich.ihook.exec"
					  preAuthorize: YES
				      allowInteraction: YES
					  extendRights: YES ];
	if ( auth == nil ) {
	    [ self.controller performSelectorOnMainThread: @selector( error: )
		    withObject: @"Authorization initialization failed."
		    waitUntilDone: YES ];
            return;
        }
	if ( ![ auth authorize ] ) {
	    [ self.controller performSelectorOnMainThread: @selector( error: )
		    withObject: @"Authorization failed."
		    waitUntilDone: YES ];
            return;
	}
	[ auth makeExternalForm: &externalForm ];
        
	if (( authtool = [[ NSBundle mainBundle ] pathForResource: @"shookexec"
							   ofType: nil ] ) == nil ) {
	    [ self.controller performSelectorOnMainThread: @selector( error: )
		    withObject: @"Couldn't find auth tool."
		    waitUntilDone: YES ];
            return;
        }
	
        av = [ av arrayByAddingObject: authtool ];
    }
    
    av = [ av arrayByAddingObject: self.executable ];
    av = [ av arrayByAddingObjectsFromArray: self.arguments ];

    ifd[ 0 ] = ifd[ 1 ] = ofd[ 0 ] = ofd[ 1 ] = efd[ 0 ] = efd[ 1 ] = -1;
    if ( pipe( ofd ) < 0 || pipe( efd ) < 0 || pipe( ifd ) < 0 ) {
	[ self.controller performSelectorOnMainThread: @selector( error: )
		withObject: [ NSString stringWithFormat: @"pipe failed: %s",
		strerror( errno ) ]
		waitUntilDone: YES ];
        return;
    }
    
    if (( ac = [ av createArgv: &execargs ] ) == 0 ) {
	[ self.controller performSelectorOnMainThread: @selector( error: )
		    withObject: @"Authorization initialization failed."
		    waitUntilDone: YES ];
	return;
    }
    
    switch (( childpgid = fork())) {
	case 0:
	    if ( close( ofd[ 0 ] ) < 0 || close( efd[ 0 ] ) < 0 ||
		close( ifd[ 1 ] ) < 0 ) {
		fprintf( stderr, "child: close failed: %s\n", strerror( errno ));
		fflush( stderr );
		exit( 2 );
	    }
	    if ( dup2( ifd[ 0 ], 0 ) < 0 || dup2( ofd[ 1 ], 1 ) < 0 ||
		dup2( efd[ 1 ], 2 ) < 0 ) {
		fprintf( stderr, "child: dup2 failed: %s\n", strerror( errno ));
		fflush( stderr );
		exit( 2 );
	    }
	    if ( close( ofd[ 1 ] ) < 0 || close( efd[ 1 ] ) < 0 ||
		close( ifd[ 0 ] ) < 0 ) {
		fprintf( stderr, "child: close failed: %s\n", strerror( errno ));
		fflush( stderr );
		exit( 2 );
	    }
	    
	    if ( setpgid( getpid(), getpid()) < 0 ) {
		fprintf( stderr, "child: setpgid failed: %s\n", strerror( errno ));
		fflush( stderr );
		exit( 2 );
	    }
	    
	    execve( execargs[ 0 ], execargs, ( char ** )environ );
	    fprintf( stderr, "execve %s: %s\n", execargs[ 0 ], strerror( errno ));
	    fflush( stderr );
	    _exit( 2 );
	    
	case -1:
	    error = [ NSString stringWithFormat: @"fork failed: %s",
		     strerror( errno ) ];
	    goto execute_script_cleanup;
	    
	default:
	    break;
    }
    
    if ( execargs != NULL ) {
	for ( i = 0; i < ac; i++ ) {
	    free( execargs[ i ] );
	}
	free( execargs );
	execargs = NULL;
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    if ( close( ofd[ 1 ] ) < 0 || close( efd[ 1 ] ) < 0 ||
	    close( ifd[ 0 ] ) < 0 ) {
	error = [ NSString stringWithFormat: @"close failed: %s",
		 strerror( errno ) ];
	goto execute_script_cleanup;
    }
    ifd[ 0 ] = ofd[ 1 ] = efd[ 1 ] = -1;
    
    if ( fcntl( ifd[ 1 ], F_SETFL, O_NONBLOCK ) < 0 ||
	    fcntl( ofd[ 0 ], F_SETFL, O_NONBLOCK ) < 0 ||
	    fcntl( efd[ 0 ], F_SETFL, O_NONBLOCK ) < 0 ) {
        error = [ NSString stringWithFormat: @"fcntl: %s", strerror( errno ) ];
	goto execute_script_cleanup;
    }
    
    /* if we're running a .shook, write the AuthRef to stdin */
    if ( [[ self.executable pathExtension ] isEqualToString: @"shook" ] ) {
        if ( write( ifd[ 1 ], &externalForm, sizeof( AuthorizationExternalForm))
		!= sizeof( AuthorizationExternalForm )) {
	    error = [ NSString stringWithFormat:
		     @"write: %s", strerror( errno ) ];
	    goto execute_script_cleanup;
        }
    }
    
    [ self.controller performSelectorOnMainThread: @selector( setIsExecuting: )
		    withObject: [ NSNumber numberWithBool: YES ]
		    waitUntilDone: YES ];
    
    if (( of = fdopen( ofd[ 0 ], "r" )) == NULL ) {
        error = [ NSString stringWithFormat: @"fdopen: %s", strerror( errno )];
	goto execute_script_cleanup;
    }
    setvbuf( of, NULL, _IONBF, 0 );
    
    pool = [[ NSAutoreleasePool alloc ] init ];
    
    for ( ;; ) {
        struct timeval		tv;
        tv.tv_sec = 1;
        tv.tv_usec = 0;
        
	FD_ZERO( &readmask );
        FD_SET( efd[ 0 ], &readmask );
        FD_SET( ofd[ 0 ], &readmask );
	
	switch ( select( MAX( ofd[ 0 ], efd[ 0 ] ) + 1,
			&readmask, NULL, NULL, &tv )) {
	case -1:
	    error = [ NSString stringWithFormat:
		     @"select failed: %s", strerror( errno ) ];
	    goto execute_script_cleanup;

	case 0:
	    if ( input_queue_count ) {
		break;
	    }
	    continue;

	default:
	    break;
	}
	
        if ( FD_ISSET( ofd[ 0 ], &readmask )) {
            if ( fgets( line, MAXPATHLEN, of ) != NULL ) {
		[ self.controller
		    performSelectorOnMainThread: @selector( setStdoutMessage: )
		    withObject: [ NSString stringWithUTF8String: line ]
		    waitUntilDone: YES ];
            }
        }
        if ( FD_ISSET( efd[ 0 ], &readmask )) {
            /* in this case, we just want raw reads: much more data here */
	    
            if (( rr = read( efd[ 0 ], ( char * )line, MAXPATHLEN )) <= 0 ) {
		break;
	    }
            line[ rr ] = '\0';
	    [ self.controller
		    performSelectorOnMainThread: @selector( addToStderrField: )
		    withObject: [ NSString stringWithUTF8String: line ]
		    waitUntilDone: YES ];
        }
	
	if ( input_queue_count ) {
	    if ( lock == nil ) {
		lock = [[ NSLock alloc ] init ];
	    }
	    
	    if ( [ lock tryLock ] ) {
		input = [[[ self.inputQueue lastObject ] copy ] autorelease ];
		[ self.inputQueue removeLastObject ];
		input_queue_count--;
		[ lock unlock ];

		if ( write( ifd[ 1 ], [ input UTF8String ],
			[ input length ] ) != [ input length ] ) {
		    error = [ NSString stringWithFormat:
			     @"write: %s", strerror( errno ) ];
		    goto execute_script_cleanup;
		}
	    }
	}

	if ( count % 1000 ) {
	    [ pool drain ];
	    pool = nil;
	    pool = [[ NSAutoreleasePool alloc ] init ];
	}
	count++;
    }
    if ( rr < 0 ) {
        error = [ NSString stringWithFormat: @"read: %s", strerror( errno ) ];
	goto execute_script_cleanup;
    }
    
    pid = wait( &status );
    
    NSLog( @"pid %d exited with status %d", pid, WEXITSTATUS( status ));
    
execute_script_cleanup:
    if ( pool != nil ) {
	[ pool drain ];
    }
    if ( lock != nil ) {
	[ lock release ];
    }
    
    if ( execargs != NULL ) {
	for ( i = 0; i < ac; i++ ) {
	    free( execargs[ i ] );
	}
	free( execargs );
    }
    
    for ( i = 0; i < 2; i++ ) {
	if ( ifd[ i ] != -1 && close( ifd[ i ] ) < 0 ) {
	    error = [ NSString stringWithFormat:
		     @"close: %s", strerror( errno ) ];
	    break;
	}
    }
    for ( i = 0; i < 2; i++ ) {
	if ( ofd[ i ] != -1 && close( ofd[ i ] ) < 0 ) {
	    error = [ NSString stringWithFormat:
		     @"close: %s", strerror( errno ) ];
	    break;
	}
    }
    for ( i = 0; i < 2; i++ ) {
	if ( efd[ i ] != -1 && close( efd[ i ] ) < 0 ) {
	    error = [ NSString stringWithFormat:
		     @"close: %s", strerror( errno ) ];
	    break;
	}
    }
    
    [ self.controller
		performSelectorOnMainThread: @selector( setExitStatus: )
		withObject: [ NSNumber numberWithInt: WEXITSTATUS( status ) ]
		waitUntilDone: YES ];
		
    if ( error || WEXITSTATUS( status ) != 0 ) {
	[ self.controller
		performSelectorOnMainThread: @selector( error: )
		withObject: ( error ? error : @"Abnormal exit" )
		waitUntilDone: YES ];
    } else {
	[ self.controller
		performSelectorOnMainThread: @selector( normalQuit: )
		withObject: nil
		waitUntilDone: NO ];
    }
}

- ( void )cancelHook
{
    AMAuthorization		*auth;
    NSArray			*args;
    NSString			*authtool;
    NSString			*error = nil;
    AuthorizationExternalForm	externalForm;
    FILE			*ef = NULL;
    char			buf[ MAXPATHLEN ];
    char			**execargs = NULL;
    int				i, ac, rc, status;
    int				ifd[ 2 ], efd[ 2 ];
    
    ac = rc = status = 0;
    ifd[ 0 ] = ifd[ 1 ] = efd[ 0 ] = efd[ 1 ] = -1;
    
    auth = [ AMAuthorization authorizationWithName: @"edu.umich.ihook.kill"
				      preAuthorize: YES
				  allowInteraction: YES
				      extendRights: YES ];
    if ( auth == nil ) {
	error = @"Authorization initialization failed";
	goto cancel_task_cleanup;
    }
    if ( ![ auth authorize ] ) {
	error = @"Authorization failed";
	goto cancel_task_cleanup;
    }
    [ auth makeExternalForm: &externalForm ];
    
    if (( authtool = [[ NSBundle mainBundle ] pathForResource: @"shookexec"
						       ofType: nil ]) == nil ) {
        error = [ NSString stringWithString:
		  @"Could not locate authorized tool" ];
        goto cancel_task_cleanup;
    }
    
    args = [ NSArray arrayWithObjects: authtool, @"--kill",
			[ NSNumber numberWithInt: hookPid ], nil ];
    
    if (( ac = [ args createArgv: &execargs ] ) <= 0 ) {
	error = [ NSString stringWithString: @"Failed to create argv" ];
	goto cancel_task_cleanup;
    }
    
    if ( pipe( ifd ) < 0 || pipe( efd ) < 0 ) {
        error = [ NSString stringWithFormat:
		  @"pipe failed: %s", strerror( errno ) ];
        goto cancel_task_cleanup;
    }
    
    switch ( fork()) {
    case 0:
	if ( close( ifd[ 1 ] ) < 0 || close( efd[ 0 ] ) < 0 ) {
	    fprintf( stderr, "child: close: %s\n", strerror( errno ));
	    fflush( stderr );
	    exit( 2 );
	}
	if ( dup2( ifd[ 0 ], 0 ) < 0 || dup2( efd[ 1 ], 2 ) < 0 ) {
	    fprintf( stderr, "child: dup2: %s\n", strerror( errno ));
	    fflush( stderr );
	    exit( 2 );
	}
	if ( close( ifd[ 0 ] ) < 0 || close( efd[ 1 ] ) < 0 ) {
	    fprintf( stderr, "child: close: %s\n", strerror( errno ));
	    fflush( stderr );
	    exit( 2 );
	}
	execve( execargs[ 0 ], execargs, NULL );
	fprintf( stderr, "child: execve: %s\n", strerror( errno ));
	fflush( stderr );
	_exit( 2 );    
	    
    case -1:
	error = [ NSString stringWithFormat: @"fork: %s", strerror( errno ) ];
	goto cancel_task_cleanup;
	    
    default:
	break;
    }
    
    signal( SIGPIPE, SIG_IGN );
    
    /* write external auth form to child's stdin */
    if ( write( ifd[ 1 ], &externalForm, sizeof( AuthorizationExternalForm ))
	    != sizeof( AuthorizationExternalForm )) {
        error = [ NSString stringWithFormat: @"write: %s", strerror( errno ) ];
        goto cancel_task_cleanup;
    }
    
    if ( close( ifd[ 1 ] ) < 0 || close( ifd[ 0 ] ) < 0 ||
	    close( efd[ 1 ] ) < 0 ) {
	error = [ NSString stringWithFormat: @"close: %s", strerror( errno ) ];
	goto cancel_task_cleanup;
    }
    ifd[ 1 ] = ifd[ 0 ] = efd[ 1 ] = -1;
    
    if (( ef = fdopen( efd[ 0 ], "r" )) == NULL ) {
	error = [ NSString stringWithFormat: @"fdopen: %s", strerror( errno )];
	goto cancel_task_cleanup;
    }
    
    /* just get first line of error output. */
    if ( fgets( buf, sizeof( buf ), ef ) != NULL ) {
	error = [ NSString stringWithUTF8String: buf ];
    }
    if ( ferror( ef )) {
	error = [ NSString stringWithFormat: @"fgets: %s", strerror( errno ) ];
    }
    
    wait( &status );
    rc = WEXITSTATUS( status );
    
cancel_task_cleanup:
    if ( execargs != NULL ) {
	for ( i = 0; i < ac; i++ ) {
	    free( execargs[ i ] );
	}
	free( execargs );
    }
    
    for ( i = 0; i < 2; i++ ) {
	if ( ifd[ i ] != -1 && close( ifd[ i ] ) != 0 ) {
	    error = [ NSString stringWithFormat:
		      @"close: %s", strerror( errno ) ];
	}
    }
    if ( efd[ 1 ] != -1 && close( efd[ 1 ] ) != 0 ) {
	error = [ NSString stringWithFormat: @"close: %s", strerror( errno ) ];
    }
    if ( ef != NULL && fclose( ef ) != 0 ) {
	error = [ NSString stringWithFormat: @"fclose: %s", strerror( errno )];
    } else if ( close( efd[ 0 ] ) != 0 ) {
	error = [ NSString stringWithFormat: @"close: %s", strerror( errno ) ];
    }
    
    if ( error ) {
	rc = 2;
    }
    
    [ self.controller performSelectorOnMainThread: @selector( setExitStatus: )
		withObject: [ NSNumber numberWithInt: rc ]
		waitUntilDone: YES ];
    if ( rc != 0 ) {
	if ( error ) {
	    [ self.controller
		    performSelectorOnMainThread: @selector( addToStderrField: )
		    withObject: error waitUntilDone: YES ];
	}
    }
}

- ( void )main
{
    if ( hookTask == HOOK_OPERATION_EXECUTE ) {
	[ self executeHook ];
    } else if ( hookTask == HOOK_OPERATION_CANCEL ) {
	[ self cancelHook ];
    }
}

- ( void )dealloc
{
    self.controller = nil;
    self.executable = nil;
    self.arguments = nil;
    self.inputQueue = nil;
    
    [ super dealloc ];
}

@end
