// This header is from java.lang.Character.java in Apache Harmony
//
// Licensed to the ASF
// Apache License
//
#ifndef IV_CHARACTER_H_
#define IV_CHARACTER_H_
#include <cassert>
#include <algorithm>
#include <iv/detail/cstdint.h>
#include <iv/detail/cinttypes.h>
#include <iv/detail/array.h>
#include <iv/utils.h>
namespace iv {
namespace core {
namespace character {
namespace code {

static const char16_t CR = 0x000D;
static const char16_t LF = 0x000A;
static const char16_t ZWNJ = 0x200C;
static const char16_t ZWJ = 0x200D;
static const char16_t BOM = 0xFEFF;
static const char16_t LINE_SEPARATOR = 0x2028;
static const char16_t PARAGRAPH_SEPARATOR = 0x2029;
static const uint64_t REMOVE = UINT64_C(0xFFFFFFFFFFFFFFFF);
static const int DEFAULT = -1;

}  // namespace code

namespace locale {

enum Locale {
  EN,
  LT,
  TR,
  AZ
};

}  // namespace locale

static const char16_t kMinValue = 0;
static const char16_t kMaxValue = 0xFFFF;

static const int kMinRadix = 2;
static const int kMaxRadix = 36;

enum Category {
  UNASSIGNED = 1,                  // Cn
  UPPERCASE_LETTER = 2,            // Lu
  LOWERCASE_LETTER = 3,            // Ll
  TITLECASE_LETTER = 4,            // Lt
  CASED_LETTER = 5,                // LC
  MODIFIER_LETTER = 6,             // Lm
  OTHER_LETTER = 7,                // Lo
  NON_SPACING_MARK = 8,            // Mn
  ENCLOSING_MARK = 9,              // Me
  COMBINING_SPACING_MARK = 10,     // Mc
  DECIMAL_DIGIT_NUMBER = 11,       // Nd
  LETTER_NUMBER = 12,              // Nl
  OTHER_NUMBER = 13,               // No
  SPACE_SEPARATOR = 14,            // Zs
  LINE_SEPARATOR = 15,             // Zl
  PARAGRAPH_SEPARATOR = 16,        // Zp
  CONTROL = 17,                    // Cc
  FORMAT = 18,                     // Cf
  PRIVATE_USE = 19,                // Co
  SURROGATE = 20,                  // Cs
  DASH_PUNCTUATION = 21,           // Pd
  START_PUNCTUATION = 22,          // Ps
  END_PUNCTUATION = 23,            // Pe
  CONNECTOR_PUNCTUATION = 24,      // Pc
  OTHER_PUNCTUATION = 25,          // Po
  MATH_SYMBOL = 26,                // Sm
  CURRENCY_SYMBOL = 27,            // Sc
  MODIFIER_SYMBOL = 28,            // Sk
  OTHER_SYMBOL = 29,               // So
  INITIAL_QUOTE_PUNCTUATION = 30,  // Pi
  FINAL_QUOTE_PUNCTUATION = 31     // Pf
};

enum CategoryFlag {
  Cn = 1 << UNASSIGNED,
  Lu = 1 << UPPERCASE_LETTER,
  Ll = 1 << LOWERCASE_LETTER,
  Lt = 1 << TITLECASE_LETTER,
  LC = 1 << CASED_LETTER,
  Lm = 1 << MODIFIER_LETTER,
  Lo = 1 << OTHER_LETTER,
  Mn = 1 << NON_SPACING_MARK,
  Me = 1 << ENCLOSING_MARK,
  Mc = 1 << COMBINING_SPACING_MARK,
  Nd = 1 << DECIMAL_DIGIT_NUMBER,
  Nl = 1 << LETTER_NUMBER,
  No = 1 << OTHER_NUMBER,
  Zs = 1 << SPACE_SEPARATOR,
  Zl = 1 << LINE_SEPARATOR,
  Zp = 1 << PARAGRAPH_SEPARATOR,
  Cc = 1 << CONTROL,
  Cf = 1 << FORMAT,
  Co = 1 << PRIVATE_USE,
  Cs = 1 << SURROGATE,
  Pd = 1 << DASH_PUNCTUATION,
  Ps = 1 << START_PUNCTUATION,
  Pe = 1 << END_PUNCTUATION,
  Pc = 1 << CONNECTOR_PUNCTUATION,
  Po = 1 << OTHER_PUNCTUATION,
  Sm = 1 << MATH_SYMBOL,
  Sc = 1 << CURRENCY_SYMBOL,
  Sk = 1 << MODIFIER_SYMBOL,
  So = 1 << OTHER_SYMBOL,
  Pi = 1 << INITIAL_QUOTE_PUNCTUATION,
  Pf = 1 << FINAL_QUOTE_PUNCTUATION
};

// unicode bidirectional constant
static const int kDirectionalityUndefined = -1;
static const int kDirectionalityLeftToRight = 0;                // L
static const int kDirectionalityRightToLeft = 1;                // R
static const int kDirectionalityRightToLeftArabic = 2;          // AL
static const int kDirectionalityEnuropeanNumber = 3;            // EN
static const int kDirectionalityEnuropeanNumberSeparator = 4;   // ES
static const int kDirectionalityEnuropeanNumberTerminator = 5;  // ET
static const int kDirectionalityEnuropeanArabicNumber = 6;      // AN
static const int kDirectionalityCommonNumberSeparator = 7;      // CS
static const int kDirectionalityNonspacingMark = 8;             // NSM
static const int kDirectionalityBoundaryNeutral = 9;            // BN
static const int kDirectionalityParagraphSeparator = 10;        // B
static const int kDirectionalitySegmentSeparator = 11;          // S
static const int kDirectionalityWhiteSpace = 12;                // WS
static const int kDirectionalityOtherNeutrals = 13;             // ON
static const int kDirectionalityLeftToRightEmbedding = 14;      // LRE
static const int kDirectionalityLeftToRightOverride = 15;       // LRO
static const int kDirectionalityRightToLeftEmbedding = 16;      // RLE
static const int kDirectionalityRightToLeftOverride = 17;       // RLO
static const int kDirectionalityPopDirectionalFormat = 18;      // PDF

static const int kSize = 16;

typedef std::array<uint8_t, 1000> CategoryCache;
static const CategoryCache kCategoryCache = { {
  17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
  17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
  14, 25, 25, 25, 27, 25, 25, 25, 22, 23, 25, 26, 25, 21, 25, 25,
  11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 25, 25, 26, 26, 26, 25,
  25,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
   2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2, 22, 25, 23, 28, 24,
  28,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3, 22, 26, 23, 26, 17,
  17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
  17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17, 17,
  14, 25, 27, 27, 27, 27, 29, 25, 28, 29,  7, 30, 26, 18, 29, 28,
  29, 26, 13, 13, 28,  3, 25, 25, 28, 13,  7, 31, 13, 13, 13, 25,
   2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
   2,  2,  2,  2,  2,  2,  2, 26,  2,  2,  2,  2,  2,  2,  2,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3, 26,  3,  3,  3,  3,  3,  3,  3,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  3,  2,  3,  2,  3,  2,  3,  2,
   3,  2,  3,  2,  3,  2,  3,  2,  3,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  2,  3,  2,  3,  2,  3,  3,
   3,  2,  2,  3,  2,  3,  2,  2,  3,  2,  2,  2,  3,  3,  2,  2,
   2,  2,  3,  2,  2,  3,  2,  2,  2,  3,  3,  3,  2,  2,  3,  2,
   2,  3,  2,  3,  2,  3,  2,  2,  3,  2,  3,  3,  2,  3,  2,  2,
   3,  2,  2,  2,  3,  2,  3,  2,  2,  3,  3,  7,  2,  3,  3,  3,
   7,  7,  7,  7,  2,  4,  3,  2,  4,  3,  2,  4,  3,  2,  3,  2,
   3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   3,  2,  4,  3,  2,  3,  2,  2,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  3,  3,  3,  3,  3,  3,  2,  2,  3,  2,  2,  3,
   3,  2,  3,  2,  2,  2,  2,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  7,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
   6,  6, 28, 28, 28, 28,  6,  6,  6,  6,  6,  6,  6,  6,  6,  6,
   6,  6, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28,
   6,  6,  6,  6,  6, 28, 28, 28, 28, 28, 28, 28,  6, 28,  6, 28,
  28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28, 28,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,  8,
   2,  3,  2,  3,  6, 28,  2,  3,  1,  1,  6,  3,  3,  3, 25,  1,
   1,  1,  1,  1, 28, 28,  2, 25,  2,  2,  2,  1,  2,  1,  2,  2,
   3,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,  2,
   2,  2,  1,  2,  2,  2,  2,  2,  2,  2,  2,  2,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,
   3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  3,  2,
   3,  3,  2,  2,  2,  3,  3,  3,  2,  3,  2,  3,  2,  3,  2,  3,
   2,  3,  2,  3,  2,  3,  2,  3
} };

typedef std::array<char16_t, 1421> CategoryKeys;
static const CategoryKeys kCategoryKeys = { {
0x03E8, 0x03EF, 0x03F4, 0x03F6, 0x03F8, 0x03F9, 0x03FB, 0x03FD,
0x0430, 0x0460, 0x0482, 0x0483, 0x0488, 0x048A, 0x04C0, 0x04C2,
0x04CE, 0x04D0, 0x0528, 0x0531, 0x0557, 0x0559, 0x055A, 0x0560,
0x0561, 0x0588, 0x058A, 0x058B, 0x058F, 0x0591, 0x05BE, 0x05C0,
0x05C1, 0x05C3, 0x05C4, 0x05C6, 0x05C8, 0x05D0, 0x05EB, 0x05F0,
0x05F3, 0x05F5, 0x0600, 0x0605, 0x0606, 0x0609, 0x060B, 0x060C,
0x060E, 0x0610, 0x061B, 0x061C, 0x061E, 0x0620, 0x0640, 0x0641,
0x064B, 0x0660, 0x066A, 0x066E, 0x0670, 0x0671, 0x06D4, 0x06D6,
0x06DD, 0x06DF, 0x06E5, 0x06E7, 0x06E9, 0x06EA, 0x06EE, 0x06F0,
0x06FA, 0x06FD, 0x06FF, 0x0700, 0x070E, 0x0710, 0x0712, 0x0730,
0x074B, 0x074D, 0x07A6, 0x07B1, 0x07B2, 0x07C0, 0x07CA, 0x07EB,
0x07F4, 0x07F6, 0x07F7, 0x07FA, 0x07FB, 0x0800, 0x0816, 0x081A,
0x081B, 0x0824, 0x0825, 0x0828, 0x0829, 0x082E, 0x0830, 0x083F,
0x0840, 0x0859, 0x085C, 0x085E, 0x085F, 0x08A0, 0x08A2, 0x08AD,
0x08E4, 0x08FF, 0x0900, 0x0903, 0x0904, 0x093A, 0x093D, 0x093E,
0x0941, 0x0949, 0x094D, 0x094E, 0x0950, 0x0951, 0x0958, 0x0962,
0x0964, 0x0966, 0x0970, 0x0972, 0x0978, 0x0979, 0x0980, 0x0982,
0x0984, 0x0985, 0x098D, 0x098F, 0x0991, 0x0993, 0x09A9, 0x09AA,
0x09B1, 0x09B3, 0x09B6, 0x09BA, 0x09BC, 0x09BE, 0x09C1, 0x09C5,
0x09C7, 0x09C9, 0x09CB, 0x09CD, 0x09CF, 0x09D7, 0x09D8, 0x09DC,
0x09DE, 0x09DF, 0x09E2, 0x09E4, 0x09E6, 0x09F0, 0x09F2, 0x09F4,
0x09FA, 0x09FC, 0x0A01, 0x0A03, 0x0A05, 0x0A0B, 0x0A0F, 0x0A11,
0x0A13, 0x0A29, 0x0A2A, 0x0A31, 0x0A32, 0x0A34, 0x0A35, 0x0A37,
0x0A38, 0x0A3A, 0x0A3C, 0x0A3E, 0x0A41, 0x0A43, 0x0A47, 0x0A49,
0x0A4B, 0x0A4E, 0x0A51, 0x0A52, 0x0A59, 0x0A5D, 0x0A5F, 0x0A66,
0x0A70, 0x0A72, 0x0A75, 0x0A76, 0x0A81, 0x0A83, 0x0A85, 0x0A8E,
0x0A8F, 0x0A92, 0x0A93, 0x0AA9, 0x0AAA, 0x0AB1, 0x0AB2, 0x0AB4,
0x0AB5, 0x0ABA, 0x0ABC, 0x0ABE, 0x0AC1, 0x0AC6, 0x0AC7, 0x0AC9,
0x0ACB, 0x0ACD, 0x0ACE, 0x0AD0, 0x0AD1, 0x0AE0, 0x0AE2, 0x0AE4,
0x0AE6, 0x0AF0, 0x0AF2, 0x0B01, 0x0B02, 0x0B04, 0x0B05, 0x0B0D,
0x0B0F, 0x0B11, 0x0B13, 0x0B29, 0x0B2A, 0x0B31, 0x0B32, 0x0B34,
0x0B35, 0x0B3A, 0x0B3C, 0x0B3E, 0x0B41, 0x0B45, 0x0B47, 0x0B49,
0x0B4B, 0x0B4D, 0x0B4E, 0x0B56, 0x0B58, 0x0B5C, 0x0B5E, 0x0B5F,
0x0B62, 0x0B64, 0x0B66, 0x0B70, 0x0B72, 0x0B78, 0x0B82, 0x0B84,
0x0B85, 0x0B8B, 0x0B8E, 0x0B91, 0x0B92, 0x0B96, 0x0B99, 0x0B9B,
0x0B9E, 0x0BA0, 0x0BA3, 0x0BA5, 0x0BA8, 0x0BAB, 0x0BAE, 0x0BBA,
0x0BBE, 0x0BC0, 0x0BC1, 0x0BC3, 0x0BC6, 0x0BC9, 0x0BCA, 0x0BCD,
0x0BCE, 0x0BD0, 0x0BD1, 0x0BD7, 0x0BD8, 0x0BE6, 0x0BF0, 0x0BF3,
0x0BF9, 0x0BFB, 0x0C01, 0x0C04, 0x0C05, 0x0C0D, 0x0C0E, 0x0C11,
0x0C12, 0x0C29, 0x0C2A, 0x0C34, 0x0C35, 0x0C3A, 0x0C3D, 0x0C3E,
0x0C41, 0x0C45, 0x0C46, 0x0C49, 0x0C4A, 0x0C4E, 0x0C55, 0x0C57,
0x0C58, 0x0C5A, 0x0C60, 0x0C62, 0x0C64, 0x0C66, 0x0C70, 0x0C78,
0x0C7F, 0x0C80, 0x0C82, 0x0C84, 0x0C85, 0x0C8D, 0x0C8E, 0x0C91,
0x0C92, 0x0CA9, 0x0CAA, 0x0CB4, 0x0CB5, 0x0CBA, 0x0CBC, 0x0CBE,
0x0CC0, 0x0CC5, 0x0CC7, 0x0CC9, 0x0CCA, 0x0CCC, 0x0CCE, 0x0CD5,
0x0CD7, 0x0CDE, 0x0CE0, 0x0CE2, 0x0CE4, 0x0CE6, 0x0CF0, 0x0CF1,
0x0CF3, 0x0D02, 0x0D04, 0x0D05, 0x0D0D, 0x0D0E, 0x0D11, 0x0D12,
0x0D3B, 0x0D3D, 0x0D3E, 0x0D41, 0x0D45, 0x0D46, 0x0D49, 0x0D4A,
0x0D4D, 0x0D4F, 0x0D57, 0x0D58, 0x0D60, 0x0D62, 0x0D64, 0x0D66,
0x0D70, 0x0D76, 0x0D79, 0x0D7A, 0x0D80, 0x0D82, 0x0D84, 0x0D85,
0x0D97, 0x0D9A, 0x0DB2, 0x0DB3, 0x0DBC, 0x0DBE, 0x0DC0, 0x0DC7,
0x0DCA, 0x0DCB, 0x0DCF, 0x0DD2, 0x0DD5, 0x0DD8, 0x0DE0, 0x0DF2,
0x0DF4, 0x0DF5, 0x0E01, 0x0E31, 0x0E32, 0x0E34, 0x0E3B, 0x0E3F,
0x0E40, 0x0E46, 0x0E47, 0x0E4F, 0x0E50, 0x0E5A, 0x0E5C, 0x0E81,
0x0E83, 0x0E85, 0x0E87, 0x0E89, 0x0E8B, 0x0E8D, 0x0E8E, 0x0E94,
0x0E98, 0x0E99, 0x0EA0, 0x0EA1, 0x0EA4, 0x0EA8, 0x0EAA, 0x0EAC,
0x0EAD, 0x0EB1, 0x0EB2, 0x0EB4, 0x0EBA, 0x0EBB, 0x0EBD, 0x0EBE,
0x0EC0, 0x0EC5, 0x0EC8, 0x0ECE, 0x0ED0, 0x0EDA, 0x0EDC, 0x0EE0,
0x0F00, 0x0F01, 0x0F04, 0x0F13, 0x0F15, 0x0F18, 0x0F1A, 0x0F20,
0x0F2A, 0x0F34, 0x0F3A, 0x0F3E, 0x0F40, 0x0F48, 0x0F49, 0x0F6D,
0x0F71, 0x0F7F, 0x0F80, 0x0F85, 0x0F86, 0x0F88, 0x0F8D, 0x0F98,
0x0F99, 0x0FBD, 0x0FBE, 0x0FC6, 0x0FC7, 0x0FCD, 0x0FCE, 0x0FD0,
0x0FD5, 0x0FD9, 0x0FDB, 0x1000, 0x102B, 0x102D, 0x1031, 0x1032,
0x1038, 0x1039, 0x103B, 0x103D, 0x103F, 0x1040, 0x104A, 0x1050,
0x1056, 0x1058, 0x105A, 0x105E, 0x1061, 0x1062, 0x1065, 0x1067,
0x106E, 0x1071, 0x1075, 0x1082, 0x1083, 0x1085, 0x1087, 0x108D,
0x108F, 0x1090, 0x109A, 0x109D, 0x109E, 0x10A0, 0x10C6, 0x10C8,
0x10CD, 0x10CE, 0x10D0, 0x10FB, 0x10FD, 0x1249, 0x124A, 0x124E,
0x1250, 0x1257, 0x125A, 0x125E, 0x1260, 0x1289, 0x128A, 0x128E,
0x1290, 0x12B1, 0x12B2, 0x12B6, 0x12B8, 0x12BF, 0x12C2, 0x12C6,
0x12C8, 0x12D7, 0x12D8, 0x1311, 0x1312, 0x1316, 0x1318, 0x135B,
0x135D, 0x1360, 0x1369, 0x137D, 0x1380, 0x1390, 0x139A, 0x13A0,
0x13F5, 0x1400, 0x1401, 0x166D, 0x166F, 0x1680, 0x1681, 0x169B,
0x169D, 0x16A0, 0x16EB, 0x16EE, 0x16F1, 0x1700, 0x170D, 0x170E,
0x1712, 0x1715, 0x1720, 0x1732, 0x1735, 0x1737, 0x1740, 0x1752,
0x1754, 0x1760, 0x176D, 0x176E, 0x1771, 0x1772, 0x1774, 0x1780,
0x17B4, 0x17B6, 0x17B7, 0x17BE, 0x17C6, 0x17C7, 0x17C9, 0x17D4,
0x17D7, 0x17D8, 0x17DB, 0x17DD, 0x17DE, 0x17E0, 0x17EA, 0x17F0,
0x17FA, 0x1800, 0x1806, 0x1807, 0x180B, 0x180E, 0x1810, 0x181A,
0x1820, 0x1843, 0x1844, 0x1878, 0x1880, 0x18A9, 0x18AB, 0x18B0,
0x18F6, 0x1900, 0x191D, 0x1920, 0x1923, 0x1927, 0x1929, 0x192C,
0x1930, 0x1932, 0x1933, 0x1939, 0x193C, 0x1940, 0x1941, 0x1944,
0x1946, 0x1950, 0x196E, 0x1970, 0x1975, 0x1980, 0x19AC, 0x19B0,
0x19C1, 0x19C8, 0x19CA, 0x19D0, 0x19DA, 0x19DB, 0x19DE, 0x1A00,
0x1A17, 0x1A19, 0x1A1C, 0x1A1E, 0x1A20, 0x1A55, 0x1A58, 0x1A5F,
0x1A61, 0x1A63, 0x1A65, 0x1A6D, 0x1A73, 0x1A7D, 0x1A7F, 0x1A80,
0x1A8A, 0x1A90, 0x1A9A, 0x1AA0, 0x1AA7, 0x1AA8, 0x1AAE, 0x1B00,
0x1B04, 0x1B05, 0x1B34, 0x1B36, 0x1B3B, 0x1B3D, 0x1B42, 0x1B43,
0x1B45, 0x1B4C, 0x1B50, 0x1B5A, 0x1B61, 0x1B6B, 0x1B74, 0x1B7D,
0x1B80, 0x1B82, 0x1B83, 0x1BA1, 0x1BA2, 0x1BA6, 0x1BA8, 0x1BAA,
0x1BAC, 0x1BAE, 0x1BB0, 0x1BBA, 0x1BE6, 0x1BE8, 0x1BEA, 0x1BED,
0x1BEF, 0x1BF2, 0x1BF4, 0x1BFC, 0x1C00, 0x1C24, 0x1C2C, 0x1C34,
0x1C36, 0x1C38, 0x1C3B, 0x1C40, 0x1C4A, 0x1C4D, 0x1C50, 0x1C5A,
0x1C78, 0x1C7E, 0x1C80, 0x1CC0, 0x1CC8, 0x1CD0, 0x1CD3, 0x1CD4,
0x1CE1, 0x1CE2, 0x1CE9, 0x1CED, 0x1CEE, 0x1CF2, 0x1CF4, 0x1CF5,
0x1CF7, 0x1D00, 0x1D2C, 0x1D6B, 0x1D78, 0x1D79, 0x1D9B, 0x1DC0,
0x1DE7, 0x1DFC, 0x1E00, 0x1E95, 0x1E9E, 0x1EFF, 0x1F08, 0x1F10,
0x1F16, 0x1F18, 0x1F1E, 0x1F20, 0x1F28, 0x1F30, 0x1F38, 0x1F40,
0x1F46, 0x1F48, 0x1F4E, 0x1F50, 0x1F58, 0x1F60, 0x1F68, 0x1F70,
0x1F7E, 0x1F80, 0x1F88, 0x1F90, 0x1F98, 0x1FA0, 0x1FA8, 0x1FB0,
0x1FB5, 0x1FB6, 0x1FB8, 0x1FBC, 0x1FBE, 0x1FBF, 0x1FC2, 0x1FC5,
0x1FC6, 0x1FC8, 0x1FCC, 0x1FCD, 0x1FD0, 0x1FD4, 0x1FD6, 0x1FD8,
0x1FDC, 0x1FDD, 0x1FE0, 0x1FE8, 0x1FED, 0x1FF0, 0x1FF2, 0x1FF5,
0x1FF6, 0x1FF8, 0x1FFC, 0x1FFD, 0x1FFF, 0x2000, 0x200B, 0x2010,
0x2016, 0x2018, 0x201A, 0x201B, 0x201D, 0x201F, 0x2020, 0x2028,
0x202A, 0x202F, 0x2030, 0x2039, 0x203B, 0x203F, 0x2041, 0x2044,
0x2046, 0x2047, 0x2052, 0x2054, 0x2055, 0x205F, 0x2060, 0x2065,
0x206A, 0x2070, 0x2072, 0x2074, 0x207A, 0x207D, 0x207F, 0x2080,
0x208A, 0x208D, 0x208F, 0x2090, 0x209D, 0x20A0, 0x20BA, 0x20D0,
0x20DD, 0x20E1, 0x20E2, 0x20E5, 0x20F1, 0x2100, 0x2102, 0x2103,
0x2107, 0x2108, 0x210A, 0x210B, 0x210E, 0x2110, 0x2113, 0x2115,
0x2116, 0x2118, 0x2119, 0x211E, 0x2124, 0x212A, 0x212E, 0x2130,
0x2134, 0x2135, 0x2139, 0x213A, 0x213C, 0x213E, 0x2140, 0x2145,
0x2146, 0x214A, 0x214C, 0x214E, 0x2150, 0x2160, 0x2183, 0x2185,
0x2189, 0x218A, 0x2190, 0x2195, 0x219A, 0x219C, 0x21A0, 0x21A1,
0x21A3, 0x21A4, 0x21A6, 0x21A7, 0x21AE, 0x21AF, 0x21CE, 0x21D0,
0x21D2, 0x21D5, 0x21F4, 0x2300, 0x2308, 0x230C, 0x2320, 0x2322,
0x2329, 0x232B, 0x237C, 0x237D, 0x239B, 0x23B4, 0x23DC, 0x23E2,
0x23F4, 0x2400, 0x2427, 0x2440, 0x244B, 0x2460, 0x249C, 0x24EA,
0x2500, 0x25B7, 0x25B8, 0x25C1, 0x25C2, 0x25F8, 0x2600, 0x266F,
0x2670, 0x2700, 0x2701, 0x2768, 0x2776, 0x2794, 0x27C0, 0x27C5,
0x27C7, 0x27E6, 0x27F0, 0x2800, 0x2900, 0x2983, 0x2999, 0x29D8,
0x29DC, 0x29FC, 0x29FE, 0x2B00, 0x2B30, 0x2B45, 0x2B47, 0x2B4D,
0x2B50, 0x2B5A, 0x2C00, 0x2C2F, 0x2C30, 0x2C5F, 0x2C61, 0x2C62,
0x2C65, 0x2C67, 0x2C6D, 0x2C71, 0x2C73, 0x2C75, 0x2C76, 0x2C7C,
0x2C7E, 0x2C81, 0x2CE3, 0x2CE5, 0x2CEB, 0x2CEF, 0x2CF2, 0x2CF4,
0x2CF9, 0x2CFD, 0x2CFE, 0x2D00, 0x2D26, 0x2D28, 0x2D2D, 0x2D2E,
0x2D30, 0x2D68, 0x2D6F, 0x2D71, 0x2D7F, 0x2D80, 0x2D97, 0x2DA0,
0x2DA7, 0x2DA8, 0x2DAF, 0x2DB0, 0x2DB7, 0x2DB8, 0x2DBF, 0x2DC0,
0x2DC7, 0x2DC8, 0x2DCF, 0x2DD0, 0x2DD7, 0x2DD8, 0x2DDF, 0x2DE0,
0x2E00, 0x2E02, 0x2E06, 0x2E09, 0x2E0B, 0x2E0D, 0x2E0E, 0x2E17,
0x2E18, 0x2E1A, 0x2E1C, 0x2E1E, 0x2E20, 0x2E22, 0x2E2A, 0x2E2F,
0x2E30, 0x2E3A, 0x2E3C, 0x2E80, 0x2E9A, 0x2E9B, 0x2EF4, 0x2F00,
0x2FD6, 0x2FF0, 0x2FFC, 0x3000, 0x3001, 0x3004, 0x3006, 0x3008,
0x3012, 0x3014, 0x301C, 0x301E, 0x3020, 0x3021, 0x302A, 0x302E,
0x3030, 0x3031, 0x3036, 0x3038, 0x303B, 0x303D, 0x303E, 0x3040,
0x3041, 0x3097, 0x3099, 0x309B, 0x309D, 0x309F, 0x30A1, 0x30FB,
0x30FC, 0x30FF, 0x3100, 0x3105, 0x312E, 0x3131, 0x318F, 0x3190,
0x3192, 0x3196, 0x31A0, 0x31BB, 0x31C0, 0x31E4, 0x31F0, 0x3200,
0x321F, 0x3220, 0x322A, 0x3248, 0x3250, 0x3251, 0x3260, 0x3280,
0x328A, 0x32B1, 0x32C0, 0x32FF, 0x3300, 0x3400, 0x4DB6, 0x4DC0,
0x4E00, 0x9FCD, 0xA000, 0xA015, 0xA016, 0xA48D, 0xA490, 0xA4C7,
0xA4D0, 0xA4F8, 0xA4FE, 0xA500, 0xA60C, 0xA60D, 0xA610, 0xA620,
0xA62A, 0xA62C, 0xA640, 0xA66E, 0xA670, 0xA673, 0xA674, 0xA67E,
0xA680, 0xA698, 0xA69F, 0xA6A0, 0xA6E6, 0xA6F0, 0xA6F2, 0xA6F8,
0xA700, 0xA717, 0xA720, 0xA722, 0xA72F, 0xA732, 0xA770, 0xA771,
0xA779, 0xA77D, 0xA77F, 0xA788, 0xA789, 0xA78B, 0xA78F, 0xA791,
0xA794, 0xA7A0, 0xA7AB, 0xA7F8, 0xA7FA, 0xA7FB, 0xA802, 0xA803,
0xA806, 0xA807, 0xA80B, 0xA80C, 0xA823, 0xA825, 0xA827, 0xA828,
0xA82C, 0xA830, 0xA836, 0xA838, 0xA83A, 0xA840, 0xA874, 0xA878,
0xA880, 0xA882, 0xA8B4, 0xA8C4, 0xA8C5, 0xA8CE, 0xA8D0, 0xA8DA,
0xA8E0, 0xA8F2, 0xA8F8, 0xA8FB, 0xA8FC, 0xA900, 0xA90A, 0xA926,
0xA92E, 0xA930, 0xA947, 0xA952, 0xA954, 0xA95F, 0xA960, 0xA97D,
0xA980, 0xA983, 0xA984, 0xA9B3, 0xA9B4, 0xA9B6, 0xA9BA, 0xA9BC,
0xA9BD, 0xA9C1, 0xA9CE, 0xA9D0, 0xA9DA, 0xA9DE, 0xA9E0, 0xAA00,
0xAA29, 0xAA2F, 0xAA31, 0xAA33, 0xAA35, 0xAA37, 0xAA40, 0xAA43,
0xAA44, 0xAA4C, 0xAA4E, 0xAA50, 0xAA5A, 0xAA5C, 0xAA60, 0xAA70,
0xAA71, 0xAA77, 0xAA7A, 0xAA7C, 0xAA80, 0xAAB0, 0xAAB2, 0xAAB5,
0xAAB7, 0xAAB9, 0xAABE, 0xAAC0, 0xAAC3, 0xAADB, 0xAADD, 0xAADE,
0xAAE0, 0xAAEB, 0xAAEC, 0xAAEE, 0xAAF0, 0xAAF2, 0xAAF3, 0xAAF5,
0xAAF7, 0xAB01, 0xAB07, 0xAB09, 0xAB0F, 0xAB11, 0xAB17, 0xAB20,
0xAB27, 0xAB28, 0xAB2F, 0xABC0, 0xABE3, 0xABE5, 0xABE6, 0xABE8,
0xABE9, 0xABEB, 0xABED, 0xABEE, 0xABF0, 0xABFA, 0xAC00, 0xD7A4,
0xD7B0, 0xD7C7, 0xD7CB, 0xD7FC, 0xD800, 0xE000, 0xF900, 0xFA6E,
0xFA70, 0xFADA, 0xFB00, 0xFB07, 0xFB13, 0xFB18, 0xFB1D, 0xFB1F,
0xFB29, 0xFB2A, 0xFB37, 0xFB38, 0xFB3D, 0xFB40, 0xFB42, 0xFB43,
0xFB45, 0xFB46, 0xFBB2, 0xFBC2, 0xFBD3, 0xFD3E, 0xFD40, 0xFD50,
0xFD90, 0xFD92, 0xFDC8, 0xFDF0, 0xFDFC, 0xFDFE, 0xFE00, 0xFE10,
0xFE17, 0xFE19, 0xFE1A, 0xFE20, 0xFE27, 0xFE30, 0xFE31, 0xFE33,
0xFE35, 0xFE45, 0xFE47, 0xFE49, 0xFE4D, 0xFE50, 0xFE53, 0xFE54,
0xFE58, 0xFE5A, 0xFE5F, 0xFE62, 0xFE64, 0xFE67, 0xFE69, 0xFE6A,
0xFE6C, 0xFE70, 0xFE75, 0xFE76, 0xFEFD, 0xFEFF, 0xFF01, 0xFF04,
0xFF05, 0xFF08, 0xFF0A, 0xFF0D, 0xFF0E, 0xFF10, 0xFF1A, 0xFF1C,
0xFF1F, 0xFF21, 0xFF3B, 0xFF3D, 0xFF3F, 0xFF41, 0xFF5B, 0xFF5D,
0xFF5F, 0xFF61, 0xFF63, 0xFF64, 0xFF66, 0xFF70, 0xFF71, 0xFF9E,
0xFFA0, 0xFFBF, 0xFFC2, 0xFFC8, 0xFFCA, 0xFFD0, 0xFFD2, 0xFFD8,
0xFFDA, 0xFFDD, 0xFFE0, 0xFFE2, 0xFFE4, 0xFFE5, 0xFFE7, 0xFFE9,
0xFFED, 0xFFEF, 0xFFF9, 0xFFFC, 0xFFFE
} };

typedef std::array<char16_t, 2842> CategoryValues;
static const CategoryValues kCategoryValues = { {
0x03EE, 0x0302, 0x03F3, 0x0003, 0x03F5, 0x0302, 0x03F7, 0x021A,
0x03F8, 0x0003, 0x03FA, 0x0002, 0x03FC, 0x0003, 0x042F, 0x0002,
0x045F, 0x0003, 0x0481, 0x0302, 0x0482, 0x001D, 0x0487, 0x0008,
0x0489, 0x0009, 0x04BF, 0x0302, 0x04C1, 0x0002, 0x04CD, 0x0203,
0x04CF, 0x0003, 0x0527, 0x0302, 0x0530, 0x0001, 0x0556, 0x0002,
0x0558, 0x0001, 0x0559, 0x0006, 0x055F, 0x0019, 0x0560, 0x0001,
0x0587, 0x0003, 0x0589, 0x1901, 0x058A, 0x0015, 0x058E, 0x0001,
0x0590, 0x1B01, 0x05BD, 0x0008, 0x05BF, 0x0815, 0x05C0, 0x0019,
0x05C2, 0x0008, 0x05C3, 0x0019, 0x05C5, 0x0008, 0x05C7, 0x0819,
0x05CF, 0x0001, 0x05EA, 0x0007, 0x05EF, 0x0001, 0x05F2, 0x0007,
0x05F4, 0x0019, 0x05FF, 0x0001, 0x0604, 0x0012, 0x0605, 0x0001,
0x0608, 0x001A, 0x060A, 0x0019, 0x060B, 0x001B, 0x060D, 0x0019,
0x060F, 0x001D, 0x061A, 0x0008, 0x061B, 0x0019, 0x061D, 0x0001,
0x061F, 0x0019, 0x063F, 0x0007, 0x0640, 0x0006, 0x064A, 0x0007,
0x065F, 0x0008, 0x0669, 0x000B, 0x066D, 0x0019, 0x066F, 0x0007,
0x0670, 0x0008, 0x06D3, 0x0007, 0x06D5, 0x0719, 0x06DC, 0x0008,
0x06DE, 0x121D, 0x06E4, 0x0008, 0x06E6, 0x0006, 0x06E8, 0x0008,
0x06E9, 0x001D, 0x06ED, 0x0008, 0x06EF, 0x0007, 0x06F9, 0x000B,
0x06FC, 0x0007, 0x06FE, 0x001D, 0x06FF, 0x0007, 0x070D, 0x0019,
0x070F, 0x1201, 0x0711, 0x0807, 0x072F, 0x0007, 0x074A, 0x0008,
0x074C, 0x0001, 0x07A5, 0x0007, 0x07B0, 0x0008, 0x07B1, 0x0007,
0x07BF, 0x0001, 0x07C9, 0x000B, 0x07EA, 0x0007, 0x07F3, 0x0008,
0x07F5, 0x0006, 0x07F6, 0x001D, 0x07F9, 0x0019, 0x07FA, 0x0006,
0x07FF, 0x0001, 0x0815, 0x0007, 0x0819, 0x0008, 0x081A, 0x0006,
0x0823, 0x0008, 0x0824, 0x0006, 0x0827, 0x0008, 0x0828, 0x0006,
0x082D, 0x0008, 0x082F, 0x0001, 0x083E, 0x0019, 0x083F, 0x0001,
0x0858, 0x0007, 0x085B, 0x0008, 0x085D, 0x0001, 0x085E, 0x0019,
0x089F, 0x0001, 0x08A1, 0x0107, 0x08AC, 0x0007, 0x08E3, 0x0001,
0x08FE, 0x0008, 0x08FF, 0x0001, 0x0902, 0x0008, 0x0903, 0x000A,
0x0939, 0x0007, 0x093C, 0x0A08, 0x093D, 0x0007, 0x0940, 0x000A,
0x0948, 0x0008, 0x094C, 0x000A, 0x094D, 0x0008, 0x094F, 0x000A,
0x0950, 0x0007, 0x0957, 0x0008, 0x0961, 0x0007, 0x0963, 0x0008,
0x0965, 0x0019, 0x096F, 0x000B, 0x0971, 0x0619, 0x0977, 0x0007,
0x0978, 0x0001, 0x097F, 0x0007, 0x0981, 0x0801, 0x0983, 0x000A,
0x0984, 0x0001, 0x098C, 0x0007, 0x098E, 0x0001, 0x0990, 0x0007,
0x0992, 0x0001, 0x09A8, 0x0007, 0x09A9, 0x0001, 0x09B0, 0x0007,
0x09B2, 0x0107, 0x09B5, 0x0001, 0x09B9, 0x0007, 0x09BB, 0x0001,
0x09BD, 0x0708, 0x09C0, 0x000A, 0x09C4, 0x0008, 0x09C6, 0x0001,
0x09C8, 0x000A, 0x09CA, 0x0001, 0x09CC, 0x000A, 0x09CE, 0x0807,
0x09D6, 0x0001, 0x09D7, 0x000A, 0x09DB, 0x0001, 0x09DD, 0x0007,
0x09DE, 0x0001, 0x09E1, 0x0007, 0x09E3, 0x0008, 0x09E5, 0x0001,
0x09EF, 0x000B, 0x09F1, 0x0007, 0x09F3, 0x001B, 0x09F9, 0x000D,
0x09FB, 0x1B1D, 0x0A00, 0x0001, 0x0A02, 0x0008, 0x0A04, 0x0A01,
0x0A0A, 0x0007, 0x0A0E, 0x0001, 0x0A10, 0x0007, 0x0A12, 0x0001,
0x0A28, 0x0007, 0x0A29, 0x0001, 0x0A30, 0x0007, 0x0A31, 0x0001,
0x0A33, 0x0007, 0x0A34, 0x0001, 0x0A36, 0x0007, 0x0A37, 0x0001,
0x0A39, 0x0007, 0x0A3B, 0x0001, 0x0A3D, 0x0108, 0x0A40, 0x000A,
0x0A42, 0x0008, 0x0A46, 0x0001, 0x0A48, 0x0008, 0x0A4A, 0x0001,
0x0A4D, 0x0008, 0x0A50, 0x0001, 0x0A51, 0x0008, 0x0A58, 0x0001,
0x0A5C, 0x0007, 0x0A5E, 0x0107, 0x0A65, 0x0001, 0x0A6F, 0x000B,
0x0A71, 0x0008, 0x0A74, 0x0007, 0x0A75, 0x0008, 0x0A80, 0x0001,
0x0A82, 0x0008, 0x0A84, 0x0A01, 0x0A8D, 0x0007, 0x0A8E, 0x0001,
0x0A91, 0x0007, 0x0A92, 0x0001, 0x0AA8, 0x0007, 0x0AA9, 0x0001,
0x0AB0, 0x0007, 0x0AB1, 0x0001, 0x0AB3, 0x0007, 0x0AB4, 0x0001,
0x0AB9, 0x0007, 0x0ABB, 0x0001, 0x0ABD, 0x0708, 0x0AC0, 0x000A,
0x0AC5, 0x0008, 0x0AC6, 0x0001, 0x0AC8, 0x0008, 0x0ACA, 0x0A01,
0x0ACC, 0x000A, 0x0ACD, 0x0008, 0x0ACF, 0x0001, 0x0AD0, 0x0007,
0x0ADF, 0x0001, 0x0AE1, 0x0007, 0x0AE3, 0x0008, 0x0AE5, 0x0001,
0x0AEF, 0x000B, 0x0AF1, 0x1B19, 0x0B00, 0x0001, 0x0B01, 0x0008,
0x0B03, 0x000A, 0x0B04, 0x0001, 0x0B0C, 0x0007, 0x0B0E, 0x0001,
0x0B10, 0x0007, 0x0B12, 0x0001, 0x0B28, 0x0007, 0x0B29, 0x0001,
0x0B30, 0x0007, 0x0B31, 0x0001, 0x0B33, 0x0007, 0x0B34, 0x0001,
0x0B39, 0x0007, 0x0B3B, 0x0001, 0x0B3D, 0x0708, 0x0B40, 0x080A,
0x0B44, 0x0008, 0x0B46, 0x0001, 0x0B48, 0x000A, 0x0B4A, 0x0001,
0x0B4C, 0x000A, 0x0B4D, 0x0008, 0x0B55, 0x0001, 0x0B57, 0x0A08,
0x0B5B, 0x0001, 0x0B5D, 0x0007, 0x0B5E, 0x0001, 0x0B61, 0x0007,
0x0B63, 0x0008, 0x0B65, 0x0001, 0x0B6F, 0x000B, 0x0B71, 0x071D,
0x0B77, 0x000D, 0x0B81, 0x0001, 0x0B83, 0x0708, 0x0B84, 0x0001,
0x0B8A, 0x0007, 0x0B8D, 0x0001, 0x0B90, 0x0007, 0x0B91, 0x0001,
0x0B95, 0x0007, 0x0B98, 0x0001, 0x0B9A, 0x0007, 0x0B9D, 0x0107,
0x0B9F, 0x0007, 0x0BA2, 0x0001, 0x0BA4, 0x0007, 0x0BA7, 0x0001,
0x0BAA, 0x0007, 0x0BAD, 0x0001, 0x0BB9, 0x0007, 0x0BBD, 0x0001,
0x0BBF, 0x000A, 0x0BC0, 0x0008, 0x0BC2, 0x000A, 0x0BC5, 0x0001,
0x0BC8, 0x000A, 0x0BC9, 0x0001, 0x0BCC, 0x000A, 0x0BCD, 0x0008,
0x0BCF, 0x0001, 0x0BD0, 0x0007, 0x0BD6, 0x0001, 0x0BD7, 0x000A,
0x0BE5, 0x0001, 0x0BEF, 0x000B, 0x0BF2, 0x000D, 0x0BF8, 0x001D,
0x0BFA, 0x1B1D, 0x0C00, 0x0001, 0x0C03, 0x000A, 0x0C04, 0x0001,
0x0C0C, 0x0007, 0x0C0D, 0x0001, 0x0C10, 0x0007, 0x0C11, 0x0001,
0x0C28, 0x0007, 0x0C29, 0x0001, 0x0C33, 0x0007, 0x0C34, 0x0001,
0x0C39, 0x0007, 0x0C3C, 0x0001, 0x0C3D, 0x0007, 0x0C40, 0x0008,
0x0C44, 0x000A, 0x0C45, 0x0001, 0x0C48, 0x0008, 0x0C49, 0x0001,
0x0C4D, 0x0008, 0x0C54, 0x0001, 0x0C56, 0x0008, 0x0C57, 0x0001,
0x0C59, 0x0007, 0x0C5F, 0x0001, 0x0C61, 0x0007, 0x0C63, 0x0008,
0x0C65, 0x0001, 0x0C6F, 0x000B, 0x0C77, 0x0001, 0x0C7E, 0x000D,
0x0C7F, 0x001D, 0x0C81, 0x0001, 0x0C83, 0x000A, 0x0C84, 0x0001,
0x0C8C, 0x0007, 0x0C8D, 0x0001, 0x0C90, 0x0007, 0x0C91, 0x0001,
0x0CA8, 0x0007, 0x0CA9, 0x0001, 0x0CB3, 0x0007, 0x0CB4, 0x0001,
0x0CB9, 0x0007, 0x0CBB, 0x0001, 0x0CBD, 0x0708, 0x0CBF, 0x080A,
0x0CC4, 0x000A, 0x0CC6, 0x0108, 0x0CC8, 0x000A, 0x0CC9, 0x0001,
0x0CCB, 0x000A, 0x0CCD, 0x0008, 0x0CD4, 0x0001, 0x0CD6, 0x000A,
0x0CDD, 0x0001, 0x0CDF, 0x0107, 0x0CE1, 0x0007, 0x0CE3, 0x0008,
0x0CE5, 0x0001, 0x0CEF, 0x000B, 0x0CF0, 0x0001, 0x0CF2, 0x0007,
0x0D01, 0x0001, 0x0D03, 0x000A, 0x0D04, 0x0001, 0x0D0C, 0x0007,
0x0D0D, 0x0001, 0x0D10, 0x0007, 0x0D11, 0x0001, 0x0D3A, 0x0007,
0x0D3C, 0x0001, 0x0D3D, 0x0007, 0x0D40, 0x000A, 0x0D44, 0x0008,
0x0D45, 0x0001, 0x0D48, 0x000A, 0x0D49, 0x0001, 0x0D4C, 0x000A,
0x0D4E, 0x0807, 0x0D56, 0x0001, 0x0D57, 0x000A, 0x0D5F, 0x0001,
0x0D61, 0x0007, 0x0D63, 0x0008, 0x0D65, 0x0001, 0x0D6F, 0x000B,
0x0D75, 0x000D, 0x0D78, 0x0001, 0x0D79, 0x001D, 0x0D7F, 0x0007,
0x0D81, 0x0001, 0x0D83, 0x000A, 0x0D84, 0x0001, 0x0D96, 0x0007,
0x0D99, 0x0001, 0x0DB1, 0x0007, 0x0DB2, 0x0001, 0x0DBB, 0x0007,
0x0DBD, 0x0701, 0x0DBF, 0x0001, 0x0DC6, 0x0007, 0x0DC9, 0x0001,
0x0DCA, 0x0008, 0x0DCE, 0x0001, 0x0DD1, 0x000A, 0x0DD4, 0x0008,
0x0DD7, 0x0108, 0x0DDF, 0x000A, 0x0DF1, 0x0001, 0x0DF3, 0x000A,
0x0DF4, 0x0019, 0x0E00, 0x0001, 0x0E30, 0x0007, 0x0E31, 0x0008,
0x0E33, 0x0007, 0x0E3A, 0x0008, 0x0E3E, 0x0001, 0x0E3F, 0x001B,
0x0E45, 0x0007, 0x0E46, 0x0006, 0x0E4E, 0x0008, 0x0E4F, 0x0019,
0x0E59, 0x000B, 0x0E5B, 0x0019, 0x0E80, 0x0001, 0x0E82, 0x0007,
0x0E84, 0x0107, 0x0E86, 0x0001, 0x0E88, 0x0007, 0x0E8A, 0x0107,
0x0E8C, 0x0001, 0x0E8D, 0x0007, 0x0E93, 0x0001, 0x0E97, 0x0007,
0x0E98, 0x0001, 0x0E9F, 0x0007, 0x0EA0, 0x0001, 0x0EA3, 0x0007,
0x0EA7, 0x0701, 0x0EA9, 0x0001, 0x0EAB, 0x0007, 0x0EAC, 0x0001,
0x0EB0, 0x0007, 0x0EB1, 0x0008, 0x0EB3, 0x0007, 0x0EB9, 0x0008,
0x0EBA, 0x0001, 0x0EBC, 0x0008, 0x0EBD, 0x0007, 0x0EBF, 0x0001,
0x0EC4, 0x0007, 0x0EC7, 0x0106, 0x0ECD, 0x0008, 0x0ECF, 0x0001,
0x0ED9, 0x000B, 0x0EDB, 0x0001, 0x0EDF, 0x0007, 0x0EFF, 0x0001,
0x0F00, 0x0007, 0x0F03, 0x001D, 0x0F12, 0x0019, 0x0F14, 0x1D19,
0x0F17, 0x001D, 0x0F19, 0x0008, 0x0F1F, 0x001D, 0x0F29, 0x000B,
0x0F33, 0x000D, 0x0F39, 0x081D, 0x0F3D, 0x1716, 0x0F3F, 0x000A,
0x0F47, 0x0007, 0x0F48, 0x0001, 0x0F6C, 0x0007, 0x0F70, 0x0001,
0x0F7E, 0x0008, 0x0F7F, 0x000A, 0x0F84, 0x0008, 0x0F85, 0x0019,
0x0F87, 0x0008, 0x0F8C, 0x0007, 0x0F97, 0x0008, 0x0F98, 0x0001,
0x0FBC, 0x0008, 0x0FBD, 0x0001, 0x0FC5, 0x001D, 0x0FC6, 0x0008,
0x0FCC, 0x001D, 0x0FCD, 0x0001, 0x0FCF, 0x001D, 0x0FD4, 0x0019,
0x0FD8, 0x001D, 0x0FDA, 0x0019, 0x0FFF, 0x0001, 0x102A, 0x0007,
0x102C, 0x000A, 0x1030, 0x0008, 0x1031, 0x000A, 0x1037, 0x0008,
0x1038, 0x000A, 0x103A, 0x0008, 0x103C, 0x000A, 0x103E, 0x0008,
0x103F, 0x0007, 0x1049, 0x000B, 0x104F, 0x0019, 0x1055, 0x0007,
0x1057, 0x000A, 0x1059, 0x0008, 0x105D, 0x0007, 0x1060, 0x0008,
0x1061, 0x0007, 0x1064, 0x000A, 0x1066, 0x0007, 0x106D, 0x000A,
0x1070, 0x0007, 0x1074, 0x0008, 0x1081, 0x0007, 0x1082, 0x0008,
0x1084, 0x000A, 0x1086, 0x0008, 0x108C, 0x000A, 0x108E, 0x0807,
0x108F, 0x000A, 0x1099, 0x000B, 0x109C, 0x000A, 0x109D, 0x0008,
0x109F, 0x001D, 0x10C5, 0x0002, 0x10C7, 0x0201, 0x10CC, 0x0001,
0x10CD, 0x0002, 0x10CF, 0x0001, 0x10FA, 0x0007, 0x10FC, 0x1906,
0x1248, 0x0007, 0x1249, 0x0001, 0x124D, 0x0007, 0x124F, 0x0001,
0x1256, 0x0007, 0x1259, 0x0107, 0x125D, 0x0007, 0x125F, 0x0001,
0x1288, 0x0007, 0x1289, 0x0001, 0x128D, 0x0007, 0x128F, 0x0001,
0x12B0, 0x0007, 0x12B1, 0x0001, 0x12B5, 0x0007, 0x12B7, 0x0001,
0x12BE, 0x0007, 0x12C1, 0x0107, 0x12C5, 0x0007, 0x12C7, 0x0001,
0x12D6, 0x0007, 0x12D7, 0x0001, 0x1310, 0x0007, 0x1311, 0x0001,
0x1315, 0x0007, 0x1317, 0x0001, 0x135A, 0x0007, 0x135C, 0x0001,
0x135F, 0x0008, 0x1368, 0x0019, 0x137C, 0x000D, 0x137F, 0x0001,
0x138F, 0x0007, 0x1399, 0x001D, 0x139F, 0x0001, 0x13F4, 0x0007,
0x13FF, 0x0001, 0x1400, 0x0015, 0x166C, 0x0007, 0x166E, 0x0019,
0x167F, 0x0007, 0x1680, 0x000E, 0x169A, 0x0007, 0x169C, 0x1617,
0x169F, 0x0001, 0x16EA, 0x0007, 0x16ED, 0x0019, 0x16F0, 0x000C,
0x16FF, 0x0001, 0x170C, 0x0007, 0x170D, 0x0001, 0x1711, 0x0007,
0x1714, 0x0008, 0x171F, 0x0001, 0x1731, 0x0007, 0x1734, 0x0008,
0x1736, 0x0019, 0x173F, 0x0001, 0x1751, 0x0007, 0x1753, 0x0008,
0x175F, 0x0001, 0x176C, 0x0007, 0x176D, 0x0001, 0x1770, 0x0007,
0x1771, 0x0001, 0x1773, 0x0008, 0x177F, 0x0001, 0x17B3, 0x0007,
0x17B5, 0x0008, 0x17B6, 0x000A, 0x17BD, 0x0008, 0x17C5, 0x000A,
0x17C6, 0x0008, 0x17C8, 0x000A, 0x17D3, 0x0008, 0x17D6, 0x0019,
0x17D7, 0x0006, 0x17DA, 0x0019, 0x17DC, 0x1B07, 0x17DD, 0x0008,
0x17DF, 0x0001, 0x17E9, 0x000B, 0x17EF, 0x0001, 0x17F9, 0x000D,
0x17FF, 0x0001, 0x1805, 0x0019, 0x1806, 0x0015, 0x180A, 0x0019,
0x180D, 0x0008, 0x180F, 0x010E, 0x1819, 0x000B, 0x181F, 0x0001,
0x1842, 0x0007, 0x1843, 0x0006, 0x1877, 0x0007, 0x187F, 0x0001,
0x18A8, 0x0007, 0x18AA, 0x0807, 0x18AF, 0x0001, 0x18F5, 0x0007,
0x18FF, 0x0001, 0x191C, 0x0007, 0x191F, 0x0001, 0x1922, 0x0008,
0x1926, 0x000A, 0x1928, 0x0008, 0x192B, 0x000A, 0x192F, 0x0001,
0x1931, 0x000A, 0x1932, 0x0008, 0x1938, 0x000A, 0x193B, 0x0008,
0x193F, 0x0001, 0x1940, 0x001D, 0x1943, 0x0001, 0x1945, 0x0019,
0x194F, 0x000B, 0x196D, 0x0007, 0x196F, 0x0001, 0x1974, 0x0007,
0x197F, 0x0001, 0x19AB, 0x0007, 0x19AF, 0x0001, 0x19C0, 0x000A,
0x19C7, 0x0007, 0x19C9, 0x000A, 0x19CF, 0x0001, 0x19D9, 0x000B,
0x19DA, 0x000D, 0x19DD, 0x0001, 0x19FF, 0x001D, 0x1A16, 0x0007,
0x1A18, 0x0008, 0x1A1B, 0x000A, 0x1A1D, 0x0001, 0x1A1F, 0x0019,
0x1A54, 0x0007, 0x1A57, 0x0A08, 0x1A5E, 0x0008, 0x1A60, 0x0108,
0x1A62, 0x0A08, 0x1A64, 0x000A, 0x1A6C, 0x0008, 0x1A72, 0x000A,
0x1A7C, 0x0008, 0x1A7E, 0x0001, 0x1A7F, 0x0008, 0x1A89, 0x000B,
0x1A8F, 0x0001, 0x1A99, 0x000B, 0x1A9F, 0x0001, 0x1AA6, 0x0019,
0x1AA7, 0x0006, 0x1AAD, 0x0019, 0x1AFF, 0x0001, 0x1B03, 0x0008,
0x1B04, 0x000A, 0x1B33, 0x0007, 0x1B35, 0x0A08, 0x1B3A, 0x0008,
0x1B3C, 0x0A08, 0x1B41, 0x000A, 0x1B42, 0x0008, 0x1B44, 0x000A,
0x1B4B, 0x0007, 0x1B4F, 0x0001, 0x1B59, 0x000B, 0x1B60, 0x0019,
0x1B6A, 0x001D, 0x1B73, 0x0008, 0x1B7C, 0x001D, 0x1B7F, 0x0001,
0x1B81, 0x0008, 0x1B82, 0x000A, 0x1BA0, 0x0007, 0x1BA1, 0x000A,
0x1BA5, 0x0008, 0x1BA7, 0x000A, 0x1BA9, 0x0008, 0x1BAB, 0x080A,
0x1BAD, 0x000A, 0x1BAF, 0x0007, 0x1BB9, 0x000B, 0x1BE5, 0x0007,
0x1BE7, 0x0A08, 0x1BE9, 0x0008, 0x1BEC, 0x000A, 0x1BEE, 0x080A,
0x1BF1, 0x0008, 0x1BF3, 0x000A, 0x1BFB, 0x0001, 0x1BFF, 0x0019,
0x1C23, 0x0007, 0x1C2B, 0x000A, 0x1C33, 0x0008, 0x1C35, 0x000A,
0x1C37, 0x0008, 0x1C3A, 0x0001, 0x1C3F, 0x0019, 0x1C49, 0x000B,
0x1C4C, 0x0001, 0x1C4F, 0x0007, 0x1C59, 0x000B, 0x1C77, 0x0007,
0x1C7D, 0x0006, 0x1C7F, 0x0019, 0x1CBF, 0x0001, 0x1CC7, 0x0019,
0x1CCF, 0x0001, 0x1CD2, 0x0008, 0x1CD3, 0x0019, 0x1CE0, 0x0008,
0x1CE1, 0x000A, 0x1CE8, 0x0008, 0x1CEC, 0x0007, 0x1CED, 0x0008,
0x1CF1, 0x0007, 0x1CF3, 0x000A, 0x1CF4, 0x0008, 0x1CF6, 0x0007,
0x1CFF, 0x0001, 0x1D2B, 0x0003, 0x1D6A, 0x0006, 0x1D77, 0x0003,
0x1D78, 0x0006, 0x1D9A, 0x0003, 0x1DBF, 0x0006, 0x1DE6, 0x0008,
0x1DFB, 0x0001, 0x1DFF, 0x0008, 0x1E94, 0x0302, 0x1E9D, 0x0003,
0x1EFE, 0x0302, 0x1F07, 0x0003, 0x1F0F, 0x0002, 0x1F15, 0x0003,
0x1F17, 0x0001, 0x1F1D, 0x0002, 0x1F1F, 0x0001, 0x1F27, 0x0003,
0x1F2F, 0x0002, 0x1F37, 0x0003, 0x1F3F, 0x0002, 0x1F45, 0x0003,
0x1F47, 0x0001, 0x1F4D, 0x0002, 0x1F4F, 0x0001, 0x1F57, 0x0003,
0x1F5F, 0x0201, 0x1F67, 0x0003, 0x1F6F, 0x0002, 0x1F7D, 0x0003,
0x1F7F, 0x0001, 0x1F87, 0x0003, 0x1F8F, 0x0004, 0x1F97, 0x0003,
0x1F9F, 0x0004, 0x1FA7, 0x0003, 0x1FAF, 0x0004, 0x1FB4, 0x0003,
0x1FB5, 0x0001, 0x1FB7, 0x0003, 0x1FBB, 0x0002, 0x1FBD, 0x1C04,
0x1FBE, 0x0003, 0x1FC1, 0x001C, 0x1FC4, 0x0003, 0x1FC5, 0x0001,
0x1FC7, 0x0003, 0x1FCB, 0x0002, 0x1FCC, 0x0004, 0x1FCF, 0x001C,
0x1FD3, 0x0003, 0x1FD5, 0x0001, 0x1FD7, 0x0003, 0x1FDB, 0x0002,
0x1FDC, 0x0001, 0x1FDF, 0x001C, 0x1FE7, 0x0003, 0x1FEC, 0x0002,
0x1FEF, 0x001C, 0x1FF1, 0x0001, 0x1FF4, 0x0003, 0x1FF5, 0x0001,
0x1FF7, 0x0003, 0x1FFB, 0x0002, 0x1FFC, 0x0004, 0x1FFE, 0x001C,
0x1FFF, 0x0001, 0x200A, 0x000E, 0x200F, 0x0012, 0x2015, 0x0015,
0x2017, 0x0019, 0x2019, 0x1F1E, 0x201A, 0x0016, 0x201C, 0x001E,
0x201E, 0x1F16, 0x201F, 0x001E, 0x2027, 0x0019, 0x2029, 0x100F,
0x202E, 0x0012, 0x202F, 0x000E, 0x2038, 0x0019, 0x203A, 0x1E1F,
0x203E, 0x0019, 0x2040, 0x0018, 0x2043, 0x0019, 0x2045, 0x161A,
0x2046, 0x0017, 0x2051, 0x0019, 0x2053, 0x191A, 0x2054, 0x0018,
0x205E, 0x0019, 0x205F, 0x000E, 0x2064, 0x0012, 0x2069, 0x0001,
0x206F, 0x0012, 0x2071, 0x060D, 0x2073, 0x0001, 0x2079, 0x000D,
0x207C, 0x001A, 0x207E, 0x1617, 0x207F, 0x0006, 0x2089, 0x000D,
0x208C, 0x001A, 0x208E, 0x1617, 0x208F, 0x0001, 0x209C, 0x0006,
0x209F, 0x0001, 0x20B9, 0x001B, 0x20CF, 0x0001, 0x20DC, 0x0008,
0x20E0, 0x0009, 0x20E1, 0x0008, 0x20E4, 0x0009, 0x20F0, 0x0008,
0x20FF, 0x0001, 0x2101, 0x001D, 0x2102, 0x0002, 0x2106, 0x001D,
0x2107, 0x0002, 0x2109, 0x001D, 0x210A, 0x0003, 0x210D, 0x0002,
0x210F, 0x0003, 0x2112, 0x0002, 0x2114, 0x031D, 0x2115, 0x0002,
0x2117, 0x001D, 0x2118, 0x001A, 0x211D, 0x0002, 0x2123, 0x001D,
0x2129, 0x1D02, 0x212D, 0x0002, 0x212F, 0x031D, 0x2133, 0x0002,
0x2134, 0x0003, 0x2138, 0x0007, 0x2139, 0x0003, 0x213B, 0x001D,
0x213D, 0x0003, 0x213F, 0x0002, 0x2144, 0x001A, 0x2145, 0x0002,
0x2149, 0x0003, 0x214B, 0x1A1D, 0x214D, 0x001D, 0x214F, 0x1D03,
0x215F, 0x000D, 0x2182, 0x000C, 0x2184, 0x0203, 0x2188, 0x000C,
0x2189, 0x000D, 0x218F, 0x0001, 0x2194, 0x001A, 0x2199, 0x001D,
0x219B, 0x001A, 0x219F, 0x001D, 0x21A0, 0x001A, 0x21A2, 0x001D,
0x21A3, 0x001A, 0x21A5, 0x001D, 0x21A6, 0x001A, 0x21AD, 0x001D,
0x21AE, 0x001A, 0x21CD, 0x001D, 0x21CF, 0x001A, 0x21D1, 0x001D,
0x21D4, 0x1D1A, 0x21F3, 0x001D, 0x22FF, 0x001A, 0x2307, 0x001D,
0x230B, 0x001A, 0x231F, 0x001D, 0x2321, 0x001A, 0x2328, 0x001D,
0x232A, 0x1617, 0x237B, 0x001D, 0x237C, 0x001A, 0x239A, 0x001D,
0x23B3, 0x001A, 0x23DB, 0x001D, 0x23E1, 0x001A, 0x23F3, 0x001D,
0x23FF, 0x0001, 0x2426, 0x001D, 0x243F, 0x0001, 0x244A, 0x001D,
0x245F, 0x0001, 0x249B, 0x000D, 0x24E9, 0x001D, 0x24FF, 0x000D,
0x25B6, 0x001D, 0x25B7, 0x001A, 0x25C0, 0x001D, 0x25C1, 0x001A,
0x25F7, 0x001D, 0x25FF, 0x001A, 0x266E, 0x001D, 0x266F, 0x001A,
0x26FF, 0x001D, 0x2700, 0x0001, 0x2767, 0x001D, 0x2775, 0x1716,
0x2793, 0x000D, 0x27BF, 0x001D, 0x27C4, 0x001A, 0x27C6, 0x1617,
0x27E5, 0x001A, 0x27EF, 0x1716, 0x27FF, 0x001A, 0x28FF, 0x001D,
0x2982, 0x001A, 0x2998, 0x1617, 0x29D7, 0x001A, 0x29DB, 0x1716,
0x29FB, 0x001A, 0x29FD, 0x1716, 0x2AFF, 0x001A, 0x2B2F, 0x001D,
0x2B44, 0x001A, 0x2B46, 0x001D, 0x2B4C, 0x001A, 0x2B4F, 0x0001,
0x2B59, 0x001D, 0x2BFF, 0x0001, 0x2C2E, 0x0002, 0x2C2F, 0x0001,
0x2C5E, 0x0003, 0x2C60, 0x0102, 0x2C61, 0x0003, 0x2C64, 0x0002,
0x2C66, 0x0003, 0x2C6C, 0x0203, 0x2C70, 0x0002, 0x2C72, 0x0302,
0x2C74, 0x0003, 0x2C75, 0x0002, 0x2C7B, 0x0003, 0x2C7D, 0x0006,
0x2C80, 0x0002, 0x2CE2, 0x0302, 0x2CE4, 0x0003, 0x2CEA, 0x001D,
0x2CEE, 0x0203, 0x2CF1, 0x0008, 0x2CF3, 0x0302, 0x2CF8, 0x0001,
0x2CFC, 0x0019, 0x2CFD, 0x000D, 0x2CFF, 0x0019, 0x2D25, 0x0003,
0x2D27, 0x0301, 0x2D2C, 0x0001, 0x2D2D, 0x0003, 0x2D2F, 0x0001,
0x2D67, 0x0007, 0x2D6E, 0x0001, 0x2D70, 0x0619, 0x2D7E, 0x0001,
0x2D7F, 0x0008, 0x2D96, 0x0007, 0x2D9F, 0x0001, 0x2DA6, 0x0007,
0x2DA7, 0x0001, 0x2DAE, 0x0007, 0x2DAF, 0x0001, 0x2DB6, 0x0007,
0x2DB7, 0x0001, 0x2DBE, 0x0007, 0x2DBF, 0x0001, 0x2DC6, 0x0007,
0x2DC7, 0x0001, 0x2DCE, 0x0007, 0x2DCF, 0x0001, 0x2DD6, 0x0007,
0x2DD7, 0x0001, 0x2DDE, 0x0007, 0x2DDF, 0x0001, 0x2DFF, 0x0008,
0x2E01, 0x0019, 0x2E05, 0x1F1E, 0x2E08, 0x0019, 0x2E0A, 0x1E1F,
0x2E0C, 0x191E, 0x2E0D, 0x001F, 0x2E16, 0x0019, 0x2E17, 0x0015,
0x2E19, 0x0019, 0x2E1B, 0x1915, 0x2E1D, 0x1F1E, 0x2E1F, 0x0019,
0x2E21, 0x1F1E, 0x2E29, 0x1716, 0x2E2E, 0x0019, 0x2E2F, 0x0006,
0x2E39, 0x0019, 0x2E3B, 0x0015, 0x2E7F, 0x0001, 0x2E99, 0x001D,
0x2E9A, 0x0001, 0x2EF3, 0x001D, 0x2EFF, 0x0001, 0x2FD5, 0x001D,
0x2FEF, 0x0001, 0x2FFB, 0x001D, 0x2FFF, 0x0001, 0x3000, 0x000E,
0x3003, 0x0019, 0x3005, 0x061D, 0x3007, 0x0C07, 0x3011, 0x1716,
0x3013, 0x001D, 0x301B, 0x1716, 0x301D, 0x1615, 0x301F, 0x0017,
0x3020, 0x001D, 0x3029, 0x000C, 0x302D, 0x0008, 0x302F, 0x000A,
0x3030, 0x0015, 0x3035, 0x0006, 0x3037, 0x001D, 0x303A, 0x000C,
0x303C, 0x0607, 0x303D, 0x0019, 0x303F, 0x001D, 0x3040, 0x0001,
0x3096, 0x0007, 0x3098, 0x0001, 0x309A, 0x0008, 0x309C, 0x001C,
0x309E, 0x0006, 0x30A0, 0x0715, 0x30FA, 0x0007, 0x30FB, 0x0019,
0x30FE, 0x0006, 0x30FF, 0x0007, 0x3104, 0x0001, 0x312D, 0x0007,
0x3130, 0x0001, 0x318E, 0x0007, 0x318F, 0x0001, 0x3191, 0x001D,
0x3195, 0x000D, 0x319F, 0x001D, 0x31BA, 0x0007, 0x31BF, 0x0001,
0x31E3, 0x001D, 0x31EF, 0x0001, 0x31FF, 0x0007, 0x321E, 0x001D,
0x321F, 0x0001, 0x3229, 0x000D, 0x3247, 0x001D, 0x324F, 0x000D,
0x3250, 0x001D, 0x325F, 0x000D, 0x327F, 0x001D, 0x3289, 0x000D,
0x32B0, 0x001D, 0x32BF, 0x000D, 0x32FE, 0x001D, 0x32FF, 0x0001,
0x33FF, 0x001D, 0x4DB5, 0x0007, 0x4DBF, 0x0001, 0x4DFF, 0x001D,
0x9FCC, 0x0007, 0x9FFF, 0x0001, 0xA014, 0x0007, 0xA015, 0x0006,
0xA48C, 0x0007, 0xA48F, 0x0001, 0xA4C6, 0x001D, 0xA4CF, 0x0001,
0xA4F7, 0x0007, 0xA4FD, 0x0006, 0xA4FF, 0x0019, 0xA60B, 0x0007,
0xA60C, 0x0006, 0xA60F, 0x0019, 0xA61F, 0x0007, 0xA629, 0x000B,
0xA62B, 0x0007, 0xA63F, 0x0001, 0xA66D, 0x0302, 0xA66F, 0x0807,
0xA672, 0x0009, 0xA673, 0x0019, 0xA67D, 0x0008, 0xA67F, 0x0619,
0xA697, 0x0302, 0xA69E, 0x0001, 0xA69F, 0x0008, 0xA6E5, 0x0007,
0xA6EF, 0x000C, 0xA6F1, 0x0008, 0xA6F7, 0x0019, 0xA6FF, 0x0001,
0xA716, 0x001C, 0xA71F, 0x0006, 0xA721, 0x001C, 0xA72E, 0x0302,
0xA731, 0x0003, 0xA76F, 0x0302, 0xA770, 0x0006, 0xA778, 0x0003,
0xA77C, 0x0203, 0xA77E, 0x0002, 0xA787, 0x0302, 0xA788, 0x0006,
0xA78A, 0x001C, 0xA78E, 0x0203, 0xA790, 0x0102, 0xA793, 0x0302,
0xA79F, 0x0001, 0xA7AA, 0x0302, 0xA7F7, 0x0001, 0xA7F9, 0x0006,
0xA7FA, 0x0003, 0xA801, 0x0007, 0xA802, 0x0008, 0xA805, 0x0007,
0xA806, 0x0008, 0xA80A, 0x0007, 0xA80B, 0x0008, 0xA822, 0x0007,
0xA824, 0x000A, 0xA826, 0x0008, 0xA827, 0x000A, 0xA82B, 0x001D,
0xA82F, 0x0001, 0xA835, 0x000D, 0xA837, 0x001D, 0xA839, 0x1D1B,
0xA83F, 0x0001, 0xA873, 0x0007, 0xA877, 0x0019, 0xA87F, 0x0001,
0xA881, 0x000A, 0xA8B3, 0x0007, 0xA8C3, 0x000A, 0xA8C4, 0x0008,
0xA8CD, 0x0001, 0xA8CF, 0x0019, 0xA8D9, 0x000B, 0xA8DF, 0x0001,
0xA8F1, 0x0008, 0xA8F7, 0x0007, 0xA8FA, 0x0019, 0xA8FB, 0x0007,
0xA8FF, 0x0001, 0xA909, 0x000B, 0xA925, 0x0007, 0xA92D, 0x0008,
0xA92F, 0x0019, 0xA946, 0x0007, 0xA951, 0x0008, 0xA953, 0x000A,
0xA95E, 0x0001, 0xA95F, 0x0019, 0xA97C, 0x0007, 0xA97F, 0x0001,
0xA982, 0x0008, 0xA983, 0x000A, 0xA9B2, 0x0007, 0xA9B3, 0x0008,
0xA9B5, 0x000A, 0xA9B9, 0x0008, 0xA9BB, 0x000A, 0xA9BC, 0x0008,
0xA9C0, 0x000A, 0xA9CD, 0x0019, 0xA9CF, 0x0601, 0xA9D9, 0x000B,
0xA9DD, 0x0001, 0xA9DF, 0x0019, 0xA9FF, 0x0001, 0xAA28, 0x0007,
0xAA2E, 0x0008, 0xAA30, 0x000A, 0xAA32, 0x0008, 0xAA34, 0x000A,
0xAA36, 0x0008, 0xAA3F, 0x0001, 0xAA42, 0x0007, 0xAA43, 0x0008,
0xAA4B, 0x0007, 0xAA4D, 0x0A08, 0xAA4F, 0x0001, 0xAA59, 0x000B,
0xAA5B, 0x0001, 0xAA5F, 0x0019, 0xAA6F, 0x0007, 0xAA70, 0x0006,
0xAA76, 0x0007, 0xAA79, 0x001D, 0xAA7B, 0x0A07, 0xAA7F, 0x0001,
0xAAAF, 0x0007, 0xAAB1, 0x0708, 0xAAB4, 0x0008, 0xAAB6, 0x0007,
0xAAB8, 0x0008, 0xAABD, 0x0007, 0xAABF, 0x0008, 0xAAC2, 0x0807,
0xAADA, 0x0001, 0xAADC, 0x0007, 0xAADD, 0x0006, 0xAADF, 0x0019,
0xAAEA, 0x0007, 0xAAEB, 0x000A, 0xAAED, 0x0008, 0xAAEF, 0x000A,
0xAAF1, 0x0019, 0xAAF2, 0x0007, 0xAAF4, 0x0006, 0xAAF6, 0x0A08,
0xAB00, 0x0001, 0xAB06, 0x0007, 0xAB08, 0x0001, 0xAB0E, 0x0007,
0xAB10, 0x0001, 0xAB16, 0x0007, 0xAB1F, 0x0001, 0xAB26, 0x0007,
0xAB27, 0x0001, 0xAB2E, 0x0007, 0xABBF, 0x0001, 0xABE2, 0x0007,
0xABE4, 0x000A, 0xABE5, 0x0008, 0xABE7, 0x000A, 0xABE8, 0x0008,
0xABEA, 0x000A, 0xABEC, 0x190A, 0xABED, 0x0008, 0xABEF, 0x0001,
0xABF9, 0x000B, 0xABFF, 0x0001, 0xD7A3, 0x0007, 0xD7AF, 0x0001,
0xD7C6, 0x0007, 0xD7CA, 0x0001, 0xD7FB, 0x0007, 0xD7FF, 0x0001,
0xDFFF, 0x0014, 0xF8FF, 0x0013, 0xFA6D, 0x0007, 0xFA6F, 0x0001,
0xFAD9, 0x0007, 0xFAFF, 0x0001, 0xFB06, 0x0003, 0xFB12, 0x0001,
0xFB17, 0x0003, 0xFB1C, 0x0001, 0xFB1E, 0x0708, 0xFB28, 0x0007,
0xFB29, 0x001A, 0xFB36, 0x0007, 0xFB37, 0x0001, 0xFB3C, 0x0007,
0xFB3F, 0x0107, 0xFB41, 0x0007, 0xFB42, 0x0001, 0xFB44, 0x0007,
0xFB45, 0x0001, 0xFBB1, 0x0007, 0xFBC1, 0x001C, 0xFBD2, 0x0001,
0xFD3D, 0x0007, 0xFD3F, 0x1716, 0xFD4F, 0x0001, 0xFD8F, 0x0007,
0xFD91, 0x0001, 0xFDC7, 0x0007, 0xFDEF, 0x0001, 0xFDFB, 0x0007,
0xFDFD, 0x1D1B, 0xFDFF, 0x0001, 0xFE0F, 0x0008, 0xFE16, 0x0019,
0xFE18, 0x1617, 0xFE19, 0x0019, 0xFE1F, 0x0001, 0xFE26, 0x0008,
0xFE2F, 0x0001, 0xFE30, 0x0019, 0xFE32, 0x0015, 0xFE34, 0x0018,
0xFE44, 0x1617, 0xFE46, 0x0019, 0xFE48, 0x1617, 0xFE4C, 0x0019,
0xFE4F, 0x0018, 0xFE52, 0x0019, 0xFE53, 0x0001, 0xFE57, 0x0019,
0xFE59, 0x1615, 0xFE5E, 0x1617, 0xFE61, 0x0019, 0xFE63, 0x151A,
0xFE66, 0x001A, 0xFE68, 0x0119, 0xFE69, 0x001B, 0xFE6B, 0x0019,
0xFE6F, 0x0001, 0xFE74, 0x0007, 0xFE75, 0x0001, 0xFEFC, 0x0007,
0xFEFE, 0x0001, 0xFF00, 0x1201, 0xFF03, 0x0019, 0xFF04, 0x001B,
0xFF07, 0x0019, 0xFF09, 0x1716, 0xFF0C, 0x1A19, 0xFF0D, 0x0015,
0xFF0F, 0x0019, 0xFF19, 0x000B, 0xFF1B, 0x0019, 0xFF1E, 0x001A,
0xFF20, 0x0019, 0xFF3A, 0x0002, 0xFF3C, 0x1619, 0xFF3E, 0x171C,
0xFF40, 0x181C, 0xFF5A, 0x0003, 0xFF5C, 0x161A, 0xFF5E, 0x171A,
0xFF60, 0x1617, 0xFF62, 0x1916, 0xFF63, 0x0017, 0xFF65, 0x0019,
0xFF6F, 0x0007, 0xFF70, 0x0006, 0xFF9D, 0x0007, 0xFF9F, 0x0006,
0xFFBE, 0x0007, 0xFFC1, 0x0001, 0xFFC7, 0x0007, 0xFFC9, 0x0001,
0xFFCF, 0x0007, 0xFFD1, 0x0001, 0xFFD7, 0x0007, 0xFFD9, 0x0001,
0xFFDC, 0x0007, 0xFFDF, 0x0001, 0xFFE1, 0x001B, 0xFFE3, 0x1C1A,
0xFFE4, 0x001D, 0xFFE6, 0x001B, 0xFFE8, 0x011D, 0xFFEC, 0x001A,
0xFFEE, 0x001D, 0xFFF8, 0x0001, 0xFFFB, 0x0012, 0xFFFD, 0x001D,
0xFFFF, 0x0001
} };

static const std::array<uint64_t, 103> kSpecialCasing = { {
UINT64_C(0x0000000000480331), UINT64_C(0x0000000000540308),
UINT64_C(0x000000000057030A), UINT64_C(0x000000000059030A),
UINT64_C(0x00000000004102BE), UINT64_C(0x0000000000530053),
UINT64_C(0x0000000000460046), UINT64_C(0x0000000000460049),
UINT64_C(0x000000000046004C), UINT64_C(0x0000004600460049),
UINT64_C(0x000000460046004C), UINT64_C(0x0000000000530054),
UINT64_C(0x0000000000530054), UINT64_C(0x0000000005440546),
UINT64_C(0x0000000005440535), UINT64_C(0x000000000544053B),
UINT64_C(0x00000000054E0546), UINT64_C(0x000000000544053D),
UINT64_C(0x000000001F0F0399), UINT64_C(0x0000000000690307),
UINT64_C(0x0000000002BC004E), UINT64_C(0x0000000003A50313),
UINT64_C(0x000003A503130300), UINT64_C(0x000003A503130301),
UINT64_C(0x000003A503130342), UINT64_C(0x000000001F280399),
UINT64_C(0x000000001F080399), UINT64_C(0x000000001F090399),
UINT64_C(0x000000001F0A0399), UINT64_C(0x000000001F0B0399),
UINT64_C(0x000000001F0C0399), UINT64_C(0x000000001F0D0399),
UINT64_C(0x000000001F0E0399), UINT64_C(0x0000000005350552),
UINT64_C(0x000000001F080399), UINT64_C(0x000000001F090399),
UINT64_C(0x000000001F0A0399), UINT64_C(0x000000001F0B0399),
UINT64_C(0x000000001F0C0399), UINT64_C(0x000000001F0D0399),
UINT64_C(0x000000001F0E0399), UINT64_C(0x000000001F0F0399),
UINT64_C(0x0000039903080301), UINT64_C(0x000000001F290399),
UINT64_C(0x000000001F2A0399), UINT64_C(0x000000001F2B0399),
UINT64_C(0x000000001F2C0399), UINT64_C(0x000000001F2D0399),
UINT64_C(0x000000001F2E0399), UINT64_C(0x000000001F2F0399),
UINT64_C(0x000000001F280399), UINT64_C(0x000000001F290399),
UINT64_C(0x000000001F2A0399), UINT64_C(0x000000001F2B0399),
UINT64_C(0x000000001F2C0399), UINT64_C(0x000000001F2D0399),
UINT64_C(0x000000001F2E0399), UINT64_C(0x000000001F2F0399),
UINT64_C(0x000000001F680399), UINT64_C(0x000000001F690399),
UINT64_C(0x000000001F6A0399), UINT64_C(0x000000001F6B0399),
UINT64_C(0x000000001F6C0399), UINT64_C(0x000000001F6D0399),
UINT64_C(0x000000001F6E0399), UINT64_C(0x000000001F6F0399),
UINT64_C(0x000000001F680399), UINT64_C(0x000000001F690399),
UINT64_C(0x000000001F6A0399), UINT64_C(0x000000001F6B0399),
UINT64_C(0x000000001F6C0399), UINT64_C(0x000000001F6D0399),
UINT64_C(0x000000001F6E0399), UINT64_C(0x000000001F6F0399),
UINT64_C(0x000003A503080301), UINT64_C(0x000000001FBA0399),
UINT64_C(0x0000000003910399), UINT64_C(0x0000000003860399),
UINT64_C(0x0000000003910342), UINT64_C(0x0000039103420399),
UINT64_C(0x0000000003910399), UINT64_C(0x000000001FCA0399),
UINT64_C(0x0000000003970399), UINT64_C(0x0000000003890399),
UINT64_C(0x0000000003970342), UINT64_C(0x0000039703420399),
UINT64_C(0x0000000003970399), UINT64_C(0x0000039903080300),
UINT64_C(0x0000039903080301), UINT64_C(0x0000000003990342),
UINT64_C(0x0000039903080342), UINT64_C(0x000003A503080300),
UINT64_C(0x000003A503080301), UINT64_C(0x0000000003A10313),
UINT64_C(0x0000000003A50342), UINT64_C(0x000003A503080342),
UINT64_C(0x00000000004A030C), UINT64_C(0x000000001FFA0399),
UINT64_C(0x0000000003A90399), UINT64_C(0x00000000038F0399),
UINT64_C(0x0000000003A90342), UINT64_C(0x000003A903420399),
UINT64_C(0x0000000003A90399)
} };

static const std::array<char16_t, 1000 - 0xC0> kLowerCaseCache = { {
0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00D7,
0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00DF,
0x00E0, 0x00E1, 0x00E2, 0x00E3, 0x00E4, 0x00E5, 0x00E6, 0x00E7,
0x00E8, 0x00E9, 0x00EA, 0x00EB, 0x00EC, 0x00ED, 0x00EE, 0x00EF,
0x00F0, 0x00F1, 0x00F2, 0x00F3, 0x00F4, 0x00F5, 0x00F6, 0x00F7,
0x00F8, 0x00F9, 0x00FA, 0x00FB, 0x00FC, 0x00FD, 0x00FE, 0x00FF,
0x0101, 0x0101, 0x0103, 0x0103, 0x0105, 0x0105, 0x0107, 0x0107,
0x0109, 0x0109, 0x010B, 0x010B, 0x010D, 0x010D, 0x010F, 0x010F,
0x0111, 0x0111, 0x0113, 0x0113, 0x0115, 0x0115, 0x0117, 0x0117,
0x0119, 0x0119, 0x011B, 0x011B, 0x011D, 0x011D, 0x011F, 0x011F,
0x0121, 0x0121, 0x0123, 0x0123, 0x0125, 0x0125, 0x0127, 0x0127,
0x0129, 0x0129, 0x012B, 0x012B, 0x012D, 0x012D, 0x012F, 0x012F,
0x0000, 0x0131, 0x0133, 0x0133, 0x0135, 0x0135, 0x0137, 0x0137,
0x0138, 0x013A, 0x013A, 0x013C, 0x013C, 0x013E, 0x013E, 0x0140,
0x0140, 0x0142, 0x0142, 0x0144, 0x0144, 0x0146, 0x0146, 0x0148,
0x0148, 0x0149, 0x014B, 0x014B, 0x014D, 0x014D, 0x014F, 0x014F,
0x0151, 0x0151, 0x0153, 0x0153, 0x0155, 0x0155, 0x0157, 0x0157,
0x0159, 0x0159, 0x015B, 0x015B, 0x015D, 0x015D, 0x015F, 0x015F,
0x0161, 0x0161, 0x0163, 0x0163, 0x0165, 0x0165, 0x0167, 0x0167,
0x0169, 0x0169, 0x016B, 0x016B, 0x016D, 0x016D, 0x016F, 0x016F,
0x0171, 0x0171, 0x0173, 0x0173, 0x0175, 0x0175, 0x0177, 0x0177,
0x00FF, 0x017A, 0x017A, 0x017C, 0x017C, 0x017E, 0x017E, 0x017F,
0x0180, 0x0253, 0x0183, 0x0183, 0x0185, 0x0185, 0x0254, 0x0188,
0x0188, 0x0256, 0x0257, 0x018C, 0x018C, 0x018D, 0x01DD, 0x0259,
0x025B, 0x0192, 0x0192, 0x0260, 0x0263, 0x0195, 0x0269, 0x0268,
0x0199, 0x0199, 0x019A, 0x019B, 0x026F, 0x0272, 0x019E, 0x0275,
0x01A1, 0x01A1, 0x01A3, 0x01A3, 0x01A5, 0x01A5, 0x0280, 0x01A8,
0x01A8, 0x0283, 0x01AA, 0x01AB, 0x01AD, 0x01AD, 0x0288, 0x01B0,
0x01B0, 0x028A, 0x028B, 0x01B4, 0x01B4, 0x01B6, 0x01B6, 0x0292,
0x01B9, 0x01B9, 0x01BA, 0x01BB, 0x01BD, 0x01BD, 0x01BE, 0x01BF,
0x01C0, 0x01C1, 0x01C2, 0x01C3, 0x01C6, 0x01C6, 0x01C6, 0x01C9,
0x01C9, 0x01C9, 0x01CC, 0x01CC, 0x01CC, 0x01CE, 0x01CE, 0x01D0,
0x01D0, 0x01D2, 0x01D2, 0x01D4, 0x01D4, 0x01D6, 0x01D6, 0x01D8,
0x01D8, 0x01DA, 0x01DA, 0x01DC, 0x01DC, 0x01DD, 0x01DF, 0x01DF,
0x01E1, 0x01E1, 0x01E3, 0x01E3, 0x01E5, 0x01E5, 0x01E7, 0x01E7,
0x01E9, 0x01E9, 0x01EB, 0x01EB, 0x01ED, 0x01ED, 0x01EF, 0x01EF,
0x01F0, 0x01F3, 0x01F3, 0x01F3, 0x01F5, 0x01F5, 0x0195, 0x01BF,
0x01F9, 0x01F9, 0x01FB, 0x01FB, 0x01FD, 0x01FD, 0x01FF, 0x01FF,
0x0201, 0x0201, 0x0203, 0x0203, 0x0205, 0x0205, 0x0207, 0x0207,
0x0209, 0x0209, 0x020B, 0x020B, 0x020D, 0x020D, 0x020F, 0x020F,
0x0211, 0x0211, 0x0213, 0x0213, 0x0215, 0x0215, 0x0217, 0x0217,
0x0219, 0x0219, 0x021B, 0x021B, 0x021D, 0x021D, 0x021F, 0x021F,
0x019E, 0x0221, 0x0223, 0x0223, 0x0225, 0x0225, 0x0227, 0x0227,
0x0229, 0x0229, 0x022B, 0x022B, 0x022D, 0x022D, 0x022F, 0x022F,
0x0231, 0x0231, 0x0233, 0x0233, 0x0234, 0x0235, 0x0236, 0x0237,
0x0238, 0x0239, 0x2C65, 0x023C, 0x023C, 0x019A, 0x2C66, 0x023F,
0x0240, 0x0242, 0x0242, 0x0180, 0x0289, 0x028C, 0x0247, 0x0247,
0x0249, 0x0249, 0x024B, 0x024B, 0x024D, 0x024D, 0x024F, 0x024F,
0x0250, 0x0251, 0x0252, 0x0253, 0x0254, 0x0255, 0x0256, 0x0257,
0x0258, 0x0259, 0x025A, 0x025B, 0x025C, 0x025D, 0x025E, 0x025F,
0x0260, 0x0261, 0x0262, 0x0263, 0x0264, 0x0265, 0x0266, 0x0267,
0x0268, 0x0269, 0x026A, 0x026B, 0x026C, 0x026D, 0x026E, 0x026F,
0x0270, 0x0271, 0x0272, 0x0273, 0x0274, 0x0275, 0x0276, 0x0277,
0x0278, 0x0279, 0x027A, 0x027B, 0x027C, 0x027D, 0x027E, 0x027F,
0x0280, 0x0281, 0x0282, 0x0283, 0x0284, 0x0285, 0x0286, 0x0287,
0x0288, 0x0289, 0x028A, 0x028B, 0x028C, 0x028D, 0x028E, 0x028F,
0x0290, 0x0291, 0x0292, 0x0293, 0x0294, 0x0295, 0x0296, 0x0297,
0x0298, 0x0299, 0x029A, 0x029B, 0x029C, 0x029D, 0x029E, 0x029F,
0x02A0, 0x02A1, 0x02A2, 0x02A3, 0x02A4, 0x02A5, 0x02A6, 0x02A7,
0x02A8, 0x02A9, 0x02AA, 0x02AB, 0x02AC, 0x02AD, 0x02AE, 0x02AF,
0x02B0, 0x02B1, 0x02B2, 0x02B3, 0x02B4, 0x02B5, 0x02B6, 0x02B7,
0x02B8, 0x02B9, 0x02BA, 0x02BB, 0x02BC, 0x02BD, 0x02BE, 0x02BF,
0x02C0, 0x02C1, 0x02C2, 0x02C3, 0x02C4, 0x02C5, 0x02C6, 0x02C7,
0x02C8, 0x02C9, 0x02CA, 0x02CB, 0x02CC, 0x02CD, 0x02CE, 0x02CF,
0x02D0, 0x02D1, 0x02D2, 0x02D3, 0x02D4, 0x02D5, 0x02D6, 0x02D7,
0x02D8, 0x02D9, 0x02DA, 0x02DB, 0x02DC, 0x02DD, 0x02DE, 0x02DF,
0x02E0, 0x02E1, 0x02E2, 0x02E3, 0x02E4, 0x02E5, 0x02E6, 0x02E7,
0x02E8, 0x02E9, 0x02EA, 0x02EB, 0x02EC, 0x02ED, 0x02EE, 0x02EF,
0x02F0, 0x02F1, 0x02F2, 0x02F3, 0x02F4, 0x02F5, 0x02F6, 0x02F7,
0x02F8, 0x02F9, 0x02FA, 0x02FB, 0x02FC, 0x02FD, 0x02FE, 0x02FF,
0x0300, 0x0301, 0x0302, 0x0303, 0x0304, 0x0305, 0x0306, 0x0307,
0x0308, 0x0309, 0x030A, 0x030B, 0x030C, 0x030D, 0x030E, 0x030F,
0x0310, 0x0311, 0x0312, 0x0313, 0x0314, 0x0315, 0x0316, 0x0317,
0x0318, 0x0319, 0x031A, 0x031B, 0x031C, 0x031D, 0x031E, 0x031F,
0x0320, 0x0321, 0x0322, 0x0323, 0x0324, 0x0325, 0x0326, 0x0327,
0x0328, 0x0329, 0x032A, 0x032B, 0x032C, 0x032D, 0x032E, 0x032F,
0x0330, 0x0331, 0x0332, 0x0333, 0x0334, 0x0335, 0x0336, 0x0337,
0x0338, 0x0339, 0x033A, 0x033B, 0x033C, 0x033D, 0x033E, 0x033F,
0x0340, 0x0341, 0x0342, 0x0343, 0x0344, 0x0345, 0x0346, 0x0347,
0x0348, 0x0349, 0x034A, 0x034B, 0x034C, 0x034D, 0x034E, 0x034F,
0x0350, 0x0351, 0x0352, 0x0353, 0x0354, 0x0355, 0x0356, 0x0357,
0x0358, 0x0359, 0x035A, 0x035B, 0x035C, 0x035D, 0x035E, 0x035F,
0x0360, 0x0361, 0x0362, 0x0363, 0x0364, 0x0365, 0x0366, 0x0367,
0x0368, 0x0369, 0x036A, 0x036B, 0x036C, 0x036D, 0x036E, 0x036F,
0x0371, 0x0371, 0x0373, 0x0373, 0x0374, 0x0375, 0x0377, 0x0377,
0x0378, 0x0379, 0x037A, 0x037B, 0x037C, 0x037D, 0x037E, 0x037F,
0x0380, 0x0381, 0x0382, 0x0383, 0x0384, 0x0385, 0x03AC, 0x0387,
0x03AD, 0x03AE, 0x03AF, 0x038B, 0x03CC, 0x038D, 0x03CD, 0x03CE,
0x0390, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7,
0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
0x03C0, 0x03C1, 0x03A2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7,
0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03AC, 0x03AD, 0x03AE, 0x03AF,
0x03B0, 0x03B1, 0x03B2, 0x03B3, 0x03B4, 0x03B5, 0x03B6, 0x03B7,
0x03B8, 0x03B9, 0x03BA, 0x03BB, 0x03BC, 0x03BD, 0x03BE, 0x03BF,
0x03C0, 0x03C1, 0x03C2, 0x03C3, 0x03C4, 0x03C5, 0x03C6, 0x03C7,
0x03C8, 0x03C9, 0x03CA, 0x03CB, 0x03CC, 0x03CD, 0x03CE, 0x03D7,
0x03D0, 0x03D1, 0x03D2, 0x03D3, 0x03D4, 0x03D5, 0x03D6, 0x03D7,
0x03D9, 0x03D9, 0x03DB, 0x03DB, 0x03DD, 0x03DD, 0x03DF, 0x03DF,
0x03E1, 0x03E1, 0x03E3, 0x03E3, 0x03E5, 0x03E5, 0x03E7, 0x03E7
} };

typedef std::array<char16_t, 151> LowerCaseKeys;
static const LowerCaseKeys kLowerCaseKeys = { {
0x0041, 0x00C0, 0x00D8, 0x0100, 0x0130, 0x0132, 0x0139, 0x014A,
0x0178, 0x0179, 0x0181, 0x0182, 0x0186, 0x0187, 0x0189, 0x018B,
0x018E, 0x018F, 0x0190, 0x0191, 0x0193, 0x0194, 0x0196, 0x0197,
0x0198, 0x019C, 0x019D, 0x019F, 0x01A0, 0x01A6, 0x01A7, 0x01A9,
0x01AC, 0x01AE, 0x01AF, 0x01B1, 0x01B3, 0x01B7, 0x01B8, 0x01BC,
0x01C4, 0x01C5, 0x01C7, 0x01C8, 0x01CA, 0x01CB, 0x01DE, 0x01F1,
0x01F2, 0x01F6, 0x01F7, 0x01F8, 0x0220, 0x0222, 0x023A, 0x023B,
0x023D, 0x023E, 0x0241, 0x0243, 0x0244, 0x0245, 0x0246, 0x0370,
0x0376, 0x0386, 0x0388, 0x038C, 0x038E, 0x0391, 0x03A3, 0x03CF,
0x03D8, 0x03F4, 0x03F7, 0x03F9, 0x03FA, 0x03FD, 0x0400, 0x0410,
0x0460, 0x048A, 0x04C0, 0x04C1, 0x04D0, 0x0531, 0x10A0, 0x10C7,
0x10CD, 0x1E00, 0x1E9E, 0x1EA0, 0x1F08, 0x1F18, 0x1F28, 0x1F38,
0x1F48, 0x1F59, 0x1F68, 0x1F88, 0x1F98, 0x1FA8, 0x1FB8, 0x1FBA,
0x1FBC, 0x1FC8, 0x1FCC, 0x1FD8, 0x1FDA, 0x1FE8, 0x1FEA, 0x1FEC,
0x1FF8, 0x1FFA, 0x1FFC, 0x2126, 0x212A, 0x212B, 0x2132, 0x2160,
0x2183, 0x24B6, 0x2C00, 0x2C60, 0x2C62, 0x2C63, 0x2C64, 0x2C67,
0x2C6D, 0x2C6E, 0x2C6F, 0x2C70, 0x2C72, 0x2C75, 0x2C7E, 0x2C80,
0x2CEB, 0x2CF2, 0xA640, 0xA680, 0xA722, 0xA732, 0xA779, 0xA77D,
0xA77E, 0xA78B, 0xA78D, 0xA790, 0xA7A0, 0xA7AA, 0xFF21
} };

typedef std::array<char16_t, 302> LowerCaseValues;
static const LowerCaseValues kLowerCaseValues = { {
0x005A, 0x0020, 0x00D6, 0x0020, 0x00DE, 0x0020, 0x812E, 0x0001,
0x0013, 0x0000, 0x8136, 0x0001, 0x8147, 0x0001, 0x8176, 0x0001,
0x0178, 0xFF87, 0x817D, 0x0001, 0x0181, 0x00D2, 0x8184, 0x0001,
0x0186, 0x00CE, 0x8187, 0x0001, 0x018A, 0x00CD, 0x018B, 0x0001,
0x018E, 0x004F, 0x018F, 0x00CA, 0x0190, 0x00CB, 0x8191, 0x0001,
0x0193, 0x00CD, 0x8194, 0x00CF, 0x0196, 0x00D3, 0x0197, 0x00D1,
0x0198, 0x0001, 0x019C, 0x00D3, 0x819D, 0x00D5, 0x019F, 0x00D6,
0x81A4, 0x0001, 0x01A6, 0x00DA, 0x81A7, 0x0001, 0x01A9, 0x00DA,
0x81AC, 0x0001, 0x01AE, 0x00DA, 0x81AF, 0x0001, 0x01B2, 0x00D9,
0x81B5, 0x0001, 0x01B7, 0x00DB, 0x01B8, 0x0001, 0x01BC, 0x0001,
0x01C4, 0x0002, 0x81C5, 0x0001, 0x01C7, 0x0002, 0x81C8, 0x0001,
0x01CA, 0x0002, 0x81DB, 0x0001, 0x81EE, 0x0001, 0x01F1, 0x0002,
0x81F4, 0x0001, 0x01F6, 0xFF9F, 0x01F7, 0xFFC8, 0x821E, 0x0001,
0x8220, 0xFF7E, 0x8232, 0x0001, 0x023A, 0x2A2B, 0x823B, 0x0001,
0x023D, 0xFF5D, 0x023E, 0x2A28, 0x8241, 0x0001, 0x0243, 0xFF3D,
0x0244, 0x0045, 0x0245, 0x0047, 0x824E, 0x0001, 0x8372, 0x0001,
0x0376, 0x0001, 0x8386, 0x0026, 0x038A, 0x0025, 0x838C, 0x0040,
0x038F, 0x003F, 0x03A1, 0x0020, 0x03AB, 0x0020, 0x03CF, 0x0008,
0x83EE, 0x0001, 0x03F4, 0xFFC4, 0x83F7, 0x0001, 0x03F9, 0xFFF9,
0x03FA, 0x0001, 0x03FF, 0xFF7E, 0x040F, 0x0050, 0x042F, 0x0020,
0x8480, 0x0001, 0x84BE, 0x0001, 0x04C0, 0x000F, 0x84CD, 0x0001,
0x8526, 0x0001, 0x0556, 0x0030, 0x10C5, 0x1C60, 0x10C7, 0x1C60,
0x10CD, 0x1C60, 0x9E94, 0x0001, 0x9E9E, 0xE241, 0x9EFE, 0x0001,
0x1F0F, 0xFFF8, 0x1F1D, 0xFFF8, 0x1F2F, 0xFFF8, 0x1F3F, 0xFFF8,
0x1F4D, 0xFFF8, 0x9F5F, 0xFFF8, 0x1F6F, 0xFFF8, 0x1F8F, 0xFFF8,
0x1F9F, 0xFFF8, 0x1FAF, 0xFFF8, 0x1FB9, 0xFFF8, 0x1FBB, 0xFFB6,
0x1FBC, 0xFFF7, 0x1FCB, 0xFFAA, 0x1FCC, 0xFFF7, 0x1FD9, 0xFFF8,
0x1FDB, 0xFF9C, 0x1FE9, 0xFFF8, 0x1FEB, 0xFF90, 0x1FEC, 0xFFF9,
0x1FF9, 0xFF80, 0x1FFB, 0xFF82, 0x1FFC, 0xFFF7, 0x2126, 0xE2A3,
0x212A, 0xDF41, 0x212B, 0xDFBA, 0x2132, 0x001C, 0x216F, 0x0010,
0x2183, 0x0001, 0x24CF, 0x001A, 0x2C2E, 0x0030, 0xAC60, 0x0001,
0x2C62, 0xD609, 0x2C63, 0xF11A, 0x2C64, 0xD619, 0xAC6B, 0x0001,
0x2C6D, 0xD5E4, 0x2C6E, 0xD603, 0x2C6F, 0xD5E1, 0xAC70, 0xD5E2,
0x2C72, 0x0001, 0x2C75, 0x0001, 0x2C7F, 0xD5C1, 0xACE2, 0x0001,
0xACED, 0x0001, 0x2CF2, 0x0001, 0x266C, 0x0001, 0x2696, 0x0001,
0x272E, 0x0001, 0x276E, 0x0001, 0x277B, 0x0001, 0xA77D, 0x75FC,
0x2786, 0x0001, 0x278B, 0x0001, 0xA78D, 0x5AD8, 0x2792, 0x0001,
0x27A8, 0x0001, 0xA7AA, 0x5ABC, 0xFF3A, 0x0020
} };

typedef std::array<char16_t, 1000 - 0xB5> UpperCaseCache;
static const UpperCaseCache kUpperCaseCache = { {
0x039C, 0x00B6, 0x00B7, 0x00B8, 0x00B9, 0x00BA, 0x00BB, 0x00BC,
0x00BD, 0x00BE, 0x00BF, 0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4,
0x00C5, 0x00C6, 0x00C7, 0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC,
0x00CD, 0x00CE, 0x00CF, 0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4,
0x00D5, 0x00D6, 0x00D7, 0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC,
0x00DD, 0x00DE, 0x0000, 0x00C0, 0x00C1, 0x00C2, 0x00C3, 0x00C4,
0x00C5, 0x00C6, 0x00C7, 0x00C8, 0x00C9, 0x00CA, 0x00CB, 0x00CC,
0x00CD, 0x00CE, 0x00CF, 0x00D0, 0x00D1, 0x00D2, 0x00D3, 0x00D4,
0x00D5, 0x00D6, 0x00F7, 0x00D8, 0x00D9, 0x00DA, 0x00DB, 0x00DC,
0x00DD, 0x00DE, 0x0178, 0x0100, 0x0100, 0x0102, 0x0102, 0x0104,
0x0104, 0x0106, 0x0106, 0x0108, 0x0108, 0x010A, 0x010A, 0x010C,
0x010C, 0x010E, 0x010E, 0x0110, 0x0110, 0x0112, 0x0112, 0x0114,
0x0114, 0x0116, 0x0116, 0x0118, 0x0118, 0x011A, 0x011A, 0x011C,
0x011C, 0x011E, 0x011E, 0x0120, 0x0120, 0x0122, 0x0122, 0x0124,
0x0124, 0x0126, 0x0126, 0x0128, 0x0128, 0x012A, 0x012A, 0x012C,
0x012C, 0x012E, 0x012E, 0x0130, 0x0049, 0x0132, 0x0132, 0x0134,
0x0134, 0x0136, 0x0136, 0x0138, 0x0139, 0x0139, 0x013B, 0x013B,
0x013D, 0x013D, 0x013F, 0x013F, 0x0141, 0x0141, 0x0143, 0x0143,
0x0145, 0x0145, 0x0147, 0x0147, 0x0000, 0x014A, 0x014A, 0x014C,
0x014C, 0x014E, 0x014E, 0x0150, 0x0150, 0x0152, 0x0152, 0x0154,
0x0154, 0x0156, 0x0156, 0x0158, 0x0158, 0x015A, 0x015A, 0x015C,
0x015C, 0x015E, 0x015E, 0x0160, 0x0160, 0x0162, 0x0162, 0x0164,
0x0164, 0x0166, 0x0166, 0x0168, 0x0168, 0x016A, 0x016A, 0x016C,
0x016C, 0x016E, 0x016E, 0x0170, 0x0170, 0x0172, 0x0172, 0x0174,
0x0174, 0x0176, 0x0176, 0x0178, 0x0179, 0x0179, 0x017B, 0x017B,
0x017D, 0x017D, 0x0053, 0x0243, 0x0181, 0x0182, 0x0182, 0x0184,
0x0184, 0x0186, 0x0187, 0x0187, 0x0189, 0x018A, 0x018B, 0x018B,
0x018D, 0x018E, 0x018F, 0x0190, 0x0191, 0x0191, 0x0193, 0x0194,
0x01F6, 0x0196, 0x0197, 0x0198, 0x0198, 0x023D, 0x019B, 0x019C,
0x019D, 0x0220, 0x019F, 0x01A0, 0x01A0, 0x01A2, 0x01A2, 0x01A4,
0x01A4, 0x01A6, 0x01A7, 0x01A7, 0x01A9, 0x01AA, 0x01AB, 0x01AC,
0x01AC, 0x01AE, 0x01AF, 0x01AF, 0x01B1, 0x01B2, 0x01B3, 0x01B3,
0x01B5, 0x01B5, 0x01B7, 0x01B8, 0x01B8, 0x01BA, 0x01BB, 0x01BC,
0x01BC, 0x01BE, 0x01F7, 0x01C0, 0x01C1, 0x01C2, 0x01C3, 0x01C4,
0x01C4, 0x01C4, 0x01C7, 0x01C7, 0x01C7, 0x01CA, 0x01CA, 0x01CA,
0x01CD, 0x01CD, 0x01CF, 0x01CF, 0x01D1, 0x01D1, 0x01D3, 0x01D3,
0x01D5, 0x01D5, 0x01D7, 0x01D7, 0x01D9, 0x01D9, 0x01DB, 0x01DB,
0x018E, 0x01DE, 0x01DE, 0x01E0, 0x01E0, 0x01E2, 0x01E2, 0x01E4,
0x01E4, 0x01E6, 0x01E6, 0x01E8, 0x01E8, 0x01EA, 0x01EA, 0x01EC,
0x01EC, 0x01EE, 0x01EE, 0x0000, 0x01F1, 0x01F1, 0x01F1, 0x01F4,
0x01F4, 0x01F6, 0x01F7, 0x01F8, 0x01F8, 0x01FA, 0x01FA, 0x01FC,
0x01FC, 0x01FE, 0x01FE, 0x0200, 0x0200, 0x0202, 0x0202, 0x0204,
0x0204, 0x0206, 0x0206, 0x0208, 0x0208, 0x020A, 0x020A, 0x020C,
0x020C, 0x020E, 0x020E, 0x0210, 0x0210, 0x0212, 0x0212, 0x0214,
0x0214, 0x0216, 0x0216, 0x0218, 0x0218, 0x021A, 0x021A, 0x021C,
0x021C, 0x021E, 0x021E, 0x0220, 0x0221, 0x0222, 0x0222, 0x0224,
0x0224, 0x0226, 0x0226, 0x0228, 0x0228, 0x022A, 0x022A, 0x022C,
0x022C, 0x022E, 0x022E, 0x0230, 0x0230, 0x0232, 0x0232, 0x0234,
0x0235, 0x0236, 0x0237, 0x0238, 0x0239, 0x023A, 0x023B, 0x023B,
0x023D, 0x023E, 0x2C7E, 0x2C7F, 0x0241, 0x0241, 0x0243, 0x0244,
0x0245, 0x0246, 0x0246, 0x0248, 0x0248, 0x024A, 0x024A, 0x024C,
0x024C, 0x024E, 0x024E, 0x2C6F, 0x2C6D, 0x2C70, 0x0181, 0x0186,
0x0255, 0x0189, 0x018A, 0x0258, 0x018F, 0x025A, 0x0190, 0x025C,
0x025D, 0x025E, 0x025F, 0x0193, 0x0261, 0x0262, 0x0194, 0x0264,
0xA78D, 0xA7AA, 0x0267, 0x0197, 0x0196, 0x026A, 0x2C62, 0x026C,
0x026D, 0x026E, 0x019C, 0x0270, 0x2C6E, 0x019D, 0x0273, 0x0274,
0x019F, 0x0276, 0x0277, 0x0278, 0x0279, 0x027A, 0x027B, 0x027C,
0x2C64, 0x027E, 0x027F, 0x01A6, 0x0281, 0x0282, 0x01A9, 0x0284,
0x0285, 0x0286, 0x0287, 0x01AE, 0x0244, 0x01B1, 0x01B2, 0x0245,
0x028D, 0x028E, 0x028F, 0x0290, 0x0291, 0x01B7, 0x0293, 0x0294,
0x0295, 0x0296, 0x0297, 0x0298, 0x0299, 0x029A, 0x029B, 0x029C,
0x029D, 0x029E, 0x029F, 0x02A0, 0x02A1, 0x02A2, 0x02A3, 0x02A4,
0x02A5, 0x02A6, 0x02A7, 0x02A8, 0x02A9, 0x02AA, 0x02AB, 0x02AC,
0x02AD, 0x02AE, 0x02AF, 0x02B0, 0x02B1, 0x02B2, 0x02B3, 0x02B4,
0x02B5, 0x02B6, 0x02B7, 0x02B8, 0x02B9, 0x02BA, 0x02BB, 0x02BC,
0x02BD, 0x02BE, 0x02BF, 0x02C0, 0x02C1, 0x02C2, 0x02C3, 0x02C4,
0x02C5, 0x02C6, 0x02C7, 0x02C8, 0x02C9, 0x02CA, 0x02CB, 0x02CC,
0x02CD, 0x02CE, 0x02CF, 0x02D0, 0x02D1, 0x02D2, 0x02D3, 0x02D4,
0x02D5, 0x02D6, 0x02D7, 0x02D8, 0x02D9, 0x02DA, 0x02DB, 0x02DC,
0x02DD, 0x02DE, 0x02DF, 0x02E0, 0x02E1, 0x02E2, 0x02E3, 0x02E4,
0x02E5, 0x02E6, 0x02E7, 0x02E8, 0x02E9, 0x02EA, 0x02EB, 0x02EC,
0x02ED, 0x02EE, 0x02EF, 0x02F0, 0x02F1, 0x02F2, 0x02F3, 0x02F4,
0x02F5, 0x02F6, 0x02F7, 0x02F8, 0x02F9, 0x02FA, 0x02FB, 0x02FC,
0x02FD, 0x02FE, 0x02FF, 0x0300, 0x0301, 0x0302, 0x0303, 0x0304,
0x0305, 0x0306, 0x0307, 0x0308, 0x0309, 0x030A, 0x030B, 0x030C,
0x030D, 0x030E, 0x030F, 0x0310, 0x0311, 0x0312, 0x0313, 0x0314,
0x0315, 0x0316, 0x0317, 0x0318, 0x0319, 0x031A, 0x031B, 0x031C,
0x031D, 0x031E, 0x031F, 0x0320, 0x0321, 0x0322, 0x0323, 0x0324,
0x0325, 0x0326, 0x0327, 0x0328, 0x0329, 0x032A, 0x032B, 0x032C,
0x032D, 0x032E, 0x032F, 0x0330, 0x0331, 0x0332, 0x0333, 0x0334,
0x0335, 0x0336, 0x0337, 0x0338, 0x0339, 0x033A, 0x033B, 0x033C,
0x033D, 0x033E, 0x033F, 0x0340, 0x0341, 0x0342, 0x0343, 0x0344,
0x0399, 0x0346, 0x0347, 0x0348, 0x0349, 0x034A, 0x034B, 0x034C,
0x034D, 0x034E, 0x034F, 0x0350, 0x0351, 0x0352, 0x0353, 0x0354,
0x0355, 0x0356, 0x0357, 0x0358, 0x0359, 0x035A, 0x035B, 0x035C,
0x035D, 0x035E, 0x035F, 0x0360, 0x0361, 0x0362, 0x0363, 0x0364,
0x0365, 0x0366, 0x0367, 0x0368, 0x0369, 0x036A, 0x036B, 0x036C,
0x036D, 0x036E, 0x036F, 0x0370, 0x0370, 0x0372, 0x0372, 0x0374,
0x0375, 0x0376, 0x0376, 0x0378, 0x0379, 0x037A, 0x03FD, 0x03FE,
0x03FF, 0x037E, 0x037F, 0x0380, 0x0381, 0x0382, 0x0383, 0x0384,
0x0385, 0x0386, 0x0387, 0x0388, 0x0389, 0x038A, 0x038B, 0x038C,
0x038D, 0x038E, 0x038F, 0x0000, 0x0391, 0x0392, 0x0393, 0x0394,
0x0395, 0x0396, 0x0397, 0x0398, 0x0399, 0x039A, 0x039B, 0x039C,
0x039D, 0x039E, 0x039F, 0x03A0, 0x03A1, 0x03A2, 0x03A3, 0x03A4,
0x03A5, 0x03A6, 0x03A7, 0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x0386,
0x0388, 0x0389, 0x038A, 0x0000, 0x0391, 0x0392, 0x0393, 0x0394,
0x0395, 0x0396, 0x0397, 0x0398, 0x0399, 0x039A, 0x039B, 0x039C,
0x039D, 0x039E, 0x039F, 0x03A0, 0x03A1, 0x03A3, 0x03A3, 0x03A4,
0x03A5, 0x03A6, 0x03A7, 0x03A8, 0x03A9, 0x03AA, 0x03AB, 0x038C,
0x038E, 0x038F, 0x03CF, 0x0392, 0x0398, 0x03D2, 0x03D3, 0x03D4,
0x03A6, 0x03A0, 0x03CF, 0x03D8, 0x03D8, 0x03DA, 0x03DA, 0x03DC,
0x03DC, 0x03DE, 0x03DE, 0x03E0, 0x03E0, 0x03E2, 0x03E2, 0x03E4,
0x03E4, 0x03E6, 0x03E6
} };

typedef std::array<char16_t, 260> UpperCaseKeys;
static const UpperCaseKeys kUpperCaseKeys = { {
0x0061, 0x00B5, 0x00DF, 0x00E0, 0x00F8, 0x00FF, 0x0101, 0x0131,
0x0133, 0x013A, 0x0149, 0x014B, 0x017A, 0x017F, 0x0180, 0x0183,
0x0188, 0x018C, 0x0192, 0x0195, 0x0199, 0x019A, 0x019E, 0x01A1,
0x01A8, 0x01AD, 0x01B0, 0x01B4, 0x01B9, 0x01BD, 0x01BF, 0x01C5,
0x01C6, 0x01C8, 0x01C9, 0x01CB, 0x01CC, 0x01CE, 0x01DD, 0x01DF,
0x01F0, 0x01F2, 0x01F3, 0x01F5, 0x01F9, 0x0223, 0x023C, 0x023F,
0x0242, 0x0247, 0x0250, 0x0251, 0x0252, 0x0253, 0x0254, 0x0256,
0x0259, 0x025B, 0x0260, 0x0263, 0x0265, 0x0266, 0x0268, 0x0269,
0x026B, 0x026F, 0x0271, 0x0272, 0x0275, 0x027D, 0x0280, 0x0283,
0x0288, 0x0289, 0x028A, 0x028C, 0x0292, 0x0345, 0x0371, 0x0377,
0x037B, 0x0390, 0x03AC, 0x03AD, 0x03B0, 0x03B1, 0x03C2, 0x03C3,
0x03CC, 0x03CD, 0x03D0, 0x03D1, 0x03D5, 0x03D6, 0x03D7, 0x03D9,
0x03F0, 0x03F1, 0x03F2, 0x03F5, 0x03F8, 0x03FB, 0x0430, 0x0450,
0x0461, 0x048B, 0x04C2, 0x04CF, 0x04D1, 0x0561, 0x0587, 0x1D79,
0x1D7D, 0x1E01, 0x1E96, 0x1E97, 0x1E98, 0x1E99, 0x1E9A, 0x1E9B,
0x1EA1, 0x1F00, 0x1F10, 0x1F20, 0x1F30, 0x1F40, 0x1F50, 0x1F51,
0x1F52, 0x1F53, 0x1F54, 0x1F55, 0x1F56, 0x1F57, 0x1F60, 0x1F70,
0x1F72, 0x1F76, 0x1F78, 0x1F7A, 0x1F7C, 0x1F80, 0x1F81, 0x1F82,
0x1F83, 0x1F84, 0x1F85, 0x1F86, 0x1F87, 0x1F88, 0x1F89, 0x1F8A,
0x1F8B, 0x1F8C, 0x1F8D, 0x1F8E, 0x1F8F, 0x1F90, 0x1F91, 0x1F92,
0x1F93, 0x1F94, 0x1F95, 0x1F96, 0x1F97, 0x1F98, 0x1F99, 0x1F9A,
0x1F9B, 0x1F9C, 0x1F9D, 0x1F9E, 0x1F9F, 0x1FA0, 0x1FA1, 0x1FA2,
0x1FA3, 0x1FA4, 0x1FA5, 0x1FA6, 0x1FA7, 0x1FA8, 0x1FA9, 0x1FAA,
0x1FAB, 0x1FAC, 0x1FAD, 0x1FAE, 0x1FAF, 0x1FB0, 0x1FB2, 0x1FB3,
0x1FB4, 0x1FB6, 0x1FB7, 0x1FBC, 0x1FBE, 0x1FC2, 0x1FC3, 0x1FC4,
0x1FC6, 0x1FC7, 0x1FCC, 0x1FD0, 0x1FD2, 0x1FD3, 0x1FD6, 0x1FD7,
0x1FE0, 0x1FE2, 0x1FE3, 0x1FE4, 0x1FE5, 0x1FE6, 0x1FE7, 0x1FF2,
0x1FF3, 0x1FF4, 0x1FF6, 0x1FF7, 0x1FFC, 0x214E, 0x2170, 0x2184,
0x24D0, 0x2C30, 0x2C61, 0x2C65, 0x2C66, 0x2C68, 0x2C73, 0x2C76,
0x2C81, 0x2CEC, 0x2CF3, 0x2D00, 0x2D27, 0x2D2D, 0xA641, 0xA681,
0xA723, 0xA733, 0xA77A, 0xA77F, 0xA78C, 0xA791, 0xA7A1, 0xFB00,
0xFB01, 0xFB02, 0xFB03, 0xFB04, 0xFB05, 0xFB06, 0xFB13, 0xFB14,
0xFB15, 0xFB16, 0xFB17, 0xFF41
} };

typedef std::array<char16_t, 520> UpperCaseValues;
static const UpperCaseValues kUpperCaseValues = { {
0x007A, 0xFFE0, 0x00B5, 0x02E7, 0x0005, 0x0000, 0x00F6, 0xFFE0,
0x00FE, 0xFFE0, 0x80FF, 0x0079, 0x812F, 0xFFFF, 0x8131, 0xFF18,
0x8137, 0xFFFF, 0x8148, 0xFFFF, 0x0014, 0x0000, 0x8177, 0xFFFF,
0x817E, 0xFFFF, 0x017F, 0xFED4, 0x0180, 0x00C3, 0x8185, 0xFFFF,
0x0188, 0xFFFF, 0x018C, 0xFFFF, 0x0192, 0xFFFF, 0x0195, 0x0061,
0x0199, 0xFFFF, 0x019A, 0x00A3, 0x019E, 0x0082, 0x81A5, 0xFFFF,
0x01A8, 0xFFFF, 0x01AD, 0xFFFF, 0x01B0, 0xFFFF, 0x81B6, 0xFFFF,
0x01B9, 0xFFFF, 0x81BD, 0xFFFF, 0x01BF, 0x0038, 0x01C5, 0xFFFF,
0x81C6, 0xFFFE, 0x01C8, 0xFFFF, 0x81C9, 0xFFFE, 0x01CB, 0xFFFF,
0x81CC, 0xFFFE, 0x81DC, 0xFFFF, 0x81DD, 0xFFB1, 0x81EF, 0xFFFF,
0x0060, 0x0000, 0x01F2, 0xFFFF, 0x81F3, 0xFFFE, 0x01F5, 0xFFFF,
0x821F, 0xFFFF, 0x8233, 0xFFFF, 0x023C, 0xFFFF, 0x0240, 0x2A3F,
0x0242, 0xFFFF, 0x824F, 0xFFFF, 0x0250, 0x2A1F, 0x0251, 0x2A1C,
0x0252, 0x2A1E, 0x0253, 0xFF2E, 0x8254, 0xFF32, 0x0257, 0xFF33,
0x8259, 0xFF36, 0x025B, 0xFF35, 0x0260, 0xFF33, 0x8263, 0xFF31,
0x0265, 0xA528, 0x8266, 0xA544, 0x0268, 0xFF2F, 0x8269, 0xFF2D,
0x026B, 0x29F7, 0x826F, 0xFF2D, 0x0271, 0x29FD, 0x0272, 0xFF2B,
0x0275, 0xFF2A, 0x027D, 0x29E7, 0x0280, 0xFF26, 0x0283, 0xFF26,
0x0288, 0xFF26, 0x0289, 0xFFBB, 0x028B, 0xFF27, 0x028C, 0xFFB9,
0x0292, 0xFF25, 0x0345, 0x0054, 0x8373, 0xFFFF, 0x0377, 0xFFFF,
0x037D, 0x0082, 0x002A, 0x0000, 0x03AC, 0xFFDA, 0x03AF, 0xFFDB,
0x004A, 0x0000, 0x03C1, 0xFFE0, 0x03C2, 0xFFE1, 0x03CB, 0xFFE0,
0x03CC, 0xFFC0, 0x03CE, 0xFFC1, 0x03D0, 0xFFC2, 0x03D1, 0xFFC7,
0x03D5, 0xFFD1, 0x03D6, 0xFFCA, 0x83D7, 0xFFF8, 0x83EF, 0xFFFF,
0x03F0, 0xFFAA, 0x03F1, 0xFFB0, 0x03F2, 0x0007, 0x03F5, 0xFFA0,
0x03F8, 0xFFFF, 0x03FB, 0xFFFF, 0x044F, 0xFFE0, 0x045F, 0xFFB0,
0x8481, 0xFFFF, 0x84BF, 0xFFFF, 0x84CE, 0xFFFF, 0x84CF, 0xFFF1,
0x8527, 0xFFFF, 0x0586, 0xFFD0, 0x0021, 0x0000, 0x1D79, 0x8A04,
0x1D7D, 0x0EE6, 0x9E95, 0xFFFF, 0x0000, 0x0000, 0x0001, 0x0000,
0x0002, 0x0000, 0x0003, 0x0000, 0x0004, 0x0000, 0x1E9B, 0xFFC5,
0x9EFF, 0xFFFF, 0x1F07, 0x0008, 0x1F15, 0x0008, 0x1F27, 0x0008,
0x1F37, 0x0008, 0x1F45, 0x0008, 0x0015, 0x0000, 0x1F51, 0x0008,
0x0016, 0x0000, 0x1F53, 0x0008, 0x0017, 0x0000, 0x1F55, 0x0008,
0x0018, 0x0000, 0x1F57, 0x0008, 0x1F67, 0x0008, 0x1F71, 0x004A,
0x1F75, 0x0056, 0x1F77, 0x0064, 0x1F79, 0x0080, 0x1F7B, 0x0070,
0x1F7D, 0x007E, 0x001A, 0x0000, 0x001B, 0x0000, 0x001C, 0x0000,
0x001D, 0x0000, 0x001E, 0x0000, 0x001F, 0x0000, 0x0020, 0x0000,
0x0012, 0x0000, 0x0022, 0x0000, 0x0023, 0x0000, 0x0024, 0x0000,
0x0025, 0x0000, 0x0026, 0x0000, 0x0027, 0x0000, 0x0028, 0x0000,
0x0029, 0x0000, 0x0019, 0x0000, 0x002B, 0x0000, 0x002C, 0x0000,
0x002D, 0x0000, 0x002E, 0x0000, 0x002F, 0x0000, 0x0030, 0x0000,
0x0031, 0x0000, 0x0032, 0x0000, 0x0033, 0x0000, 0x0034, 0x0000,
0x0035, 0x0000, 0x0036, 0x0000, 0x0037, 0x0000, 0x0038, 0x0000,
0x0039, 0x0000, 0x003A, 0x0000, 0x003B, 0x0000, 0x003C, 0x0000,
0x003D, 0x0000, 0x003E, 0x0000, 0x003F, 0x0000, 0x0040, 0x0000,
0x0041, 0x0000, 0x0042, 0x0000, 0x0043, 0x0000, 0x0044, 0x0000,
0x0045, 0x0000, 0x0046, 0x0000, 0x0047, 0x0000, 0x0048, 0x0000,
0x0049, 0x0000, 0x1FB1, 0x0008, 0x004B, 0x0000, 0x004C, 0x0000,
0x004D, 0x0000, 0x004E, 0x0000, 0x004F, 0x0000, 0x0050, 0x0000,
0x1FBE, 0xE3DB, 0x0051, 0x0000, 0x0052, 0x0000, 0x0053, 0x0000,
0x0054, 0x0000, 0x0055, 0x0000, 0x0056, 0x0000, 0x1FD1, 0x0008,
0x0057, 0x0000, 0x0058, 0x0000, 0x0059, 0x0000, 0x005A, 0x0000,
0x1FE1, 0x0008, 0x005B, 0x0000, 0x005C, 0x0000, 0x005D, 0x0000,
0x1FE5, 0x0007, 0x005E, 0x0000, 0x005F, 0x0000, 0x0061, 0x0000,
0x0062, 0x0000, 0x0063, 0x0000, 0x0064, 0x0000, 0x0065, 0x0000,
0x0066, 0x0000, 0x214E, 0xFFE4, 0x217F, 0xFFF0, 0x2184, 0xFFFF,
0x24E9, 0xFFE6, 0x2C5E, 0xFFD0, 0x2C61, 0xFFFF, 0x2C65, 0xD5D5,
0xAC66, 0xD5D8, 0xAC6C, 0xFFFF, 0x2C73, 0xFFFF, 0x2C76, 0xFFFF,
0xACE3, 0xFFFF, 0xACEE, 0xFFFF, 0x2CF3, 0xFFFF, 0x2D25, 0xE3A0,
0x2D27, 0xE3A0, 0x2D2D, 0xE3A0, 0x266D, 0xFFFF, 0x2697, 0xFFFF,
0x272F, 0xFFFF, 0x276F, 0xFFFF, 0x277C, 0xFFFF, 0x2787, 0xFFFF,
0xA78C, 0xFFFF, 0x2793, 0xFFFF, 0x27A9, 0xFFFF, 0x0006, 0x0000,
0x0007, 0x0000, 0x0008, 0x0000, 0x0009, 0x0000, 0x000A, 0x0000,
0x000B, 0x0000, 0x000C, 0x0000, 0x000D, 0x0000, 0x000E, 0x0000,
0x000F, 0x0000, 0x0010, 0x0000, 0x0011, 0x0000, 0xFF5A, 0xFFE0
} };

inline uint32_t GetCategory(char16_t c) {
  if (c < 1000) {
    return kCategoryCache[c];
  }
  const int result =
      static_cast<int>(std::upper_bound(
              kCategoryKeys.begin(),
              kCategoryKeys.end(), c) - kCategoryKeys.begin() - 1);
  assert(result < static_cast<int>(kCategoryKeys.size()));
  const int high = kCategoryValues[result * 2];
  if (c <= high) {
    const int code = kCategoryValues[result * 2 + 1];
    if (code < 0x100) {
      return code;
    }
    return ((c & 1) ? code >> 8 : code & 0xFF);
  }
  return UNASSIGNED;
}

inline bool IsASCII(char16_t c) {
  return !(c & ~0x7F);
}

inline bool IsASCIIAlpha(char16_t c) {
  return (c | 0x20) >= 'a' && (c | 0x20) <= 'z';
}

inline bool IsDecimalDigit(char16_t c) {
  return c >= '0' && c <= '9';
}

inline bool IsOctalDigit(char16_t c) {
  return c >= '0' && c <= '7';
}

inline bool IsHexDigit(char16_t c) {
  return IsDecimalDigit(c) || ((c | 0x20) >= 'a' && (c | 0x20) <= 'f');
}

inline bool IsDigit(char16_t c) {
  return IsDecimalDigit(c) ||
      (c > 1631 && GetCategory(c) == DECIMAL_DIGIT_NUMBER);
}

inline bool IsASCIIAlphanumeric(char16_t c) {
  return IsDecimalDigit(c) || IsASCIIAlpha(c);
}

inline bool IsNonASCIIIdentifierStart(char16_t c) {
  return (1U << GetCategory(c)) & (Lu | Ll | Lt | Lm | Lo | Nl);
}

inline bool IsNonASCIIIdentifierPart(char16_t c) {
  return
      ((1U << GetCategory(c)) &
       (Lu | Ll | Lt | Lm | Lo | Mn | Mc | Nd | Pc | Nl)) ||
      c == code::ZWNJ || c == code::ZWJ;
}

inline bool IsSeparatorSpace(char16_t c) {
  return GetCategory(c) == SPACE_SEPARATOR;
}

inline bool IsWhiteSpace(char16_t c) {
  return IsASCII(c) ?
      (c == ' ' || c == '\t' || c == 0xB || c == 0xC) :
      c == 0x00A0 || c == 0xFEFF || IsSeparatorSpace(c);
}

inline bool IsLineTerminator(char16_t c) {
  return
      c == code::CR ||
      c == code::LF ||
      (c & ~1) == 0x2028;  // 0x2028 or 0x2029
}

inline bool IsLineOrParagraphSeparator(char16_t c) {
  return (c & ~1) == 0x2028;  // 0x2028 or 0x2029
}

inline bool IsIdentifierStart(char16_t c) {
  return IsASCII(c) ? c == '$'  || c == '_' ||
                      c == '\\' || IsASCIIAlpha(c) :
                      IsNonASCIIIdentifierStart(c);
}

inline bool IsIdentifierPart(char16_t c) {
  return IsASCII(c) ? c == '$'  || c == '_' ||
                      c == '\\' || IsASCIIAlphanumeric(c) :
                      IsNonASCIIIdentifierPart(c);
}

inline char16_t ToASCIILowerCase(char16_t c) {
  assert(IsASCII(c));
  if ('A' <= c && c <= 'Z') {
    return c + ('a' - 'A');
  }
  return c;
}

inline char16_t ToASCIIUpperCase(char16_t c) {
  assert(IsASCII(c));
  if ('a' <= c && c <= 'z') {
    return c - ('a' - 'A');
  }
  return c;
}

inline uint64_t ToLowerCase(char16_t c) {
  if ('A' <= c && c <= 'Z') {
    return c + ('a' - 'A');
  }

  // lowercase not found until 0xC0
  if (c < 0xC0) {
    return c;
  }

  // suppress compiler warnings
  {
    const std::size_t index = c - 0xC0;
    if (index < kLowerCaseCache.size()) {
      assert(index < kLowerCaseCache.size());
      const char16_t res = kLowerCaseCache[index];
      if (res != 0) {
        return res;
      }
    }
  }

  LowerCaseKeys::const_iterator it =
      std::upper_bound(kLowerCaseKeys.begin(),
                       kLowerCaseKeys.end(), c) - 1;
  const int result = static_cast<int>(it - kLowerCaseKeys.begin());
  assert(result < static_cast<int>(kLowerCaseKeys.size()));
  if (result >= 0) {
    bool by2 = false;
    const char16_t start = kLowerCaseKeys[result];
    char16_t end = kLowerCaseValues[result * 2];
    const char16_t mapping = kLowerCaseValues[result * 2 + 1];
    if (mapping == 0) {
      // special casing
      if (start == c) {
        return kSpecialCasing[end];
      }
    } else {
      if ((start & 0x8000) != (end & 0x8000)) {
          end ^= 0x8000;
          by2 = true;
      }
      if (c <= end) {
        if (by2 && (c & 1) != (start & 1)) {
          return c;
        }
        return static_cast<char16_t>(c + mapping);
      }
    }
  }
  return c;
}

// if target character is removed, return code::REMOVE
inline uint64_t ToLocaleLowerCase(locale::Locale locale,
                                  char16_t c,
                                  int prev = code::DEFAULT,
                                  int next = code::DEFAULT) {
  switch (locale) {
    case locale::EN: {
      break;
    }

    case locale::LT: {
      //  # Introduce an explicit dot above when lowercasing capital I's and J's
      //  # whenever there are more accents above.
      //  # (of the accents used in Lithuanian: grave, acute, tilde above, and ogonek)
      //
      //  0049; 0069 0307; 0049; 0049; lt More_Above; # LATIN CAPITAL LETTER I
      //  004A; 006A 0307; 004A; 004A; lt More_Above; # LATIN CAPITAL LETTER J
      //  012E; 012F 0307; 012E; 012E; lt More_Above; # LATIN CAPITAL LETTER I WITH OGONEK
      //  00CC; 0069 0307 0300; 00CC; 00CC; lt; # LATIN CAPITAL LETTER I WITH GRAVE
      //  00CD; 0069 0307 0301; 00CD; 00CD; lt; # LATIN CAPITAL LETTER I WITH ACUTE
      //  0128; 0069 0307 0303; 0128; 0128; lt; # LATIN CAPITAL LETTER I WITH TILDE
      switch (c) {
        case 0x0049: {
          return 0x00690307;
        }
        case 0x004A: {
          return 0x006A0307;
        }
        case 0x012E: {
          return 0x012F0307;
        }
        case 0x00CC: {
          return UINT64_C(0x006903070300);
        }
        case 0x00CD: {
          return UINT64_C(0x006903070301);
        }
        case 0x0128: {
          return UINT64_C(0x006903070303);
        }
      }
      break;
    }

    case locale::TR:
    case locale::AZ: {
      switch (c) {
        //  # I and i-dotless; I-dot and i are case pairs in Turkish and Azeri
        //  # The following rules handle those cases.
        //
        //  0130; 0069; 0130; 0130; tr; # LATIN CAPITAL LETTER I WITH DOT ABOVE
        //  0130; 0069; 0130; 0130; az; # LATIN CAPITAL LETTER I WITH DOT ABOVE
        case 0x0130: {
          return 0x0069;
        }

        //  # When lowercasing, remove dot_above in the sequence I + dot_above, which will turn into i.
        //  # This matches the behavior of the canonically equivalent I-dot_above
        //
        //  0307; ; 0307; 0307; tr After_I; # COMBINING DOT ABOVE
        //  0307; ; 0307; 0307; az After_I; # COMBINING DOT ABOVE
        case 0x0307: {
          if (prev == 'I') {
            return code::REMOVE;
          }
          break;
        }

        //  # When lowercasing, unless an I is before a dot_above, it turns into a dotless i.
        //
        //  0049; 0131; 0049; 0049; tr Not_Before_Dot; # LATIN CAPITAL LETTER I
        //  0049; 0131; 0049; 0049; az Not_Before_Dot; # LATIN CAPITAL LETTER I
        case 0x0049: {
          if (next == 0x0307) {
            return code::REMOVE;
          }
          break;
        }
      }
      break;
    }
  }
  return ToLowerCase(c);
}

inline uint64_t ToUpperCase(char16_t c) {
  if ('a' <= c && c <= 'z') {
    return c - ('a' - 'A');
  }
  // uppercase not found until 0xB5
  if (c < 0xB5) {
    return c;
  }

  // suppress compiler warnings
  {
    const std::size_t index = c - 0xB5;
    if (index < kUpperCaseCache.size()) {
      assert(index < kUpperCaseCache.size());
      const char16_t res = kUpperCaseCache[index];
      if (res != 0) {
        return res;
      }
    }
  }

  UpperCaseKeys::const_iterator it =
      std::upper_bound(kUpperCaseKeys.begin(),
                       kUpperCaseKeys.end(), c) - 1;
  const int result = static_cast<int>(it - kUpperCaseKeys.begin());
  assert(result < static_cast<int>(kUpperCaseKeys.size()));
  if (result >= 0) {
    bool by2 = false;
    const char16_t start = *it;
    char16_t end = kUpperCaseValues[result * 2];
    const char16_t mapping = kUpperCaseValues[result * 2 + 1];
    if (mapping == 0) {
      // special casing
      if (start == c) {
        return kSpecialCasing[end];
      }
    } else {
      if ((start & 0x8000) != (end & 0x8000)) {
        end ^= 0x8000;
        by2 = true;
      }
      if (c <= end) {
        if (by2 && (c & 1) != (start & 1)) {
          return c;
        }
        return static_cast<char16_t>(c + mapping);
      }
    }
  }
  return c;
}

// if target character is removed, return code::REMOVE
// prev is character that is not uppercased yet.
inline uint64_t ToLocaleUpperCase(locale::Locale locale,
                                  char16_t c,
                                  int prev = code::DEFAULT,
                                  int next = code::DEFAULT) {
  switch (locale) {
    case locale::EN: {
      break;
    }

    case locale::LT: {
      //  # Remove DOT ABOVE after "i" with upper or titlecase
      //  0307; 0307; ; ; lt After_Soft_Dotted; # COMBINING DOT ABOVE
      if (c == 0x0307 && prev == 'i') {
        return code::REMOVE;
      }
      break;
    }

    case locale::TR:
    case locale::AZ: {
      //  # When uppercasing, i turns into a dotted capital I
      //  0069; 0069; 0130; 0130; tr; # LATIN SMALL LETTER I
      //  0069; 0069; 0130; 0130; az; # LATIN SMALL LETTER I
      if (c == 0x0069) {
        return 0x0130;
      }
      break;
    }
  }
  return ToUpperCase(c);
}

struct ContainsNonASCIICharacter {
  template<typename CharT>
  bool operator()(CharT ch) const {
    return !IsASCII(ch);
  }
};

template<typename Iter>
inline bool IsASCII(Iter it, Iter last) {
  return std::find_if(it, last, ContainsNonASCIICharacter()) == last;
}

}  // namespace iv::core::character
} }  // namespace iv::core
#endif  // IV_CHARACTER_H_
