/*
 * Decompiled with CFR 0.152.
 */
package org.jreleaser.sdk.signing;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.Provider;
import java.security.Security;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Optional;
import org.bouncycastle.bcpg.ArmoredOutputStream;
import org.bouncycastle.bcpg.BCPGOutputStream;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPCompressedData;
import org.bouncycastle.openpgp.PGPCompressedDataGenerator;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPPrivateKey;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPSecretKey;
import org.bouncycastle.openpgp.PGPSignature;
import org.bouncycastle.openpgp.PGPSignatureGenerator;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.PGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.PGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentSignerBuilder;
import org.bouncycastle.openpgp.operator.jcajce.JcaPGPContentVerifierBuilderProvider;
import org.bouncycastle.openpgp.operator.jcajce.JcePBESecretKeyDecryptorBuilder;
import org.jreleaser.bundle.RB;
import org.jreleaser.model.Signing;
import org.jreleaser.model.api.JReleaserContext;
import org.jreleaser.model.api.signing.Keyring;
import org.jreleaser.model.api.signing.Signing;
import org.jreleaser.model.api.signing.SigningException;
import org.jreleaser.sdk.command.CommandException;
import org.jreleaser.sdk.signing.GpgCommandSigner;
import org.jreleaser.util.StringUtils;

public final class SigningUtils {
    private SigningUtils() {
    }

    public static Optional<String> getPublicKeyID(JReleaserContext context) throws SigningException {
        if (context.getModel().getSigning().getMode() != Signing.Mode.COMMAND && context.getModel().getSigning().getMode() != Signing.Mode.COSIGN) {
            Keyring keyring = context.createKeyring();
            return Optional.of(Long.toHexString(keyring.readPublicKey().getKeyID()));
        }
        return Optional.empty();
    }

    public static Optional<Instant> getExpirationDateOfPublicKey(JReleaserContext context) throws SigningException {
        if (context.getModel().getSigning().getMode() != Signing.Mode.COMMAND && context.getModel().getSigning().getMode() != Signing.Mode.COSIGN) {
            PGPPublicKey publicKey = context.createKeyring().readPublicKey();
            if (publicKey.getValidSeconds() <= 0L) {
                return Optional.of(Instant.EPOCH);
            }
            return Optional.of(Instant.ofEpochMilli(publicKey.getCreationTime().getTime()).plus((long)publicKey.getValidDays(), ChronoUnit.DAYS));
        }
        return Optional.empty();
    }

    public static void sign(JReleaserContext context, Path file) throws SigningException {
        if (context.getModel().getSigning().getMode() == Signing.Mode.COMMAND) {
            SigningUtils.cmdSign(context, file);
        } else if (context.getModel().getSigning().getMode() != Signing.Mode.COSIGN) {
            SigningUtils.bcSign(context, file);
        }
    }

    private static void cmdSign(JReleaserContext context, Path input) throws SigningException {
        FilePair pair = SigningUtils.checkInput(context, input);
        if (pair.isValid()) {
            return;
        }
        SigningUtils.sign(context, pair);
        if (context.getModel().getSigning().isVerify()) {
            SigningUtils.verify(context, pair);
        } else {
            context.getLogger().debug(RB.$((String)"signing.verify.disabled", (Object[])new Object[0]));
        }
    }

    private static void bcSign(JReleaserContext context, Path input) throws SigningException {
        Keyring keyring = context.createKeyring();
        FilePair pair = SigningUtils.checkInput(context, input);
        if (pair.isValid()) {
            return;
        }
        SigningUtils.sign(context, keyring, pair);
        if (context.getModel().getSigning().isVerify()) {
            SigningUtils.verify(context, keyring, pair);
        } else {
            context.getLogger().debug(RB.$((String)"signing.verify.disabled", (Object[])new Object[0]));
        }
    }

    private static FilePair checkInput(JReleaserContext context, Path input) {
        Signing signing = context.getModel().getSigning();
        String extension = ".sig";
        if (signing.getMode() != Signing.Mode.COSIGN) {
            extension = signing.isArmored() ? ".asc" : ".sig";
        }
        Path output = input.getParent().resolve(input.getFileName().toString().concat(extension));
        FilePair pair = new FilePair(input, output);
        pair.setValid(SigningUtils.isValid(context, pair));
        return pair;
    }

    public static boolean verify(JReleaserContext context, Keyring keyring, FilePair filePair) throws SigningException {
        context.getLogger().setPrefix("verify");
        context.getLogger().debug("{}", new Object[]{context.relativizeToBasedir(filePair.signatureFile)});
        try {
            boolean bl;
            block20: {
                InputStream sigInputStream = PGPUtil.getDecoderStream((InputStream)new BufferedInputStream(Files.newInputStream(filePair.signatureFile, new OpenOption[0])));
                try {
                    PGPObjectFactory pgpObjFactory = new PGPObjectFactory(sigInputStream, keyring.getKeyFingerPrintCalculator());
                    Iterable pgpSigList = null;
                    Object obj = pgpObjFactory.nextObject();
                    if (obj instanceof PGPCompressedData) {
                        PGPCompressedData c1 = (PGPCompressedData)obj;
                        pgpObjFactory = new PGPObjectFactory(c1.getDataStream(), keyring.getKeyFingerPrintCalculator());
                        pgpSigList = (Iterable)pgpObjFactory.nextObject();
                    } else {
                        pgpSigList = (Iterable)obj;
                    }
                    PGPSignature sig = (PGPSignature)pgpSigList.iterator().next();
                    try (BufferedInputStream fileInputStream = new BufferedInputStream(Files.newInputStream(filePair.inputFile, new OpenOption[0]));){
                        int ch;
                        PGPPublicKey pubKey = keyring.readPublicKey();
                        sig.init((PGPContentVerifierBuilderProvider)new JcaPGPContentVerifierBuilderProvider().setProvider("BC"), pubKey);
                        while ((ch = ((InputStream)fileInputStream).read()) >= 0) {
                            sig.update((byte)ch);
                        }
                    }
                    bl = sig.verify();
                    if (sigInputStream == null) break block20;
                }
                catch (Throwable throwable) {
                    try {
                        if (sigInputStream != null) {
                            try {
                                sigInputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException | PGPException e) {
                        throw new SigningException(RB.$((String)"ERROR_signing_verify_signature", (Object[])new Object[]{context.relativizeToBasedir(filePair.inputFile)}), e);
                    }
                }
                sigInputStream.close();
            }
            return bl;
        }
        finally {
            context.getLogger().restorePrefix();
        }
    }

    public static boolean verify(JReleaserContext context, FilePair filePair) throws SigningException {
        context.getLogger().setPrefix("verify");
        try {
            context.getLogger().debug("{}", new Object[]{context.relativizeToBasedir(filePair.signatureFile)});
            GpgCommandSigner commandSigner = SigningUtils.initCommandSigner(context);
            boolean bl = commandSigner.verify(filePair.signatureFile, filePair.inputFile);
            return bl;
        }
        catch (CommandException e) {
            throw new SigningException(RB.$((String)"ERROR_signing_verify_signature", (Object[])new Object[]{context.relativizeToBasedir(filePair.inputFile)}), (Throwable)e);
        }
        finally {
            context.getLogger().restorePrefix();
        }
    }

    private static void sign(JReleaserContext context, FilePair pair) throws SigningException {
        GpgCommandSigner commandSigner = SigningUtils.initCommandSigner(context);
        SigningUtils.sign(context, commandSigner, pair.inputFile, pair.signatureFile);
    }

    public static GpgCommandSigner initCommandSigner(JReleaserContext context) {
        GpgCommandSigner cmd = new GpgCommandSigner(context.getLogger());
        Signing signing = context.getModel().getSigning();
        cmd.setExecutable(signing.getCommand().getExecutable());
        cmd.setPassphrase(signing.getPassphrase());
        cmd.setHomeDir(signing.getCommand().getHomeDir());
        cmd.setKeyName(signing.getCommand().getKeyName());
        cmd.setPublicKeyring(signing.getCommand().getPublicKeyring());
        cmd.setDefaultKeyring(signing.getCommand().isDefaultKeyring());
        cmd.setArgs(signing.getCommand().getArgs());
        return cmd;
    }

    public static void sign(JReleaserContext context, GpgCommandSigner commandSigner, Path input, Path output) throws SigningException {
        try {
            context.getLogger().info("{}", new Object[]{context.relativizeToBasedir(input)});
            commandSigner.sign(input, output);
        }
        catch (CommandException e) {
            throw new SigningException(RB.$((String)"ERROR_unexpected_error_signing", (Object[])new Object[]{input.toAbsolutePath()}), (Throwable)e);
        }
    }

    private static void sign(JReleaserContext context, Keyring keyring, FilePair pair) throws SigningException {
        PGPSignatureGenerator signatureGenerator = SigningUtils.initSignatureGenerator(context, keyring);
        SigningUtils.sign(context, signatureGenerator, pair.inputFile, pair.signatureFile);
    }

    public static PGPSignatureGenerator initSignatureGenerator(JReleaserContext context, Keyring keyring) throws SigningException {
        Signing signing = context.getModel().getSigning();
        if (context.isDryrun() && StringUtils.isBlank((String)signing.getPassphrase())) {
            return null;
        }
        try {
            PGPSecretKey pgpSecretKey = keyring.readSecretKey();
            PGPPrivateKey pgpPrivKey = pgpSecretKey.extractPrivateKey(new JcePBESecretKeyDecryptorBuilder().setProvider("BC").build(signing.getPassphrase().toCharArray()));
            PGPSignatureGenerator signatureGenerator = new PGPSignatureGenerator((PGPContentSignerBuilder)new JcaPGPContentSignerBuilder(pgpSecretKey.getPublicKey().getAlgorithm(), 2).setProvider("BC"));
            signatureGenerator.init(0, pgpPrivKey);
            return signatureGenerator;
        }
        catch (PGPException e) {
            throw new SigningException(RB.$((String)"ERROR_unexpected_error_signature_gen", (Object[])new Object[0]), (Throwable)e);
        }
    }

    public static void sign(JReleaserContext context, PGPSignatureGenerator signatureGenerator, Path input, Path output) throws SigningException {
        context.getLogger().info("{}", new Object[]{context.relativizeToBasedir(input)});
        if (null == signatureGenerator) {
            return;
        }
        PGPCompressedDataGenerator compressionStreamGenerator = new PGPCompressedDataGenerator(0);
        try (OutputStream out = SigningUtils.createOutputStream(context, output);
             BCPGOutputStream bOut = new BCPGOutputStream(compressionStreamGenerator.open(out));
             InputStream in = Files.newInputStream(input, new OpenOption[0]);){
            byte[] buffer = new byte[8192];
            int length = 0;
            while ((length = in.read(buffer)) >= 0) {
                signatureGenerator.update(buffer, 0, length);
            }
            signatureGenerator.generate().encode((OutputStream)bOut);
            compressionStreamGenerator.close();
        }
        catch (IOException | PGPException e) {
            throw new SigningException(RB.$((String)"ERROR_unexpected_error_signing", (Object[])new Object[]{input.toAbsolutePath()}), e);
        }
    }

    private static OutputStream createOutputStream(JReleaserContext context, Path output) throws IOException {
        BufferedOutputStream out = new BufferedOutputStream(Files.newOutputStream(output.toFile().toPath(), new OpenOption[0]));
        if (context.getModel().getSigning().isArmored()) {
            out = new ArmoredOutputStream((OutputStream)out);
        }
        return out;
    }

    public static boolean isValid(JReleaserContext context, FilePair pair) {
        if (Files.notExists(pair.getSignatureFile(), new LinkOption[0])) {
            context.getLogger().debug(RB.$((String)"signing.signature.not.exist", (Object[])new Object[0]), new Object[]{context.relativizeToBasedir(pair.getSignatureFile())});
            return false;
        }
        if (pair.inputFile.toFile().lastModified() > pair.signatureFile.toFile().lastModified()) {
            context.getLogger().debug(RB.$((String)"signing.file.newer", (Object[])new Object[0]), new Object[]{context.relativizeToBasedir(pair.inputFile), context.relativizeToBasedir(pair.signatureFile)});
            return false;
        }
        try {
            if (context.getModel().getSigning().isVerify()) {
                return SigningUtils.verify(context, pair);
            }
            return false;
        }
        catch (SigningException e) {
            return false;
        }
    }

    public static boolean isValid(JReleaserContext context, Keyring keyring, FilePair pair) {
        if (null == keyring) {
            return SigningUtils.isValid(context, pair);
        }
        if (Files.notExists(pair.getSignatureFile(), new LinkOption[0])) {
            context.getLogger().debug(RB.$((String)"signing.signature.not.exist", (Object[])new Object[0]), new Object[]{context.relativizeToBasedir(pair.getSignatureFile())});
            return false;
        }
        if (pair.inputFile.toFile().lastModified() > pair.signatureFile.toFile().lastModified()) {
            context.getLogger().debug(RB.$((String)"signing.file.newer", (Object[])new Object[0]), new Object[]{context.relativizeToBasedir(pair.inputFile), context.relativizeToBasedir(pair.signatureFile)});
            return false;
        }
        try {
            if (context.getModel().getSigning().isVerify()) {
                return SigningUtils.verify(context, keyring, pair);
            }
            return false;
        }
        catch (SigningException e) {
            return false;
        }
    }

    static {
        Provider bcProvider = Security.getProvider("BC");
        Security.removeProvider("BC");
        Security.setProperty("crypto.policy", "unlimited");
        Security.addProvider(null != bcProvider ? bcProvider : new BouncyCastleProvider());
    }

    public static class FilePair {
        private final Path inputFile;
        private final Path signatureFile;
        private boolean valid;

        public FilePair(Path inputFile, Path signatureFile) {
            this.inputFile = inputFile;
            this.signatureFile = signatureFile;
        }

        public Path getInputFile() {
            return this.inputFile;
        }

        public Path getSignatureFile() {
            return this.signatureFile;
        }

        public boolean isValid() {
            return this.valid;
        }

        public void setValid(boolean valid) {
            this.valid = valid;
        }

        public boolean isInvalid() {
            return !this.valid;
        }
    }
}

