/******************************************************************************
    (c) 2002 Matthew Fredette                 fredette@netbsd.org

    Some modifications:
    (c) 2003-2004 Christine Caulfield           christine.caulfield@googlemail.com

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
******************************************************************************/

#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <syslog.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/uio.h>
#include <assert.h>

#include <net/if.h>
#include <net/bpf.h>
#ifdef HAVE_NET_IF_ETHER_H
#include <net/if_ether.h>
#endif /* HAVE_NET_IF_ETHER_H */
#ifdef HAVE_NET_ETHERNET_H
#include <net/ethernet.h>
#endif /* HAVE_NET_ETHERNET_H */
#ifdef HAVE_NET_IF_DL_H
#include <net/if_dl.h>
#endif
#include <netinet/in.h>

#include <string>

#include "utils.h"
#define _LATD_INTERFACES_IMPL
#include "interfaces.h"
#include "interfaces-bpf.h"

LATinterfaces *LATinterfaces::Create()
{
    return new BPFInterfaces();
}

/* this macro helps us size a struct ifreq: */
#ifdef HAVE_SOCKADDR_SA_LEN
#define SIZEOF_IFREQ(ifr) (sizeof(ifr->ifr_name) + ifr->ifr_addr.sa_len)
#else  /* !HAVE_SOCKADDR_SA_LEN */
#define SIZEOF_IFREQ(ifr) (sizeof(ifr->ifr_name) + sizeof(struct sockaddr))
#endif /* !HAVE_SOCKADDR_SA_LEN */

#define LATD_OFFSETOF(t, m) (((char *) &(((t *) NULL)-> m)) - ((char *) ((t *) NULL)))

/* the BPF program to capture LAT packets: */
static struct bpf_insn lat_bpf_filter[] = {

  /* drop this packet if its ethertype isn't ETHERTYPE_LAT: */
  BPF_STMT(BPF_LD + BPF_H + BPF_ABS, LATD_OFFSETOF(struct ether_header, ether_type)),
  BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_LAT, 0, 1),

  /* accept this packet: */
  BPF_STMT(BPF_RET + BPF_K, (u_int) -1),

  /* drop this packet: */
  BPF_STMT(BPF_RET + BPF_K, 0),
};

/* the BPF program to capture MOP RC packets: */
static struct bpf_insn moprc_bpf_filter[] = {

  /* drop this packet if its ethertype isn't ETHERTYPE_MOPRC: */
  BPF_STMT(BPF_LD + BPF_H + BPF_ABS, LATD_OFFSETOF(struct ether_header, ether_type)),
  BPF_JUMP(BPF_JMP + BPF_JEQ + BPF_K, ETHERTYPE_MOPRC, 0, 1),

  /* accept this packet: */
  BPF_STMT(BPF_RET + BPF_K, (u_int) -1),

  /* drop this packet: */
  BPF_STMT(BPF_RET + BPF_K, 0),
};

int BPFInterfaces::Start(int proto)
{
#define DEV_BPF_FORMAT "/dev/bpf%d"
    char dev_bpf_filename[sizeof(DEV_BPF_FORMAT) + (sizeof(int) * 3) + 1];
    int minor;
    int saved_errno;
    u_int bpf_opt;
    struct bpf_version version;

    protocol = proto;

    /* loop trying to open a /dev/bpf device: */
    for(minor = 0;; minor++) {

      /* form the name of the next device to try, then try opening it.
	 if we succeed, we're done: */
      sprintf(dev_bpf_filename, DEV_BPF_FORMAT, minor);
      debuglog(("bpf: trying %s\n", dev_bpf_filename));
      if ((_latd_bpf_fd = open(dev_bpf_filename, O_RDWR)) >= 0) {
	debuglog(("bpf: opened %s as %d\n", dev_bpf_filename, _latd_bpf_fd));
	break;
      }

      /* we failed to open this device.  if this device was simply busy,
	 loop: */
      debuglog(("bpf: failed to open %s: %s\n", dev_bpf_filename, strerror(errno)));
      if (errno == EBUSY) {
	continue;
      }

      /* otherwise, we have failed: */
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* this macro helps in closing the BPF socket on error: */
#define _LATD_RAW_OPEN_ERROR(x) saved_errno = errno; x; errno = saved_errno

    /* check the BPF version: */
    if (ioctl(_latd_bpf_fd, BIOCVERSION, &version) < 0) {
      debuglog(("bpf: failed to get the BPF version on %s: %s\n",
		   dev_bpf_filename, strerror(errno)));
      _LATD_RAW_OPEN_ERROR(close(_latd_bpf_fd));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }
    if (version.bv_major != BPF_MAJOR_VERSION
	|| version.bv_minor < BPF_MINOR_VERSION) {
      debuglog(("bpf: kernel BPF version is %d.%d, my BPF version is %d.%d\n",
		   version.bv_major, version.bv_minor,
		   BPF_MAJOR_VERSION, BPF_MINOR_VERSION));
      close(_latd_bpf_fd);
      errno = ENXIO;
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* put the BPF device into immediate mode: */
    bpf_opt = true;
    if (ioctl(_latd_bpf_fd, BIOCIMMEDIATE, &bpf_opt) < 0) {
      debuglog(("bpf: failed to put %s into immediate mode: %s\n",
		   dev_bpf_filename, strerror(errno)));
      _LATD_RAW_OPEN_ERROR(close(_latd_bpf_fd));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }
#ifndef __APPLE__
    /* tell the BPF device we're providing complete Ethernet headers: */
    bpf_opt = true;
    if (ioctl(_latd_bpf_fd, BIOCSHDRCMPLT, &bpf_opt) < 0) {
      debuglog(("bpf: failed to put %s into complete-headers mode: %s\n",
		   dev_bpf_filename, strerror(errno)));
      _LATD_RAW_OPEN_ERROR(close(_latd_bpf_fd));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }
#endif
    /* done: */
    return 0;
#undef _LATD_RAW_OPEN_ERROR
}

// Return a list of valid interface numbers and the count
void BPFInterfaces::get_all_interfaces(int *ifs, int &num)
{
    num = 0;

    if (find_interface(NULL) == 0)
    {
	ifs[num++] = 0;
    }

}

// Print the name of an interface
std::string BPFInterfaces::ifname(int ifn)
{
    std::string str;

    if (ifn == 0 && _latd_bpf_interface_name != NULL) {
      str.append(_latd_bpf_interface_name);
      str.append(" ");
    }

    return str;
}

// Find an interface number by name
int BPFInterfaces::find_interface(char *ifname)
{
  int saved_errno;
  int dummy_fd;
  char ifreq_buffer[16384];  /* FIXME - magic constant. */
  struct ifconf ifc;
  struct ifreq *ifr;
  struct ifreq *ifr_user;
  size_t ifr_offset;
  struct sockaddr_in saved_ip_address;
  short	saved_flags;
#ifdef HAVE_AF_LINK
  struct ifreq *link_ifreqs[20]; /* FIXME - magic constant. */
  size_t link_ifreqs_count;
  size_t link_ifreqs_i;
  struct sockaddr_dl *sadl;
#endif /* HAVE_AF_LINK */

  /* if we have already chosen an interface, ignore this one: */
  if (_latd_bpf_interface_name != NULL) {
    return (-1);
  }

  /* make a dummy socket so we can read the interface list: */
  if ((dummy_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    return (-1);
  }

  /* read the interface list: */
  ifc.ifc_len = sizeof(ifreq_buffer);
  ifc.ifc_buf = ifreq_buffer;
  if (ioctl(dummy_fd, SIOCGIFCONF, &ifc) < 0) {
    saved_errno = errno;
    close(dummy_fd);
    errno = saved_errno;
    return (-1);
  }

#ifdef HAVE_AF_LINK
  /* start our list of link address ifreqs: */
  link_ifreqs_count = 0;
#endif /* HAVE_AF_LINK */

  /* walk the interface list: */
  ifr_user = NULL;
  for(ifr_offset = 0;; ifr_offset += SIZEOF_IFREQ(ifr)) {


    /* stop walking if we have run out of space in the buffer.  note
       that before we can use SIZEOF_IFREQ, we have to make sure that
       there is a minimum number of bytes in the buffer to use it
       (namely, that there's a whole struct sockaddr available): */
    ifr = (struct ifreq *) (ifreq_buffer + ifr_offset);
    if ((ifr_offset + sizeof(ifr->ifr_name) + sizeof(struct sockaddr)) > (size_t)ifc.ifc_len
	|| (ifr_offset + SIZEOF_IFREQ(ifr)) > (size_t)ifc.ifc_len) {
      errno = ENOENT;
      break;
    }

#ifdef HAVE_AF_LINK
    /* if this is a hardware address, save it: */
    if (ifr->ifr_addr.sa_family == AF_LINK) {
      if (link_ifreqs_count < (sizeof(link_ifreqs) / sizeof(link_ifreqs[0]))) {
	link_ifreqs[link_ifreqs_count++] = ifr;
      }
      continue;
    }
#endif /* HAVE_AF_LINK */

    /* ignore this interface if it doesn't do IP: */
    if (ifr->ifr_addr.sa_family != AF_INET) {
      continue;
    }

    /* get the interface flags, preserving the IP address in the
       struct ifreq across the call: */
    saved_ip_address = *((struct sockaddr_in *) &ifr->ifr_addr);
    if (ioctl(dummy_fd, SIOCGIFFLAGS, ifr) < 0) {
      ifr = NULL;
      break;
    }
    saved_flags = ifr->ifr_flags;
    *((struct sockaddr_in *) &ifr->ifr_addr) = saved_ip_address;

    /* ignore this interface if it isn't up and running: */
    if ((saved_flags & (IFF_UP | IFF_RUNNING)) != (IFF_UP | IFF_RUNNING)) {
      continue;
    }

    /* if we don't have an interface yet, take this one depending on
       whether the user asked for an interface by name or not.  if he
       did, and this is it, take this one.  if he didn't, and this isn't a
       loopback interface, take this one: */
    if (ifr_user == NULL
	&& (ifname != NULL
	    ? !strncmp(ifr->ifr_name, ifname, sizeof(ifr->ifr_name))
	    : !(ifr->ifr_flags & IFF_LOOPBACK))) {
      ifr_user = ifr;
#ifdef __APPLE__
      if (ifr->ifr_name[0] == 'l' && ifr->ifr_name[1] == 'o')
          ifr_user = NULL;
#endif
    }
  }

  /* close the dummy socket: */
  saved_errno = errno;
  close(dummy_fd);
  errno = saved_errno;

  /* if we don't have an interface to return: */
  if (ifr_user == NULL) {
    return (-1);
  }

#ifdef HAVE_AF_LINK

  /* we must be able to find an AF_LINK ifreq that gives us the
     interface's Ethernet address. */
  ifr = NULL;
  for(link_ifreqs_i = 0; link_ifreqs_i < link_ifreqs_count; link_ifreqs_i++) {
    if (!strncmp(link_ifreqs[link_ifreqs_i]->ifr_name,
		 ifr_user->ifr_name,
		 sizeof(ifr_user->ifr_name))) {
      ifr = link_ifreqs[link_ifreqs_i];
      break;
    }
  }
  if (ifr == NULL) {
    return (-1);
  }

  /* copy out the Ethernet address: */
  sadl = (struct sockaddr_dl *) &ifr->ifr_addr;
  memcpy(_latd_bpf_interface_addr, LLADDR(sadl), sadl->sdl_alen);

#else  /* !HAVE_AF_LINK */
#error "must have AF_LINK for now"
#endif /* !HAVE_AF_LINK */

  /* remember this single interface name: */
  if ((_latd_bpf_interface_name = strdup(ifr_user->ifr_name)) == NULL) {
    abort();
  }

  /* this zero is a fake interface index: */
  debuglog(("bpf: interface Ethernet address is %02x:%02x:%02x:%02x:%02x:%02x\n",
	    _latd_bpf_interface_addr[0],
	    _latd_bpf_interface_addr[1],
	    _latd_bpf_interface_addr[2],
	    _latd_bpf_interface_addr[3],
	    _latd_bpf_interface_addr[4],
	    _latd_bpf_interface_addr[5]));
  return (0);
}

// true if this class defines one FD for each active
// interface, false if one fd is used for all interfaces.
bool BPFInterfaces::one_fd_per_interface()
{
    return false;
}

// Return the FD for this interface (will only be called once for
// select if above returns false)
int BPFInterfaces::get_fd(int ifn)
{
    return _latd_bpf_fd;
}

// Send a packet to a given macaddr
int BPFInterfaces::send_packet(int ifn, unsigned char macaddr[], unsigned char *data, int len)
{
  struct ether_header ether_packet;
  struct iovec iov[2];

  /* we only support one interface: */
  assert(ifn == 0);

  /* make the Ethernet header: */
  memcpy(ether_packet.ether_dhost, macaddr, ETHER_ADDR_LEN);
  memcpy(ether_packet.ether_shost, _latd_bpf_interface_addr, ETHER_ADDR_LEN);
  ether_packet.ether_type = htons(protocol);

  /* write this packet: */
  iov[0].iov_base = (char* )&ether_packet;
  iov[0].iov_len = sizeof(ether_packet);
  iov[1].iov_base = (char *)data;
  iov[1].iov_len = len;
  if (writev(_latd_bpf_fd, iov, 2) < 0) {
    syslog(LOG_ERR, "writev: %m");
    return -1;
  }

  return 0;
}


#define CONT_OR_RET() \
    if (_latd_bpf_buffer_offset < _latd_bpf_buffer_end) {\
        continue;\
    } else {\
        if (_latd_bpf_buffer_offset_next <= _latd_bpf_buffer_end) \
          more = true; \
        return 0; \
    }

// Receive a packet from a given interface
int BPFInterfaces::recv_packet(int sockfd, int &ifn, unsigned char macaddr[], unsigned char *data, int maxlen, bool &more)
{
  ssize_t buffer_end;
  struct bpf_hdr the_bpf_header;
  unsigned int _latd_bpf_buffer_offset_next;

  more = false;

  /* loop until we have something to return: */
  for(;;) {

      debuglog(("buffer_offset=%d, buffer_end=%d\n", _latd_bpf_buffer_offset,_latd_bpf_buffer_end));
    /* if the buffer is empty, fill it: */
    if (_latd_bpf_buffer_offset
	>= _latd_bpf_buffer_end) {

	// Is there really anything to read? we really don't want to
	// block here.
	int bytes;
	if (ioctl(sockfd, FIONREAD, &bytes) == 0 &&
	    bytes == 0) {
	    return 0;
	}
      /* read the BPF socket: */
      debuglog(("bpf: calling read\n"));
      buffer_end = read(sockfd,
			_latd_bpf_buffer,
			_latd_bpf_buffer_size);
      if (buffer_end <= 0) {
	if (errno == EAGAIN) return 0;
	debuglog(("bpf: failed to read packets: %s\n", strerror(errno)));
	return (-1);
      }
      debuglog(("bpf: read %d bytes of packets\n", buffer_end));
      _latd_bpf_buffer_offset = 0;
      _latd_bpf_buffer_end = buffer_end;
    }

    /* if there's not enough for a BPF header, flush the buffer: */
    if ((_latd_bpf_buffer_offset
	 + sizeof(the_bpf_header))
	> _latd_bpf_buffer_end) {
      debuglog(("bpf: flushed garbage BPF header bytes\n"));
      _latd_bpf_buffer_end = 0;
      return 0;
    }

    /* get the BPF header and check it: */
    memcpy(&the_bpf_header,
	   _latd_bpf_buffer
	   + _latd_bpf_buffer_offset,
	   sizeof(the_bpf_header));
    _latd_bpf_buffer_offset_next = _latd_bpf_buffer_offset +
      BPF_WORDALIGN(the_bpf_header.bh_hdrlen + the_bpf_header.bh_caplen);
    _latd_bpf_buffer_offset += the_bpf_header.bh_hdrlen;

    /* if we're missing some part of the packet: */
    if (the_bpf_header.bh_caplen != the_bpf_header.bh_datalen
	|| ((_latd_bpf_buffer_offset + the_bpf_header.bh_datalen)
	    > _latd_bpf_buffer_end)) {
      debuglog(("bpf: flushed truncated BPF packet (caplen %d, datalen %d, offset %d, end %d\n",
		the_bpf_header.bh_caplen, the_bpf_header.bh_datalen,
		_latd_bpf_buffer_offset, _latd_bpf_buffer_end));
      _latd_bpf_buffer_offset = _latd_bpf_buffer_offset_next;
      CONT_OR_RET();
    }

    /* silently ignore packets that don't even have Ethernet headers,
       and those packets that we transmitted: */
    if (the_bpf_header.bh_datalen < sizeof(struct ether_header)
	|| !memcmp(((struct ether_header *)
		    (_latd_bpf_buffer
		     + _latd_bpf_buffer_offset))->ether_shost,
		   _latd_bpf_interface_addr,
		   ETHER_ADDR_LEN)) {
      /* silently ignore packets from us: */
	if (the_bpf_header.bh_datalen < sizeof(struct ether_header))
	    debuglog(("ignoring short packet of %d bytes\n", the_bpf_header.bh_datalen));
	else
	    debuglog(("ignoring packet from us\n"));
      _latd_bpf_buffer_offset = _latd_bpf_buffer_offset_next;
      CONT_OR_RET();
    }
    debuglog(("bpf: packet from %02x:%02x:%02x:%02x:%02x:%02x\n",
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[0],
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[1],
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[2],
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[3],
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[4],
	      ((struct ether_header *) (_latd_bpf_buffer + _latd_bpf_buffer_offset))->ether_shost[5]));

    /* if the caller hasn't provided a large enough buffer: */
    if (maxlen < 0 || (unsigned int)maxlen < the_bpf_header.bh_datalen) {
      errno = EIO;
      _latd_bpf_buffer_offset = _latd_bpf_buffer_offset_next;
      return (-1);
    }

    /* return this captured packet to the user.  the caller doesn't
       want the Ethernet header as part of the packet, but he does
       want the source address. */
    memcpy(data,
	   _latd_bpf_buffer
	   + _latd_bpf_buffer_offset
	   + sizeof(struct ether_header),
	   the_bpf_header.bh_datalen
	   - sizeof(struct ether_header));
    ifn = 0;
    memcpy(macaddr,
	   ((struct ether_header *) (_latd_bpf_buffer
				     + _latd_bpf_buffer_offset))->ether_shost,
	   ETHER_ADDR_LEN);
    _latd_bpf_buffer_offset = _latd_bpf_buffer_offset_next;
    if (_latd_bpf_buffer_offset < _latd_bpf_buffer_end)
	more = true;
    return (the_bpf_header.bh_datalen
	    - sizeof(struct ether_header));
  }
  /* NOTREACHED */
}

/* This is the LAT multicast address */
static void store_lat_multicast(unsigned char *addr)
{
    addr[0]  = 0x09;
    addr[1]  = 0x00;
    addr[2]  = 0x2b;
    addr[3]  = 0x00;
    addr[4]  = 0x00;
    addr[5]  = 0x0f;
}


// Open a connection on an interface
int BPFInterfaces::set_lat_multicast(int ifn)
{
    struct ifreq interface_ifreq;
    struct bpf_program program;
    int dummy_fd;
#ifdef HAVE_AF_LINK
    struct sockaddr_dl *sockdl;
#endif

    memset(&interface_ifreq, 0, sizeof(interface_ifreq));

    /* if we don't have an interface, bail: */
    if (ifn != 0 || _latd_bpf_interface_name == NULL) {
      syslog(LOG_ERR, "No interfaces\n");
      return -1;
    }
    strcpy(interface_ifreq.ifr_name, _latd_bpf_interface_name);

    /* point the BPF device at the interface we're using: */
    if (ioctl(_latd_bpf_fd, BIOCSETIF, &interface_ifreq) < 0) {
      debuglog(("bpf: failed to point BPF socket at %s: %s\n",
		   interface_ifreq.ifr_name, strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* set the filter on the BPF device: */
    program.bf_len = sizeof(lat_bpf_filter) / sizeof(lat_bpf_filter[0]);
    program.bf_insns = lat_bpf_filter;
    if (ioctl(_latd_bpf_fd, BIOCSETF, &program) < 0) {
      debuglog(("bpf: failed to set the filter: %s\n", strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* get the BPF read buffer size: */
    if (ioctl(_latd_bpf_fd, BIOCGBLEN, &_latd_bpf_buffer_size) < 0) {
      debuglog(("bpf: failed to read the buffer size: %s\n", strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }
    debuglog(("bpf: buffer size is %u\n", _latd_bpf_buffer_size));

    /* allocate the buffer for BPF reads: */
    if ((_latd_bpf_buffer = (unsigned char *) malloc(_latd_bpf_buffer_size)) == NULL) {
      abort();
    }
    _latd_bpf_buffer_end = _latd_bpf_buffer_offset = 0;

    // Add Multicast membership for LAT on socket

    /* make a dummy socket so we can manipulate an interface: */
    if ((dummy_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
      syslog(LOG_ERR, "Can't create a dummy socket: %m\n");
      return -1;
    }

#ifdef HAVE_AF_LINK
    sockdl = (sockaddr_dl *)&interface_ifreq.ifr_addr;
    sockdl->sdl_family = AF_LINK;
    sockdl->sdl_alen = 6;
    store_lat_multicast((unsigned char *)sockdl->sdl_data);
#else
    interface_ifreq.ifr_addr.sa_family = AF_UNSPEC;
    store_lat_multicast((unsigned char *)interface_ifreq.ifr_addr.sa_data);
#endif
#ifdef HAVE_SOCKADDR_SA_LEN
    interface_ifreq.ifr_addr.sa_len = sizeof(interface_ifreq.ifr_addr);
#endif /* HAVE_SOCKADDR_SA_LEN */


    if (ioctl(dummy_fd, SIOCADDMULTI, &interface_ifreq) < 0) {
      debuglog(("bpf: failed to add to the multicast list for interface for %s: %s\n",
		interface_ifreq.ifr_name, strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      close(dummy_fd);
      return -1;
    }

    /* close the dummy socket: */
    close(dummy_fd);

    return 0;
}

// Close an interface.
int BPFInterfaces::remove_lat_multicast(int ifn)
{
    struct ifreq interface_ifreq;
    int dummy_fd;
#ifdef HAVE_AF_LINK
    struct sockaddr_dl *sockdl;
#endif

    memset(&interface_ifreq, 0, sizeof(interface_ifreq));

    /* if we don't have an interface, bail: */
    if (ifn != 0 || _latd_bpf_interface_name == NULL) {
      syslog(LOG_ERR, "No interfaces\n");
      return -1;
    }
    strcpy(interface_ifreq.ifr_name, _latd_bpf_interface_name);

    // Delete Multicast membership for LAT on socket

    /* make a dummy socket so we can manipulate an interface: */
    if ((dummy_fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
      syslog(LOG_ERR, "Can't create a dummy socket: %m\n");
      return -1;
    }

#ifdef HAVE_AF_LINK
    sockdl = (sockaddr_dl *)&interface_ifreq.ifr_addr;
    sockdl->sdl_family = AF_LINK;
    sockdl->sdl_alen = 6;
    store_lat_multicast((unsigned char *)sockdl->sdl_data);
#else
    interface_ifreq.ifr_addr.sa_family = AF_UNSPEC;
    store_lat_multicast((unsigned char *)interface_ifreq.ifr_addr.sa_data);
#endif
#ifdef HAVE_SOCKADDR_SA_LEN
    interface_ifreq.ifr_addr.sa_len = sizeof(interface_ifreq.ifr_addr);
#endif /* HAVE_SOCKADDR_SA_LEN */

    if (ioctl(dummy_fd, SIOCDELMULTI, &interface_ifreq) < 0) {
      debuglog(("bpf: failed to delete from the multicast list for interface for %s: %s\n",
		interface_ifreq.ifr_name, strerror(errno)));
      syslog(LOG_ERR, "can't remove socket multicast: %m\n");
      close(dummy_fd);
      return -1;
    }

    /* close the dummy socket: */
    close(dummy_fd);

    return 0;
}

int BPFInterfaces::bind_socket(int ifn)
{
    struct ifreq interface_ifreq;
    struct bpf_program program;

    /* if we don't have an interface, bail: */
    if (ifn != 0 || _latd_bpf_interface_name == NULL) {
      syslog(LOG_ERR, "No interfaces\n");
      return -1;
    }
    strcpy(interface_ifreq.ifr_name, _latd_bpf_interface_name);

    /* point the BPF device at the interface we're using: */
    if (ioctl(_latd_bpf_fd, BIOCSETIF, &interface_ifreq) < 0) {
      debuglog(("bpf: failed to point BPF socket at %s: %s\n",
		   interface_ifreq.ifr_name, strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* set the filter on the BPF device: */
    program.bf_len = sizeof(moprc_bpf_filter) / sizeof(moprc_bpf_filter[0]);
    program.bf_insns = moprc_bpf_filter;
    if (ioctl(_latd_bpf_fd, BIOCSETF, &program) < 0) {
      debuglog(("bpf: failed to set the filter: %s\n", strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }

    /* get the BPF read buffer size: */
    if (ioctl(_latd_bpf_fd, BIOCGBLEN, &_latd_bpf_buffer_size) < 0) {
      debuglog(("bpf: failed to read the buffer size: %s\n", strerror(errno)));
      syslog(LOG_ERR, "Can't create LAT protocol socket: %m\n");
      return -1;
    }
    debuglog(("bpf: buffer size is %u\n", _latd_bpf_buffer_size));

    /* allocate the buffer for BPF reads: */
    if ((_latd_bpf_buffer = (unsigned char *) malloc(_latd_bpf_buffer_size)) == NULL) {
      abort();
    }
    _latd_bpf_buffer_end = _latd_bpf_buffer_offset = 0;
    return 0;
}

int LATinterfaces::ProtoLAT = ETHERTYPE_LAT;
int LATinterfaces::ProtoMOP = ETHERTYPE_MOPRC;
