/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
 
#ifndef WIN32_NO_CONFIG_H
#include "../config.h"
#else
#include "../winconf.h"
#endif
#include "DSList.h"
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

/*
 * Construct an empty list.
 */
DSList::DSList()
{
	head=NULL;
	setTail(NULL);
	numElements=0;
}

/*
 * Destroy list.
 */
DSList::~DSList()
{
	close();
}


/*
 * Construct an empty list.
 */
void DSList::init()
{
	head=NULL;
	setTail(NULL);
	numElements=0;
}

/*
 * Append an item to the *tail* of the list.
 */
void DSList::append (DSListElement * node)
{
	DSListElement* oldTail;
	node->setNext(NULL);
	oldTail=getTail();
	if (oldTail!=NULL) {
		oldTail->setNext(node);
	} else {
		setHead(node);
	}
	setTail(node);
	numElements++;
}

/*
 * Append an item to the *tail* of the list.
 */
void DSList::append (char * key, void * value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,NULL,cleanup);
	append(node);
}

/*
 * Append an item to the *tail* of the list.
 */
void DSList::append (unsigned int key, void * value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,NULL,cleanup);
	append(node);
}

/*
 * Append an item to the *tail* of the list.
 */
void DSList::append (char * key, unsigned int value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,NULL,cleanup);
	append(node);
}

/*
 * Append an item to the *tail* of the list.
 */
void DSList::append (unsigned int key, unsigned int value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,NULL,cleanup);
	append(node);
}


/*
 * Close a list.
 */
void DSList::close ()
{
	DSListElement *current,*next;
	current=getHead();
	setHead(NULL);
	setTail(NULL);
	while (current!=NULL) {
		next=current->getNext();
		delete current;
		current=next;
	}
	numElements=0;
}


/*
 * Delete an arbitary element from the list.  Internal use only.
 */
void DSList::rawDelete (DSListElement * temp, DSListElement * prev)
{
	if (!prev) {
		setHead(temp->getNext());
	} else {
		prev->setNext(temp->getNext());
	}
	if (temp->getNext()==NULL) {
		setTail(prev);
	}
	delete temp;
	numElements--;
}

/*
 * Delete the specified element.
 */
void DSList::delElement (DSListElement * ele)
{
	DSListElement * temp;
	DSListElement* prev;
	prev=NULL;
	if (!ele) return;
	for (temp=getHead();temp!=NULL;temp=temp->getNext())
	{
		if (temp==ele) {
			rawDelete(temp,prev);
			return;
		}
		prev=temp;
	}
}

/*
 * Delete the head of the list, advancing the head to the next element.
 */
void DSList::delHead ()
{
	delElement(getHead());
}

/*
 * Delete the tail of the list, advancing the head to the next element.
 */
void DSList::delTail ()
{
	delElement(getTail());
}

void DSList::delValueInt (unsigned int Key)
{
	delItem(Key);
}

void DSList::delValueString (const char* Key)
{
	delItem(Key);
}

void DSList::delItem (unsigned int Key)
{
	DSListElement* node;
	DSListElement* prev;
	DSListElement* next;
	prev=NULL;
	for (node=getHead();node!=NULL;node=next) {
		next=node->getNext();
		if (node->getKeyInt()==Key) {
			rawDelete(node, prev);
			return;
		}
		prev=node;
	}
}

void DSList::delItem (const char* Key)
{
	DSListElement* node;
	DSListElement* prev;
	DSListElement* next;
	prev=NULL;
	for (node=getHead();node!=NULL;node=next) {
		next=node->getNext();
		if (!strcmp(node->getKeyString(),Key)) {
			rawDelete(node, prev);
			return;
		}
		prev=node;
	}
}

/*
 * Return the element that is at the beginning of the list.
 */
DSListElement* DSList::getHead ()
{
	return head;
}

/*
 * Return the element that is at the tail of the list.
 */
DSListElement* DSList::getTail ()
{
	return tail;
}

DSListElement* DSList::getElement (unsigned int Key)
{
	DSListElement* node;
	for (node=getHead();node!=NULL;node=node->getNext()) {
		if (node->getKeyInt()==Key) return node;
	}
	return(NULL);
}

DSListElement* DSList::getElement (const char* Key)
{
	DSListElement* node;
	for (node=getHead();node!=NULL;node=node->getNext()) {
		if (!strcmp(node->getKeyString(),Key)) return node;
	}
	return(NULL);
}

DSListElement* DSList::getElementInt (unsigned int Key)
{
	return getElement(Key);
}

DSListElement* DSList::getElementString (const char* Key)
{
	return getElement(Key);
}

void* DSList::getPtrValue (const char* Key)
{
	DSListElement* node;
	node=getElement(Key);
	if (node) return node->getDataPtr();
	return(NULL);
}

void* DSList::getPtrValue (unsigned int Key)
{
	DSListElement* node;
	node=getElement(Key);
	if (node) return node->getDataPtr();
	return(NULL);
}

void* DSList::getValueString (const char* Key)
{
	return getPtrValue(Key);
}

void* DSList::getValueInt (unsigned int Key)
{
	return getPtrValue(Key);
}

unsigned int DSList::getNumericValue (const char* Key)
{
	DSListElement* node;
	node=getElement(Key);
	if (node) return node->getDataInt();
	return(0);
}

unsigned int DSList::getNumericValue (unsigned int Key)
{
	DSListElement* node;
	node=getElement(Key);
	if (node) return node->getDataInt();
	return(0);
}

unsigned int DSList::getIntString (const char* Key)
{
	return getNumericValue(Key);
}

unsigned int DSList::getIntInt (unsigned int Key)
{
	return getNumericValue(Key);
}

BOOL DSList::setValue (const char* Key, void * data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setDataPtr(data);
	return(TRUE);
}

BOOL DSList::setValue (const char* Key, unsigned int data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setDataInt(data);
	return(TRUE);
}

BOOL DSList::setValue (unsigned int Key, void * data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setDataPtr(data);
	return(TRUE);
}

BOOL DSList::setValue (unsigned int Key, unsigned int data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setDataInt(data);
	return(TRUE);
}


BOOL DSList::setKeyValue (const char* Key, unsigned int data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setKeyInt(data);
	return(TRUE);
}

BOOL DSList::setKeyValue (const char* Key, char * data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setKeyString(data);
	return(TRUE);
}

BOOL DSList::setKeyValue (unsigned int Key, unsigned int data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setKeyInt(data);
	return(TRUE);
}

BOOL DSList::setKeyValue (unsigned int Key, char * data)
{
	DSListElement* node;
	node=getElement(Key);
	if (!node) return FALSE;
	if (node) 
		node->setKeyString(data);
	return(TRUE);
}

void DSList::insert (DSListElement * node)
{
	node->setNext(getHead());
	setHead(node);
	numElements++;
}

void DSList::inSort (char * key, void * value, int cleanup)
{
	DSListElement* node;
	DSListElement* tmp;
	DSListElement* prev=NULL;
	for (tmp=getHead();tmp&&(strcmp(tmp->getKeyString(),key)<0);
			prev=tmp,tmp=tmp->getNext());
	node=new DSListElement(key,value,tmp,cleanup);
	if (tmp==NULL) setTail(node);
	if (prev)
		prev->setNext(node);
	else
		setHead(node);
	numElements++;
}

void DSList::inSort (unsigned int key, void * value, int cleanup)
{
	DSListElement* node;
	DSListElement* tmp;
	DSListElement* prev=NULL;
	for (tmp=getHead();tmp&&(tmp->getKeyInt()<key);
			prev=tmp,tmp=tmp->getNext());
	node=new DSListElement(key,value,tmp,cleanup);
	if (tmp==NULL) setTail(node);
	if (prev)
		prev->setNext(node);
	else
		setHead(node);
	numElements++;
}

void DSList::inSort (char * key, unsigned int value, int cleanup)
{
	DSListElement* node;
	DSListElement* tmp;
	DSListElement* prev=NULL;
	for (tmp=getHead();tmp&&(strcmp(tmp->getKeyString(),key)<0);
			prev=tmp,tmp=tmp->getNext());
	node=new DSListElement(key,value,tmp,cleanup);
	if (tmp==NULL) setTail(node);
	if (prev)
		prev->setNext(node);
	else
		setHead(node);
	numElements++;
}


void DSList::inSort (unsigned int key, unsigned int value, int cleanup)
{
	DSListElement* node;
	DSListElement* tmp;
	DSListElement* prev=NULL;
	for (tmp=getHead();tmp&&(tmp->getKeyInt()<key);
			prev=tmp,tmp=tmp->getNext());
	node=new DSListElement(key,value,tmp,cleanup);
	if (tmp==NULL) setTail(node);
	if (prev)
		prev->setNext(node);
	else
		setHead(node);
	numElements++;
}


void DSList::insert (char * key, void * value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,getHead(),cleanup);
	setHead(node);
	numElements++;
}

void DSList::insert (unsigned int key, void * value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,getHead(),cleanup);
	setHead(node);
	numElements++;
}

/*
 * Insert the item into the beginning of the list, with the specified
 * string key and int data.
 */
void DSList::insert (char * key, unsigned int value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,getHead(),cleanup);
	setHead(node);
	numElements++;
}

/*
 * Insert the item into the beginning of the list, with the specified
 * int key and int data.
 */
void DSList::insert (unsigned int key, unsigned int value, int cleanup)
{
	DSListElement* node;
	node=new DSListElement(key,value,getHead(),cleanup);
	setHead(node);
	numElements++;
}

/*
 * Set the head of the list to an arbitary element.  Be careful.
 */
void DSList::setHead (DSListElement* newHead)
{
	if (head==NULL) setTail(newHead);
	head=newHead;
}

/*
 * Set the tail of the list to an arbitary element.  Be careful.
 */
void DSList::setTail (DSListElement* newTail)
{
	tail=newTail;
}

/*
 * Count the number of times the element referenced by key exists in the list.
 */
unsigned int DSList::occurenceCount (const char * Key)
{
	DSListElement* node;
	unsigned int hits;
	hits=0;
	for (node=getHead();node!=NULL;node=node->getNext()) {
		if (strcmp(node->getKeyString(),Key)==0) hits++;
	}
	return(hits);
}

#define TYPE_NUMERIC 0
#define TYPE_STRING 1

DSListElement ** DSList::createSortArray()
{
	DSListElement ** array;
	DSListElement * tmp;
	int i;
	array=(DSListElement **)malloc(sizeof(DSListElement)*numElements);
	if (array==NULL) return NULL;
	for (i=0,tmp=getHead();tmp;tmp=tmp->getNext(),i++)
	{
		array[i]=tmp;
	}
	return array;
}

void DSList::pushArray(DSListElement ** array, BOOL bDescend)
{
	unsigned int i;
	if (numElements==0) return;
	if (bDescend) {
		for (i=1;i<numElements;i++)
			array[i]->setNext(array[i-1]);
		array[0]->setNext(NULL);
		setHead(array[numElements-1]);
		setTail(array[0]);
	} else {
		for (i=0;i<(numElements-1);i++)
			array[i]->setNext(array[i+1]);
		array[numElements-1]->setNext(NULL);
		setTail(array[numElements-1]);
		setHead(array[0]);
	}
}


int DSListQSort(DSListElement ** List, unsigned int li, int compare)
{
	DSListElement	*midpoint;
	unsigned int    FirstOffset;
	unsigned int    LastOffset;
	unsigned int    Temp;
	DSListElement   *TempElement;
	int             different = 0;
	if (li<2) return TRUE;

	Temp=rand()%li;
	midpoint=List[Temp];

	FirstOffset = 0;
	LastOffset = li - 1;

	/* Scan from the beginning till the piles meet */
	for (FirstOffset = 0; FirstOffset < LastOffset; FirstOffset++) {
		if (((compare==TYPE_STRING)&&
			(strcmp(List[FirstOffset]->getKeyString(),midpoint->getKeyString())>=0))||
			((compare==TYPE_NUMERIC)&&
			 (List[FirstOffset]->getKeyInt()>=midpoint->getKeyInt()))) {
			/*
			 * If something needs to be in the other pile, find
			 * something in the other pile and swap them over
			 */
			for (; LastOffset > FirstOffset; LastOffset--) {
				if (((compare==TYPE_STRING)&&
					(strcmp(List[LastOffset]->getKeyString(),midpoint->getKeyString())<0))||
					((compare==TYPE_NUMERIC)&&
					 (List[LastOffset]->getKeyInt()<midpoint->getKeyInt()))) {
					TempElement = List[LastOffset];
					List[LastOffset] = List[FirstOffset];
					List[FirstOffset] = TempElement;
					LastOffset--;
					break;
				}
			}
		}
	}
	FirstOffset = LastOffset;
	if (((compare==TYPE_STRING)&&
		(strcmp(List[LastOffset]->getKeyString(),midpoint->getKeyString())<0))||
		((compare==TYPE_NUMERIC)&&
		 (List[LastOffset]->getKeyInt()<midpoint->getKeyInt())))
		FirstOffset++;
	/*
	 * If there's only one pile left (should be small) check it for
	 * equality.  If the elements are equal we're finished, if they're
	 * not we need to break again.
	 * String Addendum: This is much more costly than in the int version.
	 *  strcmp() is slow, and the increased chance of false breaks here
	 *  mean this happens more often.
	 */
	if (FirstOffset == 0 || FirstOffset == li) {
		for (Temp = 0; (Temp < li) && (!different); Temp++)
			if (((compare==TYPE_STRING)&&
				(strcmp(List[Temp]->getKeyString(),List[0]->getKeyString())))||
				((compare==TYPE_NUMERIC)&&
				 (List[Temp]->getKeyInt()!=List[0]->getKeyInt())))
				different = 1;
	}
	if ((FirstOffset && (li - FirstOffset)) || different) {
		/* If there are stuff in the piles, keep going */
		if (FirstOffset > 0)
			DSListQSort(List, FirstOffset, compare);
		if ((li - FirstOffset) > 0) {
			DSListQSort(&List[FirstOffset], li - FirstOffset, compare);
		}
	}
	return TRUE;
}

BOOL DSList::sortAsNumeric(BOOL bDescend)
{
	DSListElement ** array;
	array=createSortArray();
	if (array==NULL) return FALSE;
	DSListQSort(array,numElements,TYPE_NUMERIC);
	pushArray(array,bDescend);
	free(array);
	return TRUE;
}

BOOL DSList::sortAsString(BOOL bDescend)
{
	DSListElement ** array;
	array=createSortArray();
	if (array==NULL) return FALSE;
	DSListQSort(array,numElements,TYPE_STRING);
	pushArray(array,bDescend);
	free(array);
	return TRUE;
}

