/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#ifndef DS_LIST_H
#define DS_LIST_H

#include <DSCore.h>
#include <DSListElement.h>
/*!
 * @header DSList
 * @discussion List implements a simple linked list for both string and integer keys and values.
 */

/*!
 * @class DSList
 * @abstract A single-threaded simple implementation of a Linked List.
 * @discussion List implements a simple linked list for both string and integer keys and values.  Note that this class contains undocumented, DEPRECATED functions.
 */
class EXPSYM DSList {
	private:
		DSListElement* head;
		DSListElement* tail;
		unsigned int numElements;
		DSListElement** createSortArray();
		void pushArray(DSListElement ** array, BOOL bDescend);
	protected:
		void         setHead(DSListElement* newHead);
		void         setTail(DSListElement* newTail);
		void         rawDelete(DSListElement* target, DSListElement * prev);
	public:
		/*!
		 * @function DSList
		 * @discussion Creates a new empty List.
		 */
		DSList();

		/*!
		 * @function ~List
		 * @discussion Destroys a List.
		 */
		~DSList();

		/*!
		 * @function append
		 * @discussion Appends the specified ListElement to the end of the list, as opposed to insert, which inserts it at the top of the list.
		 * @param node Specifies the new element.  This is not copied, and is inserted directly into the list.
		 */
		void         append(DSListElement * node);

		/*!
		 * @function append
		 * @discussion Appends the specified key/value to the end of the list, as opposed to insert, which inserts it at the top of the list.
		 * @param Key Specifies the key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         append(char* Key, void* Data, int cleanup=0);

		/*!
		 * @function append
		 * @discussion Appends the specified key/value to the end of the list, as opposed to insert, which inserts it at the top of the list.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         append(unsigned int Key, void* Data, int cleanup=0);

		/*!
		 * @function append
		 * @discussion Appends the specified key/value to the end of the list, as opposed to insert, which inserts it at the top of the list.
		 * @param Key Specifies the key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         append(char* Key, unsigned int Data, int cleanup=0);
		/*!
		 * @function append
		 * @discussion Appends the specified key/value to the end of the list, as opposed to insert, which inserts it at the top of the list.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.  In this call, as both values are numeric, this parameter has no effect. but is retained for consistency.
		 */
		void         append(unsigned int Key, unsigned int Data, int cleanup=0);

		/*!
		 * @function init
		 * @discussion Creates a new linked list where the constructor can't be used, ie., after allocating a List using malloc.  This should not be used in any other case, and the use of malloc is strongly discouraged in favor of new.
		 */
		void         init();

		/*!
		 * @function close
		 * @discussion Removes all the elements from the list, and destroys data in accordance with the cleanup flags of each element.  The List remains available for use.
		 */
		void         close();

		/*!
		 * @function delElement
		 * @discussion Removes the specified element from the List.
		 * @param ele Points to a ListElement which is to be removed from the list.  Its data will be dealt with according to its cleanup flags.
		 */
		void         delElement(DSListElement* ele);

		/*!
		 * @function delHead
		 * @discussion Removes the first element from the List.
		 */
		void         delHead();

		/*!
		 * @function delTail
		 * @discussion Removes the last element from the List.
		 */
		void         delTail();

		DEPRECATED void delValueInt(unsigned int Key);
		DEPRECATED void delValueString(const char* Key);

		/*!
		 * @function delItem
		 * @discussion Removes elements with the specified Key from the List.
		 * @param Key Specifies a numeric key corresponding to the items to be removed.  Its data will be dealt with according to its cleanup flags.
		 */
		void         delItem(unsigned int Key);

		/*!
		 * @function delItem
		 * @discussion Removes elements with the specified Key from the List.
		 * @param Key Specifies a string key corresponding to the items to be removed.  Its data will be dealt with according to its cleanup flags.
		 */
		void         delItem(const char* Key);

		/*!
		 * @function getHead
		 * @discussion Retrieves the first element in the List.
		 * @result A pointer to the first ListElement in the list, or NULL if there are no elements in the list.
		 */
		DSListElement* getHead();

		/*!
		 * @function getTail
		 * @discussion Retrieves the last element in the List.
		 * @result A pointer to the last ListElement in the list, or NULL if there are no elements in the list.
		 */
		DSListElement* getTail();

		DEPRECATED DSListElement* getElementInt(unsigned int Key);
		DEPRECATED DSListElement* getElementString(const char* Key);

		/*!
		 * @function getElement
		 * @discussion Retrieves an element from the list with the specified Key.
		 * @param Key A numeric key corresponding to the desired element.
		 * @result A pointer to the requested ListElement in the list, or NULL if the element was not found.
		 */
		DSListElement* getElement(unsigned int Key);

		/*!
		 * @function getElement
		 * @discussion Retrieves an element from the list with the specified Key.
		 * @param Key A string key corresponding to the desired element.
		 * @result A pointer to the requested ListElement in the list, or NULL if the element was not found.
		 */
		DSListElement* getElement(const char* Key);

		DEPRECATED void* getValueString(const char* Key);
		DEPRECATED void* getValueInt(unsigned int Key);

		/*!
		 * @function getPtrValue
		 * @discussion Retrieves the value of an element from the list with the specified Key.
		 * @param Key A string key corresponding to the desired value.
		 * @result A pointer to the requested value in the list, or NULL if the element was not found.
		 */
		void*        getPtrValue(const char* Key);

		/*!
		 * @function getPtrValue
		 * @discussion Retrieves the value of an element from the list with the specified Key.
		 * @param Key A numeric key corresponding to the desired value.
		 * @result A pointer to the requested value in the list, or NULL if the element was not found.
		 */
		void*        getPtrValue(unsigned int Key);

		DEPRECATED unsigned int getIntString(const char* Key);
		DEPRECATED unsigned int getIntInt(unsigned int Key);

		/*!
		 * @function getNumericValue
		 * @discussion Retrieves the value of an element from the list with the specified Key.
		 * @param Key A string key corresponding to the desired value.
		 * @result The requested value in the list, or 0 if the element was not found.
		 */
		unsigned int getNumericValue(const char* Key);

		/*!
		 * @function getNumericValue
		 * @discussion Retrieves the value of an element from the list with the specified Key.
		 * @param Key A numeric key corresponding to the desired value.
		 * @result The requested value in the list, or 0 if the element was not found.
		 */
		unsigned int getNumericValue(unsigned int Key);

		/*!
		 * @function insert
		 * @discussion Inserts the specified ListElement to the head of the list.
		 * @param node Specifies the new ListElement. Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 */
		void         insert(DSListElement * node);

		/*!
		 * @function inSort
		 * @discussion Inserts the specified key/value to the list maintaining ascending order.
		 * @param Key Specifies the string key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         inSort(char* Key, void* Data, int cleanup=0);

		/*!
		 * @function inSort
		 * @discussion Inserts the specified key/value to the list, maintaining ascending order.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         inSort(unsigned int Key, void* Data, int cleanup=0);

		/*!
		 * @function inSort
		 * @discussion Inserts the specified key/value to the list, maintaining ascending order.
		 * @param Key Specifies the string key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         inSort(char* Key, unsigned int Data, int cleanup=0);
		/*!
		 * @function inSort
		 * @discussion Inserts the specified key/value to the list, maintaining ascending order.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         inSort(unsigned int Key, unsigned int Data,
				int cleanup=0);

		/*!
		 * @function insert
		 * @discussion Inserts the specified key/value to the head of the list.
		 * @param Key Specifies the string key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         insert(char* Key, void* Data, int cleanup=0);

		/*!
		 * @function insert
		 * @discussion Inserts the specified key/value to the head of the list.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the data of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         insert(unsigned int Key, void* Data, int cleanup=0);

		/*!
		 * @function insert
		 * @discussion Inserts the specified key/value to the head of the list.
		 * @param Key Specifies the string key of the new value.  Remember, this data is not copied, but the pointer is used directly.  You should copy your own data if it isn't constant.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         insert(char* Key, unsigned int Data, int cleanup=0);

		/*!
		 * @function insert
		 * @discussion Inserts the specified key/value to the head of the list.
		 * @param Key Specifies the numeric key of the new value.
		 * @param Data Specifies the numeric data of the new value.
		 * @param cleanup Specifies the bitwise combination of flags used to clean up data. See the ListElement class for a list of permissable flags.
		 */
		void         insert(unsigned int Key, unsigned int Data,
				int cleanup=0);

		/*!
		 * @function occurenceCount
		 * @discussion Returns the number of occurences of the specified Key there are in the list.
		 * @param Specifies the Key that needs counting.
		 * @result The number of entries of the specified Key in the list.
		 */
		unsigned int occurenceCount(const char* Key);

		/*!
		 * @function setValue
		 * @discussion Changes the value of the data represented in the list by the specified key.  Any previous data is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A string key corresponding to the desired value.
		 * @param data A pointer to the new data.
		 * @result Returns TRUE if the value were successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setValue(const char* Key, void * data);

		/*!
		 * @function setValue
		 * @discussion Changes the value of the data represented in the list by the specified key.  Any previous data is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A string key corresponding to the desired value.
		 * @param data A numeric value of the new data.
		 * @result Returns TRUE if the value were successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setValue(const char* Key, unsigned int data);

		/*!
		 * @function setValue
		 * @discussion Changes the value of the data represented in the list by the specified key.  Any previous data is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A numeric key corresponding to the desired value.
		 * @param data A pointer to the new data.
		 * @result Returns TRUE if the value were successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setValue(unsigned int Key, void * data);

		/*!
		 * @function setValue
		 * @discussion Changes the value of the data represented in the list by the specified key.  Any previous data is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A numeric key corresponding to the desired value.
		 * @param data A numeric value of the new data.
		 * @result Returns TRUE if the value were successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setValue(unsigned int Key, unsigned int data);

		/*!
		 * @function setKeyValue
		 * @discussion Changes the value of the key represented in the list by the specified key.  The previous key is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A string key corresponding to the desired value.
		 * @param data A pointer to the new key string.
		 * @result Returns TRUE if the data was successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setKeyValue(const char* Key, char * data);

		/*!
		 * @function setKeyValue
		 * @discussion Changes the value of the key represented in the list by the specified key.  The previous key is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A string key corresponding to the desired value.
		 * @param data The new numeric key.
		 * @result Returns TRUE if the data was successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setKeyValue(const char* Key, unsigned int data);

		/*!
		 * @function setKeyValue
		 * @discussion Changes the value of the key represented in the list by the specified key.  The previous key is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A numeric key corresponding to the desired value.
		 * @param data A pointer to the new key string.
		 * @result Returns TRUE if the data was successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setKeyValue(unsigned int Key, char * data);

		/*!
		 * @function setKeyValue
		 * @discussion Changes the value of the key represented in the list by the specified key.  The previous key is destroyed.  The cleanup value governing the ListElement remains unaffected by the use of this call.
		 * @param Key A numeric key corresponding to the desired value.
		 * @param data The new numeric key.
		 * @result Returns TRUE if the data was successfully changed, or FALSE of the element could not be found.
		 */
		BOOL         setKeyValue(unsigned int Key, unsigned int data);

		/*!
		 * @function sortAsNumeric
		 * @discussion Sorts the linked list by numeric key.
		 * @param bDescend If TRUE, sort in descending order; if FALSE or omitted, sort in ascending order.
		 * @result Returns TRUE if the data was successfully sorted, or FALSE if an error occurred.
		 */
		BOOL         sortAsNumeric(BOOL bDescend=FALSE);


		/*!
		 * @function sortAsString
		 * @discussion Sorts the linked list by string key.
		 * @param bDescend If TRUE, sort in descending order; if FALSE or omitted, sort in ascending order.
		 * @result Returns TRUE if the data was successfully sorted, or FALSE if an error occurred.
		 */
		BOOL         sortAsString(BOOL bDescend=FALSE);

};

#endif

