/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */

#include "DSNetwork.h"

#ifndef WIN32_NO_CONFIG_H
#include "../config.h"
#else
#include "../winconf.h"
#endif

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#ifdef HAVE_IO_H
#include <io.h>
#endif

#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif
#ifdef HAVE_SYS_SELECT_H
#include <sys/select.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif


#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif


DSNetwork * DSNetwork::acceptIncoming()
{
	DSNetwork * mynet;
	mynet=new DSNetwork();
	if (!mynet) return NULL;

	mynet->sock=accept(sock,NULL,NULL);
	if (mynet->sock==INVALID_SOCKET) return NULL;
	return mynet;
}

DSNetwork * DSNetwork::Accept()
{
	return acceptIncoming();
}

BOOL
DSNetwork::connectToHost (char *szHostName, unsigned short nPort)
{
	SOCKADDR_IN sin;
	HOSTENT *phe;
	int ret;
	char *port;

#ifdef _WIN32_
	WSADATA wd;
	WSAStartup (MAKEWORD (1, 1), &wd);
#endif /*  */

	/* If the user has specified a port, find out where it is, and
	 * then pretend it isn't there.
	 */
	port = strchr (szHostName, ':');
	if (port)
		port[0] = '\0';

	/* Get the IP address for this hostname. */
	phe = gethostbyname (szHostName);
	if (!phe)
	{
		return FALSE;
	}

	/* Create a new socket for the communication. */
	sock = socket (AF_INET, SOCK_STREAM, 0);

	/* Clear the structure. */
	memset (&sin, 0, sizeof (sin));

	/* We want to connect to an IP network (AF_INET) */
	sin.sin_family = AF_INET;
	if (port)
	{

		/* If we have a port, use that */
		sin.sin_port = htons ((unsigned short) atoi (port + 1));
		port[0] = ':';
	}
	else
	{
		/* Default port for HTTP is 80 */
		sin.sin_port = htons (nPort);
	}

	linger();

	/* Use the address obtained from GetHostByName */
	memcpy (&sin.sin_addr.s_addr, phe->h_addr, phe->h_length);

	/* Try to connect */
	ret = connect (sock, (SOCKADDR *) & sin, sizeof (sin));
	if (ret == 0)
		return TRUE;

	/* We failed. */
	closesocket (sock);
	return FALSE;
}

BOOL DSNetwork::ConnectToHost(char * szHostname, unsigned short nPort)
{
	return connectToHost(szHostname, nPort);
}


/*
 * This function 'peeks' at pending data coming in from the network,
 * and tries to extract a single, solitary line and leave the rest
 * awaiting collection.
 *
 * s==socket it's coming from;
 * szDestination==where to put the line
 * nDestination==how big the line can be
 */
BOOLX
DSNetwork::getLineFromSocket (char *szDestination, int nDestination)
{
	char szBuffer[1024];
	int len;
	BOOLX dataready;
	char *ret;

	dataready=waitForDataEx(0,0);
	if (dataready!=YES) return dataready;

	/* Peek at all the available data */
	len = recv (sock, szBuffer, sizeof (szBuffer) - 1, MSG_PEEK);
	szBuffer[len] = '\0';
	if (len < 1)
		return NO;

	/* Find a newline */
	ret = strchr (szBuffer, '\n');
	if (!ret)
	{
		/* No newline, no line. */
		/* It may be smart to not fail here, but to wait
		 * for a while until a line arrives.  Not doing this
		 * may cause problems.  I guess we'll find out, eh?
		 */
		return NOTYET;
	}

	/* If we have space in the buffer */
	if (ret - szBuffer < (nDestination - 4))
	{
		/* Permanently read the data from the network */
		len = recv (sock, szDestination, ret - szBuffer + 1, 0);
		if (len < 1)
			return NO;
		szDestination[len] = '\0';
		return YES;
	}
	return NO;
}

BOOLX DSNetwork::GetLineFromSocket(char *szDestination, int nDestination)
{
	return getLineFromSocket(szDestination, nDestination);
}

/*
 * This function is just like the above, but will wait until a line is
 * present.  It should be used where a line is definitely needed before
 * proceeding.
 */
BOOL
DSNetwork::getLineFromSocketWait (char *szDestination, int nDestination)
{
	char szBuffer[1024];
	char * inptr;
	int len;
	int sizeremaining;
	char *ret;
	int i;
	i = 0;

	inptr=szBuffer;
	sizeremaining=sizeof(szBuffer)-1;
	
	while (i < 15)
	{
		if (waitForData(60*5,0)==FALSE) return FALSE;

		/* Peek at all the available data */
		len = recv (sock, inptr, sizeremaining, MSG_PEEK);
		inptr[len] = '\0';
		if (len < 1)
			return NO;

		/* Find a newline */
		ret = strchr (szBuffer, '\n');
		if (ret) {
			/* If we have space in the buffer */
			if (ret - szBuffer < (nDestination - 4))
			{
				/* Permanently read the data from the network */
				len = recv (sock, inptr, (ret-szBuffer)-(inptr-szBuffer)+1, 0);
				if (len < 1)
					return NO;
				inptr[len] = '\0';
				strcpy(szDestination, szBuffer);
				return TRUE;
			}
			return FALSE;
		} else {
			len = recv (sock, inptr, len, 0);
			inptr[len] = '\0';
			inptr+=len;
			sizeremaining-=len;
			i++;
		}
	}

	return FALSE;
}

BOOL DSNetwork::GetLineFromSocketWait (char *szDestination, int nDestination)
{
	return getLineFromSocketWait (szDestination, nDestination);
}

BOOL DSNetwork::GetChar(char * ch)
{
	return getChar((unsigned char *)ch);
}

BOOL DSNetwork::GetChar(unsigned char * ch)
{
	return getChar(ch);
}

BOOL DSNetwork::getChar(char * ch)
{
	return getChar((unsigned char *)ch);
}

BOOL DSNetwork::getChar(unsigned char * ch)
{
	int len;
	if (waitForData(60*5,0)==FALSE) return FALSE;
	len=recv(sock, (char *)ch, 1, 0);
	if (len>0) return TRUE;
	return FALSE;
}


BOOL DSNetwork::getSlab(char * dest, int size)
{
	int len;
	if (size<=0) return FALSE;

	do {
		if (waitForData(60*5,0)==FALSE) return FALSE;
		len=recv(sock, dest, size, 0);
		dest[len]='\0';
		dest+=len;
		size-=len;
	} while ((size>0)&&(len>0));

	if (size==0) return TRUE;
	return FALSE;
}

BOOL DSNetwork::GetSlab(char * dest, int size)
{
	return getSlab(dest, size);
}

SOCKET DSNetwork::GetSocket()
{
	return getSocket();
}

SOCKET DSNetwork::getSocket()
{
	return sock;
}


BOOL DSNetwork::linger()
{
	struct linger lyes;
	/*
	   Linger - if the socket is closed, ensure that data is sent/
	   received right up to the last byte.  Don't stop just because
	   the connection is closed.
	 */
	lyes.l_onoff = 1;
	lyes.l_linger = 10;
	setsockopt (sock, SOL_SOCKET, SO_LINGER, (char *) &lyes,
		    sizeof (lyes));
	return TRUE;

}

BOOL DSNetwork::Linger()
{
	return linger();
}


BOOL
DSNetwork::sendData(const char * string, int nSize)
{
	int ret;
	if (nSize<0) {
		ret=send(sock, string, strlen(string), 0);
	} else {
		ret=send(sock, string, nSize, 0);
	}
	if (ret>=0) return TRUE;
	return FALSE;
}

BOOL
DSNetwork::Send(const char * string, int nSize)
{
	return sendData(string, nSize);
}

BOOL
DSNetwork::Send(char * string, int nSize)
{
	return sendData((const char *)string, nSize);
}

/*
This function initialises the server.
*/
BOOL
DSNetwork::startServer (unsigned short nPort, unsigned long nIP)
{
	SOCKADDR_IN sin;
	int ret;
	unsigned int yes;

#ifdef _WIN32_
	WSADATA wd;
	WSAStartup (MAKEWORD (1, 1), &wd);
#endif /*  */
	yes = TRUE;

	/* Grab a place to listen for connections. */
	sock = socket (AF_INET, SOCK_STREAM, 0);
	if (sock < 0)
	{
		return FALSE;
	}

	linger();

	/* If this server has been restarted, don't wait for the old
	 * one to disappear completely */
	setsockopt (sock, SOL_SOCKET, SO_REUSEADDR, (char *) &yes,
		    sizeof (yes));

	memset (&sin, 0, sizeof (sin));
	sin.sin_family = AF_INET;
	sin.sin_port = htons (nPort);

	sin.sin_addr.s_addr = nIP;
	ret = bind (sock, (SOCKADDR *) & sin, sizeof (sin));
	if (ret < 0)
	{
		return FALSE;
	}

	/* Start listening for incoming connections. */
	ret = listen (sock, 5);
	if (ret == 0)
		return TRUE;
	return FALSE;
}

BOOL
DSNetwork::StartServer (unsigned short nPort, unsigned long nIP)
{
	return startServer(nPort, nIP);
}


BOOL DSNetwork::Terminate()
{
	return terminate();
}

BOOL DSNetwork::terminate()
{
	closesocket(sock);
	return TRUE;
}

BOOLX DSNetwork::waitForDataEx(int sec, int usec)
{
	int ret;
	fd_set readset;
	struct timeval timeout;

	FD_ZERO(&readset);
	FD_SET(sock,&readset);
	if (sec>=0) {
		timeout.tv_sec=sec;
		timeout.tv_usec=usec;
		ret=select(sock+1,&readset,NULL,NULL,&timeout);
	} else {
		ret=select(sock+1,&readset,NULL,NULL,NULL);
	}
	if (ret>0) return YES;
	if (ret==0) return NOTYET;
#ifndef _WIN32_
	if (errno==EINTR) return NOTYET;
#endif
	return NO;
}

BOOL DSNetwork::waitForData(int sec, int usec)
{
	if (waitForDataEx(sec, usec)==YES) return TRUE;
	return FALSE;
}

