/* LIBDS
 * =====
 * This software is Copyright (c) 2002-03 Malcolm Smith.
 * No warranty is provided, including but not limited to
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * This code is licenced subject to the GNU General
 * Public Licence (GPL).  See the COPYING file for more.
 */
#ifndef DS_TREEELEMENT_H
#define DS_TREEELEMENT_H

#include <DSCore.h>
#include <stdlib.h>

/*!
 * @header DSTreeElement
 * @discussion A TreeElement contains a value, and details about how to clean up after itself.  It can only have one value, numeric or string.   These are inserted into a DSTree.
 */

/*!
 * @class DSTreeElement
 * @abstract A unit capable of being stored in a Tree.
 * @discussion A TreeElement contains a value, and details about how to clean up after itself.  It can only have one value, numeric or string. 
 */
class EXPSYM DSTreeElement
{
	private:
		int flags;
		union {
			unsigned int nData;
			void* pData;
		};
		enum {
			VALUETYPE_STRING=2
		};
		void cleanupData();
	public:
		/*!
		 * @enum Cleanup Cleanup flags
		 * @constant CLEANUP_VALUE_FREE Specifies that the value should be cleaned up using free() when it's disposed.
		 */
		enum {
			CLEANUP_VALUE_FREE=64,
			CLEANUP_ALL=64
		};

		/*!
		 * @function DSTreeElement
		 * @discussion Creates a new TreeElement.
		 * @param cleanup Specifies the cleanup flags that apply to this TreeElement.
		 */
		DSTreeElement(int cleanup=0);

		/*!
		 * @function DSTreeElement
		 * @discussion Creates a new TreeElement.
		 * @param Data Specifies the value of the new element.  This pointer is used in the element, copy data yourself.
		 * @param Cleanup Specifies the cleanup flags that apply to this TreeElement.
		 */
		DSTreeElement(void* Data,int Cleanup=0);

		/*!
		 * @function DSTreeElement
		 * @discussion Creates a new TreeElement.
		 * @param Data Specifies the numeric value of the new element.
		 * @param Cleanup Specifies the cleanup flags that apply to this TreeElement.
		 */
		DSTreeElement(unsigned int Data, int Cleanup=0);

		/*!
		 * @function ~TreeElement
		 * @discussion Destroys the TreeElement.
		 */
		~DSTreeElement();

		/*!
		 * @function getDataInt
		 * @discussion Retrieves the numeric value of the TreeElement.
		 * @result The numeric value of the TreeElement, or 0, if the TreeElement contains a pointer.
		 */
		unsigned int getDataInt();

		/*!
		 * @function getDataPtr
		 * @discussion Retrieves the pointer value of the TreeElement.
		 * @result The pointer value of the TreeElement, or NULL, if the TreeElement contains a number.
		 */
		void*        getDataPtr();

		/*!
		 * @function setCleanup
		 * @discussion Modifies the cleanup flags.
		 * @param Cleanup The new cleanup flags.
		 */
		void         setCleanup(int Cleanup);

		/*!
		 * @function setDataInt
		 * @discussion Modifies the value in the TreeElement.  Any previous value is destroyed.
		 * @param Data The new numeric value.
		 */
		void         setDataInt(unsigned int Data);

		/*!
		 * @function setDataPtr
		 * @discussion Modifies the value in the TreeElement.  Any previous value is destroyed.
		 * @param Data The new pointer value.  This pointer is inserted into the TreeElement and is not copied.
		 */
		void         setDataPtr(void * Data);

		/*!
		 * @function setKeyInt
		 * @discussion Modifies the key in the TreeElement.  Any previous key is destroyed.
		 * @param Key The new numeric value.
		 */
		void         setKeyInt(unsigned int Key);

		/*!
		 * @function setKeyInt
		 * @discussion Modifies the key in the TreeElement.  Any previous key is destroyed.
		 * @param Key The new string value.  This pointer is used directly, not copied; make sure you have copied it yourself if you'd like the TreeElement to keep it.
		 */
		void         setKeyString(char * Key);
};

#endif



