/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: dbutil.c,v 1.27 2002/12/22 08:46:37 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:            dbutil.c
 * created on:          Tue May 14 21:19:12 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

# if ! defined HAVE_GETADDRINFO || ! defined HAVE_GETNAMEINFO
static struct hostent *	hostEnt = NULL;
static int		hostIndex = 0;
# endif

# if ! defined HAVE_GETADDRINFO && defined HAVE_RESOLV_H
#   include <sys/types.h>
#   include <netinet/in.h>
#   include <arpa/nameser.h>
#   include <resolv.h>
# endif

static int net_real_gethostbyname(int,char *,PNetAddr *,int);

/*----------------------------------------------------------------------*/
/* Get host address by name						*/
/*----------------------------------------------------------------------*/

int
net_gethostbyname( int fam, const char *name, PNetAddr *pa )
{
    char * local_name;
    char * p;
    int	   scope;
    int	   r = 0;
    int	   do_free = 0;

    if ( ! pnetInitDone )
	pnetInit();

# ifdef PNET_HAVE_LOCAL
    if (fam == AF_LOCAL)
    {
	addr_set_ip_addr(pa,fam,name,strlen(name));
	return 0;
    }
# endif

    p = NULL;
    scope = addr_extract_scope( name , 0);

    if ( scope > -1 )
    {
	/*
	 * Make local copy of the name, so we can extract scope
	 */
	if ( ! (local_name = (char*) strdup( name )) ) 
	    { MEMERR( "net_gethostbyname()", strlen( name ) ); return -1; }

	do_free = 1;

	/* Make sure we terminate the address if postfixed by a '%<iface>'
	 * so that the resolver won't fail.  */

	if ( (p = strrchr( local_name, pnet_ScopeSep )) )
	    *p = 0;
    }
    else
    	local_name = (char *) name;

    r = net_real_gethostbyname( fam, local_name, pa, scope );

    if ( do_free )
    	free( local_name );

    return r;
}

static int
net_real_gethostbyname( int fam, char *name, PNetAddr *pa, int scope )
{
    PNETADDR  * 	ppa;
    PNETADDR		pacurr;
    char 		buf[ PNET_ADDR_STRLEN ];

# ifdef HAVE_GETADDRINFO			/* {{ */

    struct addrinfo	ai_hints;
    struct addrinfo *	ai_res;
    struct addrinfo * 	ai_list;
    int	   		err_code;

    pdbg(E_INFO,"net_real_gethostbyname(fam=%s,name=%s)\n",net_aftoa(fam),name);
    pdbg(E_DBG4,"Resolver: using getaddrinfo()\n");

    pacurr = pa;
    ppa = &pa->pa_next;

    memset(&ai_hints,0,sizeof(ai_hints));

    ai_hints.ai_flags 	 = AI_CANONNAME;
    ai_hints.ai_family	 = fam;
    ai_hints.ai_socktype = SOCK_STREAM;
    ai_hints.ai_protocol = 0;

    err_code = getaddrinfo(name,NULL,&ai_hints,&ai_res);

    if (err_code)
    {
	perr(E_FATAL, "net_real_gethostbyname(\"%s\",\"%s\"): %s\n",
		     net_aftoa( fam ), name, gai_strerror(err_code));
	return -1;
    }

    ai_list = ai_res;

    while ( ai_list )
    {
	if ( ai_list->ai_family == AF_INET )
	{
	    InetAddr * paddr = (InetAddr*)ai_list->ai_addr;

	    addr_set_ip_addr( pacurr, ai_list->ai_family,
			      &paddr->sin_addr,sizeof(struct in_addr));
	}
	else if ( ai_list->ai_family == AF_INET6)
	{
	    InetAddr6 * paddr = (InetAddr6*)ai_list->ai_addr;

	    addr_set_ip_addr( pacurr, ai_list->ai_family, 
			      &paddr->sin6_addr, sizeof(struct in6_addr));
	    if ( scope > -1 )
		pacurr->pa_scope_id = scope;
	}

	ai_list = ai_list->ai_next;

	if ( ! ai_list )
	    break;

	STDMALLOC( pacurr, sizeof( PNetAddr ), -1 );
	*ppa = pacurr;
	ppa = &pacurr->pa_next;
    }

    pdbg(E_INFO,"Canonical name for '%s': '%s'\n", 
		 name, ai_res->ai_canonname ? ai_res->ai_canonname : name );

    freeaddrinfo(ai_res);

# else						/* }{ */

    pdbg(E_INFO,"net_real_gethostbyname(fam=%s,name=%s)\n",net_aftoa(fam),name);
    pdbg(E_DBG4,"Resolver: using gethostbyname()\n");

    hostIndex = 0;

# ifdef HAVE_RESOLV_H		/* { */

    if (!hostEnt)
    {
	res_init();
	DBG( _res.options |= RES_DEBUG );
# ifdef RES_USE_INET6
	_res.options |= RES_USE_INET6;
# endif
    }

# ifdef RES_USE_INET6
    if (fam == AF_INET)
	_res.options &= ~RES_USE_INET6;
    else if (fam == AF_INET6)
	_res.options |= RES_USE_INET6;
# endif

# endif				/* } */

# ifndef HAVE_GETHOSTBYNAME2
    fam = AF_INET;
# endif

# if defined HAVE_GETHOSTBYNAME2
    if ( ! (hostEnt = gethostbyname2(name,fam)) )
# else
    if ( ! (hostEnt = gethostbyname(name)) )
# endif
    {
	perr(E_WARN,"net_real_gethostbyname(\"%s\",\"%s\"): %s\n",
	     net_aftoa( fam ), name,hstrerror(h_errno));
	return -1;
    }

    addr_set_ip_addr(pa,fam,hostEnt->h_addr_list[0],hostEnt->h_length);

    if ( scope > -1 )
	pa->pa_scope_id = scope;

    pdbg(E_INFO,"Canonical name for '%s': '%s'\n",name,hostEnt->h_name);

# endif 					/* }} */

    pdbg(E_INFO,"Resolved: %s --> %s\n",name,
    		 pnet_ntop( pa, buf, sizeof( buf ) ));

    return 0;
}
int
net_gethostbyaddr(int fam,PNetAddr *pa,char *name,int buflen)
{
    char *	pname;
    int		cplen;
    char	buf[ PNET_ADDR_STRLEN ];

# ifdef HAVE_GETNAMEINFO			/* {{ */

    char 	host[NI_MAXHOST];

    if ( ! pnetInitDone )
	pnetInit();

    DBG(pdbg(E_DBG4,"RResolver: using getnameinfo()\n"));

    if ( getnameinfo( &pa->pa_saddr, PA_AddrLen(pa),
		    host,sizeof(host),NULL,0,NI_NAMEREQD))
    {
	perr(E_FATAL, "net_gethostbyaddr(): cannot resolve '%s'\n",
		     pnet_ntop( pa, buf, sizeof( buf ) ));
	return -1;
    }

    pname = host;

# else 						/* }{ */

    if ( ! pnetInitDone )
	pnetInit();

    hostIndex = 0;

    DBG(pdbg(E_DBG4,"RResolver: using gethostbyaddr()\n"));

    if ( fam == AF_INET || fam == AF_UNSPEC )
	hostEnt = gethostbyaddr((char*)&pa->pa_in4addr.sin_addr,
				sizeof(struct in_addr), AF_INET );

    if ( fam == AF_INET6 || (fam == AF_UNSPEC && ! hostEnt))
	hostEnt = gethostbyaddr((char*)&pa->pa_in6addr.sin6_addr,
				sizeof(struct in6_addr), AF_INET6);

    if ( ! hostEnt )
    {
	perr(E_WARN,"net_gethostbyaddr(): '%s' %s\n",name,hstrerror(h_errno));
	return -1;
    }

    pname = hostEnt->h_name;

# endif						/* }} */

    cplen = strlen( pname ) + 1;
    if (buflen <  (int) cplen)
	{ SPACEERR("net_gethostbyaddr()"); return -1; } 

    strncpy( name, pname, cplen );

    pdbg(E_INFO,"RResolved: %s --> %s\n",
    		 pnet_ntop( pa, buf, sizeof( buf ) ),pname);

    return 0;
    fam = fam;		/* Unused parameter warnings */
}

# if 0 
int
netGetHostAlias(SockAddr *sa)
{
    if (! hostEnt)
	{ return -1; }

    if ( ! hostEnt->h_addr_list[hostIndex] )
    {
	pdbg(E_WARN,"netGetHostAlias(): no more aliases for '%s'\n",
	     hostEnt->h_name);
	return -1;
    }

    if (sa->sa_family == AF_INET)
    {
        InetAddr *      ia = (InetAddr*)sa;

        ia->sin_family = hostEnt->h_addrtype;
        memcpy(&ia->sin_addr,
		hostEnt->h_addr_list[hostIndex],(size_t)hostEnt->h_length);
    }
# ifdef PNET_HAVE_IPv6
    else if (sa->sa_family == AF_INET6)
    {
        InetAddr6*      ia = (InetAddr6*)sa;

        ia->sin6_family = hostEnt->h_addrtype;
        memcpy(&ia->sin6_addr,
                hostEnt->h_addr_list[hostIndex],(size_t)hostEnt->h_length);
    }
# endif
    else
	{ FAMERR("netGetHostAlias()"); return -1; }

    pdbg(E_INFO,"Next alias for '%s': '%s'\n",
    	hostEnt->h_name,hostEnt->h_addr_list[hostIndex]);

    hostIndex++;

    return 0;
}

# endif
/*----------------------------------------------------------------------*/
/* Get my host address 							*/
/*----------------------------------------------------------------------*/
int
pnetGetLocalAddress( char *buf, int bufsiz )
{
    PNetAddr	pa ;
    char 	hname[128];
    char 	abuf[PNET_ADDR_STRLEN];

    DBG(dbg("pnetGetLocalAddr( buf=%lX, bufsize=%d )\n", XX(buf), bufsiz ));

    if ( ! pnetInitDone )
	pnetInit();

    if ( gethostname( hname, sizeof(hname) ) )
	{ NETERR("pnetGetLocalAddr()"); return -1; }

    pdbg(E_MSG,"pnetGetLocalAddress(): unqual'd (?) hostname: '%s'\n",hname);

    if ( net_gethostbyname( AF_UNSPEC, hname, &pa) )
	{ return -1; }
    
    /* If the next function fails, just use the name from gethostname() */
    net_gethostbyaddr( pa.pa_family, &pa, hname, sizeof(hname) );

    pdbg(E_MSG,"FQDN is %s, %s\n",hname,
	pnetAddrToString( &pa, abuf, sizeof(abuf) ) );

    if (buf)
	strncpy(buf,(const char*)hname,(size_t)bufsiz);

    return 0;
}

/*
 * Services
 */

static struct servent*	servEnt = NULL;
static int		servIndex = 0;

int
net_getservbyname(const char *name,const char *proto,int *pport)
{
    DBG(dbg("net_getservbyname(name=%s,proto=%s)\n",name,proto));

    servIndex = 0;

    if ( ! pnetInitDone )
	pnetInit();

    if ( ! (servEnt = getservbyname(name,proto)) )
    {
        perr(E_WARN,"pnet_getservbyname(): %s (%s): %s\n",name,proto,neterr()); 
        return -1;
    }

    pdbg(E_MSG,"Official name for service '%s': '%s'\n",name,servEnt->s_name);
    pdbg(E_MSG,"Port for service '%s': %d\n",servEnt->s_name,
	 (int)ntohs((unsigned short)servEnt->s_port));

    if (pport)
	*pport = htons(servEnt->s_port);

    return 0;
}
