/*
 * This file is part of
 *
 * PNET6: a Portable Network Library
 *
 * PNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: pnetif.c,v 1.17 2002/10/03 06:24:54 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            pnetif.c
 * created on:          Mon May 27 17:29:17 CEST 2002
 * created by:          peter
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*/
/*									*/
/* Interface information.						*/
/* Non-thread safe currently.						*/
/*									*/
/*----------------------------------------------------------------------*/

# include "../local.h"

# include "iflocal.h"

# ifdef HAVE_SYS_IOCTL_H

#     include <sys/ioctl.h>

/*----------------------------------------------------------------------*/
/* IOCTL wrapper							*/
/*----------------------------------------------------------------------*/

int
if_ioctl( int sd, int req, void *v )
{
    if ( ioctl(sd, req, v) < 0 )
    {
	perr(E_WARN,"ioctl(%d,%X,%X): %s (errno=%d)\n",
		sd,req,v,SYSERR(),errno);
	return -1;
    }
    return 0;
}

# endif
/*----------------------------------------------------------------------*/
/* Get all interfaces information, and store them in the global		*/
/* variable pif_head. As soon as thread support is added, make sure	*/
/* this function is only invoked by a single thread at a time.		*/
/*----------------------------------------------------------------------*/
struct pnet_if*	pif_head = NULL;

/*----------------------------------------------------------------------*/
/* Free the list of interfaces						*/
/*----------------------------------------------------------------------*/

void
free_ifs_info(struct pnet_if *fst)
{
    struct pnet_if * p = fst;
    struct pnet_if * t = p;

    while ( p )
    {
	t = t->next;
	STDFREE( p );
	p = t;
    }
}

/*----------------------------------------------------------------------*/
/* Print miscellaneous information about an interface			*/
/*----------------------------------------------------------------------*/

void
if_debug( PNetIf *pif )
{
    byte *p = pif->hwaddr;

    pdbg(E_DBG4,"Interface %s: idx = %d, tp = %d, mtu = %d\n",
    		pif->pif_name,pif->pif_index,pif->pif_type,pif->pif_mtu );

    if ( pif->pif_pflags & PIF_HAS_HWADDR )
    {
	if ( pif->hwaddr_len == 6 )
	    pdbg(E_DBG4,"Hardware address for %s: "
		    "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X\n",
		    pif->pif_name,p[0],p[1],p[2],p[3],p[4],p[5]);
	else if ( pif->hwaddr_len == 8 )
	    pdbg(E_DBG4,"Hardware address for %s: "
		    "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X:%.2X\n",
		    pif->pif_name,p[0],p[1],p[2],p[3],p[4],p[5],p[6],p[7]);
	else 
	{
	    perr(E_WARN,"Hardware address for %s has a weird length of %d\n",
			pif->pif_name, pif->hwaddr_len ); 	
	}
    }

    return;
}

/*----------------------------------------------------------------------*/
/* Interface access functions						*/
/*----------------------------------------------------------------------*/

PNETIF       pnetIfNext ( PNETIF pif ) { return pif->next ; }
const char * pnetIfName ( PNETIF pif ) { return pif->pif_name ; }
int          pnetIfIndex( PNETIF pif ) { return pif->pif_index  ; }
int	     pnetIfType ( PNETIF pif ) { return pif->pif_type; }

/*----------------------------------------------------------------------*/
/* Simple PNETIFADDR accessors 						*/
/*----------------------------------------------------------------------*/
int
pnetIfAddrGetFamily( PNETIFADDR pia )
{
    return fam2pfam( pia->pia_family );
}
/*----------------------------------------------------------------------*/
/* Copy this IfAddr to an InetAddr.					*/
/*----------------------------------------------------------------------*/
int
pnetIfAddrToInetAddr( PNETIFADDR pia, PNETADDR pa )
{
    int			len = 0;
    const pnet_byte * 	in_addr;

    in_addr = pnetIfAddrGetBinary( pia, &len );
    if ( ! in_addr )
    {
	perr(E_FATAL,"pnetIfAddrToInetAddr(): ifaddr corrupt\n");
	return -1;
    }
    addr_set_ip_addr( pa, pia->pia_family, in_addr, len );

    if ( pnetAddrIsLinkLocal( pa ) )
	pa->pa_scope_id = pia->pia_if->pif_index;
    return 0;
}
const char *
pnetIfAddrToString( PNETIFADDR pia, char *buf, int buflen )
{
    if ( pia->pia_family  == AF_INET)
    	return inet_ntop( AF_INET, (const void*)&pia->pia_addr4.sin_addr,
		   	  buf, (size_t) buflen); 
    else if (pia->pia_family == AF_INET6)
	return inet_ntop( AF_INET6, (const void*)&pia->pia_addr6.sin6_addr,
			  buf,(size_t)  buflen);

    FAMERR("pnetIfAddrToString()");

    return NULL;
}

const pnet_byte*
pnetIfAddrGetNetmask( PNETIFADDR pia )
{
    if ( pia->pia_netmask[0] )
	return pia->pia_netmask + 1;
    return NULL;
}
const pnet_byte*
pnetIfAddrGetBroadcast( PNETIFADDR pia )
{
    if ( pia->pia_broadaddr[0] )
	return pia->pia_broadaddr + 1;
    return NULL;
}
PNETIFADDR
pnetIfAddrGetNext( PNETIFADDR pia )
{
    return pia->pia_next;
}
int
pnetIfAddrGetScopeId( PNETIFADDR pia ) 
{
    if ( pia->pia_family == AF_INET6 )
	return pia->pia_scope_id;

    return -1;
}
int
pnetIfAddrGetPrefixLength( PNETIFADDR pia ) 
{
    if ( pia->pia_family == AF_INET6 )
	return pia->pia_prefixlen;

    return -1;
}
const pnet_byte*
pnetIfAddrGetBinary( PNETIFADDR pia, int * len )
{
    if ( pia->pia_family == AF_INET6 )
    {
	if ( len )
	    *len = 16;
	return (pnet_byte*)&pia->pia_addr6.sin6_addr;
    }
    else if ( pia->pia_family == AF_INET )
    {
	if ( len )
	    *len = 4;
	return (pnet_byte*)&pia->pia_addr4.sin_addr;
    }

    return NULL;
}

/*----------------------------------------------------------------------*/
/* Get an interface's addresses.					*/
/*----------------------------------------------------------------------*/

PNETIFADDR
pnetIfGetNextAddr( PNETIF pif, PNETIFADDR pia )
{
    return net_if_get_next_address( pif, pia );
}

const char *
pnetIfAddrToAscii( PNETIF pif, int which, char *buf, int buflen)
{
    DBG(dbg("pnetIfAddrToAscii(pif=%X,which=%d,buf=%X,buflen=%d)\n",
	    XX(pif),which,XX(buf),buflen));

    if (!pif)
    	{ XLOG(pif); return NULL; }

    switch (which)
    {
    case PNETIF_HWADDR:
	{
	    char * pbuf = buf;
	    int n = 0;

	    if ( ! (pif->pif_pflags & PIF_HAS_HWADDR))
		return NULL;

	    if (buflen < (int)PNETIF_HWADDRLEN)
	    {
		perr(E_WARN,"pnetIfAddrToAscii(): %s\n",pneterr(PNETerrSpace));
		return NULL;
	    }

	    if (pif->hwaddr_len <= (byte)6)
		n = sprintf(pbuf,"%.2X:%.2X:%.2X:%.2X:%.2X:%.2X",
			    pif->hwaddr[0],pif->hwaddr[1],
			    pif->hwaddr[2],pif->hwaddr[3],
			    pif->hwaddr[4],pif->hwaddr[5]);
	    if (pif->hwaddr_len == (byte)8)
		sprintf(pbuf + n,"%.2X:%.2X",pif->hwaddr[6],pif->hwaddr[7]);
		
	    SLOG(buf);

	    return buf;
	}

    case PNETIF_BADDR:
	    if ( ! (pif->pif_pflags & PIF_HAS_BADDR))
		return NULL;

	    return inet_ntop( AF_INET,
			     (void*)( pif->pif_addrs->pia_broadaddr + 1 ),	
			      buf, buflen);
    case PNETIF_DADDR:

	if ( ! (pif->pif_pflags & PIF_HAS_DADDR))
	    return NULL;

	if ( pif->pif_daddr.sa_family  == AF_INET)
	    return inet_ntop( AF_INET, (void*)&pif->pif_daddr4.sin_addr,
			      buf, buflen);
	else if (pif->pif_daddr.sa_family == AF_INET6)
	    return inet_ntop( AF_INET6, (void*)&pif->pif_daddr6.sin6_addr,
			      buf, buflen);

	FAMERR("pnetIfAddrToAscii()");
	return NULL;

    default:
	perr(E_FATAL,"pnetIfAddrToAscii(): unknown interface address type\n");
	return NULL;
    }

    /* NOTREACHED */
    return NULL;
}

/*----------------------------------------------------------------------*/
/* Get a pointer to an interface. First, we search until we find an	*/
/* interface with a matching name, or index. Then, if fam is not 	*/
/* AF_UNSPEC, we search the interface's addresses looking for one that	*/
/* matches the requested family. If such an address is found, the 	*/
/* interface is returned. Useful if you need an interface that has	*/
/* a particular address family.						*/
/*----------------------------------------------------------------------*/

# define FORALLADDRS( pif, pia )		\
	for ( pia = pif->pif_addrs; pia; pia = pia->pia_next )

/* If the bootstrap flag is on, we are being called from get_ifs_info().
 * Do not call that again.
 */
PNetIf*
net_get_if_by_name_bootstrap( int fam, const char * name, int bootstrap )
{
    struct pnet_if * 	pif;
    unsigned		idx;
    char		c;

    if ( ! name )
	{ XLOG( name ); return NULL; }

    if ( ! bootstrap && ! pif_head )
    {
	/* Fetch interface info here, using the unspecified family, since
	 * we want to fetch information about all interfaces */

	if ( get_ifs_info( AF_UNSPEC ) )
	    return NULL;			/* Failed. */
    }

    /* See if the given name is not actually just an interface index	*/
    if ( sscanf( name, "%u%c", &idx, &c ) != 1 )
	idx = -1;

    /* Look through interface list, and return the interface they're 	*/
    /* asking for 							*/

    for ( pif = pif_head; pif; pif = pif->next )
    {
	if ( ( idx > 0 && pif->pif_index == idx) ||
	     ( ! strcmp(name,pif->pif_name) )	)
	{
	    /* Got an interface with matching name,  find out if one 	*/
	    /* of its addresses has a matching family 			*/
	    PNetIfAddr	*pia;

	    if ( fam == AF_UNSPEC )
		return pif;

	    FORALLADDRS( pif, pia )
		if ( pia->pia_family == fam )
		    return pif;
	}
    }

    XLOG( NULL );
    return NULL;
}

PNetIf*
net_get_if_by_name( int fam, const char *name )
{
    return net_get_if_by_name_bootstrap( fam, name, 0 );
}

PNetIf*
net_get_if_by_index( int fam, u_long idx )
{
    struct pnet_if * pif;

    if (idx == 0)
	return NULL;

    /* If no info yet, fetch it here 	*/
    if ( ! pif_head )
    {
	if ( get_ifs_info( AF_UNSPEC ) )
	    return NULL;			/* Failed. */
    }

    for (pif = pif_head; pif; pif = pif->next)
	if (idx == pif->pif_index)
	{
	    PNetIfAddr	*pia;

	    if ( fam == AF_UNSPEC )
		return pif;

	    FORALLADDRS( pif, pia )
		if ( pia->pia_family == fam )
		    return pif;
	}

    return NULL;
}
/*----------------------------------------------------------------------*/
/* Exported functionality 						*/
/*----------------------------------------------------------------------*/

/* Get interface information for all interfaces on system that support  */
/* given address family 						*/

PNETIF
pnetGetIfInfo( int pfam )
{
    pdbg(E_INFO,"pnetGetInterfaces(%s)\n",net_paftoa(pfam));

    pnetInit( );
    if ( get_ifs_info( pfam2fam(pfam) ) )
	return NULL;

    return pif_head;
}
PNETIF
pnetIfGetByName( int pfam, const char *name )
{
    DBG(dbg("pnetIfGetByName(pfam=%s, name=%s)\n",net_paftoa(pfam),name));

    return net_get_if_by_name(  pfam2fam(pfam), name );
}
PNETIF
pnetIfGetByIndex( int pfam, int idx )
{
    DBG(dbg("pnetIfGetByIndex(pfam=%s, idx=%d)\n",net_paftoa(pfam),idx));

    return net_get_if_by_index( pfam2fam(pfam), idx);
}
int
pnetIfNameToIndex( const char *name)
{
    struct pnet_if * pif;

    SLOG( name );
    if( !(pif = net_get_if_by_name( AF_UNSPEC, name )) )
    	{ XLOG(pif); return 0; }
    return pif->pif_index;
}
const char *
pnetIfIndexToName( int idx )
{
    struct pnet_if * pif;

    if( !(pif = net_get_if_by_index( AF_UNSPEC, idx)) )
    	{ XLOG(idx); return 0; }

    return pif->pif_name;
}
const pnet_byte *
pnetIfGetLinkAddress(PNETIF pif,int *len )
{
    if ( !pif )
	return NULL;

    if ( ! (pif->pif_pflags & PIF_HAS_HWADDR) )
	return NULL;		/* No hwaddr for this interface */

    if ( len )
	*len = pif->hwaddr_len;

    return pif->hwaddr;
}
int
pnetIfGetMTU( PNETIF pif )
{
    return pif->pif_mtu;
}
/*----------------------------------------------------------------------*/
/* Get IP address of given interface. sa holds the interface's address 	*/
/* upon successful return.						*/
/* 1. Find interface by its name;					*/
/* 2. Loop through its available addresses, looking for one with a 	*/
/*    matching address family;						*/
/* 3. Copy the appropriate address to caller's buffer.			*/
/*----------------------------------------------------------------------*/
int
net_get_if_ipaddr(int fam, const char *name,SockAddr *sa)
{
    struct pnet_if * pif = NULL;
    PNetIfAddr*	     pia = NULL;

    pif = net_get_if_by_name( fam, name );

    if (!pif)
	return -1;

    FORALLADDRS( pif, pia )
	if ( pia->pia_family == fam )
	{
	    if ( fam == AF_INET )
		memcpy( sa, &pia->pia_addr4, sizeof( InetAddr ) );
	    else if ( fam == AF_INET6 )
	    	memcpy( sa, &pia->pia_addr6, sizeof( InetAddr6) );
	    else
	    	{ FAMERR( "net_get_if_ipaddr()" ); return -1; }
	}
    return 0;
}
int
net_bind_to_interface(int fam, int sd, const char *name, SockAddr *sa)
{
    struct pnet_if * pif = NULL;
    PNetIfAddr *     pia = NULL;
    int 	     len = 0;

    if ( (len = addr_len( fam )) == 0 )
    {
	perr(E_FATAL,"net_bind_to_interface(): %s: unknown address family %d\n",
		      pif->pif_name, pif->pif_family );
	return -1;
    }

    pif = net_get_if_by_name( fam, name );

    if ( ! pif )
    {
	perr(E_FATAL,"net_bind_to_interface(): interface %s unknown,\n",name);
	perr(E_FATAL,"or not configured for given family %s\n",net_aftoa(fam));
	return -1;
    }

    /* Find first address that matches the requested family 	*/

    FORALLADDRS( pif, pia )
	if ( pia->pia_family == fam )
	{
	    if (! bind( sd, &pia->pia_saddr, len ) )
		{ NETERR("bind()"); return -1; }

	    /* Copy into return buffer */
	    if (sa)
		memcpy(sa,&pia->pia_saddr,len);

	    return 0;
	}

    return -1;
}

PNetIfAddr*
net_if_get_next_address( PNETIF pif, PNetIfAddr* pia )
{
    if ( ! pia )
	return pif->pif_addrs;

    return pia->pia_next;
}
PNetIfAddr*
net_if_get_next_address_type( PNETIF pif, PNetIfAddr *pia, int fam )
{
    if ( ! pia )
	pia = pif->pif_addrs;

    while ( pia && pia->pia_family != fam )
	pia = pia->pia_next;

    /* pia will be NULL if no match is found */
    return pia;
}
int
pnetBindToInterface(PNETSOCK ps,const char *name)
{
    /* Bind a socket to an interface */

    if ( net_bind_to_interface( ps->fam, ps->sd,
    	 			name, &ps->local_addr.pa_saddr ) )
	{ NETERR("pnetBindToInteface()"); return -1; }

    addr_from_sockaddr( &ps->local_addr, &ps->local_addr.pa_saddr );

    return 0;
}
int
pnetBindToAllInterfaces( PNETSOCK ps )
{
    PNetIf * pif;
    if ( ! pif_head )
    {
	if (get_ifs_info( AF_UNSPEC ) )
	    return -1;			/* Failed. */
    }
    for ( pif = pif_head; pif; pif = pif->next )
    {
	if ( pif->pif_family == ps->fam )
	{
	    PNetIfAddr * pia;

	    FORALLADDRS( pif, pia )
	    {
		if ( pia->pia_family == ps->fam )
		{
		    if (! bind( ps->sd, &pia->pia_saddr, addr_len(ps->fam)))
		    {
			NETERR("pnetBindToAllInterfaces(): bind()");
		    }
		    pdbg(E_INFO,"Bound to %s %s\n",
		    	pif->pif_name,net_sockaddrtop( &pia->pia_saddr ));
		    break;	/* Done with this interface */
		}
	    }
	}
    }
    return 0;
}

const char*
pnetIfTypeString(PNetIf *pif)
{
    return if_get_description( pif );
}
