/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * $Id: listen.c,v 1.31 2002/11/05 08:35:35 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            listen.c
 * created on:          Sat May 25 13:15:21 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*----------------------------- (  10 lines) --------------------------*/
static int sock_accept(PNETSOCK,PNetAddr *);
static int lseAdd(PNETSOCK);
static int net_make_listen(PNETSOCK,const char *,const char *,int *);
static int net_make_listen_address(PNETSOCK,PNETADDR,int *);
static int handle_client(int,PNetSocket *,PNetAddr *);
static int fork_handle_client(int,PNetSocket *,PNetAddr *);
static void * thrd_func(void *);
static int thread_handle_client(int,PNetSocket *,PNetAddr *);
static int net_listen_all(int);

static int
sock_accept(PNETSOCK ps,PNetAddr* cl)
{
    AddrUnX	addr;
    socklen_t	len;
    int 	fd;

    len = addr_len( ps->fam );

    if ( (fd = accept(ps->sd,&addr.saddr, &len)) < 0 )
        { NETERR("sock_accept()"); }

    addr_from_sockaddr(cl,&addr.saddr);

    return fd;
}
/*----------------------------------------------------------------------*/
/* Table of sockets to listen at.					*/
/*----------------------------------------------------------------------*/
typedef struct lsock_entry
{
    PNetSocket *	ps;
    struct lsock_entry * nxt;
} LSockEntry;

static LSockEntry*	lsockHead;
static LSockEntry**	plsockCurr = &lsockHead;
static int		lseNum = 0;

static int
lseAdd( PNETSOCK ps )
{
    STDMALLOC(*plsockCurr,sizeof(LSockEntry),-1);

    (*plsockCurr)->ps = ps;
    plsockCurr = &(*plsockCurr)->nxt;

    pdbg(E_INFO,"lseAdd(ps=%lX (sd=%d)) added to listen administration.\n",
    	XX(ps),ps->sd);
    lseNum++;

    return 0;
}
/*----------------------------------------------------------------------*/
/* port is given as char, so it can double as local path (for local sockets)*/
/* Makes a listen socket and stores it into ps.				*/
/*----------------------------------------------------------------------*/
static int
net_make_listen(PNETSOCK ps,const char* ifname,
		const char *sport,int *port)
{
    PNetAddr  psrc;

    DBG(dbg("net_make_listen(ps=%lX,ifname=%s,sport=%s)\n",XX(ps),ifname,sport));

    /* Make a listen address for the given port number */
    if ( addr_make_listen(ps->fam,&psrc, ifname,sport) )
        { XLOG(ps); return -1; }

    return net_make_listen_address( ps, &psrc, port );
}
static int
net_make_listen_address( PNETSOCK ps, PNETADDR psrc, int * port )
{
    char 	buf[ PNET_ADDR_STRLEN ];

    /* Copy address to socket's local address */
    memcpy( &ps->local_addr, psrc, sizeof( PNetAddr ) );

    /* Bind socket to the new address */
    if ( sock_bind( ps, psrc ) )
        { XLOG(ps); return -1; }

    /* Get the real socket name, after kernel has made the bind() */

    if (sock_getsockname(ps,psrc))
        { XLOG(ps); return -1; }

    if ( port )
	*port = psrc->pa_sin_port;

    if ( ps->type == SOCK_STREAM )
    {
	if (listen(ps->sd,5))
	{
	    perr(E_FATAL,"net_make_listen(): listen(): %s\n",SYSERR());
	    return -1;
	}

        ps->connected = 1;
    }

    /* Mark as non-blocking. Since we're using select prior to 		*/
    /* any input operation, we need to make sure we never block.	*/
    /* A malicious client can cause the server to block in a call to 	*/
    /* accept by initiating the connection, causing our server to 	*/
    /* break out of the select(), then resetting it, before the server 	*/
    /* has had time to call accept(). If the kernel dismantles the 	*/
    /* (now defunct) connection, then our server might block in the	*/
    /* accept(), waiting for a connection that will not arrive.		*/

    if ( pnetSockSetNonBlocking(ps,1) )
	pdbg(E_MSG,"Cannot set non-blocking flag for socket=%lX\n",XX(ps));

    pdbg(E_INFO,"Listen address %s initialized\n",
    		 pnet_ntop( psrc, buf, sizeof( buf ) ));

    return 0;
}
static int
handle_client( int sd, PNetSocket *lps,PNetAddr *client)
{
    PNETSOCK 	psock;

    if (! (psock = sock_socket(sd,lps->fam,lps->type,lps->proto)) )
	{ close(sd); return -1; }

    pnetAddrCopy(&psock->local_addr,&lps->local_addr);
    pnetAddrCopy(&psock->peer_addr , client);
# ifdef PNET_HAVE_LOCAL
    psock->copied_path = 1;
# endif

    psock->read_callback = lps->read_callback;
    psock->callback_data = lps->callback_data;

    psock->connected = 1;

    netStatsConnOpened();

    if (psock->sd > 0 && psock->read_callback)
	psock->read_callback(psock,psock->callback_data);

    netStatsConnClosed();
    pnetClose(psock);

    return 0;
}

static int
fork_handle_client(int sd,PNetSocket *lps,PNetAddr *client)
{
    pid_t 	child_pid; 
    char 	buf[ PNET_ADDR_STRLEN ];

    switch ( (child_pid = sys_fork()) )
    {
    case -1:
	FATALERR("fork_handle_client(): fork()");
	return -1;
    case 0: 
	{
	    int ret;
	    /* Child, handles new client 			*/
	    /* Create a new socket to communicate with client 	*/

	    ret = handle_client( sd, lps, client );

	    pnetClose(lps);

	    pdbg(E_DBG1,"Child %d: closed connection to %s\n",
			 sys_getpid(),pnet_ntop( client, buf, sizeof( buf )));
	    exit( ret );
	    return 0;
	}
    default:
	/* Parent closes new socket and returns immediately */
	close(sd);
	pdbg(E_DBG4,"Child %d forked to handle connection from %s\n",
		child_pid,pnet_ntop(client, buf, sizeof( buf )));
	return 1;
    }

    return 0;
}
static void *
thrd_func( void * ppsock )
{
    PNetSocket * psock = (PNetSocket*)ppsock;
    char 	 buf[ PNET_ADDR_STRLEN ];
    int		 ret = 0;

    netStatsConnOpened();

    if ( psock->sd > 0 && psock->read_callback)
   	ret = psock->read_callback( psock, psock->callback_data );

    netStatsConnClosed();

    pdbg( E_DBG1, "Thread %u: closed connection to %s\n", pnetThreadId(),
		  pnet_ntop( &psock->peer_addr, buf, sizeof( buf ) ) );
    pnetClose( psock );

    return NULL;
}
static int
thread_handle_client( int sd, PNetSocket *lps, PNetAddr *client )
{
    pnet_thread_id 	thd_id;
    PNetSocket * 	psock;
    char 		buf[ PNET_ADDR_STRLEN ];

    /* Create a new socket for the given thread 	*/
    /* The thread must close that prior to exiting	*/

    if (! (psock = sock_socket(sd,lps->fam,lps->type,lps->proto)) )
	{ close(sd); return -1; }

    pnetAddrCopy(&psock->local_addr,&lps->local_addr);
    pnetAddrCopy(&psock->peer_addr , client);
# ifdef PNET_HAVE_LOCAL
    psock->copied_path = 1;
# endif


    psock->read_callback = lps->read_callback;
    psock->callback_data = lps->callback_data;
    psock->connected = 1;

    if ( pnetThreadCreate( &thd_id, thrd_func, (void*) psock ) )
	{ pnetClose( psock ); return -1; } 

    pdbg(E_DBG4,"Thread %u created to handle connection from %s\n",
    		thd_id, pnet_ntop( client , buf, sizeof( buf )) );

    return 0;
}

/*
 * Listen on more than one socket.
 */
static int listen_sem = 0;

static int
net_listen_all( int action )
{
    /* Roll up all sockets currently in the listen_tab. */

    pdbg(E_INFO,"Starting listen() on %d sockets\n",lseNum);

    if ( action == PNET_LISTEN_FORK )
    {
	/* Need to establish a sighandler for our children */
	if ( pnetSetSighandler( SIGCHLD, net_wait_for_child ) == PNET_SIG_ERR )
	    perr(E_WARN,"net_listen_all(): Cannot set handler for SIGCHLD\n");
    }

    while ( 1 )
    {
        fd_set          r_fds;
        int             ret;
	pnetsock_t	max_sd = 0;
	PNetSocket*	ps;
	LSockEntry *plse = lsockHead;
	int		num_desc = 0;

	if ( listen_sem <= 0 )
	    break;
	/* Fill in the read descpriptors */
        FD_ZERO(&r_fds);

	DBG(dbg("Descriptors for select(): "));

	while (plse)
	{
	    FD_SET(plse->ps->sd,&r_fds);

	    DBG(dbg("%d ",plse->ps->sd));

	    max_sd = PMAX( plse->ps->sd, max_sd );

	    plse = plse->nxt;

	    num_desc++;
	}
	DBG(dbg("\n"));

	pdbg(E_DBG4,"Entering select() on %d descriptors, max_sd = %d\n",
		num_desc,max_sd);

        ret = select(max_sd + 1,&r_fds,(fd_set*)0,(fd_set*)0,0);

	plse = lsockHead;

        if (ret > 0)
        {
	    for ( ; plse; plse = plse->nxt )
		if (FD_ISSET(plse->ps->sd,&r_fds))
		    break;

	    if (!plse)
	    {
		perr(E_FATAL,"no descriptor set by select()\n");
		continue;
	    }

	    ps = plse->ps;

	    if (ps->type == SOCK_STREAM)
	    {
		int 	 fd;
		PNetAddr client;

		if ( (fd = sock_accept(ps,&client)) < 0 )
		{
		    perr(E_FATAL,"Failed to initiate new connection\n");
		    netStatsConnDropped();
		    continue;
		}

		if ( action == PNET_LISTEN_FORK )
		    fork_handle_client(fd,ps,&client);
		else if ( action == PNET_LISTEN_THREAD )
		    thread_handle_client( fd, ps, &client );
	        else if ( action == PNET_LISTEN_SELF )
		    handle_client( fd, ps, &client );
	    }
	    else
	    {
		netStatsConnOpened();

		if (ps && ps->read_callback)
		    ps->read_callback(ps,ps->callback_data);

		netStatsConnClosed();
	    }
        }
        else if (ret == 0)              /* Timeout expired */
        {
            dbg("Timer expired in net_listen_all()\n");
        }
        else                            /* Error condition */
        {
            perr(E_WARN,"net_listen_all(): error in select(): %s\n",SYSERR());
        }
    }

    pdbg(E_INFO,"net_listen_all(): listen aborted due to user request\n");
    return 0;
}
/*
 * Listen at wildcard address
 */
int
pnetListen(PNETSOCK ps,const char *sport)
{
    int port;

    if ( ps->multicast )
    {
	lseAdd( ps );
	return 0;
    }
    /* Raw sockets: no need to make a listen socket */
    if ( ps->type == SOCK_RAW )
    {
	lseAdd(ps);
	return 0;
    }
    if ( net_make_listen(ps,NULL,sport,&port) )
	return -1;

    lseAdd(ps);
    return 0;
}
/*
 * Listen at the given interface.
 */
int
pnetListenAt(PNETSOCK ps,const char * ifname,const char *sport)
{
    int port;

    if ( net_make_listen(ps,ifname,sport,&port) )
	return -1;

    lseAdd(ps);
    return 0;
}
int
pnetListenAtAddress( PNETSOCK ps, PNETADDR pa )
{
    int port;

    if ( net_make_listen_address( ps, pa, &port ) )
	return -1;

    lseAdd(ps);

    return 0;
}

void
pnetStopListen( void )
{
    listen_sem--;
}
int
pnetStartListen( int mode )
{
    listen_sem++;

# if defined HAVE_FORK 

    net_listen_all( mode );

# elif defined HAVE_WINDOWS_THREADS

    if ( mode == PNET_LISTEN_FORK )
    {
    	perr(E_FATAL,"Listening in FORK mode not supported on Windows.\n");
    	perr(E_FATAL,"Use PNET_LISTEN_THREAD instead.\n");
	listen_sem --;
	return -1;
    }

    net_listen_all( PNET_LISTEN_THREAD );

# endif

    return 0;
}
