/* * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: sendrecv.c,v 1.71 2002/12/02 08:22:03 kingofgib Exp $
 */


/*----------------------------------------------------------------------*
 * filename:		sendrecv.c
 * created on:		Thu Jun  6 07:02:03 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*----------------------------- (    7 lines ) --------------------------*/
static void dgram_log_flags(int);
static int dgram_recv(PNetSocket *,char *,int);
static int net_recvfrom(PNETSOCK,char *,int,int);
static int socket_read(PNETSOCK,char *,int,int,long);
static void swap_ip_fields(pnet_ip *);
static pnetsock_t wait_for_read(pnetsock_t *,int,long);

/*----------------------------------------------------------------------*/
/* Wait for an input operation on a number of socket descriptors. A msec*/
/* argument of less than 0 means wait for ever.				*/
/*----------------------------------------------------------------------*/

static pnetsock_t
wait_for_read(pnetsock_t *sd,int num_sds,long msec)
{
    struct timeval  	tv,*ptv;
    fd_set          	rfds;
    int             	ret,i;
    pnetsock_t		max_sd = 0;

    DBG(dbg("wait_for_read(sd=%lX,msec=%ld)\n",XX(sd),msec));

    ptv = NULL;

    if (msec >= 0)
    {
	tv.tv_sec       = msec / 1000;
	tv.tv_usec      = msec % 1000 * 1000;
	ptv = &tv;
    }

    /* Clear read fds */
    FD_ZERO(&rfds);

    /* Load all the sds given */
    i = num_sds;

    while( i-- )
    {
	LLOG(sd[i]);
	FD_SET(sd[i],&rfds);
	max_sd = PMAX(sd[i],max_sd);
    }

    ret = select(max_sd + 1,&rfds,(fd_set*)0,(fd_set*)0,ptv);

    if (ret == 0)
    {
	DBG(dbg("wait_for_read(): timed out after %ld msec\n",msec));
	return PNET_READ_TIMED_OUT;
    }
    else if (ret == -1)
    {
	perr(E_FATAL,"wait_for_read(): error in select(): %s\n",neterr());
	return PNET_READ_ERROR;
    }
    
    for (i = 0; i < num_sds; i++)
    {
	if (FD_ISSET( sd[i], &rfds )) 
	{
	    DBG(dbg("wait_for_read(): pending input on sd=%d\n",sd[i]));
	    return sd[i];
	}
    }

    /* If we get here, then none of the descriptors were set. This is 
     * an error (?)
     */
    
    perr(E_FATAL,"wait_for_read(): no socket descriptor set.\n");
    perr(E_FATAL,"wait_for_read(): %s\n",neterr());

    return PNET_READ_ERROR;
}

/*----------------------------------------------------------------------*/
/* The generic read function. Read len bytes into the provided buffer	*/
/* If block is not 0, block until the given len bytes have been		*/
/* read in.								*/
/* If msec is 0, means return immediately, even if no data present	*/
/* If msec is < 0, means wait forever for data to arrive.		*/
/* If msec is > 0, means wait for the given amount of milliseconds for 	*/
/* data to arrive, then return.						*/
/*----------------------------------------------------------------------*/

static int
socket_read( PNETSOCK ps, char *buf, int blen, int block, long msec)
{
    int         total,last;
    pnetsock_t	ret = 0;

    DBG(dbg("socket_read(ps=%lX,buf=%lX,blen=%d,block=%d,msec=%ld)\n",
        XX(ps),XX(buf),blen,block,msec));

    DBG(pdbg(E_DBG4,"socket_read(): socket type %s\n",net_sttoa(ps->type)));

    total = last = 0;

    ret = wait_for_read( &ps->sd, 1, msec);

    if (ret != ps->sd)
    {
	/* wait_for_read will have set the error condition, and / or 
	 * logged it */
	return ret;
    }

    if ( ps->type != SOCK_STREAM )
	return dgram_recv(ps,buf,blen);

    while (total < blen)
    {
	last = recv(ps->sd,(void*) buf,blen - total,0);

        if (last >= 0)
	{
	    netStatsRecvData( last );
            total += last,buf += last;
	}
        else if (last < 0)
        {
            perr(E_FATAL,"socket_read(): recv of %d bytes failed.\n",
                 blen - total);
	    NETERR( "socket_read()" );
        }

        if (!block)
            break;
    }

    return total;
}

/*----------------------------------------------------------------------*/
/* This is writev() using a connectionless socket			*/
/*----------------------------------------------------------------------*/

# define PNET_ALIGNMENT	(sizeof( pnet_ulong ) - 1)

# ifndef CMSG_ALIGN
# define CMSG_ALIGN(b)	( ((pnet_uint)(b) + PNET_ALIGNMENT) & ~PNET_ALIGNMENT )
# endif
# ifndef CMSG_LEN
# define CMSG_LEN( len ) CMSG_ALIGN( sizeof( struct cmsghdr ) + len )
# endif
# ifndef CMSG_SPACE
# define CMSG_SPACE(len) (CMSG_ALIGN( sizeof( struct cmsghdr ) ) + \
			  CMSG_ALIGN( len ) )
# endif

int
net_writev( PNETSOCK ps, PNETADDR pa,
	    struct iovec * iov, int iovcnt, int flags )
{
# if defined HAVE_STRUCT_MSGHDR
    struct msghdr 	msghdr;
# endif
    int			sent;

    DBG( dbg("net_writev(ps=%lX,pa=%lX,iov=%lX,iovcnt=%d)\n", 
	XX(ps), XX(pa), XX( iov ), iovcnt ) );

    sent = 0;

    /*
     * If ps is a raw socket, and it has the own header flag set, then
     * we assume the first iov buffer contains a valid IPv4 header. We
     * also assume the user has built the header using pnetIP4_Build().
     * In that case, all fields are in network byte order.
     * However, we need to convert a few IP fields to host byte order
     * on some kernels.
     */
    if ( ps->own_header && ps->fam == AF_INET )
    {
	pnet_ip * ip = (pnet_ip*) iov[0].iov_base;

	if ( iov[0].iov_len < (int)sizeof( pnet_ip ) )
	{
	    perr( E_FATAL,"net_writev(): ip header too small.\n");
	    return -1;
	}

	swap_ip_fields( ip );
    }


# ifdef HAVE_SENDMSG						/* {{ */
    memset( &msghdr, 0, sizeof( msghdr ) );

    msghdr.msg_name 	= (void*)&pa->pa_saddr;
    msghdr.msg_namelen  = (size_t) pa->pa_len;
    msghdr.msg_iov  	= iov;
    msghdr.msg_iovlen 	= (pnet_uint)iovcnt;

# if defined PNET_HAVE_IPv6 && defined HAVE_MSGHDR_MSG_CONTROL 	/* {{ */
    if ( ps->own_header && ps->fam == AF_INET6 )
    {
	pnet_ip6*	ip6 = (struct pnet_ip6*)iov[0].iov_base;
	struct cmsghdr*	cm;
	char		buf[  CMSG_SPACE( sizeof( int ) )
# ifdef IPV6_FLOWINFO
			    + CMSG_SPACE( sizeof( pnet_uint ) )
# endif
# ifdef HAVE_STRUCT_IN6_PKTINFO
			    + CMSG_SPACE( sizeof( struct in6_pktinfo ) )
# endif
 			   ];

# ifdef HAVE_STRUCT_IN6_PKTINFO
	struct in6_pktinfo *pkt;
# endif

	cm = (struct cmsghdr*) buf;
	msghdr.msg_control = cm;
	msghdr.msg_controllen = 0;

# ifdef IPV6_HOPLIMIT
	/* Put HOPLIMIT in there */
	cm->cmsg_len    = CMSG_LEN( sizeof( int ) );
	cm->cmsg_level  = IPPROTO_IPV6;
	cm->cmsg_type   = IPV6_HOPLIMIT;
	*(int*)CMSG_DATA( cm ) = (int)ip6->ip6_hlim;
	msghdr.msg_controllen += cm->cmsg_len;
# endif

# ifdef IPV6_FLOWINFO
	/* Put flow info in there */
	cm = (struct cmsghdr*) (buf + cm->cmsg_len);
	cm->cmsg_len    = CMSG_LEN( sizeof( pnet_uint ) );
	cm->cmsg_level  = IPPROTO_IPV6;
	cm->cmsg_type   = IPV6_FLOWINFO;
	*(pnet_uint*)(CMSG_DATA(cm)) = (pnet_uint) ip6->ip6_flow;
	msghdr.msg_controllen += cm->cmsg_len;
# else
	pa->pa_flowinfo = pnetIP6_Flow( ip6 );
# endif

# ifdef IPV6_PKTINFO 		/* { */
	cm 			= (struct cmsghdr*) (buf + cm->cmsg_len);
	cm->cmsg_len 		= CMSG_LEN( sizeof( struct in6_pktinfo ) );
	cm->cmsg_level  	= IPPROTO_IPV6;
	cm->cmsg_type   	= IPV6_PKTINFO;
	msghdr.msg_controllen  += cm->cmsg_len;

	/*
	 * Specify source datagram address:
	 *  If src address is link (or site local?), assign the outgoing
	 *  interface according to the interface corresponding to the 
	 *  destination address.
	 */
	pkt = (struct in6_pktinfo*)CMSG_DATA(cm);

	memcpy( &pkt->ipi6_addr, &ip6->ip6_src, sizeof( pkt->ipi6_addr ) );

	if ( IN6_IS_ADDR_LINKLOCAL(    (struct in6_addr*)&ip6->ip6_src ) ||
	     IN6_IS_ADDR_MC_LINKLOCAL( (struct in6_addr*)&ip6->ip6_src ) ||
	     IN6_IS_ADDR_SITELOCAL(    (struct in6_addr*)&ip6->ip6_src ) )
	    pkt->ipi6_ifindex = pa->pa_scope_id;
	else
	    pkt->ipi6_ifindex = 0;

# endif				/* } */

#ifdef STDLinux
	/*
	 * On (some?) Linux, next header (i.e protocol field) is contained in 
	 * ip6->ip6_nxt. Is this behavior portable or it is simply Linux ?
	 */
	pa->pa_sin6_port= ntohs( ip6->ip6_nxt );
# endif
	/*
	 * Shift the iov vector by one (remember, IPv6 does not actually
	 * allow a self-made IPv6 header on the wire.
	 */
	msghdr.msg_iov  	= iov + 1;
	msghdr.msg_iovlen 	= (pnet_uint)iovcnt - 1;
    }
# endif								/* }} */

    sent = sendmsg( ps->sd, &msghdr, flags );

# elif defined PNET_WIN32					/* }{ */

    if ( WSASendTo( ps->sd, iov, iovcnt, &sent, flags,
	 	    &pa->pa_saddr, pa->pa_len, 0, 0 ) == SOCKET_ERROR )
	sent = -1;

# else								/* }{ */
    {
	/* Got a better idea? I'd love to here about it */
	byte * buf;
	int i,len;

	/*
	 * Find out what the size of the buffer is, then copy all
	 * iov_base buffers into 'buf'
	 */
	for ( i = 0, len = 0; i < iovcnt; i++ )
	    len += iov[i].iov_len;

	STDMALLOC( buf, len, -1 );

	for ( i = 0,len = 0; i < iovcnt; len += iov[i].iov_len, i++ )
	    memcpy( buf + len, iov[i].iov_base, iov[i].iov_len );

	sent = sendto( ps->sd, buf, len, 0, &pa->pa_saddr, pa->pa_len );

	STDFREE( buf );
    }
# endif								/* }} */

    if ( sent < 0 )
	{ NETERR( "net_writev()" ); return -1; }

    netStatsSentData( sent );

    return sent;
}

/*----------------------------------------------------------------------*/
/* This is readv, using a connectionless socket				*/
/*----------------------------------------------------------------------*/

int
net_readv( PNETSOCK ps, PNETADDR pa,
	   struct iovec * iov, int iovcnt, int *pflags )
{
    int 		flags = 0;
# if defined HAVE_STRUCT_MSGHDR
    struct msghdr 	msghdr;
# endif
    int			got = 0;

    DBG( dbg("net_readv(ps=%lX,pa=%lX,iov=%lX,iovcnt=%d)\n", 
	XX(ps), XX(pa), XX( iov ), iovcnt ) );

    got = 0;

    if ( *pflags )
	flags = *pflags;

# ifdef HAVE_RECVMSG						/* {{ */
    memset( &msghdr, 0, sizeof( msghdr ) );
    msghdr.msg_name 	= (void*)&pa->pa_saddr;
    msghdr.msg_namelen  = (size_t) pa->pa_len;
    msghdr.msg_iov  	= iov;
    msghdr.msg_iovlen 	= iovcnt;

    got = recvmsg( ps->sd, &msghdr, flags );

# ifdef HAVE_MSGHDR_MSG_FLAGS
    flags = msghdr.msg_flags;
# endif

# elif defined PNET_WIN32					/* }{ */
    if ( WSARecvFrom( ps->sd, iov, iovcnt, &got, &flags,
		      &pa->pa_saddr, &pa->pa_len, 0, 0 ) == SOCKET_ERROR )
	got = -1;
# else								/* }{ */
    {
	/* Got a better idea? I'd love to here about it */
	byte * buf;
	int i,len;

	/*
	 * Find out what the size of the buffer is, use readfrom() to get
	 * at all the data, then copy the buffer into the chunks defined
	 * by the iov_base vectors.
	 */
	for ( i = 0, len = 0; i < iovcnt; i++ )
	    len += iov[i].iov_len;

	STDMALLOC( buf, len, -1 );

	/* Set length of receiving address 	*/
	pa->pa_len = addr_len( ps->fam );

	got = recvfrom( ps->sd, buf, len, flags,
			&pa->pa_saddr, (socklen_t*) &pa->pa_len );

	for ( i = 0, len = 0; i < iovcnt; len += iov[i].iov_len, i++ )
	    memcpy( iov[i].iov_base, buf + len, iov[i].iov_len );

	STDFREE( buf );
    }
# endif								/* }} */

    if ( got < 0 )
	{ NETERR( "net_writev()" ); return -1; }

    if ( *pflags )
	*pflags = flags;

    return got;
}

/*----------------------------------------------------------------------*/
/* RECVFROM() wrapper							*/
/*----------------------------------------------------------------------*/

static int
net_recvfrom( PNETSOCK ps, char * buf, int buflen, int flags )
{
    int got;

    DBG( dbg("net_recvfrom(ps=%lX,buf=%lX,buflen=%d,flags=%lX)\n",
	     XX(ps),XX(buf),buflen,flags) );

    /* Set length of peer socket, ready to recv the peer's address 	*/
    ps->peer_addr.pa_len     = addr_len( ps->fam );

    got = recvfrom( ps->sd, buf, (size_t) buflen, flags,
		    &ps->peer_addr.pa_saddr,
		    (socklen_t*)&ps->peer_addr.pa_len );

    if ( got < 0 )
	{ NETERR("net_recvfrom()"); return -1; }

    /* This really needs to be fixed. */
    addr_from_sockaddr( &ps->peer_addr, &ps->peer_addr.pa_saddr );

    if ( ps->aux_info )
    {
	ps->aux_info->aux_pkt_hoplimit = -1;

	if ( ps->fam == AF_INET )
	{
	    if ( net_getsockopt( ps, IPPROTO_IP, IP_TTL,
				&ps->aux_info->aux_pkt_hoplimit ) )
		perr(E_DBG1,"Cannot read TTL value off IP datagram\n");
	}
    }

    return got;
}

/*----------------------------------------------------------------------*/
/* dgram_recv(): recv message on an unconnected socket. Save sender 	*/
/* address, dst address (if possible), interface on which it came, and  */
/* what type of datagram it is (brdcst, mltcst)				*/
/* Parse a whole bunch of control information, and save it in IP	*/
/* headers for raw sockets with own IP header turned on.		*/
/*----------------------------------------------------------------------*/
# if defined IP_RECVIF
# include <net/if_dl.h>
# endif

/* A desperate attempt to get this to work on MS-Win. It failed */
# ifdef STDWin32
# define msghdr		_WSAMSG
# define msg_name	name
# define msg_namelen	namelen
# define msg_iovlen	dwBufferCount
# define msg_iov	lpBuffers
# define msg_flags	dwFlags

# define iovec		_WSABUF
# endif

static void
swap_ip_fields( pnet_ip * ip )
{
# if ! defined STDLinux && ! defined STDWin32 && ! defined STDOpenBSD
    /* 
     * Linux and Windows seem to keep the following few values in network
     * byte order. Most BSD derived code doesn't. We prefer to keep 
     * everything in network byte order, so a conversion is in place here.
     */
    ip->ip_len = htons( ip->ip_len );
    ip->ip_off = htons( ip->ip_off );
/*  ip->ip_id  = htons( ip->ip_id  ); */
# endif
}

static int
dgram_recv( PNetSocket *ps, char *buf, int blen )
{
    char 	  addrbuf[ PNET_ADDR_STRLEN ];

# if defined HAVE_RECVMSG		/* {{ */
    int 	  receiving_udp;

    struct msghdr mhdr;
    struct iovec  iov[1];
    int 	  got;		/* Returned by recvmsg */
    int		  flags;	/* Needed by recvmsg */
    PNetAddr	* pa;		/* Work pointer 	*/
    PNetAddr	* pdsta	= NULL;	/* For destination address of datagram */
    int		  ifindex = -1; /* Datagram arrived on this interface  */

# if defined HAVE_MSGHDR_MSG_CONTROL	/* {{ */

    /*
     * We want to receive the destination address for the datagram 
     * and the interface index on which it arrived. The first is returned
     * as ancillary data of size 4 (4 address bytes), the second is a
     * truncated sockaddr_dl structure ( len = 8, AF_LINK, if_index,
     * type = IFT_NONE, name_len = addr_len = sel_len = 0). However, the
     * entire struct sockaddr_dl is passed into our buffer, so we need
     * to allocate enough space for that.
     */
    struct cmsghdr * pcmh;

    union {
	struct cmsghdr cm;

	char control4[	0
# ifdef IP_RECVIF
			+ CMSG_SPACE( sizeof( pnet_uint ) ) 
# endif
# ifdef IP_RECVDSTADDR
			+ CMSG_SPACE( sizeof( struct sockaddr_dl ) ) 
# endif
# ifdef IP_PKTINFO
			+ CMSG_SPACE( sizeof( struct in_pktinfo ) ) 
# endif
		    ];

# ifdef PNET_HAVE_IPv6			/* { */
	char control6[ 0
# ifdef IPV6_PKTINFO
			+ CMSG_SPACE( sizeof( struct in6_pktinfo ) ) 
# endif
# ifdef IPV6_HOPLIMIT
			+ CMSG_SPACE( sizeof( pnet_uint ) ) 
# endif
		    ];
# endif					/* } */
    } ctrl;

# ifdef PNET_HAVE_IPv6
    if ( ps->fam == AF_INET6 )
    {
	mhdr.msg_control = ctrl.control6;
	mhdr.msg_controllen = sizeof( ctrl.control6 );
    }
    else
# endif
    {
	mhdr.msg_control = ctrl.control4;
	mhdr.msg_controllen = sizeof( ctrl.control4 );
    }

    mhdr.msg_flags   = 0;

# else					/* }{ */

    memset( &mhdr,0,sizeof(mhdr));

# endif					/* }} */

    DBG(dbg("dgram_recv(ps=%lX,buf=%lX,blen=%d)\n",XX(ps),XX(buf),blen));
    DBG(pdbg(E_DBG4,"DGRAM_RECV: local  %s\n",
    		pnet_ntop(&ps->local_addr, addrbuf, sizeof( addrbuf ))));

    receiving_udp = ps->type == SOCK_DGRAM;

    pa = &ps->peer_addr;
    pa->pa_len = sizeof(InetAddr6);

    /* Flags can be one of MSG_DONTWAIT, MSG_PEEK, MSG_WAITALL, MSG_OOB */
    /* Need to set those in a future version */
    flags = 0;

    mhdr.msg_name = (void*)&pa->pa_saddr;
    mhdr.msg_namelen  = (size_t) pa->pa_len;

    iov[0].iov_base = buf;
    iov[0].iov_len  = blen;
    mhdr.msg_iov  = iov;
    mhdr.msg_iovlen = 1;

    if ( ( got = recvmsg(ps->sd,&mhdr,flags) ) < 0 )
    {
	NETERR("dgram_recv()");
	return -1;
    }

    netStatsRecvData( got );

    DBG( dgram_log_flags( mhdr.msg_flags ) );

    /* Datagram source address has been stored by recvmsg */
    addr_from_sockaddr( pa, &pa->pa_saddr );

    DBG(pdbg(E_DBG4,"DGRAM_RECV: remote %s\n",
    		     pnet_ntop( &ps->peer_addr, addrbuf, sizeof( addrbuf ))));

    if ( !ps->aux_info )
	return got;

    if ( ps->type == SOCK_DGRAM )
	ps->aux_info->ai_type = PNET_IPPROTO_UDP;
    else if ( ps->type == SOCK_RAW )
	ps->aux_info->ai_type = ps->fam;

    /* For IPv4 sockets, we must extract the TTL field here, since it won't
     * be available as ancillary data */
    if ( receiving_udp )
    {
	ps->aux_info->aux_pkt_hoplimit = -1;

	if ( ps->fam == AF_INET )
	{
	    if ( net_getsockopt( ps, IPPROTO_IP, IP_TTL,
				&ps->aux_info->aux_pkt_hoplimit ) )
		perr(E_DBG1,"Cannot read TTL value off IP datagram\n");
	}
    }
    else 
    {
	if ( ps->fam == AF_INET )
	{
	    /*
	     * IP vs. IPv6 part X: IP will return the entire datagram,
	     * including IP options. It seems IPv6 does not do that.
	     * We do a substantial bit of trickery to be able to simulate
	     * the same effect w/ IPv6: in particular, we parse a whole
	     * bunch of ancillary data in order to be able to extract some
	     * sort of info on the IPv6 header. See further on.
	     */

	    pnet_ip *	ip;
	    int		hdr_len;
     
	    /*
	     * Get pointer to IP header. Read length of ip header and copy
	     * the header into our buffer.
	     */
	    ip = (pnet_ip*)buf;

	    hdr_len = pnetIP_HeaderLength( ip );

	    if ( hdr_len < 0 || hdr_len > MAXIPHDRSIZE )
	    {
		perr(E_FATAL,"dgram_recv(): unexpected ip hdr len\n");
		return -1;
	    }

	    ps->aux_info->ai_type = ps->fam;

	    memcpy( ps->aux_info->aux_ip, ip, hdr_len );

	    ps->aux_info->aux_pkt_hoplimit = ip->ip_ttl;

	    ip =  ps->aux_info->aux_ip;

	    swap_ip_fields( ip );

	    /*
	     * XXX This is a dirty hack.
	     */
	    if ( ip->ip_len != htons( (pnet_ushort)got ) )
	    {
		ip->ip_len = htons( (pnet_ushort)got );
	    }
	}
# if defined PNET_HAVE_IPv6
	else
	{
	    /* 
	     * We need to fill in the IPv6 header. Since this is not returned
	     * by the call to recvmsg(), we fill it in ourselves. This happens
	     * first of all here, where we set version, length and next header,
	     * then below, while parsing the CMSG header data, we fill in
	     * the destination address and hoplimit. No flow label or traffic
	     * class is done yet. 
	     */
	    pnet_ip6  * ip6 = (pnet_ip6*)ps->aux_info->aux_ip6;

	    /* Set the first part of the IPv6 header */

            memset( ip6, 0, sizeof( pnet_ip6 ) );

            ip6->ip6_vfc = (6 << 4) & 0xF0;
            ip6->ip6_plen= htons(got);
            ip6->ip6_nxt = ps->proto;   /* XXX --- HACK FIX ME !!! */

	    /* Fill in source of datagram, returned by recvmsg() */
	    memcpy( ip6->ip6_src, ps->peer_addr.pa_s6_addr , 16 );
	}
# endif
    }


# if defined HAVE_MSGHDR_MSG_CONTROL		/* { */

    ps->aux_info->aux_pkt_flags = mhdr.msg_flags;

    LLOG(mhdr.msg_controllen);
    LLOG(sizeof(struct cmsghdr));
    XLOG( mhdr.msg_flags );

    if ( mhdr.msg_flags & MSG_TRUNC )
    {
	pdbg(E_WARN,"dgram_recv(): datagram truncated\n");
    }

    if (mhdr.msg_controllen < sizeof(struct cmsghdr))
    {
	pdbg(E_DBG1,"dgram_recv(): no control information received\n");
	return got;
    }

    if ( mhdr.msg_flags & MSG_CTRUNC )
    {
	perr(E_FATAL,"dgram_recv(): control information truncated\n");
	return got;
    }

    /* Parse the control information */

    for (pcmh = CMSG_FIRSTHDR(&mhdr); pcmh; pcmh = CMSG_NXTHDR(&mhdr,pcmh))
    {
	if ( pcmh->cmsg_level == IPPROTO_IP)
	{
# ifdef IP_RECVDSTADDR	/* { */
	    if (pcmh->cmsg_type == IP_RECVDSTADDR)
	    {
		pdsta = &ps->aux_info->aux_pkt_dst_addr;
		
		addr_set_ip_addr(pdsta,AF_INET,
				 CMSG_DATA(pcmh),sizeof(struct in_addr));
		continue;
	    }
# endif			/* } */
# ifdef IP_RECVIF	/* { */
	    if (pcmh->cmsg_type == IP_RECVIF)
	    {
		struct sockaddr_dl *sdl = (struct sockaddr_dl*)CMSG_DATA(pcmh);

		ifindex = sdl->sdl_index;
		ps->aux_info->aux_pkt_if =
			net_get_if_by_index( AF_INET, ifindex );

		continue;
	    }
# endif			/* } */
# ifdef IP_PKTINFO	/* { */
	    if (pcmh->cmsg_type == IP_PKTINFO)
	    {
		struct in_pktinfo *pkt = (struct in_pktinfo*)CMSG_DATA(pcmh);
		pdsta = &ps->aux_info->aux_pkt_dst_addr;

		addr_set_ip_addr(pdsta,ps->fam,
				 &pkt->ipi_addr,sizeof(struct in_addr));
		ifindex = pkt->ipi_ifindex;

		ps->aux_info->aux_pkt_if = 
			net_get_if_by_index( AF_INET, ifindex );

		continue;
	    }
# endif			/* } */
	    perr(E_DBG1,"DGRAM_RECV: Unknown ancillary data: level=IPPROTO_IP,"
	    	 "type=%d, len=%d\n",pcmh->cmsg_type,pcmh->cmsg_len);
	    continue;
	}

# ifdef IPPROTO_IPV6	/* { */
 	if ( pcmh->cmsg_level == IPPROTO_IPV6 )
	{
# ifdef IPV6_PKTINFO	/* { */
	    if ( pcmh->cmsg_type == IPV6_PKTINFO )
	    {
		struct in6_pktinfo *pkt  = (struct in6_pktinfo*)CMSG_DATA(pcmh);
		pnet_byte *	    paddr= (pnet_byte*)&pkt->ipi6_addr;

		ifindex = pkt->ipi6_ifindex;

		pdsta = &ps->aux_info->aux_pkt_dst_addr;

		addr_set_ip_addr(pdsta,ps->fam,(void*)paddr,
				 sizeof(struct in6_addr));
		ps->aux_info->aux_pkt_if = 
				net_get_if_by_index( AF_INET6, ifindex );

		if ( ps->type == SOCK_RAW )
		{
		    /*
		     * For a raw socket, copy destination address into our
		     * pseudo-ip6 header
		     */
		    pnet_ip6  *	    ip6  = (pnet_ip6*)ps->aux_info->aux_ip6;
		    memcpy( ip6->ip6_dst, paddr, sizeof( struct in6_addr ) );
		}
		continue;
	    }
# endif			/* } */
# ifdef IPV6_HOPLIMIT	/* { */
	    if (pcmh->cmsg_type == IPV6_HOPLIMIT)
	    {
		if ( receiving_udp )
		    ps->aux_info->aux_pkt_hoplimit = 
			(int) ( CMSG_DATA( pcmh )[0] ) ;
		else
		{
		    ps->aux_info->aux_ip6->ip6_hlim =
			(int) ( CMSG_DATA( pcmh )[0] ) ;
		    ps->aux_info->aux_pkt_hoplimit = 
			ps->aux_info->aux_ip6->ip6_hlim;
		}
		continue;
	    }
# endif			/* } */

	    perr(E_DBG1,"DGRAM_RECV: Unknown ancillary data: "
			"level=IPPROTO_IPV6,type=%d, len=%d\n",
			pcmh->cmsg_type,pcmh->cmsg_len);

	    continue;
	}
# endif			/* } */

	perr(E_DBG1,"DGRAM_RECV: Unknown ancillary data: len=%d, "
		    "level=%d, type=%d\n",
		     pcmh->cmsg_len,pcmh->cmsg_level,pcmh->cmsg_type);
    }
# endif 					/* } */

    if ( pdsta )
	pdbg(E_DBG1,"DGRAM_RECV: datagram destination %s\n",
		     pnet_ntop( pdsta, addrbuf, sizeof( addrbuf )));

    if (ps->aux_info->aux_pkt_if)
    {
	pdbg(E_DBG4,"DGRAM_RECV: %d bytes on %s (idx=%d,type=%s), "
		    "hoplimit = %d\n",
		     got,ps->aux_info->aux_pkt_if->pif_name,
		     ps->aux_info->aux_pkt_if->pif_index,
		     pnetIfTypeString(ps->aux_info->aux_pkt_if),
		     ps->aux_info->aux_pkt_hoplimit );
    }
    return got;
# else 					/* }{ */

    DBG(dbg("dgram_recv(ps=%lX,buf=%lX,blen=%d)\n",XX(ps),XX(buf),blen));
    DBG(pdbg(E_DBG4,"DGRAM_RECV: local  %s\n",
	    pnet_ntop( &ps->local_addr, addrbuf, sizeof( addrbuf ) )));

    return net_recvfrom( ps, (void*)buf, blen, 0 );

# endif 				/* }} */
}

static void
dgram_log_flags(int flags)
{
    char buf[sizeof " EOR OOB BCST MCST TRNC CTRNC " ];
    char *p = buf;

    pdbg( E_DBG4, "Flags: 0x%X\n",flags );
    strcpy(buf,"NONE");

# ifdef MSG_EOR
    if (flags & MSG_EOR)	
	p += (int)sprintf(p," EOR");
# endif

# ifdef MSG_OOB
    if (flags & MSG_OOB)
    	p += (int)sprintf(p," OOB");
# endif

# ifdef MSG_BCAST
    if (flags & MSG_BCAST)
    	p += (int)sprintf(p," BCST");
# endif
# ifdef MSG_MCAST
    if (flags & MSG_MCAST)
    	p += (int)sprintf(p," MCST");
# endif

# ifdef MSG_TRUNC
    if (flags & MSG_TRUNC)
    	p += (int)sprintf(p," TRUNC");
# endif

# ifdef MSG_CTRUNC
    if (flags & MSG_CTRUNC)
	p += (int)sprintf(p," CTRUNC");
# endif

    pdbg(E_DBG4,"DGRAM_RECV: msg flags: %s\n",buf);
}

/*----------------------------------------------------------------------*/
/* Exported								*/
/*----------------------------------------------------------------------*/

int
pnetRead(PNETSOCK ps,char *buf,int blen)
{
    return socket_read(ps,buf,blen,0,-1);
}
int
pnetReadBlock(PNETSOCK ps,char *buf,int blen)
{
    return socket_read(ps,buf,blen,1,-1);
}
int
pnetReadTimed(PNETSOCK ps,char *buf,int blen,int block,long msec)
{
    return socket_read(ps,buf,blen,block,msec);
}

/*----------------------------------------------------------------------*/
/* Auxiliary socket data						*/
/*----------------------------------------------------------------------*/

PNETSOCK_AUXINFO
pnetSockAuxInfo(PNETSOCK ps)
{
    return ps->aux_info;
}

int
pnetAuxType( PNETSOCK_AUXINFO paux )
{
    if ( paux->ai_type == AF_INET )
	return PNET_IPv4;
    else if ( paux->ai_type == AF_INET6 )
    	return PNET_IPv6;

    return paux->ai_type;
}
/*----------------------------------------------------------------------*/
/* Auxiliary raw info 							*/
/*----------------------------------------------------------------------*/
pnet_ip *
pnetAuxIP( PNETSOCK_AUXINFO paux )
{
    return paux->aux_ip;
}
pnet_ip6 *
pnetAuxIP6( PNETSOCK_AUXINFO paux )
{
    return paux->aux_ip6;
}
/*----------------------------------------------------------------------*/
/* Auxiliary info 							*/
/*----------------------------------------------------------------------*/
int
pnetAuxFlagEOR(PNETSOCK_AUXINFO paux)
{
# ifdef MSG_EOR
    return paux && paux->aux_pkt_flags & MSG_EOR;
# else
    return paux != paux;
# endif
}
int
pnetAuxFlagOOB(PNETSOCK_AUXINFO paux)
{
# ifdef MSG_OOB
    return paux && paux->aux_pkt_flags & MSG_OOB;
# else
    return paux != paux;
# endif
}
int
pnetAuxFlagBCST(PNETSOCK_AUXINFO paux)
{
# ifdef MSG_BCAST
    return paux && paux->aux_pkt_flags & MSG_BCAST;
# else
    return paux != paux;
# endif
}
int
pnetAuxFlagMCST(PNETSOCK_AUXINFO paux)
{
# ifdef MSG_MCAST
    return paux && paux->aux_pkt_flags & MSG_MCAST;
# else
    return paux != paux;
# endif
}
int
pnetAuxFlagTRUNC(PNETSOCK_AUXINFO paux)
{
# ifdef MSG_TRUNC
    return paux && paux->aux_pkt_flags & MSG_TRUNC;
# else
    return paux != paux;
# endif
}

/* Get the actual destination address of the datagram */
PNETADDR
pnetAuxDstAddr(PNETSOCK_AUXINFO paux)
{
    if (paux)
	return &paux->aux_pkt_dst_addr;
    return NULL;
}

/* Get interface on which datagram was received */
PNETIF
pnetAuxRecvIf(PNETSOCK_AUXINFO paux)
{
    if (paux)
	return  paux->aux_pkt_if;
    return NULL;
}
int
pnetAuxHoplimit( PNETSOCK_AUXINFO paux )
{
    if ( paux )
	return paux->aux_pkt_hoplimit;
    return -1;
}


/*----------------------------------------------------------------------*/
/* Encapsulation of send on a PNetSock					*/
/* This function's generic, it's basically a sendto().			*/
/* Three cases can be distinguished:					*/
/*  1. A connected TCP socket. In this case call sendto with NULL,0	*/
/*	as final arguments.						*/
/*  2. A connected UDP socket. In this case call sendto with NULL,0	*/
/*  3. A non-connected UDP socket. Use the specified PNETADDR		*/
/*----------------------------------------------------------------------*/

int
pnetWriteTo( PNETSOCK ps, PNETADDR pa, const char *buf, int blen )
{
    int sent,last,tries;
    char addrbuf[ PNET_ADDR_STRLEN ];

    DBG(dbg("pnetWriteTo(ps=%lX,pa=%lX,buf=%lX,blen=%d)\n",
	    XX(ps),XX(pa),XX(buf),blen));

    DBG(pdbg(E_DBG4,"WRITETO: local  %s\n",
    		     pnet_ntop( &ps->local_addr, addrbuf, sizeof( addrbuf ) )));
    DBG(pdbg(E_DBG4,"WRITETO: remote %s\n",
		     pnet_ntop( pa, addrbuf, sizeof( addrbuf ))));

    if ( ps->type == SOCK_RAW && ps->fam == AF_INET6 && ps->own_header )
    {
	struct iovec iov[2];
	iov[0].iov_base = (char*)buf;
	iov[0].iov_len  = sizeof( pnet_ip6 );
	iov[1].iov_base = (char*) (buf + sizeof( pnet_ip6 ));
	iov[1].iov_len  = blen - sizeof( pnet_ip6 );
	return net_writev( ps, pa, iov, 2, 0 );
    }
    tries = last = sent = 0;
 
    /* Note that calling this for a non-TCP datagram will only run once
     * through the loop.  */
    while (sent < blen)
    {
	if ( ps->connected )
	    last = send( ps->sd,(void*) buf, (size_t) (blen - sent), 0);
	else
	    last = sendto(ps->sd,(void*) buf,(size_t) (blen - sent) , 0,
			  &pa->pa_saddr, pa->pa_len ) ;
        if (last > 0)
	{
	    netStatsSentData( last );
            sent += last,buf += last,tries++;
	}
	else if ( last < 0 )
	{
	    if ( errno == EINTR )
		continue;

            perr(E_FATAL,"pnetWriteTo(): send of %d bytes to %s failed.\n",
                 blen - sent, pnet_ntop( pa, addrbuf, sizeof( addrbuf ) ));
	    NETERR( "pnetWriteTo()" );
            perr(E_FATAL,"pnetWriteTo(): last=%d,sent=%d,tries=%d\n",
                last,sent,tries);
            break;
        }
    }
    return sent;
}
/*----------------------------------------------------------------------*/
/* Write on a connected socket, or a socket with a valid peer address   */
/*----------------------------------------------------------------------*/
int
pnetWrite( PNETSOCK ps, const char *buf, int blen )
{
    if ( ! ps->connected && ! ps->peer_addr.init_addr )
    {
	perr(E_FATAL,"pnetWrite(): socket not connected or no peer address.\n");
	return -1;
    }

    return pnetWriteTo( ps, &ps->peer_addr, buf, blen );
}
