/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


/*
 * $Id: sockopts.c,v 1.15 2002/10/10 12:27:09 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:		sockopts.c
 * created on:		Thu Jun  6 07:03:46 CEST 2002
 * created by:		teddykgb
 * project: 		Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*----------------------------------------------------------------------*/
/* Getting/Setting of socket options.					*/
/*----------------------------------------------------------------------*/

int
pnetSockSetLinger(PNETSOCK ps,int sec)
{  
    struct linger ling;
 
    ling.l_onoff = sec > 0;
    ling.l_linger= sec;

    LLOG(ling.l_onoff);
    LLOG(ling.l_linger);

    if (setsockopt(ps->sd,SOL_SOCKET,SO_LINGER,
                   (const void*)&ling,1))
    {
        perr(E_FATAL,"pnetSockSetLinger(): %s\n",neterr());
        return -1;
    }

    return 0;
}  
int
pnetSockSetKeepalive(PNETSOCK ps,int on)
{
    return net_setsockopt(ps,SOL_SOCKET,SO_KEEPALIVE,on);
}
int
pnetSockSetReuseaddr(PNETSOCK ps,int on)
{
    return net_setsockopt(ps,SOL_SOCKET,SO_REUSEADDR,on);
}  
int
pnetSockSetBroadcast(PNETSOCK ps,int on)
{
    return net_setsockopt(ps,SOL_SOCKET,SO_BROADCAST,on);
}
int
pnetSockSetSendbuf( PNETSOCK ps, int size )
{
    if ( net_setsockopt(ps,SOL_SOCKET,SO_SNDBUF, size ) )
	return -1;

    pdbg(E_INFO,"Send buffer for socket %lX set to %d\n",XX(ps),size);

    return 0;
}
int
pnetSockSetRecvbuf( PNETSOCK ps, int size )
{
    if ( net_setsockopt(ps,SOL_SOCKET,SO_RCVBUF, size ) )
	return -1;
    pdbg(E_INFO,"Receive buffer for socket %lX set to %d\n",XX(ps),size);
    return 0;
}
int
pnetSockSetHops( PNETSOCK ps, int hop )
{
    int ret = 0;

# ifdef PNET_HAVE_IPv6		/* { */
    if ( ps->fam == AF_INET6 )
    {
#   ifdef IPV6_UNICAST_HOPS
	return net_setsockopt(ps,IPPROTO_IPV6,IPV6_UNICAST_HOPS,hop);
#   else
	perr(E_FATAL,"pnetSockSetHops(): IPPROTO_IPV6 not supported\n");
	return -1;
	ps = ps; hop = hop;
#   endif
    }
    else 
# endif				/* } */
    if ( ps->fam == AF_INET )
	ret = net_setsockopt(ps,IPPROTO_IP,IP_TTL,hop);

    return ret;
}
int
pnetSockGetHops( PNETSOCK ps, int* pHop )
{
    int ret = 0;

# ifdef PNET_HAVE_IPv6		/* { */
    if ( ps->fam == AF_INET6 )
    {
# ifdef IPV6_UNICAST_HOPS
	return net_getsockopt(ps,IPPROTO_IPV6,IPV6_UNICAST_HOPS,pHop);
# else
	perr(E_FATAL,"pnetSockGetHops(): IPPROTO_IPV6 not supported\n");
	return -1;
	ps = ps; pHop = pHop;
# endif
    }
    else
# endif				/* } */
    if ( ps->fam == AF_INET )
	ret = net_getsockopt(ps,IPPROTO_IP,IP_TTL,pHop);

    return ret;
}
int
net_setsockopt_iprecvdstaddr(PNETSOCK ps,int on)
{
# ifdef IP_RECVDSTADDR

    return net_setsockopt(ps,IPPROTO_IP,IP_RECVDSTADDR,on);

# else
    perr(E_FATAL,"net_setsockopt_iprecvdstaddr(): IP_RECVDSTADDR not supported\n");
    return -1;
    ps = ps; on = on;
# endif
}
int
net_setsockopt_iprecvif(PNETSOCK ps,int on)
{
# ifdef IP_RECVIF

    return net_setsockopt(ps,IPPROTO_IP,IP_RECVIF,on);

# else
    perr(E_FATAL,"net_setsockopt_iprecvif(): IP_RECVIF not supported\n");
    return -1;
    ps = ps; on = on;
# endif
}
int
net_setsockopt_ippktinfo(PNETSOCK ps,int on)
{
# ifdef IP_PKTINFO
    return net_setsockopt(ps,IPPROTO_IP,IP_PKTINFO,on);
# else
    perr(E_FATAL,"net_setsockopt_ippktinfo(): IP_PKTINFO not supported\n");
    return -1;
    ps = ps; on = on;
# endif
}
int
net_setsockopt_ipv6pktinfo(PNETSOCK ps,int on)
{
# if defined IPV6_RECVPKTINFO 
    return net_setsockopt(ps,IPPROTO_IPV6,IPV6_RECVPKTINFO, !!on);
# elif defined IPV6_PKTINFO
    return net_setsockopt(ps,IPPROTO_IPV6,IPV6_PKTINFO, !!on);
# else
    perr(E_FATAL,"net_setsockopt_ipv6pktinfo(): "
	 "IPV6_PKTINFO not supported\n");
    return -1;
    ps = ps; on = on;
# endif
}
int
net_setsockopt_iphdrincl(PNETSOCK ps,int on)
{
# ifdef IP_HDRINCL

    return net_setsockopt(ps,IPPROTO_IP,IP_HDRINCL,on);
	    
# else
    perr(E_FATAL,"net_setsockopt_iphdrincl(): IP_HDRINCL not supported\n");
    return -1;
    ps = ps; on = on;
# endif
}

static int net_sockopt(PNETSOCK,int,int,int *,int);

int
net_setsockopt(PNETSOCK ps,int lev,int opt,int on)
{
    DBG(dbg("net_setsockopt(ps=%lX,lev=%lX,opt=%lX,on=%d)\n",
	    XX(ps),lev,opt,on));

    return net_sockopt(ps,lev,opt,&on,0);
}
int
net_getsockopt(PNETSOCK ps,int lev,int opt,int *on)
{
    return net_sockopt(ps,lev,opt,on,1);
}
static int
net_sockopt(PNETSOCK ps,int optlev,int opt,int *pon,int get)
{
    const char *optnam;
    int		not_tcp = 0;
    int		not_udp = 0;
    int		not_raw = 0;

    int		getonly = 0;
    int		on = *pon;
    int		ret = 0;
    unsigned 	len;

    len = sizeof(on);

    if (optlev == SOL_SOCKET)
    {
	switch (opt)
	{
	case SO_DEBUG:
	    optnam = "SO_DEBUG";
	    break;
# ifdef SO_ACCEPTCONN
	case SO_ACCEPTCONN:
	    optnam = "SO_ACCEPTCONN";
	    getonly = 1;
	    not_udp = not_raw = 1;
	    break;
# endif 
	case SO_REUSEADDR:
	    optnam = "SO_REUSEADDR";
	    break;
	case SO_KEEPALIVE:
	    optnam = "SO_KEEPALIVE";
	    not_udp = not_raw = 1;
	    break;
	case SO_DONTROUTE:
	    optnam = "SO_DONTROUTE";
	    break;
	case SO_SNDBUF:
	    optnam = "SO_SNDBUF";
	    break;
	case SO_RCVBUF:
	    optnam = "SO_RCVBUF";
	    break;
	case SO_BROADCAST:
	    optnam = "SO_BROADCAST";
	    not_tcp = not_raw = 1;
	    break;
# ifdef SO_USELOOPBACK
	case SO_USELOOPBACK:
	    optnam = "SO_USELOOPBACK";
	    break;
# endif
	default:
	    perr(E_FATAL,"net_sockopt(): option %d unknown or unsupported at"
			 " level SOL_SOCKET.\n",opt);
	    return -1;
	}
    }
    else if (optlev == IPPROTO_IP)
    {
	switch (opt)
	{
# ifdef IP_RECVDSTADDR
	case IP_RECVDSTADDR:
	    not_tcp = 1;
	    optnam = "IP_RECVDSTADDR";
	    break;
# endif
# ifdef IP_RECVIF
	case IP_RECVIF:
	    not_tcp = 1;
	    optnam = "IP_RECVIF";
	    break;
# endif
# ifdef IP_HDRINCL
	case IP_HDRINCL:
	    not_udp = not_tcp = 1;
	    optnam = "IP_HDRINCL";
	    break;
# endif
# ifdef IP_PKTINFO
	case IP_PKTINFO:
	    not_tcp = 1;
	    optnam = "IP_PKTINFO";
	    break;
# endif
# ifdef IP_TTL
	case IP_TTL:
	    optnam = "IP_TTL";
	    break;
# endif
	default:
	    perr(E_FATAL,"net_sockopt(): option %d unknown or unsupported at"
			 " level IPPROTO_IP.\n",opt);
	    return -1;
	}
    }
# ifdef IPPROTO_IPV6
    else if (optlev == IPPROTO_IPV6)
    {
	switch (opt)
	{
# ifdef IPV6_PKTINFO
	case IPV6_PKTINFO:
	    not_tcp = 1;
	    optnam = "IPV6_PKTINFO";
	    break;
# endif
# ifdef IPV6_RECVPKTINFO
	case IPV6_RECVPKTINFO:
	    not_tcp = 1;
	    optnam = "IPV6_RECVPKTINFO";
	    break;
# endif
# ifdef IPV6_UNICAST_HOPS
	case IPV6_UNICAST_HOPS:
	    optnam = "IPV6_UNICAST_HOPS";
	    break;
# endif
# ifdef IPV6_NEXTHOP
	case IPV6_NEXTHOP:
	    optnam = "IPV6_NEXTHOP";
	    break;
# endif
# ifdef IPV6_HOPLIMIT
	case IPV6_HOPLIMIT:
	    optnam = "IPV6_HOPLIMIT";
	    break;
# endif
	default:
	    perr(E_FATAL,"net_sockopt(): option %d unknown or unsupported at"
			 " level IPPROTO_IPV6.\n",opt);
	    return -1;
	}
    }
# endif /* ! IPPROTO_IPV6 */
    else
    {
	perr(E_FATAL,
	    "net_sockopt(): option level %d unknown or unsupported\n.",optlev);
	return -1;
    }

    if ( not_udp && ps->type == SOCK_DGRAM)
    	{ TYPEERR("net_sockopt()"); return -1; }
    if ( not_tcp && ps->type == SOCK_STREAM)
    	{ TYPEERR("net_sockopt()"); return -1; }
    if ( not_raw && ps->type == SOCK_RAW)
    	{ TYPEERR("net_sockopt()"); return -1; }

    if (get)
	ret = getsockopt(ps->sd,optlev,opt,(void*)&on,&len);
    else
	ret = setsockopt(ps->sd,optlev,opt,(const void*)&on,len);

    if (ret)
    {
	perr(E_FATAL,"net_sockopt(ps=%lX,opt=%s,on=%d,get=%s): %s\n",
	     XX(ps),optnam,on,PYESNO(get),neterr());
	return -1;
    }
    
    pdbg(E_DBG4,"net_sockopt(ps=%lX,opt=%s,on=%d,get=%s): success.\n",
	 XX(ps),optnam, on,PYESNO(get));

    if (get)
	*pon = on;

    return 0;
}
