/*
 * This file is part of
 *
 * LIBPNET6: a Portable Network Library
 *
 * LIBPNET6 is Copyright (c) 2002, Peter Bozarov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by Peter Bozarov.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 * ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * $Id: sysutil.c,v 1.29 2002/12/15 11:22:14 kingofgib Exp $
 */

/*----------------------------------------------------------------------*
 * filename:            sysutil.c
 * created on:          Thu Jul  4 15:52:54 CEST 2002
 * created by:          teddykgb
 * project:             Portable Network Library
 *----------------------------------------------------------------------*/

# include "local.h"

/*
 * Separtor to use for appending interface names to link local IPv6
 * addresses.
 */

pnet_ushort pnet_htons( pnet_ushort s ) { return htons( s ); }
pnet_ulong  pnet_htonl( pnet_ulong  l ) { return htonl( l ); }
pnet_ushort pnet_ntohs( pnet_ushort s ) { return htons( s ); }
pnet_ulong  pnet_ntohl( pnet_ulong  l ) { return htonl( l ); }

/*----------------------------------------------------------------------*/
/* Miscellaneous system utilities					*/
/*----------------------------------------------------------------------*/
pid_t
sys_getpid( void )
{
# ifdef STDWin32
    return (pid_t) GetCurrentProcessId();
# else
    return getpid();
# endif
}

char*
sys_getenv( const char * var )
{
    return getenv( var );
}
/*----------------------------------------------------------------------*/
/* Create a child process. sys_fork() will do this and return the 	*/
/* process id of the created child, or -1 if the operation failed.	*/
/*----------------------------------------------------------------------*/
pid_t
sys_fork( void )
{
    pid_t pid;

# ifdef HAVE_FORK
    pid = fork();
# endif

/* NOT READY YET: probably won't bother for know, we can just use Threads */
# ifdef STDWin32
    int                 ret;
    STARTUPINFO         si;
    PROCESS_INFORMATION pi;
 
    /* Create a new process */

    memset(&pi, 0, sizeof(PROCESS_INFORMATION));
    memset(&si, 0, sizeof(STARTUPINFO));
    si.cb = sizeof(STARTUPINFO);
    si.dwFlags = STARTF_USESHOWWINDOW;
    si.wShowWindow = SW_HIDE;

    ret = CreateProcess(NULL,"?",NULL,NULL,0,0,NULL,NULL,&si,&pi);

    if (!ret)           /* Create Process failed */
        { LLOG(ret); return -1; }

    pid = (pid_t) pi.dwProcessId;

    CloseHandle(pi.hProcess);
    CloseHandle(pi.hThread);
# endif

    return pid;
}

/*----------------------------------------------------------------------*/
/* Return the time offset from UTC on the local system. Value returned  */
/* is in minutes east from UTC. (+ towards Moscow, - towards New York	*/
/*----------------------------------------------------------------------*/

int
sys_timediff( void )
{
    int diff;
# ifdef STDWin32
    TIME_ZONE_INFORMATION	tzinfo;

    GetTimeZoneInformation( &tzinfo );

    /* utc = local time + bias */

    diff = - (tzinfo.Bias + tzinfo.DaylightBias);
# else
    time_t       	t;
    register struct tm * gtm;
    register struct tm * local_tm;
    struct tm            utc_tm;

    t= time( &t );

    /*
     * Read UTC time and store it locally.
     */

    if ( ! (gtm = gmtime( &t )) )
        return 0;

    utc_tm = *gtm;

    /*
     * Read localtime.
     */

    if ( ! (local_tm = localtime( &t )) )
        return 0;

    /*
     * Account for possible spanning of a day. If the years are different,
     * use that, else use the absolute year day difference.
     */

    if ( ! (diff = local_tm->tm_year - utc_tm.tm_year) )
        diff = local_tm->tm_yday - utc_tm.tm_yday;

    return (local_tm->tm_hour - utc_tm.tm_hour - diff * 24) * 60;

# endif
    return diff;
}
# ifndef HAVE_GETTIMEOFDAY		/* { */

int
gettimeofday( struct timeval * tv, struct timezone * tz )
{
# ifdef STDWin32
    SYSTEMTIME	stime;

    tv->tv_sec = time( NULL );
    GetLocalTime( &stime );
    tv->tv_usec= stime.wMilliseconds * 1000;
# endif

    return 0;
}
# endif					/* } */

/*----------------------------------------------------------------------*/
/* Get the current time, hh:mm:ss.milliseconds.				*/
/*----------------------------------------------------------------------*/
char *
sys_get_current_time( char *buf, unsigned int len )
{
    static int first_time = 1;
    static int utc_offset = 0;

    struct timeval 	tv;
    int	l;

    if ( first_time )
    {
	first_time = 0;
	utc_offset = sys_timediff() * 60; /* We need utc offset in seconds */
	LLOG( utc_offset );
    }

    if ( len < sizeof( "XX:XX:XX.XXXXXX" ) )
	return (char*)"";

    if ( gettimeofday( &tv, NULL ) )
	return (char*)"";

    /* Get day portion of time, accounting for the timezone */

    l = ( tv.tv_sec + utc_offset) % ( 60 * 60 * 24 );

    sprintf( buf, "%02d:%02d:%02d.%06d",
	    l / 3600, (l % 3600) / 60, l % 60, (int) tv.tv_usec );

    return buf;
}

char*
pnetGetTimeStamp( char * buf, unsigned int len )
{
    return sys_get_current_time( buf, len );
}

/*----------------------------------------------------------------------*/
/* NOTE: this is in NO way a getopt() "compliant" function!		*/
/* This is just a quick hack to enable simple option processing.	*/
/* Supported:								*/
/* 	-p<arg> or -p arg:	option w/ an argument			*/
/*	-sSfascv:		one or more switches together		*/
/*----------------------------------------------------------------------*/
const char *	pnetOptarg = NULL;
int		pnetOptind = 1;
int		pnetOpterr = 1;
int    		pnetOptopt = 0;

int
pnetGetopt( int argc, const char ** argv, const char *opts )
{
    const char * curr;
    const char * p;
    static const char *next_opt = NULL;
    int 	 ret_opt;

    if ( !opts )
        { XLOG( opts ); return -1; }

    if ( pnetOptind == 1 )		    /* First time around */
    {
	if ( *opts == ':' )		/* Set error flag to OFF */
	    pnetOpterr = 0;
    }

    if ( pnetOptind >= argc )              /* Nothing left to do */
        { LLOG( pnetOptind >= argc ); return -1; }

    if ( next_opt && *next_opt )
	{ curr = next_opt; next_opt++; }
    else
    {
	curr = argv[pnetOptind];

	if ( *curr !=  '-' || !*(curr+1) )
	    return -1;              /* First non option argument */

	curr++;
	next_opt = curr + 1;

    }

    if ( !*next_opt )
	pnetOptind++;

    if ( !(p = strchr( opts, *curr )) )
    {
	if ( pnetOpterr )
	    fprintf(stderr, "Unknown option %c\n", *curr );
        pnetOptopt = *curr;
	return '?';
    }

    ret_opt = *p;

    if ( *(p + 1) == ':' )      /* Option has an argument */
    {
	/* Agument is either glued to the option letter, of follows it 	*/
	/* separated by a space						*/

	curr++;

	next_opt = NULL;

        if ( *curr )
            pnetOptarg = curr;
        else
            pnetOptarg = pnetOptind <= argc ? argv[pnetOptind] : NULL;

	pnetOptind++;

        if ( !pnetOptarg || *pnetOptarg == '-' )
	{
	    if ( pnetOpterr )
		fprintf( stderr, "Option %c requires an argument\n", ret_opt );

	    pnetOptopt = ret_opt;
	    ret_opt = '!';
	    pnetOptarg = NULL;
	}  
    }

    return (ret_opt);
}

/*----------------------------------------------------------------------*/
/* Detach the current process from any terminals (i.e. become a daemon)	*/
/*  1. fork() once and have the parent process exit.			*/
/*  2. Try to make this process session group leader (drops the		*/
/*     controlling terminal). Becomes only process in session or process*/
/*     group.								*/
/*  3. Ignore SIGHUP (see next step for the reason)			*/
/*  3a.While we're at it, ignore SIGINT, and SIGWINCH.			*/
/*  4. fork() once more, and have the first child exit. The grandchild  */
/*     of the original process is the program that will eventually run. */
/*     Ignoring SIGHUP in Step 3 ensures that this program will never	*/
/*     reacquire a controlling terminal should it open a terminal 	*/
/*     in the future: that's because the second fork() makes this child */
/*     no longer a session leader.					*/
/*  5. Change to the given directory, or / if nothing is specified.	*/
/*  6. Call umask( 0 ) to clear the umask, i.e. don't use the umask	*/
/*     inherited from our ancestors.					*/
/*  7. Close stdin, stdout and stderr.					*/
/*									*/
/* Well, this seems like a lot of work, but, once we've done it, we'll  */
/* have a properly detached daemon. 					*/
/*----------------------------------------------------------------------*/

# ifndef STDWin32
#   include <sys/resource.h>
#   include <sys/stat.h>
# endif
int
pnetDetachProcess( const char * progname, const char * work_dir )
{
# ifndef STDWin32					/* { */

    PNET_SIGFUNC sigfunc;
    struct rlimit rlim;
    int i;

    /* 1 */
    switch (fork())
    {
    case -1:
	LOGSYSERR( "pnetDetachProcess()" );
	return -1;
    case 0:
	break;
    default:
	exit(0);
    }

    /* 2 */
    if (setsid() == -1)
	{ LOGSYSERR( "pnetDetachProcess()" ); return -1; }

    pdbg( E_DBG4, "setsid() OK\n" );
    /* 3 */
    if ( (sigfunc = signal( SIGHUP, SIG_IGN )) == SIG_ERR )
	{ LOGSYSERR( "pnetDetachProcess()" ); return -1; }

    pdbg( E_DBG4, "Ignore SIGHUP: OK\n" );

    if ( (sigfunc = signal( SIGINT, SIG_IGN )) == SIG_ERR )
	{ LOGSYSERR( "pnetDetachProcess()" ); }

    pdbg( E_DBG4, "Ignore SIGINT: %s\n", sigfunc == SIG_ERR ? "FAILED" : "OK" );

# ifdef SIGWINCH
    if ( (sigfunc = signal( SIGINT, SIG_IGN )) == SIG_ERR )
	{ LOGSYSERR( "pnetDetachProcess()" ); }

    pdbg( E_DBG4, "Ignore SIGWINCH: %s\n", sigfunc == SIG_ERR ? "FAILED" : "OK" );
# endif

    /* 4 */
    switch (fork())
    {
    case -1:
	LOGSYSERR( "pnetDetachProcess()" );
	return -1;
    case 0:
	break;
    default:
	exit(0);
    }

    /* 5 */
    if ( ! work_dir )
	work_dir = "/";

    if ( chdir( work_dir ) )
	{ LOGSYSERR( "pnetDetachProcess()" ); return -1; }

    pdbg( E_INFO, "Process %d successfully detached, directory is now %s\n",
		  sys_getpid(), work_dir );

    /* 6. */
    umask( 0 );

    /* 7. Close all files. */

    if ( getrlimit( RLIMIT_NOFILE, &rlim ) < 0 ||
	 rlim.rlim_max == RLIM_INFINITY )
	rlim.rlim_max = 1024;
    for ( i = 0; i < rlim.rlim_max; i++ )
	close( i );

    i = open( "/dev/null", O_RDWR );
    if ( i < 0 )
	{ LOGSYSERR( "pnetDetachProcess()" ); return -1; }

    dup2( i, 0 );
    dup2( i, 1 );
    dup2( i, 2 );
 
    if ( i > 2 )
	close( i );

# endif 						/* } */

    print_to_syslog( 1, progname );

    return 0;
}
