# include <pnet6.h>
# include <pnet6tlp.h>
# include <string.h>
# include <stdlib.h>

/*----------------------------------------------------------------------*/
/* Open a raw socket and listen for raw messages. These will normally	*/
/* only include IP messages with protocols other than TCP/UPD.		*/
/* If you want to listen or write all types of raw datagrams, then check*/
/* out the ../pktaccess directory for examples on how to do that.	*/
/*----------------------------------------------------------------------*/
static int 	datagram_proto = PNET_IPPROTO_ICMP;

static int
usage( const char *progname )
{
    printf("%s usage:\n", progname );
    printf("    -a address_family: inet4 or inet6, default inet4\n");
    printf("    -p protocol type:  a number, default ICMP (1)\n");
    printf("    -h host to send datagram to: a string, no default value\n");
    printf("    -s use this address as source address\n");
    printf("    -m ICMP message type to be filtered out, default 0 (none)\n");
    return 1;
}

static int
handle_packet( PNETSOCK_AUXINFO paux )
{
    PNETADDRSTORAGE	pas1;
    PNETADDRSTORAGE	pas2;
    PNETADDR 		pa1	= (PNETADDR)pas1;
    PNETADDR 		pa2	= (PNETADDR)pas2;
    pnet_ip* 		pip;
    pnet_ip6* 		pip6;
    char		addrbuf[ PNET_ADDR_BUFSIZ ];

    if ( pnetAuxType( paux ) == PNET_IPv4 )
    {
	/* We have a IP packet */
	int proto = 0;

	pip = pnetAuxIP( paux );

	pnetIP_Source( pip, pa1 );
	pnetIP_Destination( pip, pa2 );

	proto= pnetIP_Protocol( pip );

	printf( "IP %d, hlen %d, plen %d, proto %d, hlim = %d, ",
		pnetIP_Version( pip ), pnetIP_HeaderLength( pip ),
		pnetIP_PayloadLength( pip ), proto, pnetIP_Hoplimit( pip ) );
	printf( "src %s, ",
		pnet_ntop( pa1, addrbuf, sizeof( addrbuf ) ) );
	printf( "dst %s\n",
		pnet_ntop( pa2, addrbuf, sizeof( addrbuf ) ) );

	if ( proto == PNET_IPPROTO_ICMP )
	{
	    pnet_icmp *icmp= (pnet_icmp*)pnetIP_Payload( pip );

	    printf("ICMP, type %s, code %s\n",
		    pnetICMP_TypeDesc( PNET_IPPROTO_ICMP, icmp->icmp_type ),
		    pnetICMP_CodeDesc( PNET_IPPROTO_ICMP, icmp->icmp_type,
				       icmp->icmp_code ) );
	}
    }
    else if ( pnetAuxType( paux ) == PNET_IPv6 )
    {
	int proto;

	pip6= pnetAuxIP6( paux );

	pnetIP_Source( pip6, pa1 );
	pnetIP_Destination( pip6, pa2 );

	proto= pnetIP_Protocol( pip6 );

	printf( "IP %d, hlen %d, plen %d, proto %d, hlim = %d ",
		pnetIP_Version( pip6 ), pnetIP_HeaderLength( pip6 ),
		pnetIP_PayloadLength( pip6 ), proto , pnetIP_Hoplimit( pip6 ));
	printf( "src %s, ",
		pnet_ntop( pa1, addrbuf, sizeof( addrbuf ) ) );
	printf( "dst %s\n",
		pnet_ntop( pa2, addrbuf, sizeof( addrbuf ) ) );

	if ( proto == PNET_IPPROTO_ICMPV6 )
	{
	    pnet_icmp6 *icmp= (pnet_icmp6*)pnetIP_Payload( pip6 );

	    printf("ICMPv6, type %s, code %s\n",
		    pnetICMP_TypeDesc( PNET_IPPROTO_ICMP, icmp->icmp6_type ),
		    pnetICMP_CodeDesc( PNET_IPPROTO_ICMP, icmp->icmp6_type,
				       icmp->icmp6_code ) );
	}
    }
    else
    {
	printf("Packet type %d unrecognized\n", pnetAuxType( paux ) );
    }

    return 0;
}
/* Receive a raw packet */
static int
raw_recv( PNETSOCK ps, void * data )
{
    int  		ret;
    char 		buf[128];
    PNETSOCK_AUXINFO	paux;

    /* Enter non-blocking read (i.e. don't wait for sizeof(buf) bytes	*/
    /* to arrive, but return immediately upon arrival of any data) 	*/ 

    ret = pnetRead( ps, buf, sizeof( buf ) );

    if ( ret == PNET_READ_TIMED_OUT )
    {
	printf("pnetRead() timed out\n"); /* Should not happen */
	return -1;
    }
    else if ( ret == PNET_READ_ERROR )
    {
	printf("pnetRead(): input error\n");
	return -1;
    }

    printf( "Got a raw packet of length = %d\n", ret );

    if ( (paux = pnetSockAuxInfo( ps )) )
    {
	char   abuf[128];
	PNETIF pif = pnetAuxRecvIf( paux );

	if ( pif )
	    printf("Packet received on interface %s\n", pnetIfName(pif));
	printf("Packet destination was       %s\n",
		pnetAddrToString( pnetAuxDstAddr( paux ), abuf, sizeof( abuf ) ));
	handle_packet( paux );
    }
    return 0;
}
static int
raw_send( PNETSOCK ps, PNETADDR pa, PNETADDR src )
{
    pnet_ip  	ip;
    pnet_ip6	ip6;
    void *	pip;
    pnet_tcp 	tcp;
    pnet_udp 	udp;
    char	buf[] = "Hello";

    int		dglen = sizeof( pnet_ip ) + strlen( buf );

    dglen += sizeof( pnet_tcp );

    printf( "Sending datagram of length %d\n", dglen );

    if ( ps && pnetSockFamily( ps ) == PNET_IPv4 )
    {
	pnetIP4_Build( &ip, 0, dglen, 7890, 0, 0, 49,
		       datagram_proto, src, pa );
	pip = (void*)&ip;
    }
    else
    {
	pnetIP6_Build( &ip6, 0xAB, 0x0FF00, sizeof( tcp ) + strlen( buf ),
		      datagram_proto, 143, src, pa );
	pip = (void*)&ip6;
    }

    pnetTCP_Build( &tcp, 34565, 9999, 998, 123,
		   0 , TH_SYN |TH_FIN , 4500, 0x0, 0 );

    pnetUDP_Build( &udp, 5678, 9999, sizeof( udp ) + strlen( buf ), 0x0 );

    pnetHexdumpX( stdout, (pnet_byte*)pip, sizeof( ip6 ), 4, 0 );

    printf("Version  = %d\n", pnetIP_Version( pip ));
    printf("Headerlen= %d\n", pnetIP_HeaderLength( pip ));
    printf("Payloadl = %d\n", pnetIP_PayloadLength( pip ));
    printf("Protocol = %d\n", pnetIP_Protocol( pip ));
    printf("Hlim     = %d\n", pnetIP_Hoplimit( pip ));

    if ( ps )
	/*
	return pnetTCP_Send( ps, pa, pip, &tcp,
				    (pnet_byte*)buf, strlen( buf ) );
			    */
	return pnetIP_Send( ps, pa, pip,
			    (pnet_byte*) &tcp, sizeof( tcp ), 16, 1,
			    /* (pnet_byte*) &udp, sizeof( udp ), 6, 1, */
			    /* 0, 0, 0, 0, */
			    (pnet_byte*)buf, strlen( buf ) );
    return 0;
}

int
main(int argc, const char **argv)
{
    PNETSOCK	rawps = NULL;
    PNETADDR	pa,psrc;
    const char *host = NULL;
    const char *thishost= "localhost";
    int 	fam = PNET_IPv4;
    int		msg_type = 0;
    int		c;
    char 	addrbuf[ PNET_ADDR_BUFSIZ ];

    while ( (c = pnetGetopt( argc, argv, ":a:h:m:p:s:")) != -1 )
    {
	switch ( c )
	{
	case 'a':
	    if ( ! pnetOptarg )
	    	return usage( argv[0] );
	    if ( ! strcmp( pnetOptarg, "inet" ) )
		fam = PNET_IPv4;
	    else if ( ! strcmp( pnetOptarg, "inet6" ) )
	    	fam = PNET_IPv6;
	    else
	    {
		printf("Unknown address family %s\n",pnetOptarg );
		return usage( argv[0] );
	    }
	    break;
	case 'h':
	    if ( ! pnetOptarg )
	    	return usage( argv[0] );
	    host = strdup( pnetOptarg );
	    break;
	case 'm':
	    if ( ! pnetOptarg )
	    	return usage( argv[0] );
	    msg_type = atoi( pnetOptarg );
	    break;
	
	case 'p':
	    if ( ! pnetOptarg )
	    	return usage( argv[0] );
	    datagram_proto = atoi( pnetOptarg );
	    break;

	case 's':
	    thishost = strdup( pnetOptarg );
	    break;
	    
	
	case '!':
	    printf(" Option %c requires an argument\n", pnetOptopt );
	    return 1;

	case '?':
	    printf("Unknown argument %c\n", pnetOptopt );
	    return 1;
	}
    }
    if ( !host )
    	return usage( argv[0] );

    if ( ! (pa = pnetAddrResolve( fam, host ) ) )
    {
	printf("Cannot resolve %s\n",host);
	return 1;
    }
    printf("Resolved '%s' --> '%s'\n",host,
	    pnetAddrToString( pa, addrbuf, sizeof( addrbuf ) )  );

    if ( ! (psrc = pnetAddrResolve( fam, thishost )) )
    {
	printf("Cannot resolve %s\n",thishost);
	return 1;
    }
    printf("Resolved '%s' --> '%s'\n",thishost,
	    pnetAddrToString( psrc, addrbuf, sizeof( addrbuf ) )  );

    if ( ! (rawps = pnetRAWSocket2( fam, datagram_proto )) )
    {
	printf("Failed to open RAW socket. Do you have root privileges?\n");
	return 1;
    }

    pnetSockOwnIPHeader( rawps , 1 );

    printf("msg_type %d, fam = %d, datagram_proto = %d\n",msg_type, fam, datagram_proto );

    if ( msg_type && fam == PNET_IPv6 )
    {
	/* Install ICMPv6 Filter on the socket */

	PNET_ICMPv6_Filter filter;
	filter = pnetICMP_FilterOpen(); 

	pnetICMP_FilterSetPass( filter, msg_type );
	printf("Installing ICMPv6 filter for message type %d\n", msg_type );
	if ( pnetICMP_FilterInstall( rawps, filter ) )
	    printf("Cannot install filter \n");
	pnetICMP_FilterClose( filter );
    }

    printf("Raw socket opened successfully.\n");
    printf("Listening for raw datagrams with protocol %d\n",datagram_proto);

    /* Add the callback to be invoked when some data arrives on the socket*/
    pnetSockAddReadcallback( rawps, raw_recv, NULL );

    printf("Resolved '%s' --> '%s'\n",thishost,
	    pnetAddrToString( psrc, addrbuf, sizeof( addrbuf ) )  );

    raw_send( rawps, pa, psrc );

    if ( pnetListen( rawps, NULL ) || pnetStartListen( 0 ) ) 
	printf("Can't establish listen socket \n");

    pnetClose(rawps);
    pnetAddrFree( pa );

    return 0;
}
