/*
 * $Id: server.c,v 1.38 2002/11/05 08:35:35 kingofgib Exp $
 */
# include <pnet6.h>
# include <stdio.h>

# include <string.h>

/*----------------------------------------------------------------------*/
/* A simple implementation of an echo server. The server listens on 	*/
/* a number of ports (just so that we can demonstrate the ease with	*/
/* which a single listen loop can handle different types of sockets).	*/
/* As long as the client does not close the connection, communication   */
/* remains possible is active. This makes is sort of a ``multi-echo''   */
/* server.								*/
/*----------------------------------------------------------------------*/

static FILE *	outstream = 0;	/* Stream for output */

/*
 * The socket read callback. The pointer 'data' has been passed to PNET6
 * during the assignment of the read callback for this socket. See below
 * in main().
 * Note that, with the exception of highly protocol specific information 
 * (the auxiliary socket info stuff), at no time are we aware of the
 * actual type of address family (IP/IPv6/Local) and/or protocol (UDP/TCP).
 *
 * We explicitly use a small buffer of 10 bytes. Any datagram larger than
 * that will be truncatated. We can find out if a datagram was truncated
 * by using the pnetAuxFlagTRUNC() function (see below).
 */

static int
Read(PNETSOCK ps,void *data)
{
    PNETADDR		from;		/* Connection came from here */
    PNETADDR		to  ;		/* Connected locally here */
    PNETSOCK_AUXINFO 	paux ;		/* For socket auxiliary data */
    char 		buf[10];	/* Read buffer */
    char		addrbuf[PNET_ADDR_STRLEN]; /* To print addresses */
    int 		ret;
    int			timeout = 10000;

    /*
     * If data is 0, then the read callback is meant for a UDP socket.
     * We set the timeout to 0 in that case, since, when Read() is invoked
     * for a UDP socket we are guaranteed that there is a datagram for us
     * to read immediately. For TCP, the read callback is called as soon
     * as the TCP hand-shake is completed, which does not necessarily 
     * mean there is already data for us to read. In that case we set the
     * timeout on the read operation to 10 seconds.
     */
    if ( ! data )
	timeout = 0;
    /* 
     * Enter a loop, in which we listen for more client input. When
     * the client closes the connection, pnetRead() returns 0 to indicate
     * that. Everytime the client sends something we echo it back.
     */
    while ( 1 ) 
    {
	/* Wait for input on the socket for up to timeout seconds */
	/* 
	 * NOTE: the actual value of a peer address for a UDP socket is
	 * is only filled in after Read() has returned. That's why, in
	 * this dual TCP/UDP handler we only print the value of the peer
	 * address after read has returned.
	 * All PNET read routines return PNET_READ_TIMED_OUT if no input
	 * is present after the given time period; PNET_READ_ERROR if an
	 * error occurred, and 0 if the connection was closed.
	 */

	switch ( (ret = pnetReadTimed(ps,buf,sizeof(buf),0,timeout)) )
	{
	case 0:
	    fprintf(  outstream, "Client closed connection\n" );
	    return 0;
	case PNET_READ_ERROR:
	    fprintf(  outstream, "Error reading data\n" );
	    return -1;
	case PNET_READ_TIMED_OUT:
	    fprintf(  outstream, "Timed out\n" );
	    return -1;
	default:
	    break;
	}

	/* Get address of remote host */
	from = pnetSockGetPeerAddr(ps);

	/*
	 * Get actual address that input (a datagram) came in on.
	 * This is not necessarily the address that the datagram was
	 * sent to (think about a datagram sent to a multicast or 
	 * broadcast address vs. the address on which it was received
	 * locally). This is not applicable to TCP, where this duality
	 * is non-existent: datagram destination and actual destination
	 * address is always the same with TCP.
	 */
	to   = pnetSockGetLocalAddr(ps);

	fprintf( outstream, "Incoming request from %s: %d bytes\n",
		    pnetAddrToString( from, addrbuf, sizeof( addrbuf )), ret );
	/*
	 * Get a pointer to the socket's auxiliary info.
	 */
	paux = pnetSockAuxInfo(ps);

	/*
	 * A datagram socket can return additional information, if it
	 * has been created with the pnetUDPSocketX() function, which marks
	 * it for receiving ancillary data such as interface index on which
	 * the datagram has been received, and the actual datagram 
	 * destination address. See discussion above.
	 * This is also available for IPv6 sockets.
	 */
	if ( paux )
	{
	    PNETADDR 	pdst;		/* Actual datagram destination */
	    char 	buff[PNET_ADDR_STRLEN]; /* Buffer to print pdest */
	    PNETIF 	pif;		/* Interface datagram arrived on */

	    if ( pnetAuxFlagTRUNC( paux ) )
		fprintf( outstream, "Datagram was truncated\n" );

	    /* Get actual datagram address */
	    pdst = pnetAuxDstAddr( paux );
	    fprintf( outstream, "Actual datagram destination address: %s\n",
		    pnetAddrToString( pdst, buff, sizeof( buff )));

	    /* Get interface index on which datagram arrived */
	    pif = pnetAuxRecvIf( paux );
	    if ( pif )
		fprintf( outstream, "Datagram received on interface %s\n",pnetIfName(pif));

	    /* Read the datagram's hoplimit. */
	    fprintf( outstream, "Datagram hop limit = %d\n", pnetAuxHoplimit( paux ) );
	}

	/* 
	 * Write back the current data we've received
	 */
	if ( pnetWrite(ps,buf,ret) != ret )
	    fprintf( outstream, "Error sending data to client %s\n", addrbuf );
    }

    return 0;
}
static int
usage( const char * progname )
{
    printf( "%s: usage\n", progname );
    printf( 
    	"    [-i] : interface name or address (default=*)\n"
	"    [-t] : use tcp\n" 
	"    [-u] : use udp (default)\n" 
	"    [-d] : detach from terminal (become daemon)\n"
	);
    return 1;  
}
/*
 * We create three sockets: IP, IPv6 and Unix. If the -t flag is given
 * we make 'em TCP sockets, else we make 'em UDP sockets.
 * IPv4 packets can be accepted on port 44444, IPv6 packets on port 44446, 
 * and UNIX domain packets on path "/tmp/pnet6time"
 * In addition, the -i option lets you specify the interface on which to 
 * listen to, that is a specific network card address. In this simple 
 * example we just pick the first IP address that the interface has. In a
 * more advanced example, we need to open a listen socket for each of the
 * ip addresses an interface might have. See 'examples/udpserv' for a more
 * advanced server example If the interface is not given,
 * the default is to listen on the wildcard address (i.e. accept packets 
 * incoming on any of the available network interfaces).
 */
int
main( int argc, const char **argv )
{
    PNETSOCK 	ps_4;		/* Accept IPv4 */
    PNETSOCK 	ps_6;		/* Accept IPv6 */
    PNETSOCK 	ps_U;		/* Accept Unix */

    const char*	ifname = "*";
    const char* logfilename = "server.log";
    int		stream = 0;
    int		daemon = 0;

    char	buf[256];	/* For name of local host */
    int		sendsize = 45000;
    int		c;

    while ( (c = pnetGetopt( argc, argv, ":i:tud" )) != -1 )
    {
	switch ( c )
	{
	case 'd': daemon = 1; break;
	case 'i': ifname = strdup( pnetOptarg ); break;
	case 't': stream = 1; break;
	case 'u': stream = 0; break;
	case 'l': logfilename = strdup( pnetOptarg ); break;

	case '!':
	    printf("Option '%c' requires an argument\n",pnetOptopt);
	    usage( argv[0] );
	    return 1;
	case '?':
	    printf("Unknown argument '%c'\n",pnetOptopt);
	    usage( argv[0] );
	    return 1;
	}
    }

    /*
     * Open log file
    outstream = fopen( logfilename, "w+" );
     */

    if ( !outstream )
	outstream = stderr;

# if 0
    pnetBindOutputStreams( outstream, outstream );
# endif

    /* Convert arguments to useful things */
    if ( ifname && *ifname == '*')
	ifname = NULL;

    /* Print some info on the current host */

    if ( pnetGetLocalAddress( buf, sizeof(buf) ) )
	strcpy(buf, "localhost" );

    pnetLog( "Server running on host %s\n",buf);

    if ( daemon )
	pnetDetachProcess( "pnet6-server", "/" );

    /*------------------------------------------------------------------*/
    /* The following is all you need to do to create a listen server	*/
    /* The server will listen on as many sockets as you register	*/
    /* with pnetListenAt(). They don't need to be of the same type.	*/
    /*------------------------------------------------------------------*/

    if ( stream )
    {
	ps_4 = pnetTCPSocket2( PNET_IPv4 );
	ps_6 = pnetTCPSocket2( PNET_IPv6 );
	ps_U = pnetTCPSocket2( PNET_LOCAL );
    }
    else
    {
	/* We use pnetUDPSocketX() for UDP sockets so that we will be
	 * able to receive additional UDP datagram information (see the
	 * Read() handler above */

	ps_4 = pnetUDPSocketX( PNET_IPv4 );
	ps_6 = pnetUDPSocketX( PNET_IPv6 );
	ps_U = pnetUDPSocketX( PNET_LOCAL );
    }

    if ( ! ps_4 || ! ps_6 || ! ps_U )
	{ fprintf( outstream, "Cannot create sockets.\n"); return 1; }

    /*
     * Assign read callbacks to all sockets. In this program they use the
     * same function. The last argument is some user provided data, which
     * is passed back whenever the callback is invoked.
     */

    pnetSockAddReadcallback( ps_4, Read, (void*) stream );
    pnetSockAddReadcallback( ps_6, Read, (void*) stream );
    pnetSockAddReadcallback( ps_U, Read, (void*) stream );

    /*
     * Mark sockets as listen sockets, giving address and port on which to
     * listen. 'ifname' must be an interface name, such as 'rl0' or 'eth0', or
     * it can be NULL, in which case the wildcard address is bound to the
     * socket. It can also be an string containing an IP address.
     * For a UNIX socket, it can be set to NULL, or "/unix" or "/local"
     * If you want to enable large socket buffers, to speed up a connection
     * (if your have a fast connection) call 
     * pnetSockSetSendbuf( ps, size = whatever ) prior to calling the
     * listen. This is needed because the tcp handshake and window negotiation
     * will use the values for the buffers as they are at the time the 
     * connection is being established, i.e. during accept() .
     */

    pnetSockSetSendbuf( ps_4, sendsize );

    /* Mark the IPv4 socket as a listen socket */

    if ( pnetListenAt( ps_4, ifname, "44444" ) )
    {
	fprintf( outstream, "Unable to start a listening IPv4 socket\n");
	return 1;
    }
    /*
     * The following can fail if IPv6 is not supported. We don't 
     * fail so that the server can run even on non-IPv6 enabled hosts.
     */
    pnetListenAt( ps_6, ifname, "44446");

    /*
     * Idem for Unix sockets
     */
    pnetListenAt( ps_U, "/unix", "/tmp/pnet6time");

    /*
     * Enter our listen loop here. PNET_LISTEN_SELF tells the server to handle
     * each connection itself. PNET_LISTEN_FORK tells the server to 
     * fork a child for each connection, while PNET_LISTEN_THREAD tells
     * the server to spawn a thread to handle each connection.
     * NOTE: these values are ignored for UDP servers, since there is no
     * port demultiplexing with UDP, so a UDP server handles all new
     * connections iteratively. We provide THREAD for TCP connections.
     */

    pnetStartListen( PNET_LISTEN_THREAD );

    return 0;
}
