// license:BSD-3-Clause
// copyright-holders:Alex Pasadyn,Zsolt Vasvari
/***************************************************************************

    TMS34010: Portable Texas Instruments TMS34010 emulator

    Copyright Alex Pasadyn/Zsolt Vasvari
    Parts based on code by Aaron Giles

***************************************************************************/

#ifndef MAME_CPU_TMS34010_TMS34010_H
#define MAME_CPU_TMS34010_TMS34010_H

#pragma once


/* register indexes for get_reg and set_reg */
enum
{
	TMS34010_PC,
	TMS34010_SP,
	TMS34010_ST,
	TMS34010_A0,
	TMS34010_A1,
	TMS34010_A2,
	TMS34010_A3,
	TMS34010_A4,
	TMS34010_A5,
	TMS34010_A6,
	TMS34010_A7,
	TMS34010_A8,
	TMS34010_A9,
	TMS34010_A10,
	TMS34010_A11,
	TMS34010_A12,
	TMS34010_A13,
	TMS34010_A14,
	TMS34010_B0,
	TMS34010_B1,
	TMS34010_B2,
	TMS34010_B3,
	TMS34010_B4,
	TMS34010_B5,
	TMS34010_B6,
	TMS34010_B7,
	TMS34010_B8,
	TMS34010_B9,
	TMS34010_B10,
	TMS34010_B11,
	TMS34010_B12,
	TMS34010_B13,
	TMS34010_B14
};


/***************************************************************************
    INTERNAL I/O CONSTANTS
***************************************************************************/

enum
{
	REG_HESYNC = 0,
	REG_HEBLNK,
	REG_HSBLNK,
	REG_HTOTAL,
	REG_VESYNC,
	REG_VEBLNK,
	REG_VSBLNK,
	REG_VTOTAL,
	REG_DPYCTL,
	REG_DPYSTRT,
	REG_DPYINT,
	REG_CONTROL,
	REG_HSTDATA,
	REG_HSTADRL,
	REG_HSTADRH,
	REG_HSTCTLL,

	REG_HSTCTLH,
	REG_INTENB,
	REG_INTPEND,
	REG_CONVSP,
	REG_CONVDP,
	REG_PSIZE,
	REG_PMASK,
	REG_UNK23,
	REG_UNK24,
	REG_UNK25,
	REG_UNK26,
	REG_DPYTAP,
	REG_HCOUNT,
	REG_VCOUNT,
	REG_DPYADR,
	REG_REFCNT
};

/* Interrupts that are generated by the processor internally */
#define TMS34010_INT1       0x0002      /* External Interrupt 1 */
#define TMS34010_INT2       0x0004      /* External Interrupt 2 */
#define TMS34010_HI         0x0200      /* Host Interrupt */
#define TMS34010_DI         0x0400      /* Display Interrupt */
#define TMS34010_WV         0x0800      /* Window Violation Interrupt */


#define TMS340X0_SCANLINE_IND16_CB_MEMBER(_name) void _name(screen_device &screen, bitmap_ind16 &bitmap, int scanline, const tms340x0_device::display_params *params)
#define TMS340X0_SCANLINE_RGB32_CB_MEMBER(_name) void _name(screen_device &screen, bitmap_rgb32 &bitmap, int scanline, const tms340x0_device::display_params *params)
#define TMS340X0_TO_SHIFTREG_CB_MEMBER(_name) void _name(offs_t address, uint16_t *shiftreg)
#define TMS340X0_FROM_SHIFTREG_CB_MEMBER(_name) void _name(offs_t address, uint16_t *shiftreg)


class tms340x0_device : public cpu_device,
						public device_video_interface
{
public:
	// Configuration structure
	struct display_params
	{
		uint16_t  vcount;                             /* most recent VCOUNT */
		uint16_t  veblnk, vsblnk;                     /* start/end of VBLANK */
		uint16_t  heblnk, hsblnk;                     /* start/end of HBLANK */
		uint16_t  rowaddr, coladdr;                   /* row/column addresses */
		uint8_t   yoffset;                            /* y offset from addresses */
		uint8_t   enabled;                            /* video enabled */
	};

	typedef device_delegate<void (screen_device &screen, bitmap_ind16 &bitmap, int scanline, const display_params *params)> scanline_ind16_cb_delegate;
	typedef device_delegate<void (screen_device &screen, bitmap_rgb32 &bitmap, int scanline, const display_params *params)> scanline_rgb32_cb_delegate;
	typedef device_delegate<void (offs_t address, uint16_t *shiftreg)> shiftreg_in_cb_delegate;
	typedef device_delegate<void (offs_t address, uint16_t *shiftreg)> shiftreg_out_cb_delegate;

	void set_halt_on_reset(bool halt_on_reset) { m_halt_on_reset = halt_on_reset; }
	void set_pixel_clock(uint32_t pixclock) { m_pixclock = pixclock; }
	void set_pixel_clock(const XTAL &xtal) { set_pixel_clock(xtal.value()); }
	void set_pixels_per_clock(int pixperclock) { m_pixperclock = pixperclock; }

	auto output_int() { return m_output_int_cb.bind(); }
	auto ioreg_pre_write() { return m_ioreg_pre_write_cb.bind(); }

	// Setter for ind16 scanline callback
	template <typename... T>
	void set_scanline_ind16_callback(T &&... args)
	{
		m_scanline_ind16_cb.set(std::forward<T>(args)...);
	}

	// Setter for rgb32 scanline callback
	template <typename... T>
	void set_scanline_rgb32_callback(T &&... args)
	{
		m_scanline_rgb32_cb.set(std::forward<T>(args)...);
	}

	// Setter for shift register input callback
	template <typename... T>
	void set_shiftreg_in_callback(T &&... args)
	{
		m_to_shiftreg_cb.set(std::forward<T>(args)...);
	}

	// Setters for shift register output callback
	template <typename... T>
	void set_shiftreg_out_callback(T &&... args)
	{
		m_from_shiftreg_cb.set(std::forward<T>(args)...);
	}

	void get_display_params(display_params *params);

	uint32_t tms340x0_ind16(screen_device &screen, bitmap_ind16 &bitmap, const rectangle &cliprect);
	uint32_t tms340x0_rgb32(screen_device &screen, bitmap_rgb32 &bitmap, const rectangle &cliprect);

	virtual void io_register_w(offs_t offset, u16 data, u16 mem_mask = ~u16(0)) = 0;
	virtual u16 io_register_r(offs_t offset) = 0;

	void host_w(offs_t offset, u16 data, u16 mem_mask = ~u16(0));
	u16 host_r(offs_t offset);

	TIMER_CALLBACK_MEMBER(internal_interrupt_callback);
	TIMER_CALLBACK_MEMBER(scanline_callback);

protected:
	enum
	{
		REG020_VESYNC,
		REG020_HESYNC,
		REG020_VEBLNK,
		REG020_HEBLNK,
		REG020_VSBLNK,
		REG020_HSBLNK,
		REG020_VTOTAL,
		REG020_HTOTAL,
		REG020_DPYCTL,      /* matches 010 */
		REG020_DPYSTRT,     /* matches 010 */
		REG020_DPYINT,      /* matches 010 */
		REG020_CONTROL,     /* matches 010 */
		REG020_HSTDATA,     /* matches 010 */
		REG020_HSTADRL,     /* matches 010 */
		REG020_HSTADRH,     /* matches 010 */
		REG020_HSTCTLL,     /* matches 010 */

		REG020_HSTCTLH,     /* matches 010 */
		REG020_INTENB,      /* matches 010 */
		REG020_INTPEND,     /* matches 010 */
		REG020_CONVSP,      /* matches 010 */
		REG020_CONVDP,      /* matches 010 */
		REG020_PSIZE,       /* matches 010 */
		REG020_PMASKL,
		REG020_PMASKH,
		REG020_CONVMP,
		REG020_CONTROL2,
		REG020_CONFIG,
		REG020_DPYTAP,      /* matches 010 */
		REG020_VCOUNT,
		REG020_HCOUNT,
		REG020_DPYADR,      /* matches 010 */
		REG020_REFADR,

		REG020_DPYSTL,
		REG020_DPYSTH,
		REG020_DPYNXL,
		REG020_DPYNXH,
		REG020_DINCL,
		REG020_DINCH,
		REG020_RES0,
		REG020_HESERR,
		REG020_RES1,
		REG020_RES2,
		REG020_RES3,
		REG020_RES4,
		REG020_SCOUNT,
		REG020_BSFLTST,
		REG020_DPYMSK,
		REG020_RES5,

		REG020_SETVCNT,
		REG020_SETHCNT,
		REG020_BSFLTDL,
		REG020_BSFLTDH,
		REG020_RES6,
		REG020_RES7,
		REG020_RES8,
		REG020_RES9,
		REG020_IHOST1L,
		REG020_IHOST1H,
		REG020_IHOST2L,
		REG020_IHOST2H,
		REG020_IHOST3L,
		REG020_IHOST3H,
		REG020_IHOST4L,
		REG020_IHOST4H
	};

	// construction/destruction
	tms340x0_device(const machine_config &mconfig, device_type type, const char *tag, device_t *owner, uint32_t clock, address_map_constructor internal_regs_map, bool is_34020);

	// device-level overrides
	virtual void device_start() override;
	virtual void device_reset() override;
	virtual void device_post_load() override;

	// device_execute_interface overrides
	virtual uint32_t execute_min_cycles() const noexcept override { return 1; }
	virtual uint32_t execute_max_cycles() const noexcept override { return 10000; }
	virtual void execute_run() override;
	virtual void execute_set_input(int inputnum, int state) override;

	// device_memory_interface overrides
	virtual space_config_vector memory_space_config() const override;

	// device_state_interface overrides
	virtual void state_string_export(const device_state_entry &entry, std::string &str) const override;

	typedef void (tms340x0_device::*pixel_write_func)(offs_t offset, uint32_t data);
	typedef uint32_t (tms340x0_device::*pixel_read_func)(offs_t offset);
	typedef uint32_t (tms340x0_device::*raster_op_func)(uint32_t newpix, uint32_t oldpix);
	typedef void (tms340x0_device::*wfield_func)(offs_t offset, uint32_t data);
	typedef uint32_t (tms340x0_device::*rfield_func)(offs_t offset);
	typedef uint32_t (tms340x0_device::*pixel_op_func)(uint32_t, uint32_t, uint32_t);
	typedef void (tms340x0_device::*pixblt_op_func)(int, int);
	typedef void (tms340x0_device::*pixblt_b_op_func)(int);
	typedef void (tms340x0_device::*word_write_func)(offs_t offset, uint16_t data);
	typedef uint16_t (tms340x0_device::*word_read_func)(offs_t offset);

	static const wfield_func s_wfield_functions[32];
	static const rfield_func s_rfield_functions[64];
	static const pixel_op_func s_pixel_op_table[32];
	static const uint8_t s_pixel_op_timing_table[33];
	static const pixblt_op_func s_pixblt_op_table[];
	static const pixblt_op_func s_pixblt_r_op_table[];
	static const pixblt_b_op_func s_pixblt_b_op_table[];
	static const pixblt_b_op_func s_fill_op_table[];
	static const pixel_write_func s_pixel_write_ops[4][6];
	static const pixel_read_func s_pixel_read_ops[6];
	static const raster_op_func s_raster_ops[32];

	address_space_config m_program_config;

	uint32_t           m_pc;
	uint32_t           m_ppc;
	uint32_t           m_st;
	pixel_write_func m_pixel_write;
	pixel_read_func  m_pixel_read;
	raster_op_func   m_raster_op;
	pixel_op_func    m_pixel_op;
	uint32_t           m_pixel_op_timing;
	uint32_t           m_convsp;
	uint32_t           m_convdp;
	uint32_t           m_convmp;
	int32_t            m_gfxcycles;
	uint8_t            m_pixelshift;
	const bool         m_is_34020;
	bool             m_reset_deferred;
	bool             m_halt_on_reset; /* /HCS pin, which determines HALT state after reset */
	uint8_t            m_hblank_stable;
	uint8_t            m_external_host_access;
	uint8_t            m_executing;

	uint32_t  m_pixclock;                           /* the pixel clock (0 means don't adjust screen size) */
	int     m_pixperclock;                        /* pixels per clock */
	emu_timer *m_scantimer;
	int m_icount;

	scanline_ind16_cb_delegate m_scanline_ind16_cb;
	scanline_rgb32_cb_delegate m_scanline_rgb32_cb;
	devcb_write_line m_output_int_cb; /* output interrupt callback */
	devcb_write16 m_ioreg_pre_write_cb;
	shiftreg_in_cb_delegate m_to_shiftreg_cb;  /* shift register write */
	shiftreg_out_cb_delegate m_from_shiftreg_cb; /* shift register read */

	struct XY
	{
#ifdef LSB_FIRST
		int16_t x;
		int16_t y;
#else
		int16_t y;
		int16_t x;
#endif
	};

	/* A registers 0-15 map to regs[0]-regs[15] */
	/* B registers 0-15 map to regs[30]-regs[15] */
	union
	{
		int32_t reg;
		XY xy;
	} m_regs[31];

	uint16_t m_IOregs[64];
	uint16_t              m_shiftreg[(8 * 512 * sizeof(uint16_t))/2];


	virtual uint32_t TMS34010_RDMEM_WORD(offs_t A) = 0;
	virtual uint32_t TMS34010_RDMEM_DWORD(offs_t A) = 0;
	virtual void TMS34010_WRMEM_WORD(offs_t A, uint32_t V) = 0;
	virtual void TMS34010_WRMEM_DWORD(offs_t A, uint32_t V) = 0;
	void SET_ST(uint32_t st);
	void RESET_ST();
	virtual uint16_t ROPCODE() = 0;
	virtual void execute_op(uint16_t op) = 0;
	virtual int16_t PARAM_WORD() = 0;
	virtual int32_t PARAM_LONG() = 0;
	virtual int16_t PARAM_WORD_NO_INC() = 0;
	virtual int32_t PARAM_LONG_NO_INC() = 0;
	uint32_t RBYTE(offs_t offset);
	void WBYTE(offs_t offset, uint32_t data);
	uint32_t RLONG(offs_t offset);
	void WLONG(offs_t offset, uint32_t data);
	void PUSH(uint32_t data);
	int32_t POP();
	uint32_t read_pixel_1(offs_t offset);
	uint32_t read_pixel_2(offs_t offset);
	uint32_t read_pixel_4(offs_t offset);
	uint32_t read_pixel_8(offs_t offset);
	uint32_t read_pixel_16(offs_t offset);
	uint32_t read_pixel_32(offs_t offset);
	uint32_t read_pixel_shiftreg(offs_t offset);
	void write_pixel_1(offs_t offset, uint32_t data);
	void write_pixel_2(offs_t offset, uint32_t data);
	void write_pixel_4(offs_t offset, uint32_t data);
	void write_pixel_8(offs_t offset, uint32_t data);
	void write_pixel_16(offs_t offset, uint32_t data);
	void write_pixel_32(offs_t offset, uint32_t data);
	void write_pixel_t_1(offs_t offset, uint32_t data);
	void write_pixel_t_2(offs_t offset, uint32_t data);
	void write_pixel_t_4(offs_t offset, uint32_t data);
	void write_pixel_t_8(offs_t offset, uint32_t data);
	void write_pixel_t_16(offs_t offset, uint32_t data);
	void write_pixel_t_32(offs_t offset, uint32_t data);
	void write_pixel_r_1(offs_t offset, uint32_t data);
	void write_pixel_r_2(offs_t offset, uint32_t data);
	void write_pixel_r_4(offs_t offset, uint32_t data);
	void write_pixel_r_8(offs_t offset, uint32_t data);
	void write_pixel_r_16(offs_t offset, uint32_t data);
	void write_pixel_r_32(offs_t offset, uint32_t data);
	void write_pixel_r_t_1(offs_t offset, uint32_t data);
	void write_pixel_r_t_2(offs_t offset, uint32_t data);
	void write_pixel_r_t_4(offs_t offset, uint32_t data);
	void write_pixel_r_t_8(offs_t offset, uint32_t data);
	void write_pixel_r_t_16(offs_t offset, uint32_t data);
	void write_pixel_r_t_32(offs_t offset, uint32_t data);
	void write_pixel_shiftreg(offs_t offset, uint32_t data);
	uint32_t raster_op_1(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_2(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_3(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_4(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_5(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_6(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_7(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_8(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_9(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_10(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_11(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_12(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_13(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_14(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_15(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_16(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_17(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_18(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_19(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_20(uint32_t newpix, uint32_t oldpix);
	uint32_t raster_op_21(uint32_t newpix, uint32_t oldpix);
	void wfield_01(offs_t offset, uint32_t data);
	void wfield_02(offs_t offset, uint32_t data);
	void wfield_03(offs_t offset, uint32_t data);
	void wfield_04(offs_t offset, uint32_t data);
	void wfield_05(offs_t offset, uint32_t data);
	void wfield_06(offs_t offset, uint32_t data);
	void wfield_07(offs_t offset, uint32_t data);
	void wfield_08(offs_t offset, uint32_t data);
	void wfield_09(offs_t offset, uint32_t data);
	void wfield_10(offs_t offset, uint32_t data);
	void wfield_11(offs_t offset, uint32_t data);
	void wfield_12(offs_t offset, uint32_t data);
	void wfield_13(offs_t offset, uint32_t data);
	void wfield_14(offs_t offset, uint32_t data);
	void wfield_15(offs_t offset, uint32_t data);
	void wfield_16(offs_t offset, uint32_t data);
	void wfield_17(offs_t offset, uint32_t data);
	void wfield_18(offs_t offset, uint32_t data);
	void wfield_19(offs_t offset, uint32_t data);
	void wfield_20(offs_t offset, uint32_t data);
	void wfield_21(offs_t offset, uint32_t data);
	void wfield_22(offs_t offset, uint32_t data);
	void wfield_23(offs_t offset, uint32_t data);
	void wfield_24(offs_t offset, uint32_t data);
	void wfield_25(offs_t offset, uint32_t data);
	void wfield_26(offs_t offset, uint32_t data);
	void wfield_27(offs_t offset, uint32_t data);
	void wfield_28(offs_t offset, uint32_t data);
	void wfield_29(offs_t offset, uint32_t data);
	void wfield_30(offs_t offset, uint32_t data);
	void wfield_31(offs_t offset, uint32_t data);
	void wfield_32(offs_t offset, uint32_t data);
	uint32_t rfield_z_01(offs_t offset);
	uint32_t rfield_z_02(offs_t offset);
	uint32_t rfield_z_03(offs_t offset);
	uint32_t rfield_z_04(offs_t offset);
	uint32_t rfield_z_05(offs_t offset);
	uint32_t rfield_z_06(offs_t offset);
	uint32_t rfield_z_07(offs_t offset);
	uint32_t rfield_z_08(offs_t offset);
	uint32_t rfield_z_09(offs_t offset);
	uint32_t rfield_z_10(offs_t offset);
	uint32_t rfield_z_11(offs_t offset);
	uint32_t rfield_z_12(offs_t offset);
	uint32_t rfield_z_13(offs_t offset);
	uint32_t rfield_z_14(offs_t offset);
	uint32_t rfield_z_15(offs_t offset);
	uint32_t rfield_z_16(offs_t offset);
	uint32_t rfield_z_17(offs_t offset);
	uint32_t rfield_z_18(offs_t offset);
	uint32_t rfield_z_19(offs_t offset);
	uint32_t rfield_z_20(offs_t offset);
	uint32_t rfield_z_21(offs_t offset);
	uint32_t rfield_z_22(offs_t offset);
	uint32_t rfield_z_23(offs_t offset);
	uint32_t rfield_z_24(offs_t offset);
	uint32_t rfield_z_25(offs_t offset);
	uint32_t rfield_z_26(offs_t offset);
	uint32_t rfield_z_27(offs_t offset);
	uint32_t rfield_z_28(offs_t offset);
	uint32_t rfield_z_29(offs_t offset);
	uint32_t rfield_z_30(offs_t offset);
	uint32_t rfield_z_31(offs_t offset);
	uint32_t rfield_32(offs_t offset);
	uint32_t rfield_s_01(offs_t offset);
	uint32_t rfield_s_02(offs_t offset);
	uint32_t rfield_s_03(offs_t offset);
	uint32_t rfield_s_04(offs_t offset);
	uint32_t rfield_s_05(offs_t offset);
	uint32_t rfield_s_06(offs_t offset);
	uint32_t rfield_s_07(offs_t offset);
	uint32_t rfield_s_08(offs_t offset);
	uint32_t rfield_s_09(offs_t offset);
	uint32_t rfield_s_10(offs_t offset);
	uint32_t rfield_s_11(offs_t offset);
	uint32_t rfield_s_12(offs_t offset);
	uint32_t rfield_s_13(offs_t offset);
	uint32_t rfield_s_14(offs_t offset);
	uint32_t rfield_s_15(offs_t offset);
	uint32_t rfield_s_16(offs_t offset);
	uint32_t rfield_s_17(offs_t offset);
	uint32_t rfield_s_18(offs_t offset);
	uint32_t rfield_s_19(offs_t offset);
	uint32_t rfield_s_20(offs_t offset);
	uint32_t rfield_s_21(offs_t offset);
	uint32_t rfield_s_22(offs_t offset);
	uint32_t rfield_s_23(offs_t offset);
	uint32_t rfield_s_24(offs_t offset);
	uint32_t rfield_s_25(offs_t offset);
	uint32_t rfield_s_26(offs_t offset);
	uint32_t rfield_s_27(offs_t offset);
	uint32_t rfield_s_28(offs_t offset);
	uint32_t rfield_s_29(offs_t offset);
	uint32_t rfield_s_30(offs_t offset);
	uint32_t rfield_s_31(offs_t offset);
	void unimpl(uint16_t op);
	void illop(uint16_t op);
	void pixblt_l_l(uint16_t op); /* 0f00 */
	void pixblt_l_xy(uint16_t op); /* 0f20 */
	void pixblt_xy_l(uint16_t op); /* 0f40 */
	void pixblt_xy_xy(uint16_t op); /* 0f60 */
	void pixblt_b_l(uint16_t op); /* 0f80 */
	void pixblt_b_xy(uint16_t op); /* 0fa0 */
	void fill_l(uint16_t op);   /* 0fc0 */
	void fill_xy(uint16_t op);  /* 0fe0 */
	void line(uint16_t op);     /* df10/df90 */
	void add_xy_a(uint16_t op); /* e000/e100 */
	void add_xy_b(uint16_t op); /* e000/e100 */
	void sub_xy_a(uint16_t op); /* e200/e300 */
	void sub_xy_b(uint16_t op); /* e200/e300 */
	void cmp_xy_a(uint16_t op); /* e400/e500 */
	void cmp_xy_b(uint16_t op); /* e400/e500 */
	void cpw_a(uint16_t op);    /* e600/e700 */
	void cpw_b(uint16_t op);    /* e600/e700 */
	void cvxyl_a(uint16_t op);  /* e800/e900 */
	void cvxyl_b(uint16_t op);  /* e800/e900 */
	void movx_a(uint16_t op);   /* ec00/ed00 */
	void movx_b(uint16_t op);   /* ec00/ed00 */
	void movy_a(uint16_t op);   /* ee00/ef00 */
	void movy_b(uint16_t op);   /* ee00/ef00 */
	void pixt_ri_a(uint16_t op); /* f800/f900 */
	void pixt_ri_b(uint16_t op); /* f800/f900 */
	void pixt_rixy_a(uint16_t op); /* f000/f100 */
	void pixt_rixy_b(uint16_t op); /* f000/f100 */
	void pixt_ir_a(uint16_t op); /* fa00/fb00 */
	void pixt_ir_b(uint16_t op); /* fa00/fb00 */
	void pixt_ii_a(uint16_t op); /* fc00/fd00 */
	void pixt_ii_b(uint16_t op); /* fc00/fd00 */
	void pixt_ixyr_a(uint16_t op); /* f200/f300 */
	void pixt_ixyr_b(uint16_t op); /* f200/f300 */
	void pixt_ixyixy_a(uint16_t op); /* f400/f500 */
	void pixt_ixyixy_b(uint16_t op); /* f400/f500 */
	void drav_a(uint16_t op); /* f600/f700 */
	void drav_b(uint16_t op); /* f600/f700 */
	void abs_a(uint16_t op); /* 0380 */
	void abs_b(uint16_t op); /* 0390 */
	void add_a(uint16_t op); /* 4000/4100 */
	void add_b(uint16_t op); /* 4000/4100 */
	void addc_a(uint16_t op); /* 4200/4200 */
	void addc_b(uint16_t op); /* 4200/4200 */
	void addi_w_a(uint16_t op); /* 0b00 */
	void addi_w_b(uint16_t op); /* 0b10 */
	void addi_l_a(uint16_t op); /* 0b20 */
	void addi_l_b(uint16_t op); /* 0b30 */
	void addk_a(uint16_t op); /* 1000-1300 */
	void addk_b(uint16_t op); /* 1000-1300 */
	void and_a(uint16_t op); /* 5000/5100 */
	void and_b(uint16_t op); /* 5000/5100 */
	void andi_a(uint16_t op); /* 0b80 */
	void andi_b(uint16_t op); /* 0b90 */
	void andn_a(uint16_t op); /* 5200-5300 */
	void andn_b(uint16_t op); /* 5200-5300 */
	void btst_k_a(uint16_t op); /* 1c00-1f00 */
	void btst_k_b(uint16_t op); /* 1c00-1f00 */
	void btst_r_a(uint16_t op); /* 4a00-4b00 */
	void btst_r_b(uint16_t op); /* 4a00-4b00 */
	void clrc(uint16_t op); /* 0320 */
	void cmp_a(uint16_t op); /* 4800/4900 */
	void cmp_b(uint16_t op); /* 4800/4900 */
	void cmpi_w_a(uint16_t op); /* 0b40 */
	void cmpi_w_b(uint16_t op); /* 0b50 */
	void cmpi_l_a(uint16_t op); /* 0b60 */
	void cmpi_l_b(uint16_t op); /* 0b70 */
	void dint(uint16_t op);
	void divs_a(uint16_t op); /* 5800/5900 */
	void divs_b(uint16_t op); /* 5800/5900 */
	void divu_a(uint16_t op); /* 5a00/5b00 */
	void divu_b(uint16_t op); /* 5a00/5b00 */
	void eint(uint16_t op);
	void exgf0_a(uint16_t op);  /* d500 */
	void exgf0_b(uint16_t op);    /* d510 */
	void exgf1_a(uint16_t op);    /* d700 */
	void exgf1_b(uint16_t op);    /* d710 */
	void lmo_a(uint16_t op);  /* 6a00/6b00 */
	void lmo_b(uint16_t op);  /* 6a00/6b00 */
	void mmfm_a(uint16_t op); /* 09a0 */
	void mmfm_b(uint16_t op); /* 09b0 */
	void mmtm_a(uint16_t op); /* 0980 */
	void mmtm_b(uint16_t op); /* 0990 */
	void mods_a(uint16_t op); /* 6c00/6d00 */
	void mods_b(uint16_t op); /* 6c00/6d00 */
	void modu_a(uint16_t op); /* 6e00/6f00 */
	void modu_b(uint16_t op); /* 6e00/6f00 */
	void mpys_a(uint16_t op); /* 5c00/5d00 */
	void mpys_b(uint16_t op); /* 5c00/5d00 */
	void mpyu_a(uint16_t op); /* 5e00/5e00 */
	void mpyu_b(uint16_t op); /* 5e00/5f00 */
	void neg_a(uint16_t op); /* 03a0 */
	void neg_b(uint16_t op); /* 03b0 */
	void negb_a(uint16_t op); /* 03c0 */
	void negb_b(uint16_t op); /* 03d0 */
	void nop(uint16_t op); /* 0300 */
	void not_a(uint16_t op); /* 03e0 */
	void not_b(uint16_t op); /* 03f0 */
	void or_a(uint16_t op); /* 5400-5500 */
	void or_b(uint16_t op); /* 5400-5500 */
	void ori_a(uint16_t op); /* 0ba0 */
	void ori_b(uint16_t op); /* 0bb0 */
	void rl_k_a(uint16_t op); /* 3000-3300 */
	void rl_k_b(uint16_t op); /* 3000-3300 */
	void rl_r_a(uint16_t op); /* 6800/6900 */
	void rl_r_b(uint16_t op); /* 6800/6900 */
	void setc(uint16_t op); /* 0de0 */
	void setf0(uint16_t op);
	void setf1(uint16_t op);
	void sext0_a(uint16_t op); /* 0500 */
	void sext0_b(uint16_t op); /* 0510 */
	void sext1_a(uint16_t op); /* 0700 */
	void sext1_b(uint16_t op); /* 0710 */
	void sla_k_a(uint16_t op); /* 2000-2300 */
	void sla_k_b(uint16_t op); /* 2000-2300 */
	void sla_r_a(uint16_t op); /* 6000/6100 */
	void sla_r_b(uint16_t op); /* 6000/6100 */
	void sll_k_a(uint16_t op); /* 2400-2700 */
	void sll_k_b(uint16_t op); /* 2400-2700 */
	void sll_r_a(uint16_t op); /* 6200/6300 */
	void sll_r_b(uint16_t op); /* 6200/6300 */
	void sra_k_a(uint16_t op); /* 2800-2b00 */
	void sra_k_b(uint16_t op); /* 2800-2b00 */
	void sra_r_a(uint16_t op); /* 6400/6500 */
	void sra_r_b(uint16_t op); /* 6400/6500 */
	void srl_k_a(uint16_t op); /* 2c00-2f00 */
	void srl_k_b(uint16_t op); /* 2c00-2f00 */
	void srl_r_a(uint16_t op); /* 6600/6700 */
	void srl_r_b(uint16_t op); /* 6600/6700 */
	void sub_a(uint16_t op); /* 4400/4500 */
	void sub_b(uint16_t op); /* 4400/4500 */
	void subb_a(uint16_t op); /* 4600/4700 */
	void subb_b(uint16_t op); /* 4600/4700 */
	void subi_w_a(uint16_t op); /* 0be0 */
	void subi_w_b(uint16_t op); /* 0bf0 */
	void subi_l_a(uint16_t op); /* 0d00 */
	void subi_l_b(uint16_t op); /* 0d10 */
	void subk_a(uint16_t op); /* 1400-1700 */
	void subk_b(uint16_t op); /* 1400-1700 */
	void xor_a(uint16_t op); /* 5600-5700 */
	void xor_b(uint16_t op); /* 5600-5700 */
	void xori_a(uint16_t op); /* 0bc0 */
	void xori_b(uint16_t op); /* 0bd0 */
	void zext0_a(uint16_t op); /* 0520 */
	void zext0_b(uint16_t op); /* 0530 */
	void zext1_a(uint16_t op); /* 0720 */
	void zext1_b(uint16_t op); /* 0720 */
	void movi_w_a(uint16_t op);
	void movi_w_b(uint16_t op);
	void movi_l_a(uint16_t op);
	void movi_l_b(uint16_t op);
	void movk_a(uint16_t op);
	void movk_b(uint16_t op);
	void movb_rn_a(uint16_t op); /* 8c00-8d00 */
	void movb_rn_b(uint16_t op); /* 8c00-8d00 */
	void movb_nr_a(uint16_t op); /* 8e00-8f00 */
	void movb_nr_b(uint16_t op); /* 8e00-8f00 */
	void movb_nn_a(uint16_t op); /* 9c00-9d00 */
	void movb_nn_b(uint16_t op); /* 9c00-9d00 */
	void movb_r_no_a(uint16_t op); /* ac00-ad00 */
	void movb_r_no_b(uint16_t op); /* ac00-ad00 */
	void movb_no_r_a(uint16_t op); /* ae00-af00 */
	void movb_no_r_b(uint16_t op); /* ae00-af00 */
	void movb_no_no_a(uint16_t op); /* bc00-bd00 */
	void movb_no_no_b(uint16_t op); /* bc00-bd00 */
	void movb_ra_a(uint16_t op);
	void movb_ra_b(uint16_t op);
	void movb_ar_a(uint16_t op);
	void movb_ar_b(uint16_t op);
	void movb_aa(uint16_t op);
	void move_rr_a(uint16_t op); /* 4c00/d00 */
	void move_rr_b(uint16_t op); /* 4c00/d00 */
	void move_rr_ax(uint16_t op); /* 4e00/f00 */
	void move_rr_bx(uint16_t op); /* 4e00/f00 */
	void move0_rn_a(uint16_t op); /* 8000 */
	void move0_rn_b(uint16_t op);
	void move1_rn_a(uint16_t op);
	void move1_rn_b(uint16_t op);
	void move0_r_dn_a(uint16_t op); /* a000 */
	void move0_r_dn_b(uint16_t op);
	void move1_r_dn_a(uint16_t op);
	void move1_r_dn_b(uint16_t op);
	void move0_r_ni_a(uint16_t op); /* 9000 */
	void move0_r_ni_b(uint16_t op);
	void move1_r_ni_a(uint16_t op);
	void move1_r_ni_b(uint16_t op);
	void move0_nr_a(uint16_t op); /* 8400-500 */
	void move0_nr_b(uint16_t op); /* 8400-500 */
	void move1_nr_a(uint16_t op); /* 8600-700 */
	void move1_nr_b(uint16_t op); /* 8600-700 */
	void move0_dn_r_a(uint16_t op); /* A400-500 */
	void move0_dn_r_b(uint16_t op); /* A400-500 */
	void move1_dn_r_a(uint16_t op); /* A600-700 */
	void move1_dn_r_b(uint16_t op); /* A600-700 */
	void move0_ni_r_a(uint16_t op); /* 9400-500 */
	void move0_ni_r_b(uint16_t op); /* 9400-500 */
	void move1_ni_r_a(uint16_t op); /* 9600-700 */
	void move1_ni_r_b(uint16_t op); /* 9600-700 */
	void move0_nn_a(uint16_t op); /* 8800 */
	void move0_nn_b(uint16_t op);
	void move1_nn_a(uint16_t op);
	void move1_nn_b(uint16_t op);
	void move0_dn_dn_a(uint16_t op); /* a800 */
	void move0_dn_dn_b(uint16_t op);
	void move1_dn_dn_a(uint16_t op);
	void move1_dn_dn_b(uint16_t op);
	void move0_ni_ni_a(uint16_t op); /* 9800 */
	void move0_ni_ni_b(uint16_t op);
	void move1_ni_ni_a(uint16_t op);
	void move1_ni_ni_b(uint16_t op);
	void move0_r_no_a(uint16_t op); /* b000 */
	void move0_r_no_b(uint16_t op);
	void move1_r_no_a(uint16_t op);
	void move1_r_no_b(uint16_t op);
	void move0_no_r_a(uint16_t op); /* b400 */
	void move0_no_r_b(uint16_t op);
	void move1_no_r_a(uint16_t op);
	void move1_no_r_b(uint16_t op);
	void move0_no_ni_a(uint16_t op); /* d000 */
	void move0_no_ni_b(uint16_t op);
	void move1_no_ni_a(uint16_t op);
	void move1_no_ni_b(uint16_t op);
	void move0_no_no_a(uint16_t op); /* b800 */
	void move0_no_no_b(uint16_t op);
	void move1_no_no_a(uint16_t op);
	void move1_no_no_b(uint16_t op);
	void move0_ra_a(uint16_t op);
	void move0_ra_b(uint16_t op);
	void move1_ra_a(uint16_t op);
	void move1_ra_b(uint16_t op);
	void move0_ar_a(uint16_t op);
	void move0_ar_b(uint16_t op);
	void move1_ar_a(uint16_t op);
	void move1_ar_b(uint16_t op);
	void move0_a_ni_a(uint16_t op); /* d400 */
	void move0_a_ni_b(uint16_t op); /* d410 */
	void move1_a_ni_a(uint16_t op); /* d600 */
	void move1_a_ni_b(uint16_t op); /* d610 */
	void move0_aa(uint16_t op); /* 05c0 */
	void move1_aa(uint16_t op); /* 07c0 */
	void call_a(uint16_t op); /* 0920 */
	void call_b(uint16_t op); /* 0930 */
	void callr(uint16_t op); /* 0d3f */
	void calla(uint16_t op); /* 0d5f */
	void dsj_a(uint16_t op);  /* 0d80 */
	void dsj_b(uint16_t op);  /* 0d90 */
	void dsjeq_a(uint16_t op); /* 0da0 */
	void dsjeq_b(uint16_t op); /* 0db0 */
	void dsjne_a(uint16_t op); /* 0dc0 */
	void dsjne_b(uint16_t op); /* 0dd0 */
	void dsjs_a(uint16_t op);
	void dsjs_b(uint16_t op);
	void emu(uint16_t op);     /* 0100 */
	void exgpc_a(uint16_t op); /* 0120 */
	void exgpc_b(uint16_t op); /* 0130 */
	void getpc_a(uint16_t op); /* 0140 */
	void getpc_b(uint16_t op); /* 0150 */
	void getst_a(uint16_t op); /* 0180 */
	void getst_b(uint16_t op); /* 0190 */
	void j_UC_0(uint16_t op);
	void j_UC_8(uint16_t op);
	void j_UC_x(uint16_t op);
	void j_P_0(uint16_t op);
	void j_P_8(uint16_t op);
	void j_P_x(uint16_t op);
	void j_LS_0(uint16_t op);
	void j_LS_8(uint16_t op);
	void j_LS_x(uint16_t op);
	void j_HI_0(uint16_t op);
	void j_HI_8(uint16_t op);
	void j_HI_x(uint16_t op);
	void j_LT_0(uint16_t op);
	void j_LT_8(uint16_t op);
	void j_LT_x(uint16_t op);
	void j_GE_0(uint16_t op);
	void j_GE_8(uint16_t op);
	void j_GE_x(uint16_t op);
	void j_LE_0(uint16_t op);
	void j_LE_8(uint16_t op);
	void j_LE_x(uint16_t op);
	void j_GT_0(uint16_t op);
	void j_GT_8(uint16_t op);
	void j_GT_x(uint16_t op);
	void j_C_0(uint16_t op);
	void j_C_8(uint16_t op);
	void j_C_x(uint16_t op);
	void j_NC_0(uint16_t op);
	void j_NC_8(uint16_t op);
	void j_NC_x(uint16_t op);
	void j_EQ_0(uint16_t op);
	void j_EQ_8(uint16_t op);
	void j_EQ_x(uint16_t op);
	void j_NE_0(uint16_t op);
	void j_NE_8(uint16_t op);
	void j_NE_x(uint16_t op);
	void j_V_0(uint16_t op);
	void j_V_8(uint16_t op);
	void j_V_x(uint16_t op);
	void j_NV_0(uint16_t op);
	void j_NV_8(uint16_t op);
	void j_NV_x(uint16_t op);
	void j_N_0(uint16_t op);
	void j_N_8(uint16_t op);
	void j_N_x(uint16_t op);
	void j_NN_0(uint16_t op);
	void j_NN_8(uint16_t op);
	void j_NN_x(uint16_t op);
	void jump_a(uint16_t op); /* 0160 */
	void jump_b(uint16_t op); /* 0170 */
	void popst(uint16_t op); /* 01c0 */
	void pushst(uint16_t op); /* 01e0 */
	void putst_a(uint16_t op); /* 01a0 */
	void putst_b(uint16_t op); /* 01b0 */
	void reti(uint16_t op); /* 0940 */
	void rets(uint16_t op); /* 0960/70 */
	void rev_a(uint16_t op); /* 0020 */
	void rev_b(uint16_t op); /* 0030 */
	void trap(uint16_t op); /* 0900/10 */
	int apply_window(const char *inst_name,int srcbpp, uint32_t *srcaddr, XY *dst, int *dx, int *dy);
	int compute_fill_cycles(int left_partials, int right_partials, int full_words, int op_timing);
	int compute_pixblt_cycles(int left_partials, int right_partials, int full_words, int op_timing);
	int compute_pixblt_b_cycles(int left_partials, int right_partials, int full_words, int rows, int op_timing, int bpp);
	void memory_w(offs_t offset, uint16_t data);
	uint16_t memory_r(offs_t offset);
	void shiftreg_w(offs_t offset, uint16_t data);
	uint16_t shiftreg_r(offs_t offset);
	uint16_t dummy_shiftreg_r(offs_t offset);
	uint32_t pixel_op00(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op01(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op02(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op03(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op04(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op05(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op06(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op07(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op08(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op09(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op10(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op11(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op12(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op13(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op14(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op15(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op16(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op17(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op18(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op19(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op20(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	uint32_t pixel_op21(uint32_t dstpix, uint32_t mask, uint32_t srcpix);
	void pixblt_1_op0(int src_is_linear, int dst_is_linear);
	void pixblt_2_op0(int src_is_linear, int dst_is_linear);
	void pixblt_4_op0(int src_is_linear, int dst_is_linear);
	void pixblt_8_op0(int src_is_linear, int dst_is_linear);
	void pixblt_16_op0(int src_is_linear, int dst_is_linear);
	void pixblt_r_1_op0(int src_is_linear, int dst_is_linear);
	void pixblt_r_2_op0(int src_is_linear, int dst_is_linear);
	void pixblt_r_4_op0(int src_is_linear, int dst_is_linear);
	void pixblt_r_8_op0(int src_is_linear, int dst_is_linear);
	void pixblt_r_16_op0(int src_is_linear, int dst_is_linear);
	void pixblt_b_1_op0(int dst_is_linear);
	void pixblt_b_2_op0(int dst_is_linear);
	void pixblt_b_4_op0(int dst_is_linear);
	void pixblt_b_8_op0(int dst_is_linear);
	void pixblt_b_16_op0(int dst_is_linear);
	void fill_1_op0(int dst_is_linear);
	void fill_2_op0(int dst_is_linear);
	void fill_4_op0(int dst_is_linear);
	void fill_8_op0(int dst_is_linear);
	void fill_16_op0(int dst_is_linear);
	void pixblt_1_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_2_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_4_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_8_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_16_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_1_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_2_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_4_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_8_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_16_op0_trans(int src_is_linear, int dst_is_linear);
	void pixblt_b_1_op0_trans(int dst_is_linear);
	void pixblt_b_2_op0_trans(int dst_is_linear);
	void pixblt_b_4_op0_trans(int dst_is_linear);
	void pixblt_b_8_op0_trans(int dst_is_linear);
	void pixblt_b_16_op0_trans(int dst_is_linear);
	void fill_1_op0_trans(int dst_is_linear);
	void fill_2_op0_trans(int dst_is_linear);
	void fill_4_op0_trans(int dst_is_linear);
	void fill_8_op0_trans(int dst_is_linear);
	void fill_16_op0_trans(int dst_is_linear);
	void pixblt_1_opx(int src_is_linear, int dst_is_linear);
	void pixblt_2_opx(int src_is_linear, int dst_is_linear);
	void pixblt_4_opx(int src_is_linear, int dst_is_linear);
	void pixblt_8_opx(int src_is_linear, int dst_is_linear);
	void pixblt_16_opx(int src_is_linear, int dst_is_linear);
	void pixblt_r_1_opx(int src_is_linear, int dst_is_linear);
	void pixblt_r_2_opx(int src_is_linear, int dst_is_linear);
	void pixblt_r_4_opx(int src_is_linear, int dst_is_linear);
	void pixblt_r_8_opx(int src_is_linear, int dst_is_linear);
	void pixblt_r_16_opx(int src_is_linear, int dst_is_linear);
	void pixblt_b_1_opx(int dst_is_linear);
	void pixblt_b_2_opx(int dst_is_linear);
	void pixblt_b_4_opx(int dst_is_linear);
	void pixblt_b_8_opx(int dst_is_linear);
	void pixblt_b_16_opx(int dst_is_linear);
	void fill_1_opx(int dst_is_linear);
	void fill_2_opx(int dst_is_linear);
	void fill_4_opx(int dst_is_linear);
	void fill_8_opx(int dst_is_linear);
	void fill_16_opx(int dst_is_linear);
	void pixblt_1_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_2_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_4_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_8_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_16_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_1_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_2_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_4_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_8_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_r_16_opx_trans(int src_is_linear, int dst_is_linear);
	void pixblt_b_1_opx_trans(int dst_is_linear);
	void pixblt_b_2_opx_trans(int dst_is_linear);
	void pixblt_b_4_opx_trans(int dst_is_linear);
	void pixblt_b_8_opx_trans(int dst_is_linear);
	void pixblt_b_16_opx_trans(int dst_is_linear);
	void fill_1_opx_trans(int dst_is_linear);
	void fill_2_opx_trans(int dst_is_linear);
	void fill_4_opx_trans(int dst_is_linear);
	void fill_8_opx_trans(int dst_is_linear);
	void fill_16_opx_trans(int dst_is_linear);
	void check_interrupt();
	void set_pixel_function();
	void set_raster_op();

};


class tms34010_device : public tms340x0_device
{
public:
	tms34010_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock);

	/* Reads & writes to the 34010 I/O registers; place at 0xc0000000 */
	virtual void io_register_w(offs_t offset, u16 data, u16 mem_mask = ~u16(0)) override;
	virtual u16 io_register_r(offs_t offset) override;

protected:
	// device-level overrides
	virtual void device_start() override;

	virtual uint64_t execute_clocks_to_cycles(uint64_t clocks) const noexcept override { return (clocks + 8 - 1) / 8; }
	virtual uint64_t execute_cycles_to_clocks(uint64_t cycles) const noexcept override { return (cycles * 8); }
	virtual std::unique_ptr<util::disasm_interface> create_disassembler() override;
	void internal_regs_map(address_map &map);

	virtual uint16_t ROPCODE() override;
	virtual void execute_op(uint16_t op) override;
	virtual int16_t PARAM_WORD() override;
	virtual int32_t PARAM_LONG() override;
	virtual int16_t PARAM_WORD_NO_INC() override;
	virtual int32_t PARAM_LONG_NO_INC() override;
	virtual uint32_t TMS34010_RDMEM_WORD(offs_t A) override;
	virtual uint32_t TMS34010_RDMEM_DWORD(offs_t A) override;
	virtual void TMS34010_WRMEM_WORD(offs_t A, uint32_t V) override;
	virtual void TMS34010_WRMEM_DWORD(offs_t A, uint32_t V) override;

	typedef void (tms34010_device::*opcode_func)(uint16_t op);

private:
	static const opcode_func s_opcode_table[65536 >> 4];

	memory_access<32, 1, 3, ENDIANNESS_LITTLE>::cache m_cache;
	memory_access<32, 1, 3, ENDIANNESS_LITTLE>::specific m_program;
};

DECLARE_DEVICE_TYPE(TMS34010, tms34010_device)

class tms34020_device : public tms340x0_device
{
public:
	tms34020_device(const machine_config &mconfig, const char *tag, device_t *owner, uint32_t clock);

	/* Reads & writes to the 34020 I/O registers; place at 0xc0000000 */
	virtual void io_register_w(offs_t offset, u16 data, u16 mem_mask = ~u16(0)) override;
	virtual u16 io_register_r(offs_t offset) override;

protected:
	// device-level overrides
	virtual void device_start() override;

	virtual uint64_t execute_clocks_to_cycles(uint64_t clocks) const noexcept override { return (clocks + 4 - 1) / 4; }
	virtual uint64_t execute_cycles_to_clocks(uint64_t cycles) const noexcept override { return (cycles * 4); }
	virtual std::unique_ptr<util::disasm_interface> create_disassembler() override;
	void internal_regs_map(address_map &map);

	virtual uint16_t ROPCODE() override;
	virtual void execute_op(uint16_t op) override;
	virtual int16_t PARAM_WORD() override;
	virtual int32_t PARAM_LONG() override;
	virtual int16_t PARAM_WORD_NO_INC() override;
	virtual int32_t PARAM_LONG_NO_INC() override;
	virtual uint32_t TMS34010_RDMEM_WORD(offs_t A) override;
	virtual uint32_t TMS34010_RDMEM_DWORD(offs_t A) override;
	virtual void TMS34010_WRMEM_WORD(offs_t A, uint32_t V) override;
	virtual void TMS34010_WRMEM_DWORD(offs_t A, uint32_t V) override;

	typedef void (tms34020_device::*opcode_func)(uint16_t op);

	void addxyi_a(uint16_t op);
	void addxyi_b(uint16_t op);
	void blmove(uint16_t op);
	void cexec_l(uint16_t op);
	void cexec_s(uint16_t op);
	void clip(uint16_t op);
	void cmovcg_a(uint16_t op);
	void cmovcg_b(uint16_t op);
	void cmovcm_f(uint16_t op);
	void cmovcm_b(uint16_t op);
	void cmovgc_a(uint16_t op);
	void cmovgc_b(uint16_t op);
	void cmovgc_a_s(uint16_t op);
	void cmovgc_b_s(uint16_t op);
	void cmovmc_f(uint16_t op);
	void cmovmc_f_va(uint16_t op);
	void cmovmc_f_vb(uint16_t op);
	void cmovmc_b(uint16_t op);
	void cmp_k_a(uint16_t op);
	void cmp_k_b(uint16_t op);
	void cvdxyl_a(uint16_t op);
	void cvdxyl_b(uint16_t op);
	void cvmxyl_a(uint16_t op);
	void cvmxyl_b(uint16_t op);
	void cvsxyl_a(uint16_t op);
	void cvsxyl_b(uint16_t op);
	void exgps_a(uint16_t op);
	void exgps_b(uint16_t op);
	void fline(uint16_t op);
	void fpixeq(uint16_t op);
	void fpixne(uint16_t op);
	void getps_a(uint16_t op);
	void getps_b(uint16_t op);
	void idle(uint16_t op);
	void linit(uint16_t op);
	void mwait(uint16_t op);
	void pfill_xy(uint16_t op);
	void pixblt_l_m_l(uint16_t op);
	void retm(uint16_t op);
	void rmo_a(uint16_t op);
	void rmo_b(uint16_t op);
	void rpix_a(uint16_t op);
	void rpix_b(uint16_t op);
	void setcdp(uint16_t op);
	void setcmp(uint16_t op);
	void setcsp(uint16_t op);
	void swapf_a(uint16_t op);
	void swapf_b(uint16_t op);
	void tfill_xy(uint16_t op);
	void trapl(uint16_t op);
	void vblt_b_l(uint16_t op);
	void vfill_l(uint16_t op);
	void vlcol(uint16_t op);

private:
	static const opcode_func s_opcode_table[65536 >> 4];

	memory_access<32, 2, 3, ENDIANNESS_LITTLE>::cache m_cache;
	memory_access<32, 2, 3, ENDIANNESS_LITTLE>::specific m_program;
};

DECLARE_DEVICE_TYPE(TMS34020, tms34020_device)



/* Host control interface */
#define TMS34010_HOST_ADDRESS_L     0
#define TMS34010_HOST_ADDRESS_H     1
#define TMS34010_HOST_DATA          2
#define TMS34010_HOST_CONTROL       3

#endif // MAME_CPU_TMS34010_TMS34010_H
