%-----------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%-----------------------------------------------------------------------------%
% Copyright (C) 2006-2010 The University of Melbourne.
% This file may only be copied under the terms of the GNU General
% Public License - see the file COPYING in the Mercury distribution.
%-----------------------------------------------------------------------------%
%
% File: implicit_parallelism.m.
% Authors: tannier, pbone.
%
% This module uses deep profiling feedback information generated by
% mdprof_feedback to introduce parallel conjunctions where it could be
% worthwhile (implicit parallelism). It deals with both independent and
% dependent parallelism.
%
%-----------------------------------------------------------------------------%

:- module transform_hlds.implicit_parallelism.
:- interface.

:- import_module hlds.hlds_module.

:- import_module io.

%-----------------------------------------------------------------------------%

    % apply_implicit_parallelism_transformation(!ModuleInfo, !IO)
    %
    % Apply the implicit parallelism transformation using the specified
    % feedback file.
    %
:- pred apply_implicit_parallelism_transformation(
    module_info::in, module_info::out, io::di, io::uo) is det.

%-----------------------------------------------------------------------------%
%-----------------------------------------------------------------------------%

:- implementation.

:- import_module check_hlds.inst_match.
:- import_module check_hlds.mode_util.
:- import_module hlds.goal_util.
:- import_module hlds.hlds_goal.
:- import_module hlds.hlds_pred.
:- import_module hlds.instmap.
:- import_module hlds.pred_table.
:- import_module hlds.quantification.
:- import_module libs.compiler_util.
:- import_module libs.globals.
:- import_module libs.options.
:- import_module ll_backend.
:- import_module ll_backend.prog_rep.
:- import_module ll_backend.stack_layout.
:- import_module mdbcomp.feedback.
:- import_module mdbcomp.feedback.automatic_parallelism.
:- import_module mdbcomp.prim_data.
:- import_module mdbcomp.program_representation.
:- import_module parse_tree.error_util.
:- import_module parse_tree.prog_data.
:- import_module transform_hlds.dep_par_conj.

:- import_module assoc_list.
:- import_module bool.
:- import_module char.
:- import_module counter.
:- import_module int.
:- import_module list.
:- import_module map.
:- import_module maybe.
:- import_module pair.
:- import_module require.
:- import_module set.
:- import_module string.
:- import_module svmap.
:- import_module term.
:- import_module varset.

%-----------------------------------------------------------------------------%

apply_implicit_parallelism_transformation(!ModuleInfo, !IO) :-
    module_info_get_globals(!.ModuleInfo, Globals),
    lookup_bool_option(Globals, old_implicit_parallelism,
        UseOldImplicitParallelism),
    (
        UseOldImplicitParallelism = yes,
        apply_old_implicit_parallelism_transformation(!.ModuleInfo,
            MaybeModuleInfo),
        (
            MaybeModuleInfo = ok(!:ModuleInfo)
        ;
            MaybeModuleInfo = error(Error),
            sorry(this_file, "The old implicit parallelism code is not "
                ++ "supported: " ++ Error)
        )
    ;
        UseOldImplicitParallelism = no,
        io_get_maybe_source_file_map(MaybeSourceFileMap, !IO),
        (
            MaybeSourceFileMap = yes(SourceFileMap)
        ;
            MaybeSourceFileMap = no,
            error(this_file ++ "could not retrieve the source file map")
        ),
        apply_new_implicit_parallelism_transformation(SourceFileMap, Specs,
            !ModuleInfo),
        write_error_specs(Specs, Globals, 0, _, 0, NumErrors, !IO),
        module_info_incr_num_errors(NumErrors, !ModuleInfo)
    ).

%-----------------------------------------------------------------------------%

    % This type is used to track whether parallelism has been introduced by a
    % predicate.
    %
:- type introduced_parallelism
    --->    have_not_introduced_parallelism
    ;       introduced_parallelism.

:- pred apply_new_implicit_parallelism_transformation(source_file_map::in,
    list(error_spec)::out, module_info::in, module_info::out) is det.

apply_new_implicit_parallelism_transformation(SourceFileMap, Specs,
        !ModuleInfo) :-
    module_info_get_globals(!.ModuleInfo, Globals0),
    globals.get_maybe_feedback_info(Globals0, MaybeFeedbackInfo),
    module_info_get_name(!.ModuleInfo, ModuleName),
    (
        yes(FeedbackInfo) = MaybeFeedbackInfo,
        get_implicit_parallelism_feedback(ModuleName, FeedbackInfo,
            ParallelismInfo)
    ->
        % Retrieve and process predicates.
        module_info_get_valid_predids(PredIds, !ModuleInfo),
        module_info_get_predicate_table(!.ModuleInfo, PredTable0),
        predicate_table_get_preds(PredTable0, PredMap0),
        list.foldl3(maybe_parallelise_pred(!.ModuleInfo, ParallelismInfo),
            PredIds, PredMap0, PredMap,
            have_not_introduced_parallelism, IntroducedParallelism,
            [], Specs),
        (
            IntroducedParallelism = have_not_introduced_parallelism
        ;
            IntroducedParallelism = introduced_parallelism,
            predicate_table_set_preds(PredMap, PredTable0, PredTable),
            module_info_set_predicate_table(PredTable, !ModuleInfo),
            module_info_set_contains_par_conj(!ModuleInfo)
        )
    ;
        map.lookup(SourceFileMap, ModuleName, ModuleFilename),
        Context = context(ModuleFilename, 1),
        Peices = [words("Implicit parallelism was requested but the"),
            words("feedback file does not the candidate parallel"),
            words("conjunctions feedback information.")],
        Specs = [error_spec(severity_error, phase_auto_parallelism,
            [simple_msg(Context, [always(Peices)])])]
    ).

    % Information retrieved from the feedback system to be used for
    % parallelising this module.
    %
:- type parallelism_info
    --->    parallelism_info(
                pi_parameters           :: candidate_par_conjunctions_params,

                pi_cpc_map              :: module_candidate_par_conjs_map
                    % A map of candidate parallel conjunctions in this module
                    % indexed by their procedure.
            ).

:- type intra_module_proc_label
    --->    intra_module_proc_label(
                im_pred_name            :: string,
                im_arity                :: int,
                im_pred_or_func         :: pred_or_func,
                im_mode                 :: int
            ).

:- type candidate_par_conjunction == candidate_par_conjunction(pard_goal).

:- type seq_conj == seq_conj(pard_goal).

    % A map of the candidate parallel conjunctions indexed by the procedure
    % label for a given module.
    %
:- type module_candidate_par_conjs_map
    == map(intra_module_proc_label, candidate_par_conjunctions_proc).

:- pred get_implicit_parallelism_feedback(module_name::in, feedback_info::in,
    parallelism_info::out) is semidet.

get_implicit_parallelism_feedback(ModuleName, FeedbackInfo, ParallelismInfo) :-
    FeedbackData =
        feedback_data_candidate_parallel_conjunctions(_, _),
    get_feedback_data(FeedbackInfo, FeedbackData),
    FeedbackData =
        feedback_data_candidate_parallel_conjunctions(Parameters, AssocList),
    make_module_candidate_par_conjs_map(ModuleName, AssocList,
        CandidateParConjsMap),
    ParallelismInfo = parallelism_info(Parameters, CandidateParConjsMap).

:- pred make_module_candidate_par_conjs_map(module_name::in,
    assoc_list(string_proc_label, candidate_par_conjunctions_proc)::in,
    module_candidate_par_conjs_map::out) is det.

make_module_candidate_par_conjs_map(ModuleName,
        CandidateParConjsAssocList0, CandidateParConjsMap) :-
    ModuleNameStr = sym_name_to_string(ModuleName),
    filter_map(cpc_proc_is_in_module(ModuleNameStr),
        CandidateParConjsAssocList0, CandidateParConjsAssocList),
    CandidateParConjsMap = map.from_assoc_list(CandidateParConjsAssocList).

:- pred cpc_proc_is_in_module(string::in,
    pair(string_proc_label, candidate_par_conjunctions_proc)::in,
    pair(intra_module_proc_label, candidate_par_conjunctions_proc)::out)
    is semidet.

cpc_proc_is_in_module(ModuleName, ProcLabel - CPC, IMProcLabel - CPC) :-
    (
        ProcLabel = str_ordinary_proc_label(PredOrFunc, _, DefModule, Name,
            Arity, Mode)
    ;
        ProcLabel = str_special_proc_label(_, _, DefModule, Name, Arity, Mode),
        PredOrFunc = pf_predicate
    ),
    ModuleName = DefModule,
    IMProcLabel = intra_module_proc_label(Name, Arity, PredOrFunc, Mode).

%-----------------------------------------------------------------------------%

:- pred maybe_parallelise_pred(module_info::in, parallelism_info::in,
    pred_id::in, pred_table::in, pred_table::out,
    introduced_parallelism::in, introduced_parallelism::out,
    list(error_spec)::in, list(error_spec)::out) is det.

maybe_parallelise_pred(ModuleInfo, ParallelismInfo, PredId, !PredTable,
        !IntroducedParallelism, !Specs) :-
    map.lookup(!.PredTable, PredId, PredInfo0),
    ProcIds = pred_info_non_imported_procids(PredInfo0),
    pred_info_get_procedures(PredInfo0, ProcTable0),
    list.foldl3(maybe_parallelise_proc(ModuleInfo, ParallelismInfo, PredId),
        ProcIds, ProcTable0, ProcTable, have_not_introduced_parallelism,
        ProcIntroducedParallelism, !Specs),
    (
        ProcIntroducedParallelism = have_not_introduced_parallelism
    ;
        ProcIntroducedParallelism = introduced_parallelism,
        !:IntroducedParallelism = introduced_parallelism,
        pred_info_set_procedures(ProcTable, PredInfo0, PredInfo),
        svmap.det_update(PredId, PredInfo, !PredTable)
    ).

:- pred maybe_parallelise_proc(module_info::in, parallelism_info::in,
    pred_id::in, proc_id::in, proc_table::in, proc_table::out,
    introduced_parallelism::in, introduced_parallelism::out,
    list(error_spec)::in, list(error_spec)::out) is det.

maybe_parallelise_proc(ModuleInfo, ParallelismInfo, PredId, ProcId, !ProcTable,
        !IntroducedParallelism, !Specs) :-
    module_info_pred_proc_info(ModuleInfo, PredId, ProcId,
        PredInfo, ProcInfo0),

    % Lookup the Candidate Parallel Conjunction (CPC) Map for this procedure.
    Name = pred_info_name(PredInfo),
    Arity = pred_info_orig_arity(PredInfo),
    PredOrFunc = pred_info_is_pred_or_func(PredInfo),
    Mode = proc_id_to_int(ProcId),
    IMProcLabel = intra_module_proc_label(Name, Arity, PredOrFunc, Mode),
    CPCMap = ParallelismInfo ^ pi_cpc_map,
    ( map.search(CPCMap, IMProcLabel, CPCProc) ->
        proc_info_get_has_parallel_conj(ProcInfo0, HasParallelConj),
        (
            HasParallelConj = yes,
            Spec = report_already_parallelised(PredInfo),
            !:Specs = [Spec | !.Specs]
        ;
            HasParallelConj = no,

            proc_info_get_goal(ProcInfo0, Goal0),
            CPCProc = candidate_par_conjunctions_proc(VarTable, CPCs0),

            Context = goal_info_get_context(Goal0 ^ hlds_goal_info),
            term.context_file(Context, FileName),
            proc_info_get_vartypes(ProcInfo0, VarTypes),
            % VarNumRep is not used by goal_to_goal_rep, var_num_1_byte
            % is an arbitrary value. XXX zs: I don't think this is true.
            VarNumRep = var_num_1_byte,
            proc_info_get_headvars(ProcInfo0, HeadVars),
            proc_info_get_varset(ProcInfo0, VarSet),
            compute_var_number_map(HeadVars, VarSet, [], Goal0, VarNumMap),
            ProgRepInfo = prog_rep_info(FileName, VarTypes, VarNumMap,
                VarNumRep, ModuleInfo),
            proc_info_get_initial_instmap(ProcInfo0, ModuleInfo, Instmap),

            % Sort the candidate parallelisations so that we introduce
            % parallelisations in an order that allows us to continue to insert
            % parallelisations even as the goal tree changes. In particular,
            % insert deeper parallelisations before shallower ones, and later
            % ones before earlier ones.
            sort_and_remove_dups(compare_candidate_par_conjunctions,
                CPCs0, CPCs),
            foldl3(
                maybe_parallelise_goal(PredInfo, ProgRepInfo, VarTable,
                    Instmap),
                CPCs, Goal0, Goal, !IntroducedParallelism, !Specs),
            (
                !.IntroducedParallelism = introduced_parallelism,
                % In the future we'll specialise the procedure for parallelism,
                % We don't do that now so simply replace the procedure's body.
                proc_info_set_goal(Goal, ProcInfo0, ProcInfo1),
                proc_info_set_has_parallel_conj(yes, ProcInfo1, ProcInfo),
                svmap.det_update(ProcId, ProcInfo, !ProcTable)
            ;
                !.IntroducedParallelism = have_not_introduced_parallelism
            )
        )
    ;
        true
    ).

:- pred compare_candidate_par_conjunctions(candidate_par_conjunction::in,
    candidate_par_conjunction::in, comparison_result::out) is det.

compare_candidate_par_conjunctions(CPCA, CPCB, Result) :-
    goal_path_from_string_det(CPCA ^ cpc_goal_path, fgp(StepsA)),
    goal_path_from_string_det(CPCB ^ cpc_goal_path, fgp(StepsB)),
    compare_goal_paths(StepsA, StepsB, Result).

:- pred compare_goal_paths(list(goal_path_step)::in, list(goal_path_step)::in,
    comparison_result::out) is det.

compare_goal_paths(StepsA, StepsB, Result) :-
    (
        StepsA = [FirstStepA | LaterStepsA],
        (
            StepsB = [FirstStepB | LaterStepsB],
            compare(Result0, FirstStepA, FirstStepB),
            % Reverse the ordering here so that later goals are sorted before
            % earlier ones. This way parallelisations are placed inside later
            % goals first.
            (
                Result0 = (=),
                compare_goal_paths(LaterStepsA, LaterStepsB, Result)
            ;
                Result0 = (<),
                Result = (>)
            ;
                Result0 = (>),
                Result = (<)
            )
        ;
            StepsB = [],
            % StepsA is longer than StepsB. Make A 'less than' B so that
            % deeper parallelisations are insearted first.
            Result = (<)
        )
    ;
        StepsA = [],
        (
            StepsB = [_ | _],
            % B is 'less than' A, see above.
            Result = (>)
        ;
            StepsB = [],
            % Both goal paths are empty.
            Result = (=)
        )
    ).

    % maybe_parallelise_goal(ProgRepInfo, VarTable, CPC, !Goal,
    %   !IntroducedParallelism).
    %
    % Attempt to parallelise some part of !.Goal returning !:Goal.
    % If !.IntroducedParallelism = have_not_introduced_parallelism then !Goal
    % will be unmodified.
    %
:- pred maybe_parallelise_goal(pred_info::in, prog_rep_info::in,
    var_table::in, instmap::in, candidate_par_conjunction::in,
    hlds_goal::in, hlds_goal::out,
    introduced_parallelism::in, introduced_parallelism::out,
    list(error_spec)::in, list(error_spec)::out) is det.

maybe_parallelise_goal(PredInfo, ProgRepInfo, VarTable, Instmap0, CPC, Goal0,
        Goal, !IntroducedParallelism, !Specs) :-
    TargetGoalPathString = CPC ^ cpc_goal_path,
    ( goal_path_from_string(TargetGoalPathString, TargetGoalPathPrime) ->
        TargetGoalPath = TargetGoalPathPrime
    ;
        unexpected(this_file,
            "Invalid goal path in CPC Feedback Information")
    ),
    maybe_transform_goal_at_goal_path_with_instmap(
        maybe_parallelise_conj(ProgRepInfo, VarTable, CPC),
        TargetGoalPath, Instmap0, Goal0, MaybeGoal),
    (
        MaybeGoal = ok(Goal),
        !:IntroducedParallelism = introduced_parallelism
    ;
        (
            MaybeGoal = error(Error)
        ;
            MaybeGoal = goal_not_found,
            Error = "Could not find goal in procedure; "
                ++ "perhaps the program has changed"
        ),
        Goal = Goal0,
        Spec = report_failed_parallelisation(PredInfo, TargetGoalPathString,
            Error),
        !:Specs = [Spec | !.Specs]
    ).

:- func report_failed_parallelisation(pred_info, string, string) =
    error_spec.

report_failed_parallelisation(PredInfo, GoalPath, Error) = Spec :-
    % Should the severity be informational?
    PredOrFunc = pred_info_is_pred_or_func(PredInfo),
    ModuleName = pred_info_module(PredInfo),
    PredName = pred_info_name(PredInfo),
    Arity = pred_info_orig_arity(PredInfo),
    Peices = [words("In"), p_or_f(PredOrFunc),
        sym_name_and_arity(qualified(ModuleName, PredName) / Arity),
        suffix(":"), nl,
        words("Warning: could not auto-parallelise"), quote(GoalPath),
        suffix(":"), words(Error)],
    pred_info_get_context(PredInfo, Context),
    % XXX Make this a warning or error if the user wants compilation to
    % abort.
    Spec = error_spec(severity_informational, phase_auto_parallelism,
        [simple_msg(Context, [always(Peices)])]).

:- func report_already_parallelised(pred_info) = error_spec.

report_already_parallelised(PredInfo) = Spec :-
    % Should the severity be informational?
    PredOrFunc = pred_info_is_pred_or_func(PredInfo),
    ModuleName = pred_info_module(PredInfo),
    PredName = pred_info_name(PredInfo),
    Arity = pred_info_orig_arity(PredInfo),
    Peices = [words("In"), p_or_f(PredOrFunc),
        sym_name_and_arity(qualified(ModuleName, PredName) / Arity),
        suffix(":"), nl,
        words("Warning: this procedure contains explicit parallel"),
        words("conjunctions, it will not be automatically parallelised.")],
    pred_info_get_context(PredInfo, Context),
    Spec = error_spec(severity_warning, phase_auto_parallelism,
        [simple_msg(Context, [always(Peices)])]).

:- pred maybe_parallelise_conj(prog_rep_info::in, var_table::in,
    candidate_par_conjunction::in, instmap::in, hlds_goal::in,
    maybe_error(hlds_goal)::out) is det.

maybe_parallelise_conj(ProgRepInfo, VarTable, CPC, Instmap0,
        Goal0, MaybeGoal) :-
    Goal0 = hlds_goal(GoalExpr0, _GoalInfo0),
    % We've reached the point indicated by the goal path, Find the
    % conjuncts that we wish to parallelise.
    cpc_get_first_goal(CPC, FirstGoalRep),
    (
        GoalExpr0 = conj(plain_conj, Conjs0),
        flatten_conj(Conjs0, Conjs1),
        find_first_goal(FirstGoalRep, Conjs1, ProgRepInfo, VarTable, Instmap0,
            found_first_goal(GoalsBefore, FirstGoal, OtherGoals))
    ->
        GoalsBeforeInstDeltas = map(
            (func(G) = goal_info_get_instmap_delta(G ^ hlds_goal_info)),
            GoalsBefore),
        foldl(apply_instmap_delta_sv, GoalsBeforeInstDeltas,
            Instmap0, Instmap),
        build_par_conjunction(ProgRepInfo, VarTable, Instmap,
            [FirstGoal | OtherGoals], CPC, MaybeParConjunction),
        (
            MaybeParConjunction = ok(
                par_conjunction_and_remaining_goals(ParConjunction,
                RemainingGoals)),
            Conjs = GoalsBefore ++ ParConjunction ++ RemainingGoals,
            GoalExpr = conj(plain_conj, Conjs),
            MaybeGoal = ok(hlds_goal(GoalExpr, Goal0 ^ hlds_goal_info))
        ;
            MaybeParConjunction = error(Error),
            MaybeGoal = error(Error)
        )
    ;
        MaybeGoal = error("Could not find partition within conjunction: "
            ++ "perhaps the program has changed")
    ).

:- pred cpc_get_first_goal(candidate_par_conjunction::in, pard_goal::out)
    is det.

cpc_get_first_goal(CPC, FirstGoal) :-
    GoalsBefore = CPC ^ cpc_goals_before,
    (
        GoalsBefore = [FirstGoal | _]
    ;
        GoalsBefore = [],
        ParConj = CPC ^ cpc_conjs,
        (
            ParConj = [FirstParConj | _],
            FirstParConj = seq_conj([FirstGoalPrime | _])
        ->
            FirstGoal = FirstGoalPrime
        ;
            error(this_file ++ "Candidate parallel conjunction is empty")
        )
    ).

:- type find_first_goal_result
    --->    did_not_find_first_goal
    ;       found_first_goal(
                ffg_goals_before        :: hlds_goals,
                ffg_goal                :: hlds_goal,
                ffg_goals_after         :: hlds_goals
            ).

:- pred find_first_goal(pard_goal::in, list(hlds_goal)::in,
    prog_rep_info::in, var_table::in, instmap::in,
    find_first_goal_result::out) is det.

find_first_goal(_, [], _, _, _, did_not_find_first_goal).
find_first_goal(GoalRep, [Goal | Goals], ProcRepInfo, VarTable, !.Instmap,
        Result) :-
    (
        pard_goal_match_hlds_goal(ProcRepInfo, VarTable, !.Instmap, GoalRep,
            Goal)
    ->
        Result = found_first_goal([], Goal, Goals)
    ;
        InstmapDelta = goal_info_get_instmap_delta(Goal ^ hlds_goal_info),
        apply_instmap_delta_sv(InstmapDelta, !Instmap),
        find_first_goal(GoalRep, Goals, ProcRepInfo, VarTable, !.Instmap,
            Result0),
        (
            Result0 = did_not_find_first_goal,
            Result = did_not_find_first_goal
        ;
            Result0 = found_first_goal(GoalsBefore0, _, _),
            Result = Result0 ^ ffg_goals_before := [Goal | GoalsBefore0]
        )
    ).

:- type par_conjunction_and_remaining_goals
    --->    par_conjunction_and_remaining_goals(
                pcrg_par_conjunction            :: hlds_goals,
                pcrg_remaining_goals            :: hlds_goals
            ).

:- pred build_par_conjunction(prog_rep_info::in, var_table::in, instmap::in,
    hlds_goals::in, candidate_par_conjunction::in,
    maybe_error(par_conjunction_and_remaining_goals)::out) is det.

build_par_conjunction(ProcRepInfo, VarTable, Instmap0, !.Goals, CPC,
        MaybeParConjunction) :-
    GoalRepsBefore = CPC ^ cpc_goals_before,
    GoalRepsAfter = CPC ^ cpc_goals_after,
    ParConjReps = CPC ^ cpc_conjs,
    some [!Instmap] (
        !:Instmap = Instmap0,
        build_seq_conjuncts(ProcRepInfo, VarTable, GoalRepsBefore,
            MaybeGoalsBefore, !Goals, !Instmap),
        build_par_conjuncts(ProcRepInfo, VarTable, ParConjReps,
            MaybeParConjuncts, !Goals, !Instmap),
        build_seq_conjuncts(ProcRepInfo, VarTable, GoalRepsAfter,
            MaybeGoalsAfter, !Goals, !Instmap),
        _ = !.Instmap
    ),
    (
        MaybeGoalsBefore = yes(GoalsBefore),
        (
            MaybeParConjuncts = yes(ParConjuncts),
            (
                MaybeGoalsAfter = yes(GoalsAfter),
                create_conj_from_list(ParConjuncts, parallel_conj,
                    ParConjunction0),
                ParConjunction = GoalsBefore ++ [ParConjunction0 | GoalsAfter],
                MaybeParConjunction = ok(
                    par_conjunction_and_remaining_goals(ParConjunction,
                    !.Goals))
            ;
                MaybeGoalsAfter = no,
                MaybeParConjunction = error("The goals after the parallel "
                    ++ "conjunction do not match those in the feedback file")
            )
        ;
            MaybeParConjuncts = no,
            MaybeParConjunction = error("The goals within the parallel "
                ++ "conjunction do not match those in the feedback file")
        )
    ;
        MaybeGoalsBefore = no,
        MaybeParConjunction = error("The goals before the parallel "
            ++ "conjunction do not match those in the feedback file")
    ).

:- pred build_par_conjuncts(prog_rep_info::in, var_table::in,
    list(seq_conj)::in, maybe(hlds_goals)::out,
    hlds_goals::in, hlds_goals::out, instmap::in, instmap::out) is det.

build_par_conjuncts(_, _, [], yes([]), !Goals, !Instmap).
build_par_conjuncts(ProcRepInfo, VarTable, [GoalRep | GoalReps], MaybeConjs,
        !Goals, !Instmap) :-
    GoalRep = seq_conj(SeqConjs),
    build_seq_conjuncts(ProcRepInfo, VarTable, SeqConjs, MaybeConj, !Goals,
        !Instmap),
    (
        MaybeConj = yes(Conj0),
        create_conj_from_list(Conj0, plain_conj, Conj),
        build_par_conjuncts(ProcRepInfo, VarTable, GoalReps, MaybeConjs0,
            !Goals, !Instmap),
        (
            MaybeConjs0 = yes(Conjs0),
            MaybeConjs = yes([Conj | Conjs0])
        ;
            MaybeConjs0 = no,
            MaybeConjs = no
        )
    ;
        MaybeConj = no,
        MaybeConjs = no
    ).

:- pred build_seq_conjuncts(prog_rep_info::in, var_table::in,
    list(pard_goal)::in, maybe(hlds_goals)::out,
    hlds_goals::in, hlds_goals::out, instmap::in, instmap::out) is det.

build_seq_conjuncts(_, _, [], yes([]), !Goals, !Instmap).
build_seq_conjuncts(ProcRepInfo, VarTable, [GoalRep | GoalReps], MaybeConjs,
        !Goals, !Instmap) :-
    (
        !.Goals = [Goal | !:Goals],
        ( pard_goal_match_hlds_goal(ProcRepInfo, VarTable, !.Instmap, GoalRep,
                Goal) ->
            InstmapDelta = goal_info_get_instmap_delta(Goal ^ hlds_goal_info),
            apply_instmap_delta_sv(InstmapDelta, !Instmap),
            build_seq_conjuncts(ProcRepInfo, VarTable, GoalReps, MaybeConjs0,
                !Goals, !Instmap),
            (
                MaybeConjs0 = yes(Conjs0),
                MaybeConjs = yes([Goal | Conjs0])
            ;
                MaybeConjs0 = no,
                MaybeConjs = no
            )
        ;
            MaybeConjs = no
        )
    ;
        !.Goals = [],
        MaybeConjs = no
    ).

:- pred pard_goal_match_hlds_goal(prog_rep_info::in, var_table::in,
    instmap::in, pard_goal::in, hlds_goal::in) is semidet.

pard_goal_match_hlds_goal(ProgRepInfo, VarTable, Instmap, GoalRep1, Goal) :-
    goal_to_goal_rep(ProgRepInfo, Instmap, Goal, GoalRep2),
    goal_reps_match(VarTable, GoalRep1, GoalRep2).

:- pred goal_reps_match(var_table::in, goal_rep(A)::in, goal_rep(B)::in)
    is semidet.

goal_reps_match(VarTable, goal_rep(GoalExpr1, Detism, _),
        goal_rep(GoalExpr2, Detism, _)) :-
    goal_expr_reps_match(VarTable, GoalExpr1, GoalExpr2).

:- pred goal_expr_reps_match(var_table::in, goal_expr_rep(A)::in,
    goal_expr_rep(B)::in) is semidet.

goal_expr_reps_match(VarTable, conj_rep(Conjs1), conj_rep(Conjs2)) :-
    zip_all_true(goal_reps_match(VarTable), Conjs1, Conjs2).
goal_expr_reps_match(VarTable, disj_rep(Disjs1), disj_rep(Disjs2)) :-
    zip_all_true(goal_reps_match(VarTable), Disjs1, Disjs2).
goal_expr_reps_match(VarTable, switch_rep(VarRep1, CanFail, Cases1),
        switch_rep(VarRep2, CanFail, Cases2)) :-
    % Note that cases can appear in a different order and goals would still be
    % equivalent.  We don't handle this.
    var_reps_match(VarTable, VarRep1, VarRep2),
    zip_all_true(case_reps_match(VarTable), Cases1, Cases2).
goal_expr_reps_match(VarTable, ite_rep(Cond1, Then1, Else1),
        ite_rep(Cond2, Then2, Else2)) :-
    goal_reps_match(VarTable, Cond1, Cond2),
    goal_reps_match(VarTable, Then1, Then2),
    goal_reps_match(VarTable, Else1, Else2).
goal_expr_reps_match(VarTable, negation_rep(SubGoal1),
        negation_rep(SubGoal2)) :-
    goal_reps_match(VarTable, SubGoal1, SubGoal2).
goal_expr_reps_match(VarTable, scope_rep(SubGoal1, MaybeCut),
        scope_rep(SubGoal2, MaybeCut)) :-
    goal_reps_match(VarTable, SubGoal1, SubGoal2).
goal_expr_reps_match(VarTable, atomic_goal_rep(_, _, _, AtomicGoal1),
        atomic_goal_rep(_, _, _, AtomicGoal2)) :-
    % We don't compare names and file numbers, a trivial change made by the
    % user could change the line number dramatically without changing how the
    % program should be parallelised.
    %
    % Vars are not matched here either, we only consider the vars within the
    % atomic_goal_rep structures.
    atomic_goal_reps_match(VarTable, AtomicGoal1, AtomicGoal2).

:- pred atomic_goal_reps_match(var_table::in, atomic_goal_rep::in,
    atomic_goal_rep::in) is semidet.

atomic_goal_reps_match(VarTable, AtomicRepA, AtomicRepB) :-
    (
        (
            AtomicRepA = unify_construct_rep(VarA, ConsId, ArgsA),
            AtomicRepB = unify_construct_rep(VarB, ConsId, ArgsB)
        ;
            AtomicRepA = unify_deconstruct_rep(VarA, ConsId, ArgsA),
            AtomicRepB = unify_deconstruct_rep(VarB, ConsId, ArgsB)
        ;
            AtomicRepA = higher_order_call_rep(VarA, ArgsA),
            AtomicRepB = higher_order_call_rep(VarB, ArgsB)
        ;
            AtomicRepA = method_call_rep(VarA, MethodNum, ArgsA),
            AtomicRepB = method_call_rep(VarB, MethodNum, ArgsB)
        ),
        var_reps_match(VarTable, VarA, VarB),
        zip_all_true(var_reps_match(VarTable), ArgsA, ArgsB)
    ;
        (
            AtomicRepA = partial_deconstruct_rep(VarA, ConsId, MaybeArgsA),
            AtomicRepB = partial_deconstruct_rep(VarB, ConsId, MaybeArgsB)
        ;
            AtomicRepA = partial_construct_rep(VarA, ConsId, MaybeArgsA),
            AtomicRepB = partial_construct_rep(VarB, ConsId, MaybeArgsB)
        ),
        var_reps_match(VarTable, VarA, VarB),
        zip_all_true(maybe_var_reps_match(VarTable), MaybeArgsA, MaybeArgsB)
    ;
        (
            AtomicRepA = unify_assign_rep(VarA1, VarA2),
            AtomicRepB = unify_assign_rep(VarB1, VarB2)
        ;
            AtomicRepA = cast_rep(VarA1, VarA2),
            AtomicRepB = cast_rep(VarB1, VarB2)
        ;
            AtomicRepA = unify_simple_test_rep(VarA1, VarA2),
            AtomicRepB = unify_simple_test_rep(VarB1, VarB2)
        ),
        var_reps_match(VarTable, VarA1, VarB1),
        var_reps_match(VarTable, VarA2, VarB2)
    ;
        (
            AtomicRepA = pragma_foreign_code_rep(ArgsA),
            AtomicRepB = pragma_foreign_code_rep(ArgsB)
        ;
            AtomicRepA = plain_call_rep(ModuleName, PredName, ArgsA),
            AtomicRepB = plain_call_rep(ModuleName, PredName, ArgsB)
        ;
            AtomicRepA = builtin_call_rep(ModuleName, PredName, ArgsA),
            AtomicRepB = builtin_call_rep(ModuleName, PredName, ArgsB)
        ;
            AtomicRepA = event_call_rep(EventName, ArgsA),
            AtomicRepB = event_call_rep(EventName, ArgsB)
        ),
        zip_all_true(var_reps_match(VarTable), ArgsA, ArgsB)
    ).

:- pred case_reps_match(var_table::in, case_rep(A)::in, case_rep(B)::in)
    is semidet.

case_reps_match(VarTable, case_rep(ConsId, OtherConsIds, GoalRep1),
        case_rep(ConsId, OtherConsIds, GoalRep2)) :-
    goal_reps_match(VarTable, GoalRep1, GoalRep2).

:- pred var_reps_match(var_table::in, var_rep::in, var_rep::in) is semidet.

var_reps_match(VarTable, VarA, VarB) :-
    ( search_var_name(VarTable, VarA, _) ->
        % Variables named by the programmer _must_ match, we expect to find
        % them in the var table, and that they would be identical.  (Since one
        % of the variables will be built using it's name and the var table
        % constructed when converting the original code to byte code).
        VarA = VarB
    ;
        % Unamed variables match implicitly.  They will usually be identical be
        % we allow this to be releaxed so that the program may change a little
        % after being profiled but before being parallelised.
        true
    ).

:- pred maybe_var_reps_match(var_table::in,
    maybe(var_rep)::in, maybe(var_rep)::in) is semidet.

maybe_var_reps_match(_, no, no).
maybe_var_reps_match(VarTable, yes(VarA), yes(VarB)) :-
    var_reps_match(VarTable, VarA, VarB).

    % zip_all_true(Pred, ListA, ListB)
    %
    % True when lists have equal length and every corresponding pair of values
    % from the lists satisifies Pred.
    %
:- pred zip_all_true(pred(A, B), list(A), list(B)).
:- mode zip_all_true(pred(in, in) is semidet, in, in) is semidet.

zip_all_true(_, [], []).
zip_all_true(Pred, [A | As], [B | Bs]) :-
    Pred(A, B),
    zip_all_true(Pred, As, Bs).

:- pred match_sym_name(sym_name::in, string::in, string::in) is semidet.

match_sym_name(unqualified(ProcName), _, ProcName).
match_sym_name(qualified(SymModuleName, ProcName), ModuleName, ProcName) :-
    ModuleNameParts = reverse(split_at_separator(unify('.'), ModuleName)),
    match_sym_module_name(SymModuleName, ModuleNameParts).

:- pred match_sym_module_name(sym_name::in, list(string)::in) is semidet.

match_sym_module_name(unqualified(Name), [Name]).
match_sym_module_name(qualified(SymName, Name), [Name | Names]) :-
    match_sym_module_name(SymName, Names).

:- pred args_match(prog_varset::in, prog_var::in, maybe(string)::in)
    is semidet.

args_match(_, _, no).
args_match(VarSet, Var, yes(Name)) :-
    varset.search_name(VarSet, Var, Name).

:- pred model_det_and_at_least_semipure(hlds_goal::in) is semidet.

model_det_and_at_least_semipure(Goal) :-
    GoalInfo = Goal ^ hlds_goal_info,
    Determinism = goal_info_get_determinism(GoalInfo),
    ( Determinism = detism_det
    ; Determinism = detism_cc_multi
    ),
    Purity = goal_info_get_purity(GoalInfo),
    ( Purity = purity_pure
    ; Purity = purity_semipure
    ).

%-----------------------------------------------------------------------------%
%
% The following code is deprecated, it is the older implicit parallelisation
% transformation developed by Jerömé.
%
% TODO
%   -   Once a call which is a candidate for implicit parallelism is found,
%       search forward AND backward for the closest goal which is also a
%       candidate for implicit parallelism/parallel conjunction and determine
%       which side is the best (on the basis of the number of shared variables).
%
% XXX Several predicates in this module repeatedly add goals to the ends of
% lists of goals, yielding quadratic behavior. This should be fixed.
%
%-----------------------------------------------------------------------------%

    % Represent a call site static which is a candidate for introducing
    % implicit parallelism.
    %
:- type candidate_call_site
    --->    candidate_call_site(
                caller      :: string,          % The caller of the call.
                slot_number :: int,             % The slot number of the call.
                kind        :: call_site_kind,  % The kind of the call.
                callee      :: string           % The callee of the call.
            ).

    % Represent the kind of a call site.
    %
:- type call_site_kind
        --->    csk_normal
        ;       csk_special
        ;       csk_higher_order
        ;       csk_method
        ;       csk_callback.

    % Construct a call_site_kind from its string representation.
    %
:- pred construct_call_site_kind(string::in, call_site_kind::out) is semidet.

construct_call_site_kind("normal_call",         csk_normal).
construct_call_site_kind("special_call",        csk_special).
construct_call_site_kind("higher_order_call",   csk_higher_order).
construct_call_site_kind("method_call",         csk_method).
construct_call_site_kind("callback",            csk_callback).

%-----------------------------------------------------------------------------%

:- pred apply_old_implicit_parallelism_transformation(
    module_info::in, maybe_error(module_info)::out) is det.

apply_old_implicit_parallelism_transformation(ModuleInfo0, MaybeModuleInfo) :-
    module_info_get_globals(ModuleInfo0, Globals),
    (
        globals.get_maybe_feedback_info(Globals, yes(FeedbackInfo)),
        FeedbackData = feedback_data_calls_above_threshold_sorted(_, _, _),
        get_feedback_data(FeedbackInfo, FeedbackData)
    ->
        some [!ModuleInfo]
        (
            !:ModuleInfo = ModuleInfo0,
            module_info_get_valid_predids(PredIds, !ModuleInfo),
            FeedbackData =
                feedback_data_calls_above_threshold_sorted(_, _, Calls),
            list.map(call_site_convert, Calls, CandidateCallSites),
            process_preds_for_implicit_parallelism(PredIds, CandidateCallSites,
                !ModuleInfo),
            MaybeModuleInfo = ok(!.ModuleInfo)
        )
    ;
        MaybeModuleInfo =
            error("Insufficient feedback information for implicit parallelism")
    ).

    % This predicate isn't really necessary as this entire module should use
    % the call_site structure defined in mdbcomp.program_representation.
    % However it's expected that the rest of this module will be replaced in
    % the near future.
    %
:- pred call_site_convert(call_site::in, candidate_call_site::out) is det.

call_site_convert(Call, CallSite) :-
    Call = call_site(Caller0, Slot, CallTypeAndCallee),
    string_proc_label_to_string(Caller0, Caller),
    (
        CallTypeAndCallee = plain_call(Callee0),
        string_proc_label_to_string(Callee0, Callee),
        CallSiteKind = csk_normal
    ;
        (
            CallTypeAndCallee = callback_call,
            CallSiteKind = csk_callback
        ;
            CallTypeAndCallee = higher_order_call,
            CallSiteKind = csk_higher_order
        ;
            CallTypeAndCallee = method_call,
            CallSiteKind = csk_method
        ;
            CallTypeAndCallee = special_call,
            CallSiteKind = csk_special
        ),
        Callee = ""
    ),
    CallSite = candidate_call_site(Caller, Slot, CallSiteKind, Callee).

:- pred string_proc_label_to_string(string_proc_label::in, string::out) is det.

string_proc_label_to_string(ProcLabel, String) :-
    (
        ProcLabel = str_ordinary_proc_label(_, Module, _, Name, Arity, Mode)
    ;
        ProcLabel = str_special_proc_label(_, _, Module, Name, Arity, Mode)
    ),
    string.format("%s.%s/%d-%d", [s(Module), s(Name), i(Arity), i(Mode)],
        String).

    % Process predicates for implicit parallelism.
    %
:- pred process_preds_for_implicit_parallelism(list(pred_id)::in,
    list(candidate_call_site)::in, module_info::in, module_info::out) is det.

process_preds_for_implicit_parallelism([],
        _CandidateCallSites, !ModuleInfo).
process_preds_for_implicit_parallelism([PredId | PredIds],
        CandidateCallSites, !ModuleInfo) :-
    module_info_pred_info(!.ModuleInfo, PredId, PredInfo),
    ProcIds = pred_info_non_imported_procids(PredInfo),
    process_procs_for_implicit_parallelism(PredId, ProcIds,
        CandidateCallSites, !ModuleInfo),
    process_preds_for_implicit_parallelism(PredIds,
        CandidateCallSites, !ModuleInfo).

    % Process procedures for implicit parallelism.
    %
:- pred process_procs_for_implicit_parallelism(pred_id::in,
    list(proc_id)::in, list(candidate_call_site)::in,
    module_info::in, module_info::out) is det.

process_procs_for_implicit_parallelism(_PredId, [],
        _CandidateCallSites, !ModuleInfo).
process_procs_for_implicit_parallelism(PredId, [ProcId | ProcIds],
        CandidateCallSites, !ModuleInfo) :-
    module_info_pred_proc_info(!.ModuleInfo, PredId, ProcId,
        PredInfo0, ProcInfo0),
    % Initialize the counter for the slot number.
    SiteNumCounter = counter.init(0),
    pred_proc_id_to_raw_id(PredInfo0, ProcId, CallerRawId),
    get_callees_feedback(CallerRawId, CandidateCallSites, [], CallSites),
    list.length(CallSites, NumCallSites),
    ( NumCallSites = 0 ->
        % No candidate calls for implicit parallelism in this procedure.
        process_procs_for_implicit_parallelism(PredId, ProcIds,
            CandidateCallSites, !ModuleInfo)
    ;
        proc_info_get_goal(ProcInfo0, Body0),
        process_goal_for_implicit_parallelism(Body0, Body, ProcInfo0,
            !ModuleInfo, no, _, 0, _, CallSites, _, SiteNumCounter, _),
        proc_info_set_goal(Body, ProcInfo0, ProcInfo1),
        proc_info_set_has_parallel_conj(yes, ProcInfo1, ProcInfo2),
        requantify_proc_general(ordinary_nonlocals_no_lambda,
            ProcInfo2, ProcInfo3),
        recompute_instmap_delta_proc(do_not_recompute_atomic_instmap_deltas,
            ProcInfo3, ProcInfo, !ModuleInfo),
        pred_info_set_proc_info(ProcId, ProcInfo, PredInfo0, PredInfo),
        module_info_set_pred_info(PredId, PredInfo, !ModuleInfo),
        process_procs_for_implicit_parallelism(PredId, ProcIds,
            CandidateCallSites, !ModuleInfo)
    ).

    % Filter the list of call site information from the feedback file so that
    % the resulting list only contains those call sites that belong to the first
    % argument, e.g. the caller.
    %
:- pred get_callees_feedback(string::in, list(candidate_call_site)::in,
    list(candidate_call_site)::in, list(candidate_call_site)::out) is det.

get_callees_feedback(_Caller, [], !ResultAcc).
get_callees_feedback(Caller, [CandidateCallSite | CandidateCallSites],
        !ResultAcc) :-
    CandidateCallSite = candidate_call_site(CSSCaller, _, _, _),
    ( Caller = CSSCaller ->
        !:ResultAcc = [CandidateCallSite | !.ResultAcc]
    ;
        true
    ),
    get_callees_feedback(Caller, CandidateCallSites, !ResultAcc).

    % Process a goal for implicit parallelism.
    % MaybeConj is the conjunction which contains Goal.
    %
:- pred process_goal_for_implicit_parallelism(hlds_goal::in, hlds_goal::out,
    proc_info::in, module_info::in, module_info::out,
    maybe(hlds_goal_expr)::in, maybe(hlds_goal_expr)::out, int ::in, int::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out,
    counter::in, counter::out) is det.

process_goal_for_implicit_parallelism(!Goal, ProcInfo, !ModuleInfo,
        !MaybeConj, !IndexInConj, !CalleesToBeParallelized, !SiteNumCounter) :-
    !.Goal = hlds_goal(GoalExpr0, GoalInfo),
    (
        GoalExpr0 = unify(_, _, _, _, _),
        increment_index_if_in_conj(!.MaybeConj, !IndexInConj)
    ;
        GoalExpr0 = plain_call(_, _, _, _, _, _),
        process_call_for_implicit_parallelism(!.Goal, ProcInfo, !ModuleInfo,
            !IndexInConj, !MaybeConj, !CalleesToBeParallelized,
            !SiteNumCounter)
        % We deal with the index in the conjunction in
        % process_call_for_implicit_parallelism.
    ;
        GoalExpr0 = call_foreign_proc(_, _, _, _, _, _, _),
        process_call_for_implicit_parallelism(!.Goal, ProcInfo, !ModuleInfo,
            !IndexInConj, !MaybeConj, !CalleesToBeParallelized,
            !SiteNumCounter)
    ;
        GoalExpr0 = generic_call(Details, _, _, _),
        (
            Details = higher_order(_, _, _, _),
            process_call_for_implicit_parallelism(!.Goal, ProcInfo,
                !ModuleInfo, !IndexInConj, !MaybeConj,
                !CalleesToBeParallelized, !SiteNumCounter)
        ;
            Details = class_method(_, _, _, _),
            process_call_for_implicit_parallelism(!.Goal, ProcInfo,
                !ModuleInfo, !IndexInConj, !MaybeConj,
                !CalleesToBeParallelized, !SiteNumCounter)
        ;
            Details = event_call(_),
            increment_index_if_in_conj(!.MaybeConj, !IndexInConj)
        ;
            Details = cast(_),
            increment_index_if_in_conj(!.MaybeConj, !IndexInConj)
        )
    ;
        % No distinction is made between plain conjunctions and parallel
        % conjunctions. We have to process the parallel conjunction for the
        % slot number.
        GoalExpr0 = conj(_, _),
        process_conj_for_implicit_parallelism(GoalExpr0, GoalExpr, 1,
            ProcInfo, !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter),
        % A plain conjunction will never be contained in an other plain
        % conjunction. As for parallel conjunctions, they will not be modified.
        % Therefore, incrementing the index suffices (no need to call
        % update_conj_and_index).
        !:Goal = hlds_goal(GoalExpr, GoalInfo),
        increment_index_if_in_conj(!.MaybeConj, !IndexInConj)
    ;
        GoalExpr0 = disj(Goals0),
        process_disj_for_implicit_parallelism(Goals0, [], Goals,
            ProcInfo, !ModuleInfo, !CalleesToBeParallelized,
            !SiteNumCounter),
        GoalExpr = disj(Goals),
        % If we are not in a conjunction, then we need to return the modified
        % value of Goal. If we are in a conjunction, that information is not
        % read (see process_conj_for_implicit_parallelism).
        !:Goal = hlds_goal(GoalExpr, GoalInfo),
        update_conj_and_index(!MaybeConj, !.Goal, !IndexInConj)
    ;
        GoalExpr0 = switch(Var, CanFail, Cases0),
        process_switch_cases_for_implicit_parallelism(Cases0, [], Cases,
            ProcInfo, !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter),
        GoalExpr = switch(Var, CanFail, Cases),
        !:Goal = hlds_goal(GoalExpr, GoalInfo),
        update_conj_and_index(!MaybeConj, !.Goal, !IndexInConj)
    ;
        GoalExpr0 = negation(SubGoal0),
        process_goal_for_implicit_parallelism(SubGoal0, SubGoal, ProcInfo,
            !ModuleInfo, !MaybeConj, !IndexInConj, !CalleesToBeParallelized,
            !SiteNumCounter),
        GoalExpr = negation(SubGoal),
        !:Goal = hlds_goal(GoalExpr, GoalInfo),
        update_conj_and_index(!MaybeConj, !.Goal, !IndexInConj)
    ;
        GoalExpr0 = scope(Reason, Goal0),
        ( Reason = from_ground_term(_, from_ground_term_construct) ->
            % Treat the scope as if it were a single unification, since
            % that is effectively what happens at runtime.
            increment_index_if_in_conj(!.MaybeConj, !IndexInConj)
        ;
            % 0 is the default value when we are not in a conjunction
            % (in this case a scope).
            process_goal_for_implicit_parallelism(Goal0, Goal, ProcInfo,
                !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized,
                !SiteNumCounter),
            GoalExpr = scope(Reason, Goal),
            !:Goal = hlds_goal(GoalExpr, GoalInfo),
            update_conj_and_index(!MaybeConj, !.Goal, !IndexInConj)
        )
    ;
        GoalExpr0 = if_then_else(Vars, Cond0, Then0, Else0),
        process_goal_for_implicit_parallelism(Cond0, Cond, ProcInfo,
            !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized,
            !SiteNumCounter),
        process_goal_for_implicit_parallelism(Then0, Then, ProcInfo,
            !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized,
            !SiteNumCounter),
        process_goal_for_implicit_parallelism(Else0, Else, ProcInfo,
            !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized,
            !SiteNumCounter),
        GoalExpr = if_then_else(Vars, Cond, Then, Else),
        !:Goal = hlds_goal(GoalExpr, GoalInfo),
        update_conj_and_index(!MaybeConj, !.Goal, !IndexInConj)
    ;
        GoalExpr0 = shorthand(_),
        % These should have been expanded out by now.
        unexpected(this_file,
            "process_goal_for_implicit_parallelism: shorthand")
    ).

    % Increment the index if we are in a conjunction.
    %
:- pred increment_index_if_in_conj(maybe(hlds_goal_expr)::in, int::in, int::out)
    is det.

increment_index_if_in_conj(MaybeConj, !IndexInConj) :-
    (
        MaybeConj = yes(_),
        !:IndexInConj = !.IndexInConj + 1
    ;
        MaybeConj = no
    ).

    % Process a call for implicit parallelism.
    %
:- pred process_call_for_implicit_parallelism(hlds_goal::in, proc_info::in,
    module_info::in, module_info::out, int::in, int::out,
    maybe(hlds_goal_expr)::in, maybe(hlds_goal_expr)::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out,
    counter::in, counter::out) is det.

process_call_for_implicit_parallelism(Call, ProcInfo, !ModuleInfo,
        !IndexInConj, !MaybeConj, !CalleesToBeParallelized, !SiteNumCounter) :-
    counter.allocate(SlotNumber, !SiteNumCounter),
    get_call_kind_and_callee(!.ModuleInfo, Call, Kind, CalleeRawId),
    (
        !.MaybeConj = yes(Conj0),
        Conj0 = conj(plain_conj, ConjGoals0)
    ->
        (
            is_in_css_list_to_be_parallelized(Kind, SlotNumber, CalleeRawId,
                !.CalleesToBeParallelized, [], !:CalleesToBeParallelized)
        ->
            (
                build_goals_surrounded_by_calls_to_be_parallelized(ConjGoals0,
                    !.ModuleInfo, [Call], Goals, !.IndexInConj + 1, End,
                    !SiteNumCounter, !CalleesToBeParallelized)
            ->
                parallelize_calls(Goals, !.IndexInConj, End, Conj0, Conj,
                    ProcInfo, !ModuleInfo),
                !:IndexInConj = End,
                !:MaybeConj = yes(Conj)
            ;
                % The next call is not in the feedback file or we've hit a
                % plain conjunction/disjunction/switch/if_then_else.
                !:IndexInConj = !.IndexInConj + 1
            )
        ;
            % Not to be parallelized.
            !:IndexInConj = !.IndexInConj + 1
        )
    ;
        % Call is not in a conjunction or the call is already in a parallel
        % conjunction.
        true
    ).

    % Give the raw id (the same as in the deep profiler) of a callee contained
    % in a call.
    %
:- pred get_call_kind_and_callee(module_info::in, hlds_goal::in,
    call_site_kind::out, string::out) is det.

get_call_kind_and_callee(ModuleInfo, Call, Kind, CalleeRawId) :-
    GoalExpr = Call ^ hlds_goal_expr,
    (
        GoalExpr = plain_call(PredId, ProcId, _, _, _, _),
        module_info_pred_proc_info(ModuleInfo, PredId, ProcId, PredInfo, _),
        pred_proc_id_to_raw_id(PredInfo, ProcId, CalleeRawId),
        Kind = csk_normal
    ;
        GoalExpr = call_foreign_proc(_, PredId, ProcId, _, _, _, _),
        module_info_pred_proc_info(ModuleInfo, PredId, ProcId, PredInfo, _),
        pred_proc_id_to_raw_id(PredInfo, ProcId, CalleeRawId),
        Kind = csk_special
    ;
        GoalExpr = generic_call(Details, _, _, _),
        (
            Details = higher_order(_, _, _, _),
            CalleeRawId = "",
            Kind = csk_higher_order
        ;
            Details = class_method(_, _, _, _),
            CalleeRawId = "",
            Kind = csk_method
        ;
            Details = event_call(_),
            unexpected(this_file, "get_call_kind_and_callee: event_call")
        ;
            Details = cast(_),
            unexpected(this_file, "get_call_kind_and_callee: cast")
        )
    ;
        % XXX Some of our callers can call us with these kinds of goals.
        ( GoalExpr = unify(_, _, _, _, _)
        ; GoalExpr = conj(_, _)
        ; GoalExpr = disj(_)
        ; GoalExpr = switch(_, _, _)
        ; GoalExpr = if_then_else(_, _, _, _)
        ; GoalExpr = negation(_)
        ; GoalExpr = scope(_, _)
        ; GoalExpr = shorthand(_)
        ),
        unexpected(this_file, "get_call_kind_and_callee")
    ).

    % Convert a pred_info and a proc_id to the raw procedure id (the same used
    % in the deep profiler).
    %
:- pred pred_proc_id_to_raw_id(pred_info::in, proc_id::in, string::out) is det.

pred_proc_id_to_raw_id(PredInfo, ProcId, RawId) :-
    ModuleName = pred_info_module(PredInfo),
    Name = pred_info_name(PredInfo),
    OrigArity = pred_info_orig_arity(PredInfo),
    IsPredOrFunc = pred_info_is_pred_or_func(PredInfo),
    ModuleString = sym_name_to_string(ModuleName),
    ProcIdInt = proc_id_to_int(ProcId),
    RawId = string.append_list([ModuleString, ".", Name, "/",
        string.int_to_string(OrigArity),
        ( IsPredOrFunc = pf_function -> "+1" ; ""), "-",
        string.from_int(ProcIdInt)]).

    % Succeeds if the caller, slot number and callee correspond to a
    % candidate_call_site in the list given as a parameter.
    % Fail otherwise.
    %
:- pred is_in_css_list_to_be_parallelized(call_site_kind::in, int::in,
    string::in, list(candidate_call_site)::in,
    list(candidate_call_site)::in, list(candidate_call_site)::out) is semidet.

is_in_css_list_to_be_parallelized(Kind, SlotNumber, CalleeRawId,
        CandidateCallSites, !ResultAcc) :-
    (
        CandidateCallSites = [],
        fail
    ;
        CandidateCallSites = [HeadCandidateCallSite | TailCandidateCallSites],
        HeadCandidateCallSite = candidate_call_site(_, CSSSlotNumber, CSSKind,
            CSSCallee),
        % =< because there is not a one to one correspondance with the source
        % code. New calls might have been added by the previous passes of the
        % compiler.
        (
            CSSSlotNumber =< SlotNumber,
            CSSKind = Kind,
            CSSCallee = CalleeRawId
        ->
            !:ResultAcc = !.ResultAcc ++ TailCandidateCallSites
        ;
            !:ResultAcc = !.ResultAcc ++ [HeadCandidateCallSite],
            is_in_css_list_to_be_parallelized(Kind, SlotNumber, CalleeRawId,
                TailCandidateCallSites, !ResultAcc)
        )
    ).

    % Build a list of goals surrounded by two calls which are in the feedback
    % file or by a call which is in the feedback file and a parallel
    % conjunction.
    %
    % Succeed if we can build that list of goals.
    % Fail otherwise.
    %
:- pred build_goals_surrounded_by_calls_to_be_parallelized(list(hlds_goal)::in,
    module_info::in, list(hlds_goal)::in, list(hlds_goal)::out,
    int::in, int::out, counter::in, counter::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out)
    is semidet.

build_goals_surrounded_by_calls_to_be_parallelized(ConjGoals, ModuleInfo,
        !ResultAcc, !Index, !SiteNumCounter, !CalleesToBeParallelized) :-
    list.length(ConjGoals, Length),
    ( !.Index > Length ->
        fail
    ;
        list.index1_det(ConjGoals, !.Index, Goal),
        GoalExpr = Goal ^ hlds_goal_expr,
        (
            ( GoalExpr = disj(_)
            ; GoalExpr = switch(_, _, _)
            ; GoalExpr = if_then_else(_, _, _, _)
            ; GoalExpr = conj(plain_conj, _)
            )
        ->
            fail
        ;
            !:ResultAcc = !.ResultAcc ++ [Goal],
            ( goal_is_conjunction(Goal, parallel_conj) ->
                true
            ;
                ( goal_is_call_or_negated_call(Goal) ->
                    counter.allocate(SlotNumber, !SiteNumCounter),
                    get_call_kind_and_callee(ModuleInfo, Goal, Kind,
                        CalleeRawId),
                    (
                        is_in_css_list_to_be_parallelized(Kind, SlotNumber,
                            CalleeRawId, !.CalleesToBeParallelized,
                            [], !:CalleesToBeParallelized)
                    ->
                        true
                    ;
                        !:Index = !.Index + 1,
                        build_goals_surrounded_by_calls_to_be_parallelized(
                            ConjGoals, ModuleInfo, !ResultAcc, !Index,
                            !SiteNumCounter, !CalleesToBeParallelized)
                    )
                ;
                    !:Index = !.Index + 1,
                    build_goals_surrounded_by_calls_to_be_parallelized(
                        ConjGoals, ModuleInfo, !ResultAcc, !Index,
                        !SiteNumCounter, !CalleesToBeParallelized)
                )
            )
        )
    ).

    % Succeeds if Goal is a conjunction and return the type of the
    % conjunction.  Fail otherwise.
    %
:- pred goal_is_conjunction(hlds_goal::in, conj_type::out) is semidet.

goal_is_conjunction(Goal, Type) :-
    GoalExpr = Goal ^ hlds_goal_expr,
    GoalExpr = conj(Type, _).

    % Succeed if Goal is a call or a negated call.
    % Call here includes higher-order and class method calls.
    % Fail otherwise.
    %
    % XXX Should be a function returning a bool or something similar.
    %
:- pred goal_is_call_or_negated_call(hlds_goal::in) is semidet.

goal_is_call_or_negated_call(Goal) :-
    GoalExpr = Goal ^ hlds_goal_expr,
    (
        GoalExpr = plain_call(_, _, _, _, _, _)
    ;
        GoalExpr = call_foreign_proc(_, _, _, _, _, _, _)
    ;
        GoalExpr = generic_call(Details, _, _, _),
        (
            Details = class_method(_, _, _, _)
        ;
            Details = higher_order(_, _, _, _)
        )
    ;
        GoalExpr = negation(GoalNeg),
        GoalNegExpr = GoalNeg ^ hlds_goal_expr,
        (
            GoalNegExpr = plain_call(_, _, _, _, _, _)
        ;
            GoalNegExpr = call_foreign_proc(_, _, _, _, _, _, _)
        ;
            GoalNegExpr = generic_call(Details, _, _, _),
            (
                Details = class_method(_, _, _, _)
            ;
                Details = higher_order(_, _, _, _)
            )
        )
    ).

    % Parallelize two calls/a call and a parallel conjunction which might have
    % goals between them. If these have no dependencies with the first call
    % then we move them before the first call and parallelize the two
    % calls/call and parallel conjunction.
    %
    % Goals is contained in Conj.
    %
:- pred parallelize_calls(list(hlds_goal)::in, int::in, int::in,
    hlds_goal_expr::in, hlds_goal_expr::out, proc_info::in,
    module_info::in, module_info::out) is det.

parallelize_calls(Goals, Start, End, !Conj, ProcInfo, !ModuleInfo) :-
    ( !.Conj = conj(plain_conj, ConjGoals0) ->
        ( ConjGoals0 = [FirstGoal, LastGoal] ->
            (
                is_worth_parallelizing(FirstGoal, LastGoal, ProcInfo,
                    !.ModuleInfo)
            ->
                ( goal_is_conjunction(LastGoal, parallel_conj) ->
                    % The parallel conjunction has to remain flatened.
                    add_call_to_parallel_conjunction(FirstGoal, LastGoal,
                        ParallelGoal),
                    !:Conj = ParallelGoal ^ hlds_goal_expr
                ;
                    !:Conj = conj(parallel_conj, ConjGoals0)
                )
            ;
                % Not worth parallelizing.
                true
            )
        ;
            % There are more than two goals in the conjunction.
            list.length(Goals, Length),
            list.index1_det(Goals, 1, FirstGoal),
            list.index1_det(Goals, Length, LastGoal),
            (
                is_worth_parallelizing(FirstGoal, LastGoal, ProcInfo,
                    !.ModuleInfo)
            ->
                GoalsInBetweenAndLast = list.det_tail(Goals),
                list.delete_all(GoalsInBetweenAndLast, LastGoal,
                    GoalsInBetween),
                % Check the dependencies of GoalsInBetween with FirstGoal.
                list.filter(goal_depends_on_goal(FirstGoal),
                    GoalsInBetween, GoalsFiltered),
                ( list.is_empty(GoalsFiltered) ->
                    ( goal_is_conjunction(LastGoal, parallel_conj) ->
                        add_call_to_parallel_conjunction(FirstGoal, LastGoal,
                            ParallelGoal)
                    ;
                        create_conj(FirstGoal, LastGoal, parallel_conj,
                            ParallelGoal)
                    ),
                    ( Start = 1 ->
                        GoalsFront = []
                    ;
                        list.det_split_list(Start - 1, ConjGoals0,
                            GoalsFront, _)
                    ),
                    list.length(ConjGoals0, ConjLength),
                    ( End = ConjLength ->
                        GoalsBack = []
                    ;
                        list.det_split_list(End, ConjGoals0, _, GoalsBack)
                    ),
                    ConjGoals = GoalsFront ++ GoalsInBetween ++
                        [ParallelGoal] ++ GoalsBack,
                    !:Conj = conj(plain_conj, ConjGoals)
                ;
                    % The goals between the two calls/call and parallel
                    % conjunction can't be moved before the first call.
                    true
                )
            ;
                % Not worth parallelizing.
                true
            )
        )
    ;
        % Conj is not a conjunction.
        unexpected(this_file, "parallelize_calls")
    ).

    % Two calls are worth parallelizing if the number of shared variables is
    % smaller than the number of argument variables of at least one of the two
    % calls.
    %
    % A call and a parallel conjunction are worth parallelizing if the number of
    % shared variables is smaller than the number of argument variables of the
    % call.
    %
    % Succeed if it is worth parallelizing the two goals.
    % Fail otherwise.
    %
:- pred is_worth_parallelizing(hlds_goal::in, hlds_goal::in, proc_info::in,
    module_info::in) is semidet.

is_worth_parallelizing(GoalA, GoalB, ProcInfo, ModuleInfo) :-
    proc_info_get_initial_instmap(ProcInfo, ModuleInfo, InstMap),
    SharedVars = find_shared_variables(ModuleInfo, InstMap, [GoalA, GoalB]),
    set.to_sorted_list(SharedVars, SharedVarsList),
    list.length(SharedVarsList, NbSharedVars),
    ( NbSharedVars = 0 ->
        % No shared vars between the goals.
        true
    ;
        ( goal_is_conjunction(GoalB, parallel_conj) ->
            get_number_args(GoalA, NbArgsA),
            NbSharedVars < NbArgsA
        ;
            (
                get_number_args(GoalA, NbArgsA),
                get_number_args(GoalB, NbArgsB)
            ->
                ( NbSharedVars < NbArgsA, NbSharedVars < NbArgsB
                ; NbSharedVars = NbArgsA, NbSharedVars < NbArgsB
                ; NbSharedVars < NbArgsA, NbSharedVars = NbArgsB
                )
            ;
                unexpected(this_file, "is_worth_parallelizing")
            )
        )
    ).

    % Give the number of argument variables of a call.
    %
:- pred get_number_args(hlds_goal::in, int::out) is semidet.

get_number_args(Call, NbArgs) :-
    CallExpr = Call ^ hlds_goal_expr,
    (
        CallExpr = plain_call(_, _, Args, _, _, _),
        list.length(Args, NbArgs)
    ;
        CallExpr = generic_call(Details, Args, _, _),
        (
            Details = higher_order(_, _, _, _),
            list.length(Args, NbArgs)
        ;
            Details = class_method(_, _, _, _),
            list.length(Args, NbArgs)
        )
    ;
        CallExpr = call_foreign_proc(_, _, _, Args, _, _, _),
        list.length(Args, NbArgs)
    ).

    % Add a call to an existing parallel conjunction.
    %
:- pred add_call_to_parallel_conjunction(hlds_goal::in, hlds_goal::in,
    hlds_goal::out) is det.

add_call_to_parallel_conjunction(Call, ParallelGoal0, ParallelGoal) :-
    ParallelGoalExpr0 = ParallelGoal0 ^ hlds_goal_expr,
    ParallelGoalInfo0 = ParallelGoal0 ^ hlds_goal_info,
    ( ParallelGoalExpr0 = conj(parallel_conj, GoalList0) ->
        GoalList = [Call | GoalList0],
        goal_list_nonlocals(GoalList, NonLocals),
        goal_list_instmap_delta(GoalList, InstMapDelta),
        goal_list_determinism(GoalList, Detism),
        goal_list_purity(GoalList, Purity),
        goal_info_set_nonlocals(NonLocals, ParallelGoalInfo0,
            ParallelGoalInfo1),
        goal_info_set_instmap_delta(InstMapDelta, ParallelGoalInfo1,
            ParallelGoalInfo2),
        goal_info_set_determinism(Detism,
            ParallelGoalInfo2, ParallelGoalInfo3),
        goal_info_set_purity(Purity, ParallelGoalInfo3, ParallelGoalInfo),
        ParallelGoalExpr = conj(parallel_conj, GoalList),
        ParallelGoal = hlds_goal(ParallelGoalExpr, ParallelGoalInfo)
    ;
        unexpected(this_file, "add_call_to_parallel_conjunction")
    ).

    % Succeed if the first goal depends on the second one.
    % Fail otherwise.
    %
:- pred goal_depends_on_goal(hlds_goal::in, hlds_goal::in) is semidet.

goal_depends_on_goal(Goal1, Goal2) :-
    Goal1 = hlds_goal(_, GoalInfo1),
    Goal2 = hlds_goal(_, GoalInfo2),
    InstmapDelta1 = goal_info_get_instmap_delta(GoalInfo1),
    instmap_delta_changed_vars(InstmapDelta1, ChangedVars1),
    NonLocals2 = goal_info_get_nonlocals(GoalInfo2),
    set.intersect(ChangedVars1, NonLocals2, Intersection),
    \+ set.empty(Intersection).

    % Process a conjunction for implicit parallelism.
    %
:- pred process_conj_for_implicit_parallelism(
    hlds_goal_expr::in, hlds_goal_expr::out, int::in,
    proc_info::in, module_info::in, module_info::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out,
    counter::in, counter::out) is det.

process_conj_for_implicit_parallelism(!GoalExpr, IndexInConj, ProcInfo,
    !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter) :-
    ( !.GoalExpr = conj(_, GoalsConj) ->
        list.length(GoalsConj, Length),
        ( IndexInConj > Length ->
            true
        ;
            MaybeConj0 = yes(!.GoalExpr),
            list.index1_det(GoalsConj, IndexInConj, GoalInConj),
            % We are not interested in the return value of GoalInConj, only
            % MaybeConj matters.
            process_goal_for_implicit_parallelism(GoalInConj, _, ProcInfo,
                !ModuleInfo, MaybeConj0, MaybeConj, IndexInConj, IndexInConj0,
                !CalleesToBeParallelized, !SiteNumCounter),
            ( MaybeConj = yes(GoalExprProcessed) ->
                !:GoalExpr = GoalExprProcessed
            ;
                unexpected(this_file, "process_conj_for_implicit_parallelism")
            ),
            process_conj_for_implicit_parallelism(!GoalExpr, IndexInConj0,
                ProcInfo, !ModuleInfo, !CalleesToBeParallelized,
                !SiteNumCounter)
        )
    ;
        unexpected(this_file, "process_conj_for_implicit_parallelism")
    ).

    % Process a disjunction for implicit parallelism.
    %
:- pred process_disj_for_implicit_parallelism(
    list(hlds_goal)::in, list(hlds_goal)::in, list(hlds_goal)::out,
    proc_info::in, module_info::in, module_info::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out,
    counter::in, counter::out) is det.

process_disj_for_implicit_parallelism([], !GoalsAcc, _ProcInfo,
        !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter).
process_disj_for_implicit_parallelism([Goal0 | Goals], !GoalsAcc,
        ProcInfo, !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter) :-
    process_goal_for_implicit_parallelism(Goal0, Goal, ProcInfo,
        !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized, !SiteNumCounter),
    !:GoalsAcc = !.GoalsAcc ++ [Goal],
    process_disj_for_implicit_parallelism(Goals, !GoalsAcc, ProcInfo,
        !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter).

    % If we are in a conjunction, update it by replacing the goal at index by
    % Goal and increment the index.
    %
:- pred update_conj_and_index(
    maybe(hlds_goal_expr)::in, maybe(hlds_goal_expr)::out,
    hlds_goal::in, int::in, int::out) is det.

update_conj_and_index(!MaybeConj, Goal, !IndexInConj) :-
    ( !.MaybeConj = yes(conj(Type, Goals0)) ->
        list.replace_nth_det(Goals0, !.IndexInConj, Goal, Goals),
        !:IndexInConj = !.IndexInConj + 1,
        !:MaybeConj = yes(conj(Type, Goals))
    ;
        true
    ).

    % Process a switch for implicit parallelism.
    %
:- pred process_switch_cases_for_implicit_parallelism(
    list(case)::in, list(case)::in, list(case)::out, proc_info::in,
    module_info::in, module_info::out,
    list(candidate_call_site)::in, list(candidate_call_site)::out,
    counter::in, counter::out) is det.

process_switch_cases_for_implicit_parallelism([], !CasesAcc, _ProcInfo,
        !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter).
process_switch_cases_for_implicit_parallelism([Case0 | Cases], !CasesAcc,
        ProcInfo, !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter) :-
    Case0 = case(MainConsId, OtherConsIds, Goal0),
    process_goal_for_implicit_parallelism(Goal0, Goal, ProcInfo,
        !ModuleInfo, no, _, 0, _, !CalleesToBeParallelized, !SiteNumCounter),
    Case = case(MainConsId, OtherConsIds, Goal),
    !:CasesAcc = !.CasesAcc ++ [Case],
    process_switch_cases_for_implicit_parallelism(Cases, !CasesAcc,
        ProcInfo, !ModuleInfo, !CalleesToBeParallelized, !SiteNumCounter).

%-----------------------------------------------------------------------------%

:- func this_file = string.

this_file = "implicit_parallelism.m".

%-----------------------------------------------------------------------------%
:- end_module transform_hlds.implicit_parallelism.
%-----------------------------------------------------------------------------%
