/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  cardbase.c -- The database of Magic cards.
 *
 *  Each card in memory is uniquely identified by a "global id".
 *  The global id should not be sent over the network or stored on disk.
 *  It only has meaning in this client during this run.
 *  It's not really global.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib.h>
#include "cardbase.h"
#include "fgetline.h"
#include "cardart.h"

static int read_int32 (FILE *fp);
static int read_int8 (FILE *fp);
static char *read_pascal_string (FILE *fp);
static char *read_c_string (FILE *fp);
static int starts_with (const char *full, const char *prefix);
static int ends_with (const char *full, const char *suffix);
static int cost_to_color_bits (char *text);
static int card_compare (const void *ap, const void *bp);
static int lookup_card (struct cardbase *db, const char *name);

/**
 *  Parse an Apprentice / E-League style cardinfo.dat.
 */
struct cardbase *cardbase_new_from_dat (const char *filename)
{
    FILE *fp;
    int location, num_cards, sorted, i, dummy;
    struct cardinfo *cards;
    struct cardbase *db;

    fp = fopen (filename, "r");
    if (fp == NULL) {
	perror (filename);
	return (NULL);
    }
    location = read_int32 (fp);
    if (fseek (fp, location, SEEK_SET) < 0) {
	perror ("fseek");
	fclose (fp);
	return (NULL);
    }
    num_cards = read_int32 (fp);
    cards = calloc (num_cards, sizeof (struct cardinfo));
    if (cards == NULL) {
	fprintf (stderr, "%s: corrupt\n", filename);
	fclose (fp);
	return (NULL);
    }
    sorted = TRUE;
    for (i = 0; i < num_cards; i++) {
	if (feof (fp)) {
	    num_cards = i;
	    break;
	}
	cards[i].name = read_pascal_string (fp);
	cards[i].location = read_int32 (fp);
	cards[i].color = read_int8 (fp);
	cards[i].expansion = read_pascal_string (fp);
	dummy = read_int8 (fp);
	dummy = read_int8 (fp);
	if ((i > 0) && (strcasecmp (cards[i-1].name, cards[i].name) > 0))
	    sorted = FALSE;
    }
    for (i = 0; i < num_cards; i++) {
	if (fseek (fp, cards[i].location, SEEK_SET) < 0)
	    continue;
	cards[i].cardtype = read_pascal_string (fp);
	cards[i].cost = read_pascal_string (fp);
	cards[i].pow_tgh = read_pascal_string (fp);
	cards[i].text = read_pascal_string (fp);
	cards[i].flavor = read_pascal_string (fp);
    }
    fclose (fp);

    if (!sorted)
	qsort (cards, num_cards, sizeof (struct cardinfo), card_compare);

    db = calloc (1, sizeof (struct cardbase));
    db->num_cards = num_cards;
    db->cards = cards;

    return (db);
}

static int read_int32 (FILE *fp)
{
    int i;
    fread (&i, sizeof (i), 1, fp);
    return GINT32_FROM_LE (i);
}

static int read_int8 (FILE *fp)
{
    unsigned char i;
    fread (&i, sizeof (i), 1, fp);
    return i;
}

static char *read_pascal_string (FILE *fp)
{
    unsigned short len, i;
    char *cp;

    fread (&len, sizeof (len), 1, fp);
    len = GINT16_FROM_LE (len);
    cp = malloc (len+1);
    fread (cp, 1, len, fp);
    while ((len > 0) && isspace (cp[len-1]))
	len--;
    cp[len] = 0;

    /* gtk+ doesn't like non-ascii characters. */
    for (i = 0; i < len; i++) {
	switch ((unsigned char) cp[i]) {
	case 0222:
	    cp[i] = '\'';
	    break;
	case 0224:
	    cp[i] = '"';
	    break;
	}
    }
    return cp;
}

/**
 *  Parse the plain text file containing the full Oracle name, cost,
 *  and text of every Magic card ever printed.
 *  
 *  The file can be downloaded from www.MagicTheGathering.com in the
 *  Rules / Tournament section.  The filename is ORTmm-dd-yy.txt.
 */
struct cardbase *cardbase_new_from_ort (const char *filename)
{
    FILE *fp;
    int max_cards, sorted, i, num_cards;
    struct cardinfo *cards;
    struct cardbase *db;
    char buf[128], *cp, *text, *full;

    fp = fopen (filename, "r");
    if (fp == NULL) {
	perror (filename);
	return (NULL);
    }
    /*
     *  Quickly calculate an upper bound on the number of cards defined
     *  in the file.  Use the number of blank lines in the file.
     */
    max_cards = 0;
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	cp = buf;
	while (isspace (*cp))
	    cp++;
	if (*cp == 0)
	    max_cards++;
    }
    rewind (fp);
    cards = calloc (max_cards, sizeof (struct cardinfo));
    if (cards == NULL) {
	perror (filename);
	fclose (fp);
	return (NULL);
    }
    i = 0;
    sorted = TRUE;
    while (!feof (fp)) {
	text = read_c_string (fp);
	if (text == NULL)
	    continue;
	if ((i > 0) && (strcasecmp (cards[i-1].name, text) > 0))
	    sorted = FALSE;
	cards[i].name = text;
	text = read_c_string (fp);
	if (text == NULL) {
	    fprintf (stderr, "%s looks corrupt.\n", filename);
	    break;
	}
	if (starts_with (text, "Basic Land") ||
	    starts_with (text, "Land -- ") || ends_with (text, "Land")) {
	    cards[i].cost = NULL;
	    cards[i].color = COL_LAND_BIT;
	    cards[i].cardtype = text;
	} else {
	    cards[i].cost = text;
	    cards[i].color = cost_to_color_bits (text);
	    text = read_c_string (fp);
	    cards[i].cardtype = text;
	}
	cards[i].pow_tgh = NULL;
	if ((text != NULL) && (starts_with (text, "Creature") ||
			       starts_with (text, "Artifact Creature"))) {
	    cards[i].pow_tgh = read_c_string (fp);
	}
	full = NULL;
	while ((text = read_c_string (fp)) != NULL) {
	    if (full == NULL) {
		full = text;
	    } else {
		full = realloc (full, strlen (full) + strlen (text) + 2);
		strcat (full, "\n");
		strcat (full, text);
	    }
	}
	cards[i].text = full;
	cards[i].flavor = NULL;
	cards[i].expansion = NULL;
	cards[i].location = 0;
	i++;
    }
    fclose (fp);
    num_cards = i;

    if (!sorted)
	qsort (cards, num_cards, sizeof (struct cardinfo), card_compare);

    db = calloc (1, sizeof (struct cardbase));
    db->num_cards = num_cards;
    db->cards = cards;

    return (db);
}

/**
 *  Read a line of text into newly allocated memory.
 *  Strip the trailing newline and any other trailing space.
 */
static char *read_c_string (FILE *fp)
{
    char *full, buf[1024];
    int len, done;

    full = NULL;
    while (fgets (buf, sizeof (buf), fp) != NULL) {
	len = strlen (buf);
	if (len == 0) {
	    done = TRUE;
	} else {
	    done = ((buf[len-1] == '\n') || (buf[len-1] == '\r'));
	}
	if (full == NULL) {
	    full = strdup (buf);
	} else {
	    full = realloc (full, strlen (full) + strlen (buf) + 2);
	    strcat (full, buf);
	}
	if (done)
	    break;
    }
    if (full == NULL)
	return (NULL);
    len = strlen (full);
    while ((len > 0) && (isspace (full[len-1])))
	len--;
    if (len == 0) {
	free (full);
	return (NULL);
    }
    full[len] = 0;
    return (full);
}

static int starts_with (const char *full, const char *prefix)
{
    return (strncasecmp (full, prefix, strlen (prefix)) == 0);
}

static int ends_with (const char *full, const char *suffix)
{
    int n1, n2;

    n1 = strlen (full);
    n2 = strlen (suffix);
    return ((n1 >= n2) && (strcasecmp (full+n1-n2, suffix) == 0));
}

/**
 *  Turn {a}{b}{c} into abc, and return an integer with the bits for
 *  a, b, and c turned on.
 */
static int cost_to_color_bits (char *text)
{
    int bits, bit, src, dst;

    bits = dst = 0;
    for (src = 0; text[src] != 0; src++) {
	if ((text[src] == '{') || (text[src] == '}'))
	    continue;
	bit = 0;
	switch (text[src]) {
	case 'W':
	    bit = COL_WHITE_BIT;
	    break;
	case 'U':
	    bit = COL_BLUE_BIT;
	    break;
	case 'B':
	    bit = COL_BLACK_BIT;
	    break;
	case 'R':
	    bit = COL_RED_BIT;
	    break;
	case 'G':
	    bit = COL_GREEN_BIT;
	    break;
	}
	if (bits == 0) {
	    bits = bit;
	} else if (bit > 0) {
	    if (bits != bit)
		bits |= COL_GOLD_BIT;
	    bits |= bit;
	}
	text[dst++] = text[src];
    }
    text[dst] = 0;
    if ((dst > 0) && (bits == 0))
	bits = COL_ARTIFACT_BIT;
    return (bits);
}

/**
 *  This is passed to qsort() if the database was not sorted in the file.
 */
static int card_compare (const void *ap, const void *bp)
{
    const struct cardinfo *c1, *c2;

    c1 = (const struct cardinfo *) ap;
    c2 = (const struct cardinfo *) bp;
    return (strcasecmp (c1->name, c2->name));
}

/**
 *  Binary search the database.  Ignore the tokens.
 */
static int lookup_card (struct cardbase *db, const char *name)
{
    int lo, hi, mid, val;

    lo = 0;
    hi = db->num_cards-1;
    while (lo <= hi) {
	mid = (lo + hi) / 2;
	val = strcasecmp (db->cards[mid].name, name);
	if (val == 0)
	    return (mid);
	if (val < 0)
	    lo = mid+1;
	else
	    hi = mid-1;
    }
    return (-1);
}

/**
 *  Look up global ids for the cards in a deck.
 *  If the card name is not found in the database, then add it to the
 *  database (with no attributes), which beats having to remove it from
 *  the deck.
 */
int cardbase_find_globalid (struct cardbase *db, const char *name)
{
    int globalid;

    globalid = lookup_card (db, name);
    if (globalid >= 0)
	return (globalid);
    return (cardbase_add_card (db, name, "", 0));
}

/**
 *  Quick O(1) lookup a card by globalid.
 */
struct cardinfo *cardbase_get_card (struct cardbase *db, int globalid)
{
    return (globalid < db->num_cards ? &db->cards[globalid] :
	    db->extras[globalid - db->num_cards]);
}

/**
 *  Add a token creature.
 */
int cardbase_add_card (struct cardbase *db, const char *name, char *pow_tgh,
		       int color)
{
    struct cardinfo *card;
    int globalid;

    db->extras = realloc (db->extras,
			  (db->num_extras+1) * sizeof (struct cardinfo *));
    card = calloc (1, sizeof (struct cardinfo));
    card->name = strdup (name);
    if (pow_tgh != NULL)
	card->pow_tgh = strdup (pow_tgh);
    card->color = color;
    /* Never try to download this card's picture. */
    card->art = cardart_get_null_art ();
    db->extras[db->num_extras] = card;
    globalid = db->num_cards + db->num_extras;
    db->num_extras++;
    return (globalid);
}
