/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  proto-appr.c -- Functions to send and respond to Apprentice protocol
 *		    requests.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "game.h"
#include "playarea.h"
#include "opponent.h"
#include "deck.h"
#include "cardbase.h"
#include "zone.h"
#include "dialogs.h"
#include "encrypt.h"

#ifdef APPRENTICE

#define APPRENTICE_VERSION "1.45"

static void send_initial_player_name (struct game *game);
static void send_cardlist (struct game *game, int pid, int is_sb);
static int string_contains (char *haystack, char *needle);

const int default_port = 4747;

void game_become_client (struct game *game)
{
    /*
     *  There are now two players.
     *  The local player is player [0], the server is player [1].
     */
    game->num_players = 2;
    game->player = realloc (game->player, sizeof (struct player *) * 2);
    game->player[1] = player_new (game, NULL);
    player_make_labels (game->player[1], game->player_vbox);
    opponent_listen (game);
    send_version_number (game);
    send_initial_player_name (game);
    start_new_game (game);
}

void game_become_server (struct game *game)
{
    game->num_players = 2;
    game->player = realloc (game->player, sizeof (struct player *) * 2);
    game->player[1] = game->player[0];
    game->player[0] = player_new (game, NULL);
    player_make_labels (game->player[0], game->player_vbox);
    game->local_player = 1;
    game->opponent->flags |= is_server_flag;
    opponent_listen (game);
    // send_version_number (game);
    // send_initial_player_name (game);
    // start_new_game (game);
}

int game_should_restart (struct game *game)
{
    return ((game->opponent->flags & opponent_wants_new) != 0);
}

void send_version_number (struct game *game)
{
    send_opponents (game, "version-number ver:\"%s\" value:1",
		    APPRENTICE_VERSION);
}

static void send_initial_player_name (struct game *game)
{
    send_opponents (game, "set-player-name pid:%d name:\"%s\"",
		    game->local_player+1,
		    game->player[game->local_player]->name);
}

static void send_cardlist (struct game *game, int pid, int is_sb)
{
    struct player *me;
    int *cardlist;
    int size, cid, qty, globalid, offset;
    struct cardinfo *tcard;
    char *command, *name;

    me = game->player[pid];
    if (is_sb) {
	command = "setup-sb-card";
	cardlist = me->deck->sb_cardlist;
	size = me->deck->sb_size;
	offset = player_sideboard_offset (game, pid);
    } else {
	command = "setup-card";
	cardlist = me->deck->cardlist;
	size = me->deck->size;
	offset = player_deck_offset (pid);
    }
    cid = 0;
    while (cid < size) {
	qty = 1;
	globalid = cardlist[cid];
	while ((cid+qty < size) && (globalid == cardlist[cid+qty]))
	    qty++;
	tcard = cardbase_get_card (game->cardbase, globalid);
	if (tcard != NULL) {
	    name = encrypt_name (me->name, tcard->name);
	    send_opponents (game, "%s pid:%d startcid:%d name:\"%s\" qty:%d",
			    command, pid+1, offset+cid, name, qty);
	    gtk_main_iteration_do (FALSE);
	}
	cid += qty;
    }
}

void send_player_setup (struct game *game)
{
    int pid;
    struct player *me;

    pid = game->local_player;
    me = game->player[pid];
    send_opponents (game, "setup-begin pid:%d", pid+1);
    send_cardlist (game, pid, FALSE);
    send_opponents (game, "sideboard qty:%d pid:%d",
		    me->deck->sb_size, pid+1);
    if (!game->no_view_sideboard)
	send_cardlist (game, pid, TRUE);

    send_opponents (game, "do-msg msg:\"%s has a %d card deck.\"",
		    me->name, me->deck->size);
}

void send_setup_end (struct game *game)
{
    send_opponents (game, "setup-end pid:%d", game->local_player+1);
}

void send_player_name (struct game *game)
{
    send_opponents (game, "change-player-name pid:%d name:\"%s\"",
		    game->local_player+1,
		    game->player[game->local_player]->name);
}

void send_draw_cards (struct game *game, int qty)
{
    int pid, i;
    char *name;

    pid = game->local_player;
    name = game->player[pid]->name;
    if (qty == 0) {
	send_opponents (game, "do-msg msg:\"%s draws no cards.\"", name);
    } else if (qty == 1) {
	send_opponents (game, "do-drawcard pid:%d msg:\"%s draws a card.\"",
			pid+1, name);
    } else {
	for (i = 0; i < qty-1; i++)
	    send_opponents (game, "do-drawcard pid:%d msg:\"nul\"", pid+1);
	send_opponents (game, "do-drawcard pid:%d msg:\"%s drew %d cards.\"",
			pid+1, name, qty);
    }
}

void send_starting_message (struct game *game, char *message)
{
    struct player *me = game->player[game->local_player];
    send_opponents (game, "do-msg msg:\"-- %s's Security Code: %d\"",
		    me->name, me->deck->security_code);
    send_opponents (game, "do-msg msg:\"-- %s says: ''%s''\"",
		    me->name, message);
}

void send_message (struct game *game, char *message)
{
    int pid = game->local_player;
    send_opponents (game, "do-msg msg:\"%s says:'%s'\" pid:%d",
		    game->player[pid]->name, message, pid+1);
}

void send_shuffle (struct game *game)
{
    struct player *me;
    struct zone *zone;
    char buf[380], *cp;
    int offset, i, j, max;

    me = game->player[game->local_player];
    send_opponents (game, "do-msg msg:\"%s is shuffling library...\"",
		    me->name);
    zone = me->zone[LIBRARY_ZONE];
    offset = player_deck_offset (game->local_player) - 1;
    for (i = 0; i < zone->size; ) {
	sprintf (buf, "do-shuffle lid:%d startpos:%d order:\"",
		 3 + game->local_player*4, i+1);
	cp = buf + strlen (buf);
	max = zone->size-i;
	if (max > 60)
	    max = 60;
	for (j = 0; j < max; j++) {
	    sprintf (cp, "%d", zone->cardlist[i+j]-offset);
	    cp += strlen (cp);
	    if (j+1 < max)
		*cp++ = ' ';
	}
	strcpy (cp, "\"");
	send_opponents (game, "%s", buf);
	i += j;
    }
}

void send_flip_coin (struct game *game, int res)
{
    send_opponents (game, "flip-coin res:%d pid:%d", res,
		    game->local_player+1);
}

void send_roll_die (struct game *game, int res, int size)
{
    send_opponents (game, "do-msg msg:\"%s rolled a %d, using a %d "
		    "sided die.\"", game->player[game->local_player]->name,
		    res, size);
}

void send_game_phase (struct game *game)
{
    send_opponents (game, "set-game-phase phase:%d", game->phase_number);
}

void send_game_turn (struct game *game)
{
    send_opponents (game, "set-game-turn turn:%d", game->turn_number);
}

void send_player_life (struct game *game)
{
    send_opponents (game, "set-player-life pid:%d life:%d",
		    game->local_player+1,
		    game->player[game->local_player]->life);
}

void send_card_counters (struct game *game, struct table_card *tcard)
{
    send_opponents (game, "set-card-counters cid:%d counters:%d",
		    tcard->cid, tcard->counters);
}

void send_card_location (struct game *game, struct table_card *tcard)
{
    int face;

    face = ((tcard->flags & CARD_FLIPPED_FLAG) != 0);
    send_opponents (game, "set-card-location cid:%d lid:0 x:%d y:%d "
		    "width:48 height:48 face:%d", tcard->cid,
		    tcard->xpos, tcard->ypos, face);
}

void send_card_zone (struct game *game, struct card_move *mp)
{
    struct player *me;
    int no_peeking, lid;
    char text[80], *cname, *place, *extra;

    me = game->player[mp->pid];
    no_peeking = ((me->zone[mp->srcznum]->flags & ZONE_PRIVATE) &&
		  (me->zone[mp->dstznum]->flags & ZONE_PRIVATE));
    lid = tux_to_lid (mp->pid, mp->dstznum);
    if (mp->bottom || no_peeking) {
	if (no_peeking) {
	    if ((me->zone[mp->srcznum]->flags & ZONE_SHOW_ORDER) &&
		(mp->idx >= 0)) {
		sprintf (text, "card #%d", mp->idx+1);
		cname = text;
	    } else {
		cname = "a card";
	    }
	} else {
	    cname = game_get_card_name (game, mp->cid);
	}
	if (mp->bottom) {
	    place = "BOTTOM";
	    extra = "x:0 ";
	} else {
	    place = "TOP";
	    extra = "";
	}
	send_opponents (game, "set-card-location cid:%d lid:%d %s"
			"msg:\"%s moves %s to the %s of %s's %s.\"",
			mp->cid, lid, extra,
			game->player[game->local_player]->name,
			cname, place, me->name, name_of_zone (mp->dstznum));
    } else {
	send_opponents (game, "set-card-location cid:%d lid:%d",
			mp->cid, lid);
    }
}

void send_unplay_card (struct game *game, int cid, int pid, int znum,
		       int bottom, int facedown)
{
    int lid;
    struct player *me;
    char *cname, *place, *extra;

    lid = tux_to_lid (pid, znum);
    if (bottom || facedown) {
	me = game->player[pid];
	if (facedown) {
	    cname = "a facedown card";
	} else {
	    cname = game_get_card_name (game, cid);
	}
	if (bottom) {
	    place = "the BOTTOM of ";
	    extra = "x:0 ";
	} else {
	    place = "";
	    extra = "";
	}
	send_opponents (game, "set-card-location cid:%d lid:%d %s"
			"msg:\"%s moves %s to %s%s's %s.\"",
			cid, lid, extra,
			game->player[game->local_player]->name,
			cname, place, me->name, name_of_zone (znum));
    } else {
	send_opponents (game, "set-card-location cid:%d lid:%d", cid, lid);
    }
}

void send_card_order_in_zone (struct game *game, int cid, int pid, int znum,
			      int order)
{
    int lid = tux_to_lid (pid, znum);
    send_opponents (game, "set-card-location cid:%d lid:%d x:%d",
		    cid, lid, order);
}

void send_card_flag (struct game *game, int cid, int flag, int state)
{
    char *ftext;

    ftext = NULL;
    switch (flag) {
    case CARD_TAPPED_FLAG:
	ftext = "Tapped";
	break;
    case CARD_ATTACK_FLAG:
	ftext = "Attack";
	break;
    case CARD_PHASED_FLAG:
	ftext = "Phased";
	break;
    case CARD_NO_UNTAP_FLAG:
	send_opponents (game, "untap-mode cid:%d pid:%d set:%d",
			cid, game->local_player+1, state);
	return;
    case CARD_FLIPPED_FLAG:
	send_opponents (game, "flip-card-over cid:%d pid:%d face:%d",
			cid, game->local_player+1, state);
	return;
    }
    if (ftext != NULL)
	send_opponents (game, "set-card-flag cid:%d flag:\"%s\" set:%d",
			cid, ftext, state);
}

void send_card_controller (struct game *game, struct table_card *tcard,
			   int take)
{
    char *tmsg;

    tmsg = (take ? " take:1" : "");
    send_opponents (game, "set-card-controller cid:%d pid:%d%s",
		    tcard->cid, tcard->controller+1, tmsg);
}

void send_sleight_card (struct game *game, struct table_card *tcard)
{
    send_opponents (game, "sleight-card cid:%d color:%d",
		    tcard->cid, tcard->color);
}

void send_random_card (struct game *game, int cid, int znum)
{
    send_opponents (game, "do-msg msg:\"%s randomly shows '%s'\"",
		    game->player[game->local_player]->name,
		    game_get_card_name (game, cid));
}

void send_inform_peeking (struct game *game, int owner, int znum,
			  int quantity, int sflag)
{
    char *pname, *oname, *zname, *desc;

    pname = game->player[game->local_player]->name;
    zname = name_of_zone (znum);
    oname = game->player[owner]->name;
    desc = alloca (strlen (oname) + 6 + strlen (zname));
    if (owner == game->local_player) {
	strcpy (desc, zname);
    } else {
	sprintf (desc, "%s's %s", oname, zname);
    }
    if (sflag) {
	if (quantity == 0)
	    send_opponents (game, "do-msg msg:\"%s is looking through "
			    "%s...\"", pname, desc);
	else
	    send_opponents (game, "do-msg msg:\"%s is looking through "
			    "top %d cards of %s...\"",
			    pname, quantity, desc);
    } else {
	if (quantity == 0)
	    send_opponents (game, "do-msg msg:\"%s stops looking through "
			    "%s.\"", pname, desc);
    }
}

void send_create_card (struct game *game, char *name, int cid, int znum,
		       char *powtuf, int color)
{
    int lid = tux_to_lid (game->local_player, znum);
    send_opponents (game, "do-createcard name:\"%s\" cid:%d pid:%d "
		    "destlid:%d powtuf:\"%s\" border:%d", name, cid,
		    game->local_player+1, lid, powtuf, color);
}

void send_show_zone (struct game *game, int znum, int quantity)
{
    int pid = game->local_player+1;
    if (znum == HAND_ZONE)
	send_opponents (game, "show-hand pid:%d", pid);
    else if ((znum == LIBRARY_ZONE) && (quantity > 0))
	send_opponents (game, "show-libtop pid:%d qty:%d", pid, quantity);
    else if (znum == LIBRARY_ZONE)
	send_opponents (game, "show-lib pid:%d", pid);
}

void send_request_new_game (struct game *game, int flag)
{
    if (flag) {
	send_opponents (game, "setup-reset");
	send_opponents (game, "setup-begin pid:%d", game->local_player+1);
    } else {
	send_opponents (game, "do-msg msg:\"%s: Cancelled start new game.\"",
			game->player[game->local_player]->name);
	send_setup_end (game);
    }
}

void send_ping (struct game *game)
{
    int pid;

    pid = pid_of_single_opponent (game);
    if (pid < 0)
	return;
    process_send_ping (game, pid);
    send_opponents (game, "init-ping pid:%d", game->local_player+1);
    display_message (game, "* Ping initiated *");
}

void send_inform_sideboard (struct game *game)
{
    send_opponents (game, "do-msg msg:\"+ %s is sideboarding.\"",
		    game->player[game->local_player]->name);
}

void send_peek_at_facedown (struct game *game)
{
    send_opponents (game, "do-msg msg:\"%s is peeking at a facedown card.\"",
		    game->player[game->local_player]->name);
}

/* ----------------------------------------------------------------- */

void
run_parsed_command (struct game *game, char *cmd, struct rcmd_args *args)
{
    struct player *me;
    int pid, globalid, idx, znum, i, flag, cid, offset, bflag;
    struct zone *zone;
    struct table_card *tcard;
    char *cp, *next, *name;
    struct card_move m;

    pid = pid_of_single_opponent (game);

    if (strcmp (cmd, "version-number") == 0) {
	/* game->opponent->protocol = strdup (args->text); */
	if (args->value == 0)
	    game->no_view_sideboard = TRUE;
	if ((game->opponent->flags & is_server_flag) != 0) {
	    send_version_number (game);
	    send_initial_player_name (game);
	    start_new_game (game);
	}
    }
    else if (strcmp (cmd, "set-player-name") == 0) {
	player_set_name (game->player[pid], args->name);
    }
    else if (strcmp (cmd, "setup-begin") == 0) {
	me = game->player[pid];
	display_message (game, "%s is setting up cards...", me->name);
	if (me->deck != NULL) {
	    if ((game->opponent->flags & local_wants_new) == 0)
		return;
	    start_new_game (game);
	} else {
	    me->deck = deck_new ();
	}
    }
    else if (strcmp (cmd, "setup-card") == 0) {
	me = game->player[pid];
	name = decrypt_name (me->name, args->name);
	globalid = cardbase_find_globalid (game->cardbase, name);
	if (globalid < 0)
	    return;
	idx = args->cid - player_deck_offset (pid);
	if (idx < 0)
	    return;
	deck_set_card (me->deck, idx, args->qty, globalid);
    }
    else if (strcmp (cmd, "sideboard") == 0) {
	me = game->player[pid];
	if (me->deck != NULL) {
	    offset = player_deck_offset (pid);
	    zone_add_deck (me->zone[LIBRARY_ZONE], offset, me->deck->size);
	}
	display_message (game, "%s has a %d card sideboard.", me->name,
			 args->qty);
    }
    else if (strcmp (cmd, "setup-sb-card") == 0) {
	me = game->player[pid];
	name = decrypt_name (me->name, args->name);
	globalid = cardbase_find_globalid (game->cardbase, name);
	if (globalid < 0)
	    return;
	idx = args->cid - player_sideboard_offset (game, pid);
	if (idx < 0)
	    return;
	deck_set_sb_card (me->deck, idx, args->qty, globalid);
    }
    else if (strcmp (cmd, "setup-end") == 0) {
	if ((game->opponent->flags & opponent_wants_new) != 0) {
	    game->opponent->flags &= ~opponent_wants_new;
	    return;
	}
	me = game->player[pid];
	if (me->deck != NULL) {
	    offset = player_sideboard_offset (game, pid);
	    zone_add_deck (me->zone[SIDEBOARD_ZONE], offset,
			   me->deck->sb_size);
	    game_save_deck (game, pid);
	}
	close_new_game_dialog (game);
    }
    else if (strcmp (cmd, "setup-reset") == 0) {
	me = game->player[pid];
	display_message (game, "%s is requesting a new game.", me->name);
	game->opponent->flags |= opponent_wants_new;
    }

    else if (strcmp (cmd, "do-msg") == 0) {
	display_message (game, "%s", args->text);
    }
    else if (strcmp (cmd, "do-createcard") == 0) {
	lid_to_tux (args->lid, &pid, &znum);
	idx = args->cid - player_token_offset (game, pid);
	if (idx >= 0) {
	    game_create_card (game, pid, args->name, idx, znum,
			      args->text, args->border);
	}
    }
    else if (strcmp (cmd, "do-drawcard") == 0) {
	me = game->player[pid];
	if (me->zone[LIBRARY_ZONE]->size == 0)
	    return;
	m.pid = pid;
	m.srcznum = LIBRARY_ZONE;
	m.idx = -1;
	m.dstznum = HAND_ZONE;
	m.bottom = FALSE;
	m.cid = 0;
	game_move_card (game, &m);
	if ((args->text != NULL) && (strcmp (args->text, "nul") != 0))
	    display_message (game, "%s", args->text);
    }
    else if (strcmp (cmd, "do-shuffle") == 0) {
	lid_to_tux (args->lid, &pid, &znum);
	zone = game->player[pid]->zone[znum];
	args->startpos--;
	cp = args->text;
	offset = player_deck_offset (pid) - 1;
	i = 0;
	while (TRUE) {
	    cid = strtol (cp, &next, 0);
	    if (cp == next)
		break;
	    zone->cardlist[args->startpos+i] = cid+offset;
	    i++;
	    cp = next;
	}
	if (zone->list != NULL)
	    zone_refresh_list_display (zone);
    }
    else if (strcmp (cmd, "flip-coin") == 0) {
	game_flip_coin (game, pid, args->value);
    }
    else if (strcmp (cmd, "sleight-card") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard != NULL)
	    set_table_card_color (game, tcard, args->value, pid);
    }
    else if (strcmp (cmd, "set-card-controller") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard != NULL)
	    set_card_controller (game, tcard, pid, args->value);
    }
    else if (strcmp (cmd, "set-card-counters") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard != NULL)
	    set_card_counters (game, tcard, args->value);
    }
    else if (strcmp (cmd, "set-card-flag") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard == NULL)
	    return;
	flag = 0;
	if (args->flag != NULL) {
	    if (strcmp (args->flag, "Tapped") == 0)
		flag = CARD_TAPPED_FLAG;
	    else if (strcmp (args->flag, "Attack") == 0)
		flag = CARD_ATTACK_FLAG;
	    else if (strcmp (args->flag, "Phased") == 0)
		flag = CARD_PHASED_FLAG;
	    else if (strcmp (args->flag, "Flipped") == 0)
		flag = CARD_FLIPPED_FLAG;
	    else if (strcmp (args->flag, "UntapMode") == 0)
		flag = CARD_NO_UNTAP_FLAG;
	}
	if (flag == 0)
	    return;
	set_table_card_flag (game, tcard, flag, args->value, pid);
    }
    else if (strcmp (cmd, "flip-card-over") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard != NULL)
	    set_table_card_flag (game, tcard, CARD_FLIPPED_FLAG,
				 args->face, pid);
    }
    else if (strcmp (cmd, "untap-mode") == 0) {
	tcard = playarea_get_table_card (game->playarea, args->cid);
	if (tcard != NULL)
	    set_table_card_flag (game, tcard, CARD_NO_UNTAP_FLAG,
				 args->value, pid);
    }
    else if (strcmp (cmd, "set-card-location") == 0) {
	bflag = string_contains (args->text, "bottom");
	set_card_location (game, pid, args->cid, args->lid,
			   args->x, args->y, args->face, bflag);
    }
    else if (strcmp (cmd, "set-game-phase") == 0) {
	set_game_phase (game, args->value);
    }
    else if (strcmp (cmd, "set-game-turn") == 0) {
	set_game_turn (game, args->value);
    }
    else if (strcmp (cmd, "set-player-life") == 0) {
	set_player_life (game, pid, args->value);
    }
    else if (strcmp (cmd, "change-player-name") == 0) {
	set_player_name (game, pid, args->name);
    }
    else if (strcmp (cmd, "show-hand") == 0) {
	do_zone_dialog (game, pid, HAND_ZONE, 0);
    }
    else if (strcmp (cmd, "show-lib") == 0) {
	do_zone_dialog (game, pid, LIBRARY_ZONE, 0);
    }
    else if (strcmp (cmd, "show-libtop") == 0) {
	do_zone_dialog (game, pid, LIBRARY_ZONE, args->qty);
    }
    else if (strcmp (cmd, "init-ping") == 0) {
	send_opponents (game, "return-ping pid:%d", game->local_player+1);
    }
    else if (strcmp (cmd, "return-ping") == 0) {
	process_return_ping (game, pid);
    }
    else {
	display_message (game, "Invalid request: \"%s\"", cmd);
    }
}

static int string_contains (char *haystack, char *needle)
{
    int i, nlength, hlength;

    if (haystack == NULL)
	return FALSE;
    nlength = strlen (needle);
    hlength = strlen (haystack);
    if (nlength > hlength)
	return FALSE;
    for (i = 0; i <= (hlength - nlength); i++) {
	if (strncasecmp (haystack+i, needle, nlength) == 0)
	    return TRUE;
    }
    return FALSE;
}

/* ----------------------------------------------------------------- */

void lid_to_tux (int lid, int *pidp, int *zonep)
{
    *pidp = *zonep = 0;
    if (lid < 1)
	return;
    if (lid <= 4) {
	*zonep = lid-1;
	return;
    }
    if (lid <= 8) {
	*pidp = 1;
	*zonep = lid-5;
	return;
    }
    if (lid == 9) {
	*pidp = 0;
	*zonep = SIDEBOARD_ZONE;
	return;
    }
    if (lid == 10) {
	*pidp = 1;
	*zonep = SIDEBOARD_ZONE;
	return;
    }
}

int tux_to_lid (int pid, int znum)
{
    if (znum < SIDEBOARD_ZONE)
	return (pid * 4 + znum + 1);
    return (pid + 9);
}

#endif /* APPRENTICE */
