/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  zone.c -- A zone is an ordered set of cards, such as a hand,
 *	      library, or graveyard.
 *
 *  A zone may be displayed in a listbox.
 *  A zone may be represented by a button in a toolbar.
 *  Cards can be dragged between any graphical widgets that represent zones.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include "deck.h"
#include "zone.h"
#include "cardbase.h"
#include "game.h"
#include "playarea.h"
#include "opponent.h"
#include "random.h"

static void zone_update_label (struct zone *zone);

struct zone *zone_new (struct game *game)
{
    struct zone *zone;

    zone = calloc (1, sizeof (struct zone));
    zone->game = game;
    return (zone);
}

void zone_add_deck (struct zone *zone, int offset, int size)
{
    int x;

    zone->alloced = size;
    zone->cardlist = realloc (zone->cardlist, zone->alloced * sizeof (int));
    for (x = 0; x < size; x++)
	zone->cardlist[x] = offset+size-x-1;
    zone->size = size;
    zone_update_label (zone);
}

void zone_shuffle (struct zone *zone)
{
    int c1, c2, card;

    for (c1 = 0; c1 < zone->size; c1++) {
	/* Swap c1 with a card between c1 and the rest of the deck. */
	c2 = c1 + random_number (zone->size-c1);
	card = zone->cardlist[c1];
	zone->cardlist[c1] = zone->cardlist[c2];
	zone->cardlist[c2] = card;
    }
}

void zone_add_card (struct zone *zone, int card, int bottom)
{
    int idx, x;
    struct cardinfo *info;
    char *sarr[2];

    while (zone->size >= zone->alloced) {
	zone->alloced += 60;
	zone->cardlist = realloc (zone->cardlist,
				  zone->alloced * sizeof (int));
    }
    idx = (bottom ? 0 : zone->size);
    for (x = zone->size; x > idx; x--)
	zone->cardlist[x] = zone->cardlist[x-1];
    zone->cardlist[idx] = card;
    zone->size++;
    zone_update_label (zone);

    if (zone->list != NULL) {
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	    idx = zone->size - 1 - idx;
	info = game_get_cardinfo (zone->game, card);
	if (info == NULL)
	    return;
	sarr[0] = info->cost;
	sarr[1] = info->name;
	gtk_clist_insert (GTK_CLIST (zone->list), idx, sarr);
    }
}

int zone_remove_card (struct zone *zone, int idx)
{
    int card, x;

    if (idx < 0)
	idx = zone->size-1;
    if (idx < 0)
	return (-1);
    if (zone->size <= idx)
	return (-1);
    card = zone->cardlist[idx];
    zone->size--;
    zone_update_label (zone);
    for (x = idx; x < zone->size; x++)
	zone->cardlist[x] = zone->cardlist[x+1];
    if (zone->list != NULL) {
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	    idx = zone->size - idx;
	gtk_clist_remove (GTK_CLIST (zone->list), idx);
    }
    return (card);
}

int zone_move_card (struct zone *zone, int idx, int other)
{
    int cid, x;

    cid = zone->cardlist[idx];
    if (idx < other) {
	for (x = idx; x < other; x++)
	    zone->cardlist[x] = zone->cardlist[x+1];
    } else if (idx > other) {
	for (x = idx; x > other; x--)
	    zone->cardlist[x] = zone->cardlist[x-1];
    }
    zone->cardlist[other] = cid;
    if (zone->list != NULL) {
	GtkCList *clist = GTK_CLIST (zone->list);
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0) {
	    idx = zone->size - 1 - idx;
	    other = zone->size - 1 - other;
	}
	if ((idx < clist->rows) && (other < clist->rows)) {
	    gtk_clist_row_move (clist, idx, other);
	} else if (idx < clist->rows) {
	    gtk_clist_remove (clist, idx);
	}
    }
    return (cid);
}

void zone_reset (struct zone *zone)
{
    int num_rows, i;

    num_rows = zone->size;
    zone->size = 0;
    zone_update_label (zone);
    if (zone->dialog != NULL) {
	gtk_widget_destroy (zone->dialog);
	zone->list = NULL;
	zone->dialog = NULL;
    }
    if (zone->list != NULL) {
	for (i = num_rows-1; i >= 0; i--)
	    gtk_clist_remove (GTK_CLIST (zone->list), i);
    }
}

void zone_destroy (struct zone *zone)
{
    if (zone->cardlist != NULL)
	free (zone->cardlist);
    if (zone->dialog != NULL) {
	gtk_widget_destroy (zone->dialog);
	zone->list = NULL;
	zone->dialog = NULL;
    }
    free (zone);
}

/* ----------------------------------------------------------------- */

void zone_set_label (struct zone *zone, GtkWidget *label)
{
    zone->label = label;
    zone_update_label (zone);
}

static void zone_update_label (struct zone *zone)
{
    char buf[12];
    if (zone->label != NULL) {
	sprintf (buf, "%d", zone->size);
	gtk_label_set_text (GTK_LABEL (zone->label), buf);
    }
}

/* ----------------------------------------------------------------- */

static void get_card_from_list (GtkWidget *, GdkDragContext *,
				GtkSelectionData *, guint, guint, gpointer);
static gint zone_start_drag (GtkWidget *widget, GdkEvent *event,
			     struct zone *zone);
static void do_drop_card (GtkWidget *, GdkDragContext *, gint, gint,
			  GtkSelectionData *, guint, guint, struct game *);

GtkWidget *zone_create_clist (struct zone *zone, int top_cards_requested)
{
    GtkWidget *clist;
    int size, card, x, idx;
    struct cardinfo *info;
    char *sarr[2];

    clist = gtk_clist_new (2);
    gtk_clist_set_selection_mode (GTK_CLIST (clist),
				  GTK_SELECTION_SINGLE);
    gtk_clist_set_column_width (GTK_CLIST (clist), 0, 35);
    gtk_clist_set_column_width (GTK_CLIST (clist), 1, 0);

    size = top_cards_requested;
    if ((size == 0) || (size > zone->size))
	size = zone->size;
    for (x = 0; x < size; x++) {
	idx = x;
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	    idx = zone->size - 1 - x;
	card = zone->cardlist[idx];
	info = game_get_cardinfo (zone->game, card);
	if (info == NULL)
	    continue;
	sarr[0] = info->cost;
	sarr[1] = info->name;
	gtk_clist_append (GTK_CLIST (clist), sarr);
    }

    zone_set_list (zone, clist);
    zone->top_cards_requested = top_cards_requested;
    return (clist);
}

void zone_set_list (struct zone *zone, GtkWidget *list)
{
    zone->list = list;
    zone->top_cards_requested = 0;
    if (list == NULL)
	return;

    gtk_clist_set_use_drag_icons (GTK_CLIST (list), FALSE);

    /* gtk_drag_source_set (...); */
    gtk_signal_connect (GTK_OBJECT (list), "button_press_event",
			GTK_SIGNAL_FUNC (zone_start_drag), zone);
    gtk_signal_connect (GTK_OBJECT (list), "motion_notify_event",
			GTK_SIGNAL_FUNC (zone_start_drag), zone);
    gtk_signal_connect (GTK_OBJECT (list), "drag_data_get",
			GTK_SIGNAL_FUNC (get_card_from_list), NULL);

    gtk_drag_dest_set (list, GTK_DEST_DEFAULT_ALL, target_table, num_targets,
		       GDK_ACTION_MOVE);
    gtk_signal_connect (GTK_OBJECT (list), "drag_data_received",
			GTK_SIGNAL_FUNC (do_drop_card), zone->game);
}

void zone_set_box (struct zone *zone, GtkWidget *box, int sourcep)
{
    zone->box = box;

    if (sourcep) {
	gtk_widget_set_events (box, gtk_widget_get_events (box) |
			   GDK_BUTTON_PRESS_MASK | GDK_BUTTON1_MOTION_MASK);
	gtk_signal_connect (GTK_OBJECT (box), "button_press_event",
			    GTK_SIGNAL_FUNC (zone_start_drag), zone);
	gtk_signal_connect (GTK_OBJECT (box), "motion_notify_event",
			    GTK_SIGNAL_FUNC (zone_start_drag), zone);
	gtk_signal_connect (GTK_OBJECT (box), "drag_data_get",
			    GTK_SIGNAL_FUNC (get_card_from_list), NULL);
    }

    gtk_drag_dest_set (box, GTK_DEST_DEFAULT_ALL, target_table, num_targets,
		       GDK_ACTION_MOVE);
    gtk_signal_connect (GTK_OBJECT (box), "drag_data_received",
			GTK_SIGNAL_FUNC (do_drop_card), zone->game);
}

static void get_card_from_list (GtkWidget *w, GdkDragContext *context,
				GtkSelectionData *selection, guint info,
				guint time, gpointer data)
{
    gtk_selection_data_set (selection, selection->target, 8, NULL, 0);
}

int zone_dragged_card_idx (struct zone *zone)
{
    int idx = 0;

    if (zone->drag_from_list) {
	if (zone->list == NULL)
	    return (-1);
	idx = GTK_CLIST (zone->list)->click_cell.row;
	if (idx < 0)
	    return (idx);
    }
    if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	idx = zone->size - 1 - idx;
    return (idx);
}

/*
 *  zone_start_drag () -- This is called (by the main loop) when
 *	a mouse button is pressed in the zone's list box.
 */
static gint zone_start_drag (GtkWidget *widget, GdkEvent *event,
			     struct zone *zone)
{
    int delta_x, delta_y, rnum;
    GtkTargetList *tlist;

    switch (event->type) {
    case GDK_BUTTON_PRESS:
	if (event->button.button == 1) {
	    zone->drag_beginning = TRUE;
	    zone->x = event->button.x;
	    zone->y = event->button.y;
	}
	break;
    case GDK_BUTTON_RELEASE:
	zone->drag_beginning = FALSE;
	break;
    case GDK_MOTION_NOTIFY:
	if (zone->drag_beginning &&
	    (event->motion.state & GDK_BUTTON1_MASK) != 0) {
	    delta_x = zone->x - event->motion.x;
	    delta_y = zone->y - event->motion.y;
	    if ((delta_x > 3) || (delta_x < -3) ||
		(delta_y > 3) || (delta_y < -3)) {
		zone->drag_beginning = FALSE;
		zone->drag_from_list = GTK_IS_CLIST (widget);
		if (zone->drag_from_list) {
		    rnum = GTK_CLIST (widget)->click_cell.row;
		    gtk_clist_select_row (GTK_CLIST (widget), rnum, 0);
		}
		tlist = gtk_target_list_new (target_table, num_targets);
		gtk_drag_begin (widget, tlist, GDK_ACTION_MOVE, 1, event);
		return TRUE;
	    }
	    break;
	}
    default:
	break;
    }
    return FALSE;
}

/*
 *  do_drop_card () -- When a card is dropped on any zone widget,
 *	that widget calls this function to move the card from its
 *	source zone to that widget's zone.
 */
static void do_drop_card (GtkWidget *wdst, GdkDragContext *context,
			  gint x, gint y, GtkSelectionData *data,
			  guint info, guint time, struct game *game)
{
    GtkWidget *wsrc;
    struct card_move m;
    int card, owner;
    struct table_card *tcard;

    m.bottom = is_shift_down ();
    wsrc = gtk_drag_get_source_widget (context);
    if (wsrc == NULL)
	return;

    m.dstznum = widget_to_zone (game, wdst, &m.pid);
    if (m.dstznum < 0)
	return;

    if (wsrc == game->playarea->darea) {
	/*
	 *  You may move a card you control to any of its owner's zones.
	 */
	card = playarea_dragged_card (game->playarea);
	if (card <= 0)
	    return;
	tcard = playarea_get_table_card (game->playarea, card);
	if (tcard == NULL)
	    return;
	if (tcard->controller != game->local_player)
	    return;
	owner = owner_of_card (game, card);
	if (owner < 0)
	    return;
	game_unplay_card (game, card, owner, m.dstznum, game->local_player,
			  m.bottom);
	send_unplay_card (game, card, owner, m.dstznum, m.bottom,
			  ((tcard->flags & CARD_FLIPPED_FLAG) != 0));
	return;
    }

    if (m.pid != game->local_player)
	return;
    m.srcznum = widget_to_zone (game, wsrc, &m.pid);
    if ((m.srcznum < 0) || (m.pid != game->local_player))
	return;
    if (m.srcznum == m.dstznum)
	return;
    m.idx = zone_dragged_card_idx (game->player[m.pid]->zone[m.srcznum]);
    game_move_card (game, &m);
    if (m.cid > 0) {
	game_move_card_message (game, &m);
	send_card_zone (game, &m);
    }
}

/* ----------------------------------------------------------------- */

static void gtk_clist_set_num_rows (GtkCList *clist, int rows)
{
    char *text[2] = { "", "" };

    while (clist->rows > rows)
	gtk_clist_remove (clist, clist->rows-1);
    while (clist->rows < rows)
	gtk_clist_append (clist, text);
}

void zone_refresh_list_display (struct zone *zone)
{
    int size, x, idx, card;
    struct cardinfo *info;

    if (zone->list == NULL)
	return;
    size = zone->top_cards_requested;
    if ((size == 0) || (size > zone->size))
	size = zone->size;
    gtk_clist_set_num_rows (GTK_CLIST (zone->list), size);
    for (x = 0; x < size; x++) {
	idx = x;
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	    idx = zone->size - 1 - x;
	card = zone->cardlist[idx];
	info = game_get_cardinfo (zone->game, card);
	if (info == NULL)
	    continue;
	gtk_clist_set_text (GTK_CLIST (zone->list), x, 0, info->cost);
	gtk_clist_set_text (GTK_CLIST (zone->list), x, 1, info->name);
    }
}
