/*
 *  Copyright (c) 2002  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  cardart.c -- Downloads card images from www.wizards.com.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include "prefs.h"
#include "cardbase.h"
#include "cardart.h"
#include "http_fetcher.h"

struct cardart {
    GdkPixbuf *card_image;
    GdkPixbuf *big_art;
    GdkPixbuf *small_art;
    short width, height;
};

/*
 *  Default values.  Prefs can override.
 */
const char *url_prefix = "http://www.wizards.com/global/images/magic/general/";
const char *url_suffix = ".jpg";
const char *cache_component = ".mindlesscache";

const char *download_cardart_key = "download_cardart";
const char *cache_directory_key = "cache_directory";

/*
 *  Global state.  Set by cardart_initialized().
 */
static int do_download = 0;
static char *cachedir = NULL;

static struct cardart *cardart_open_file (char *, struct cardart *);
static int is_old_card (char *expansion);
static int is_old_set (char *expansion, int len);

void cardart_initialize (struct prefs *prefs)
{
    char *home, *cp;
    int len;

    do_download = get_int_preference (prefs, download_cardart_key, 0);

    cachedir = get_preference (prefs, cache_directory_key);
    if (cachedir == NULL) {
	home = getenv ("HOME");
	if (home == NULL)
	    return;
	len = strlen (home);
	cachedir = malloc (len + strlen (cache_component) + 2);
	strcpy (cachedir, home);
	if (cachedir[len-1] != '/')
	    cachedir[len++] = '/';
	strcpy (cachedir+len, cache_component);
    }
    if ((mkdir (cachedir, 0755) < 0) && (errno != EEXIST)) {
	perror (cachedir);
	do_download = 0;
    }
    if (do_download) {
	cp = get_preference (prefs, "cardart_url_prefix");
	if (cp != NULL)
	    url_prefix = cp;
	cp = get_preference (prefs, "cardart_url_suffix");
	if (cp != NULL)
	    url_suffix = cp;
    }
}

struct cardart *
cardart_load_cardart (char *cardname, char *expansions)
{
    return (cardart_load_cardart3 (cardname, expansions, FALSE));
}

struct cardart *
cardart_load_cardart3 (char *cardname, char *expansions, int force_download)
{
    char *newname, *dp, *sp, *pathname, *url, *data;
    struct cardart *art;
    int size, result;
    FILE *fp;

    if (cachedir == NULL)
	return NULL;

    /*
     *  Translate card name to file name and try to load it from cache.
     */
    newname = alloca (strlen (cardname) + strlen (url_suffix) + 2);
    dp = newname;
    for (sp = cardname; *sp != 0; sp++) {
	if (isalnum (*sp))
	    *dp++ = tolower (*sp);
	else if ((*sp == ' ') || (*sp == '-'))
	    *dp++ = '_';
    }
    strcpy (dp, url_suffix);

    pathname = alloca (strlen (cachedir) + strlen (newname) + 4);
    strcpy (pathname, cachedir);
    strcat (pathname, "/");
    strcat (pathname, newname);

    art = cardart_open_file (pathname, NULL);
    if (art != NULL)
	return art;

    if (!(do_download || force_download))
	return NULL;
    /*
     *  Ignore split cards until I come up with a better solution.
     */
    if (strchr (cardname, '/') != NULL)
	return NULL;
    /*
     *  Create a cardart structure to hold the downloaded image.
     *  If the image can not be downloaded, return the empty cardart
     *  structure (instead of NULL) so that we don't try to download
     *  the same image again.
     */
    art = calloc (1, sizeof (struct cardart));
    url = alloca (strlen (url_prefix) + strlen (newname) + 4);
    strcpy (url, url_prefix);
    strcat (url, newname);
    puts (url);
    size = http_fetch (url, &data);
    if (size < 0) {
	http_perror ("http_fetch");
	return art;
    }
    if (size == 0) {
	return art;
    }
    fp = fopen (pathname, "w");
    if (fp == NULL) {
	perror (pathname);
	free (data);
	return art;
    }
    dp = data;
    while (size > 0) {
	result = fwrite (dp, 1, size, fp);
	if (result <= 0) {
	    perror (pathname);
	    break;
	}
	dp += result;
	size -= result;
    }
    fclose (fp);
    free (data);

    return (cardart_open_file (pathname, art));
}

static struct cardart *
cardart_open_file (char *pathname, struct cardart *art)
{
    GdkPixbuf *pix;

#if GDK_PIXBUF_MAJOR < 2
    pix = gdk_pixbuf_new_from_file (pathname);
#else
    pix = gdk_pixbuf_new_from_file (pathname, NULL);
#endif
    if (pix == NULL)
	return art;
    if (art == NULL)
	art = calloc (1, sizeof (struct cardart));
    art->card_image = pix;
    return (art);
}

void
cardart_destroy (struct cardart *art)
{
    if (art != NULL) {
	if (art->card_image != NULL)
	    gdk_pixbuf_unref (art->card_image);
	if (art->big_art != NULL)
	    gdk_pixbuf_unref (art->big_art);
	if (art->small_art != NULL)
	    gdk_pixbuf_unref (art->small_art);
	free (art);
    }
}

GdkPixbuf *
cardart_get_card_image (struct cardinfo *info)
{
    struct cardart *art = info->art;
    return (art == NULL ? NULL : art->card_image);
}

GdkPixbuf *
cardart_get_unscaled_art (struct cardinfo *info)
{
    struct cardart *art = info->art;
    int full_width, full_height, src_x, src_y, crop_width, crop_height;
    GdkColorspace colorspace;
    gboolean has_alpha;
    int bits_per_sample;

    if (art == NULL)
	return (NULL);
    if ((art->big_art == NULL) && (art->card_image != NULL)) {
	full_width = gdk_pixbuf_get_width (art->card_image);
	full_height = gdk_pixbuf_get_height (art->card_image);
	if ((info->expansion != NULL) && is_old_card (info->expansion)) {
	    src_x = full_width * 19 / 200;
	    src_y = full_height * 23 / 284;
	    crop_width = full_width * 162 / 200;
	    crop_height = full_height * 131 / 284;
	} else {
	    src_x = full_width * 9 / 200;
	    src_y = full_height * 26 / 284;
	    crop_width = full_width * 182 / 200;
	    crop_height = full_height * 132 / 284;
	}
	colorspace = gdk_pixbuf_get_colorspace (art->card_image);
	has_alpha = gdk_pixbuf_get_has_alpha (art->card_image);
	bits_per_sample = gdk_pixbuf_get_bits_per_sample (art->card_image);
	art->big_art = gdk_pixbuf_new (colorspace, has_alpha, bits_per_sample,
				       crop_width, crop_height);
	gdk_pixbuf_copy_area (art->card_image,
			      src_x, src_y, crop_width, crop_height,
			      art->big_art, 0, 0);
    }
    return (art->big_art);
}

GdkPixbuf *
cardart_get_small_art (struct cardinfo *info, int width, int height)
{
    struct cardart *art = info->art;
    GdkPixbuf *big_art;

    if (art == NULL)
	return (NULL);
    if ((art->small_art != NULL) &&
	(art->width != width || art->height != height)) {
	gdk_pixbuf_unref (art->small_art);
	art->small_art = NULL;
	art->width = art->height = 0;
    }
    if (art->small_art == NULL) {
	big_art = cardart_get_unscaled_art (info);
	if (big_art == NULL)
	    return (NULL);
	art->small_art = gdk_pixbuf_scale_simple (big_art, width, height,
						  GDK_INTERP_BILINEAR);
	art->width = width;
	art->height = height;
    }
    return (art->small_art);
}

struct cardart *
cardart_get_null_art (void)
{
    return calloc (1, sizeof (struct cardart));
}

/**
 * Return true if this card has not appeared in any set since 8th Edition
 * was released.
 */
static int is_old_card (char *expansion)
{
    char *cp;
    int found, len;

    cp = expansion;
    found = FALSE;
    while (*cp != 0) {
	len = 0;
	while (isalnum (cp[len]))
	    len++;
	if (!is_old_set (cp, len))
	    return (FALSE);
	found = TRUE;
	cp += len;
	if (*cp == '-') {
	    cp++;
	    /* skip over commonality */
	    while (isalnum (*cp))
		cp++;
	}
	/* skip over commas, spaces, and garbage */
	while ((*cp != 0) && !isalnum (*cp))
	    cp++;
    }
    return (found);
}

static int is_old_set (char *expansion, int len)
{
    const char *old_sets[] = {
	"1E", "2E", "3E", "AN", "AQ", "LG", "DK", "FE", "4E",
	"IA", "CH", "HL", "AL", "MI", "VI", "5E", "WL", "TE", "ST", "EX",
	"US", "UL", "6E", "UD", "MM", "NE", "PR", "IN", "PS", "7E", "AP",
	"OD", "TO", "JD", "ON", "LE", "SC", NULL
    };
    int i;
    if (len == 2)
	for (i = 0; old_sets[i] != NULL; i++)
	    if (strncmp (old_sets[i], expansion, len) == 0)
		return (TRUE);
    return (FALSE);
}
