/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  deck.c -- A deck is an ordered set of cards.
 *
 *  At the start of a game, each card in each player's deck is given
 *  a unique integer id.  A deck does not change during a game.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "deck.h"
#include "cardbase.h"
#include "fgetline.h"

static void cardlist_set_card (int *size, int *alloced, int **cardlist,
			       int idx, int qty, int globalid);

struct deck *deck_new (void)
{
    return (calloc (1, sizeof (struct deck)));
}

struct deck *deck_read_file (const char *filename, struct cardbase *db)
{
    FILE *fp;
    struct deck *deck;
    char buf[160], *cp, *name;
    int globalid, num, is_sideboard, sectemp;

    fp = fopen (filename, "r");
    if (fp == NULL)
	return (NULL);
    deck = deck_new ();
    deck->filename = strdup (filename);

    while ((cp = fgetline (buf, sizeof (buf), fp)) != NULL) {
	while (isspace (*cp))
	    cp++;
	if (strncmp (cp, "//", 2) == 0)
	    continue;
	is_sideboard = (strncasecmp (cp, "SB:", 3) == 0);
	if (is_sideboard) {
	    cp += 3;
	    while (isspace (*cp))
		cp++;
	}
	num = 0;
	while (isdigit (*cp)) {
	    num = num*10 + *cp-'0';
	    cp++;
	}
	while (isspace (*cp))
	    cp++;
	if (*cp == 0)
	    continue;
	name = cp;
	globalid = cardbase_find_globalid (db, name);
	if (globalid < 0) {
	    /* show error */
	    continue;
	}
	/* apprentice security code */
	sectemp = 0;	
	for (cp = name; *cp != 0; cp++) {
	    sectemp += *cp;
	}
	if (is_sideboard) {
	    cardlist_set_card (&deck->sb_size, &deck->sb_alloced,
			       &deck->sb_cardlist,
			       deck->sb_size, num, globalid);
	    deck->security_code += num * sectemp;
	} else {
	    cardlist_set_card (&deck->size, &deck->alloced, &deck->cardlist,
			       deck->size, num, globalid);
	    deck->security_code += num * sectemp * 2;
	}
    }

    if (deck->name == NULL) {
	cp = strrchr (filename, '/');
	if (cp == NULL)
	    deck->name = strdup (filename);
	else
	    deck->name = strdup (cp+1);
    }
    return (deck);
}

void deck_set_card (struct deck *deck, int idx, int qty, int globalid)
{
    cardlist_set_card (&deck->size, &deck->alloced, &deck->cardlist,
		       idx, qty, globalid);
}

void deck_set_sb_card (struct deck *deck, int idx, int qty, int globalid)
{
    cardlist_set_card (&deck->sb_size, &deck->sb_alloced, &deck->sb_cardlist,
		       idx, qty, globalid);
}

void deck_set_token (struct deck *deck, int idx, int qty, int globalid)
{
    cardlist_set_card (&deck->tk_size, &deck->tk_alloced, &deck->tk_cardlist,
		       idx, qty, globalid);
}

static void
cardlist_set_card (int *size, int *alloced, int **cardlist,
		   int idx, int qty, int globalid)
{
    int max, i;

    max = idx + qty;
    if (*alloced < max) {
	*alloced = max + 3;
	*cardlist = realloc (*cardlist, *alloced * sizeof (int));
    }
    /*
     *  Add blank cards up to idx if necessary.
     */
    while (*size < max) {
	(*cardlist)[*size] = 0;
	*size = *size + 1;
    }
    /*
     *  Overwrite cards in the cardlist.
     *  Hopefully, we just created these cards.
     */
    for (i = 0; i < qty; i++) {
	(*cardlist)[idx+i] = globalid;
    }
}

int deck_lookup (struct deck *deck, int cid)
{
    if (cid < 0)
	return (-1);
    if (cid < deck->size)
	return (deck->cardlist[cid]);
    cid -= deck->size;
    if (cid < deck->sb_size)
	return (deck->sb_cardlist[cid]);
    cid -= deck->sb_size;
    if (cid < deck->tk_size)
	return (deck->tk_cardlist[cid]);
    return (-1);
}

void deck_remove_tokens (struct deck *deck)
{
    deck->tk_size = 0;
}

void deck_destroy (struct deck *deck)
{
    if (deck->name != NULL)
	free (deck->name);
    if (deck->cardlist != NULL)
	free (deck->cardlist);
    if (deck->sb_cardlist != NULL)
	free (deck->sb_cardlist);
    if (deck->tk_cardlist != NULL)
	free (deck->tk_cardlist);
    free (deck);
}

void deck_write_file (const char *filename, struct deck *deck,
		      struct cardbase *cardbase)
{
    FILE *fp;
    int i, globalid, count;
    struct cardinfo *tcard;

    fp = fopen (filename, "w");
    if (fp == NULL)
	return;
    i = 0;
    while (i < deck->size) {
	globalid = deck->cardlist[i];
	count = 1;
	while (deck->cardlist[i+count] == globalid)
	    count++;
	tcard = cardbase_get_card (cardbase, globalid);
	fprintf (fp, "%d %s\n", count, tcard->name);
	i += count;
    }
    i = 0;
    while (i < deck->sb_size) {
	globalid = deck->sb_cardlist[i];
	count = 1;
	while (deck->sb_cardlist[i+count] == globalid)
	    count++;
	tcard = cardbase_get_card (cardbase, globalid);
	fprintf (fp, "SB: %d %s\n", count, tcard->name);
	i += count;
    }
    fclose (fp);
}
