/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *  some portions copyright (c) 2005 Federico Poloni <f.poloni@sns.it>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  dialogs.c -- Dialogs for setting life total, card counters, etc.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "game.h"
#include "dialogs.h"
#include "opponent.h"
#include "zone.h"
#include "deck.h"
#include "cardbase.h"
#include "playarea.h"
#include "cardart.h"
#include "prefs.h"
#include "expansions.h"
#include "viewcard.h"

#if GTK_MAJOR_VERSION >= 2
#define KEY_EVENT "key_release_event"
#else
#define KEY_EVENT "key_press_event"
#endif

/* ----------------------------------------------------------------- */

void do_message_dialog (const char *message)
{
    GtkWidget *dialog, *label, *ok_button;
    GtkDialog *dp;

    dialog = gtk_dialog_new ();
    /* gtk_window_set_title (GTK_WINDOW (dialog), ""); */
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);

    label = gtk_label_new (message);
    gtk_container_add (GTK_CONTAINER (dp->vbox), label);
    ok_button = gtk_button_new_with_label ("OK");
    gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_container_add (GTK_CONTAINER (dp->action_area), ok_button);
    gtk_widget_show_all (dialog);
}

/* ----------------------------------------------------------------- */

struct input_args {
    GtkWidget *dialog;
    GtkWidget *entry;
    struct game *game;
    input_callback_t callback;
    gpointer data;
};

static int input_box_key_pressed (GtkWidget *, GdkEventKey *,
				  struct input_args *);
static void do_input_ok (GtkWidget *w, struct input_args *args);
static void do_input_cancel (GtkWidget *w, struct input_args *args);

void do_input_dialog (struct game *game, const char *title,
		      const char *prompt, const char *def,
		      input_callback_t callback, gpointer data)
{
    GtkWidget *dialog, *label, *entry, *ok_button, *cancel_button;
    GtkDialog *dp;
    struct input_args *args;

    dialog = gtk_dialog_new ();
    game_make_transient (game, dialog);
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dp->vbox), 10);
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    label = gtk_label_new (prompt);
    gtk_box_pack_start (GTK_BOX (dp->vbox), label, FALSE, FALSE, 0);
    entry = gtk_entry_new ();
    if (def != NULL)
	gtk_entry_set_text (GTK_ENTRY (entry), def);
    gtk_box_pack_start (GTK_BOX (dp->vbox), entry, FALSE, FALSE, 5);

    args = malloc (sizeof (struct input_args));
    args->dialog = dialog;
    args->entry = entry;
    args->game = game;
    args->callback = callback;
    args->data = data;

    gtk_signal_connect (GTK_OBJECT (entry), KEY_EVENT,
			GTK_SIGNAL_FUNC (input_box_key_pressed), args);
    ok_button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_input_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), ok_button,
			TRUE, TRUE, 0);
    cancel_button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
			GTK_SIGNAL_FUNC (do_input_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), cancel_button,
			TRUE, TRUE, 0);
    gtk_widget_show_all (dialog);
    gtk_widget_grab_focus (entry);
}

static int input_box_key_pressed (GtkWidget *w, GdkEventKey *event,
				  struct input_args *args)
{
    if ((event->keyval == GDK_KP_Enter) || (event->keyval == GDK_Return))
	do_input_ok (w, args);
    return (TRUE);
}

static void do_input_ok (GtkWidget *w, struct input_args *args)
{
    const gchar *text;
    int good;

    text = gtk_entry_get_text (GTK_ENTRY (args->entry));
    good = (*args->callback)(args->game, args->data, text);
    if (!good)
	return;
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void do_input_cancel (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static void do_verify_ok (GtkWidget *w, struct input_args *args);
static void do_verify_cancel (GtkWidget *w, struct input_args *args);

void do_verify_dialog (struct game *game, const char *title,
		       const char *prompt, input_callback_t callback)
{
    GtkWidget *dialog, *label, *ok_button, *cancel_button;
    GtkDialog *dp;
    struct input_args *args;

    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    game_make_transient (game, dialog);
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);

    label = gtk_label_new (prompt);
    gtk_box_pack_start (GTK_BOX (dp->vbox), label, FALSE, FALSE, 0);

    args = malloc (sizeof (struct input_args));
    args->dialog = dialog;
    args->entry = NULL;
    args->game = game;
    args->callback = callback;
    args->data = NULL;

    ok_button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_verify_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), ok_button,
			TRUE, TRUE, 0);
    cancel_button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
			GTK_SIGNAL_FUNC (do_verify_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), cancel_button,
			TRUE, TRUE, 0);
    gtk_widget_show_all (dialog);
}

static void do_verify_ok (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    (*args->callback)(args->game, NULL, NULL);
    free (args);
}

static void do_verify_cancel (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static int life_dialog_callback (struct game *, gpointer, const char *);

void do_life_dialog (struct game *game, int what_todo)
{
    char *title, prompt[75];

    sprintf(prompt, "You have %d life.\n",
                    game->player[game->local_player]->life);
    title = NULL;
    switch (what_todo) {
    case SET_LIFE:
	title = "Set Life";
	strcat (prompt, "Enter your new life total.");
	break;
    case GAIN_LIFE:
	title = "Add Life";
	strcat (prompt, "You GAINED how many life points?");
	break;
    case LOSE_LIFE:
	title = "Subtract Life";
	strcat (prompt, "You LOST how many life points?");
	break;
    }
    do_input_dialog (game, title, prompt, NULL, life_dialog_callback,
		     GINT_TO_POINTER(what_todo));
}

static int
life_dialog_callback (struct game *game, gpointer data, const char *text)
{
    int what_todo, old_life, new_life;

    what_todo = GPOINTER_TO_INT(data);
    old_life = game->player[game->local_player]->life;
    if (*text == '+') {
	what_todo = GAIN_LIFE;
	text++;
    } else if (*text == '-') {
	what_todo = LOSE_LIFE;
	text++;
    }
    new_life = atoi (text);
    switch (what_todo) {
    case GAIN_LIFE:
	new_life = old_life + new_life;
	break;
    case LOSE_LIFE:
	new_life = old_life - new_life;
	break;
    }
    if (old_life != new_life) {
	set_player_life (game, game->local_player, new_life);
	send_player_life (game);
    }
    return (TRUE);
}

/* ----------------------------------------------------------------- */

static void view_button_callback (GtkWidget *dialog, gpointer data);
static void shuffle_button_callback (GtkWidget *dialog, gpointer data);

void do_library_dialog (struct game *game)
{
    GtkWidget *dialog, *label, *button;
    GtkDialog *dp;
    GtkContainer *area;

    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (dialog), "Library Options");
    game_make_transient (game, dialog);
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);
    gtk_object_set_user_data (GTK_OBJECT (dialog), game);

    label = gtk_label_new ("What would you like to do?");
    gtk_container_add (GTK_CONTAINER (dp->vbox), label);
    area = GTK_CONTAINER (dp->action_area);
    button = gtk_button_new_with_label ("View...");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (view_button_callback),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    button = gtk_button_new_with_label ("Shuffle");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (shuffle_button_callback),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    gtk_widget_show_all (dialog);
}

static void view_button_callback (GtkWidget *dialog, gpointer data)
{
    struct game *game;

    game = gtk_object_get_user_data (GTK_OBJECT (dialog));
    gtk_widget_destroy (dialog);
    do_zone_dialog (game, game->local_player, LIBRARY_ZONE, 0);
}

static void shuffle_button_callback (GtkWidget *dialog, gpointer data)
{
    struct game *game;

    game = gtk_object_get_user_data (GTK_OBJECT (dialog));
    gtk_widget_destroy (dialog);
    local_player_shuffle (game);
}

/* ----------------------------------------------------------------- */

struct zone_dialog_args {
    GtkWidget *dialog;
    GtkWidget *shuffle_box;
    GtkWidget *up_button;
    struct game *game;
    int pid;
    int znum;
    int private;
};

static gint zone_dialog_delete (GtkWidget *w, GdkEvent *event,
				struct zone_dialog_args *args);
static void zone_dialog_ok (GtkWidget *, struct zone_dialog_args *);
static void close_zone_dialog (struct zone_dialog_args *, int);
static void do_view_card (GtkWidget *, struct zone_dialog_args *);
static void do_zone_move_card (GtkWidget *, struct zone_dialog_args *);
static void do_move_all_to_bottom (GtkWidget *, struct zone_dialog_args *);

void do_zone_dialog (struct game *game, int pid, int znum, int quantity)
{
    struct zone *zone = game->player[pid]->zone[znum];
    do_qzone_dialog (game, pid, znum, quantity,
		     ((zone->flags & ZONE_PRIVATE) != 0));
}

void do_qzone_dialog (struct game *game, int pid, int znum, int quantity,
		      int private)
{
    struct player *me;
    struct zone *zone;
    char title[256];
    GtkWidget *dialog, *table, *scrollbox, *clist, *button, *arrow;
    struct zone_dialog_args *args;
    const char *zone_names[] = { "Hand", "Graveyard", "Library",
				 "Removed Cards", "Sideboard" };

    me = game->player[pid];
    zone = me->zone[znum];
    if (zone == NULL)
	return;
    if (zone->dialog != NULL) {
	gtk_window_present (GTK_WINDOW (zone->dialog));
	return;
    }
    if (pid == game->local_player) {
	strcpy (title, "Viewing My ");
	strcat (title, zone_names[znum]);
    } else {
	strcpy (title, "Viewing ");
	strcat (title, me->name);
	strcat (title, "'s ");
	strcat (title, zone_names[znum]);
    }
    if (quantity > 0)
	sprintf (title+strlen (title), " (%d)", quantity);
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    game_make_transient (game, dialog);

    args = malloc (sizeof (struct zone_dialog_args));
    args->dialog = dialog;
    args->shuffle_box = NULL;
    args->game = game;
    args->pid = pid;
    args->znum = znum;
    args->private = private;

    gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
			GTK_SIGNAL_FUNC (zone_dialog_delete), args);
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    table = gtk_table_new (3, 4, FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (table), 10);
    gtk_container_add (GTK_CONTAINER (dialog), table);

    scrollbox = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollbox),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize (GTK_WIDGET (scrollbox), 210, 250);
    clist = zone_create_clist (zone, quantity);
    gtk_container_add (GTK_CONTAINER (scrollbox), clist);
    gtk_table_attach (GTK_TABLE (table), scrollbox, 0, 2, 0, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
    if ((pid == game->local_player) && (quantity == 0) &&
	(zone->flags & ZONE_SHUFFLE_CLOSE) != 0) {
	button = gtk_check_button_new_with_label ("Shuffle When Done");
	args->shuffle_box = button;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
	align_and_attach (table, button, 0, 1, 3, 4);
    }
    if ((pid == game->local_player) && (quantity > 0)) {
	button = new_wide_button ("Move All to Bottom", 6);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (do_move_all_to_bottom), args);
	align_and_attach (table, button, 0, 1, 3, 4);
    }
    button = new_wide_button ("View", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_view_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 0, 1,
		      0, 0, 0, 0);
    arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), arrow);
    args->up_button = button;
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_zone_move_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 1, 2,
		      0, 0, 0, 0);
    arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), arrow);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_zone_move_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 2, 3,
		      0, 0, 0, 0);
    button = new_wide_button ("Ok", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (zone_dialog_ok), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 3, 4,
		      0, 0, 0, 0);

    gtk_table_set_row_spacings (GTK_TABLE (table), 2);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    zone->dialog = dialog;
    gtk_widget_show_all (dialog);

    if (args->private) {
	game_inform_peeking (game, game->local_player, pid, znum, quantity, 1);
	send_inform_peeking (game, pid, znum, quantity, 1);
    }
}

void align_and_attach (GtkWidget *table, GtkWidget *child,
		       int left, int right, int top, int bottom)
{
    GtkWidget *w;

    w = gtk_alignment_new (0, 0, 0, 0);
    gtk_container_add (GTK_CONTAINER (w), child);
    gtk_table_attach (GTK_TABLE (table), w, left, right, top, bottom,
		      GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

GtkWidget *new_wide_button (const char *text, int pad)
{
    GtkWidget *button, *label;

    button = gtk_button_new ();
    label = gtk_label_new (text);
    gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0.5);
    gtk_misc_set_padding (GTK_MISC (label), pad, 1);
    gtk_container_add (GTK_CONTAINER (button), label);
    return (button);
}

static gint zone_dialog_delete (GtkWidget *w, GdkEvent *event,
				struct zone_dialog_args *args)
{
    close_zone_dialog (args, FALSE);
    return (FALSE);
}

static void zone_dialog_ok (GtkWidget *w, struct zone_dialog_args *args)
{
    close_zone_dialog (args, TRUE);
}

static void close_zone_dialog (struct zone_dialog_args *args, int do_close)
{
    struct game *game = args->game;
    struct zone *zone;
    int qty, do_shuffle;

    zone = game->player[args->pid]->zone[args->znum];
    if (args->private) {
	qty = zone->top_cards_requested;
	game_inform_peeking (game, game->local_player, args->pid,
			     args->znum, qty, 0);
	send_inform_peeking (game, args->pid, args->znum, qty, 0);
    }
    zone->list = NULL;
    zone->dialog = NULL;
    do_shuffle = FALSE;
    if (do_close) {
	if (args->shuffle_box != NULL)
	    do_shuffle = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
						       (args->shuffle_box));
	gtk_widget_destroy (args->dialog);
    }
    free (args);
    if (do_shuffle) {
	/* Let's just assume this zone is the local player's library. */
	local_player_shuffle (game);
    }
}

static void do_view_card (GtkWidget *w, struct zone_dialog_args *args)
{
    view_card_selected_in_zone_list (args->game, args->pid, args->znum);
}

void view_card_selected_in_zone_list (struct game *game, int pid, int znum)
{
    struct zone *zone;
    int idx;

    zone = game->player[pid]->zone[znum];
    if (zone->list == NULL)
	return;
    idx = gtk_clist_get_selected_row (GTK_CLIST (zone->list));
    if (idx < 0)
	return;
    if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	idx = zone->size - 1 - idx;
    do_view_card_dialog (game, zone->cardlist[idx]);
}

int gtk_clist_get_selected_row (GtkCList *clist)
{
    GList *selection;

    selection = clist->selection;
    if (selection == NULL)
	return -1;
    if (selection->next != NULL)
	return -1;
    return GPOINTER_TO_INT (selection->data);
}

static void do_zone_move_card (GtkWidget *w, struct zone_dialog_args *args)
{
    struct game *game = args->game;
    struct zone *zone;
    int idx, other, cid, rows;

    if (args->pid != game->local_player)
	return;
    zone = game->player[args->pid]->zone[args->znum];
    if (zone->list == NULL)
	return;
    idx = gtk_clist_get_selected_row (GTK_CLIST (zone->list));
    if (idx < 0)
	return;
    rows = GTK_CLIST (zone->list)->rows;
    other = idx + (w == args->up_button ? -1 : +1);
    if ((other < 0) || (other >= zone->size) || (other >= rows))
	return;
    if ((zone->flags & ZONE_DISPLAY_HAND) == 0) {
	idx = zone->size - 1 - idx;
	other = zone->size - 1 - other;
    }
    cid = zone_move_card (zone, idx, other);

    game_rearrange_message (game, game->local_player, args->pid, args->znum);
    send_card_order_in_zone (game, cid, args->pid, args->znum, other);
}

static void
do_move_all_to_bottom (GtkWidget *w, struct zone_dialog_args *args)
{
    struct game *game = args->game;
    struct zone *zone;
    int idx, other, cid, rows;

    if (args->pid != game->local_player)
	return;
    zone = game->player[args->pid]->zone[args->znum];
    if ((zone->list == NULL) || (zone->top_cards_requested == 0))
	return;
    for (rows = GTK_CLIST (zone->list)->rows; rows > 0; rows--) {
	idx = 0;
	other = zone->size-1;
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0) {
	    idx = zone->size - 1 - idx;
	    other = zone->size - 1 - other;
	}
	cid = zone_move_card (zone, idx, other);
	game_rearrange_message (game, game->local_player,
				args->pid, args->znum);
	send_card_order_in_zone (game, cid, args->pid, args->znum, other);
    }
}

/* ----------------------------------------------------------------- */

struct create_args {
    GtkWidget *dialog;
    GtkWidget *name_entry;
    GtkWidget *pow_entry;
    GtkWidget *color_menu;
    struct game *game;
};

static void create_card_ok (GtkWidget *dialog, struct create_args *args);
static void create_card_cancel (GtkWidget *dialog, struct create_args *args);

static char *create_card_name;
static char *create_card_power;
static int create_card_color;

void do_create_card_dialog (struct game *game)
{
    GtkWidget *table, *label, *button;
    GtkDialog *dp;
    struct create_args *args;
    GList *items;
    int i;

    args = malloc (sizeof (struct create_args));
    args->dialog = gtk_dialog_new ();
    dp = GTK_DIALOG (args->dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dp->vbox), 10);
    gtk_window_set_title (GTK_WINDOW (args->dialog), "Create Card");
    game_make_transient (game, args->dialog);

    table = gtk_table_new (2, 3, FALSE);

    label = gtk_label_new ("Name:");
    align_and_attach (table, label, 0, 1, 0, 1);
    label = gtk_label_new ("Power/Toughness:");
    align_and_attach (table, label, 0, 1, 1, 2);
    label = gtk_label_new ("Color:");
    align_and_attach (table, label, 0, 1, 2, 3);
    gtk_container_add (GTK_CONTAINER (dp->vbox), table);

    args->name_entry = gtk_entry_new ();
    align_and_attach (table, args->name_entry, 1, 2, 0, 1);
    args->pow_entry = gtk_entry_new ();
    align_and_attach (table, args->pow_entry, 1, 2, 1, 2);
    args->color_menu = gtk_combo_new ();
    items = NULL;
    for (i = 0; color_names[i] != NULL; i++)
	items = g_list_append (items, color_names[i]);
    gtk_combo_set_popdown_strings (GTK_COMBO (args->color_menu), items);
    align_and_attach (table, args->color_menu, 1, 2, 2, 3);
    args->game = game;

    if (create_card_name != NULL) {
	gtk_entry_set_text (GTK_ENTRY (args->name_entry), create_card_name);
	gtk_editable_select_region (GTK_EDITABLE (args->name_entry), 0,
				    strlen(create_card_name));
    }
    gtk_widget_grab_focus (GTK_WIDGET (args->name_entry));
    if (create_card_power != NULL)
	gtk_entry_set_text (GTK_ENTRY (args->pow_entry), create_card_power);
    if (create_card_color > 0)
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (args->color_menu)->entry),
			    color_names[create_card_color-1]);

    button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (create_card_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), button, TRUE, TRUE, 15);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (create_card_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), button, TRUE, TRUE, 15);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 25);

    gtk_table_set_row_spacings (GTK_TABLE (table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    gtk_widget_show_all (args->dialog);
}

static void create_card_ok (GtkWidget *dialog, struct create_args *args)
{
    struct game *game = args->game;
    char *name, *power, *ctext;
    int color, cid, znum;
    GtkCombo *combo;
    struct deck *deck;

    name = gtk_editable_get_chars (GTK_EDITABLE (args->name_entry), 0, -1);
    power = gtk_editable_get_chars (GTK_EDITABLE (args->pow_entry), 0, -1);
    combo = GTK_COMBO (args->color_menu);
    ctext = gtk_editable_get_chars (GTK_EDITABLE (combo->entry), 0, -1);
    color = color_name_to_number (ctext);
    g_free (ctext);

    deck = game->player[game->local_player]->deck;
    znum = HAND_ZONE;
    cid = game_create_card (game, game->local_player, name, deck->tk_size,
			    znum, power, color);
    send_create_card (game, name, cid, znum, power, color);

    if (create_card_name != NULL)
	g_free (create_card_name);
    create_card_name = name;
    if (create_card_power != NULL)
	g_free (create_card_power);
    create_card_power = power;
    create_card_color = color;

    gtk_widget_destroy (args->dialog);
    free (args);
}

static void create_card_cancel (GtkWidget *dialog, struct create_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static void clear_view_card_dialog (GtkWidget *w, struct game *game);

/**
 * Display the given card in the game's View Card dialog.
 * Be sure to note when the window is created and destroyed.
 */
void
do_view_card_dialog (struct game *game, int cid)
{
    do_view_card_dialog_from_cardinfo (game, game_get_cardinfo (game, cid));
}

void
do_view_card_dialog_from_cardinfo (struct game *game, struct cardinfo *info)
{
    GtkWidget *dialog;

    if (get_int_preference (game->prefs, "view_card_windows", 0) != 0) {
	dialog = viewcard_new_window ();
    } else {
	if (game->view_card_dialog == NULL) {
	    game->view_card_dialog = viewcard_new_window ();
	    gtk_signal_connect (GTK_OBJECT (game->view_card_dialog),
				"destroy",
				GTK_SIGNAL_FUNC (clear_view_card_dialog),
				game);
	}
	dialog = game->view_card_dialog;
    }
    game_make_transient (game, dialog);
    viewcard_display_cardinfo (dialog, info);
}

static void
clear_view_card_dialog (GtkWidget *w, struct game *game)
{
    game->view_card_dialog = NULL;
}

/* ----------------------------------------------------------------- */

struct preferences_args {
    struct game *game;
    GtkWidget *dialog;
    GtkWidget *download_button;
    GtkWidget *picture_button;
    GtkWidget *name_button;
    GtkWidget *both_button;
    GtkWidget *zoom_entry;
    GtkWidget *image_entry;
    GtkWidget *tile_button;
    GtkWidget *center_button;
};

static void preferences_apply (GtkWidget *w, struct preferences_args *args);
static void preferences_ok (GtkWidget *w, struct preferences_args *args);
static void preferences_cancel (GtkWidget *w, struct preferences_args *args);
static void image_browse (GtkWidget *w, struct preferences_args *args);
static void image_browse_ok (GtkWidget *dialog, gpointer data);
#define is_button_active(x) \
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(x))
#define toggle_button_set_active(x, val) \
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(x), val)

void do_preferences_dialog (struct game *game)
{
    struct preferences_args *args;
    int do_download, pic_type, background_type;
    GtkWidget *items, *radios, *button, *zoombox, *label;
    GtkWidget *frame, *vbox, *hbox, *alignment;
    GtkDialog *dp;
    char *text, *background;
    GSList *group;

    args = malloc (sizeof (struct preferences_args));
    args->game = game;

    args->dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (args->dialog), "Preferences");
    game_make_transient (game, args->dialog);
    dp = GTK_DIALOG (args->dialog);
    gtk_container_set_border_width (GTK_CONTAINER (args->dialog), 10);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);

    /* frame = gtk_frame_new ("Card Display"); */
    items = gtk_vbox_new (FALSE, 10);
    gtk_container_set_border_width (GTK_CONTAINER (items), 6);

    text = "Download card pictures from www.wizards.com";
    args->download_button = gtk_check_button_new_with_label (text);
    gtk_container_add (GTK_CONTAINER (items), args->download_button);

    radios = gtk_vbox_new (FALSE, 0);
    text = "Display only card picture";
    args->picture_button = gtk_radio_button_new_with_label (NULL, text);
    gtk_container_add (GTK_CONTAINER (radios), args->picture_button);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->picture_button));
    text = "Display only card name";
    args->name_button = gtk_radio_button_new_with_label (group, text);
    gtk_container_add (GTK_CONTAINER (radios), args->name_button);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->name_button));
    text = "Display card picture and name";
    args->both_button = gtk_radio_button_new_with_label (group, text);
    gtk_container_add (GTK_CONTAINER (radios), args->both_button);
    gtk_container_add (GTK_CONTAINER (items), radios);

    zoombox = gtk_hbox_new (FALSE, 0);
    text = "Zoom ratio:";
    label = gtk_label_new (text);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
    gtk_box_pack_start (GTK_BOX (zoombox), label, TRUE, TRUE, 5);
    args->zoom_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (zoombox), args->zoom_entry, FALSE, FALSE, 5);
    gtk_container_add (GTK_CONTAINER (items), zoombox);

    gtk_container_add (GTK_CONTAINER (dp->vbox), items);

    do_download = get_int_preference (game->prefs, download_cardart_key, 0);
    if (do_download) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(args->download_button),
				      TRUE);
    }

    pic_type = get_int_preference (game->prefs, card_pic_type_key,
				   PLAYAREA_ART_ONLY);
    switch (pic_type) {
    case PLAYAREA_ART_ONLY:
	toggle_button_set_active (args->picture_button, TRUE);
	break;
    case PLAYAREA_NAME_ONLY:
	toggle_button_set_active (args->name_button, TRUE);
	break;
    case PLAYAREA_ART_AND_NAME:
	toggle_button_set_active (args->both_button, TRUE);
	break;
    }

    text = get_preference (game->prefs, zoom_ratio_key);
    if (text == NULL)
	text = "1";
    gtk_entry_set_text (GTK_ENTRY (args->zoom_entry), text);

    frame = gtk_frame_new ("Play Area");
    vbox = gtk_vbox_new (FALSE, 6);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
    hbox = gtk_hbox_new (FALSE, 12);
    label = gtk_label_new ("Image:");
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    args->image_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), args->image_entry, TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (vbox), hbox);
    hbox = gtk_hbox_new (TRUE, 0);
    args->tile_button = gtk_radio_button_new_with_label (NULL, "Tile");
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->tile_button));
    alignment = gtk_alignment_new (0.75, 0.5, 0, 0);
    gtk_container_add (GTK_CONTAINER (alignment), args->tile_button);
    gtk_container_add (GTK_CONTAINER (hbox), alignment);
    args->center_button = gtk_radio_button_new_with_label (group, "Center");
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->center_button));
    alignment = gtk_alignment_new (0.25, 0.5, 0, 0);
    gtk_container_add (GTK_CONTAINER (alignment), args->center_button);
    gtk_container_add (GTK_CONTAINER (hbox), alignment);
    button = gtk_button_new_with_label ("Browse...");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (image_browse), args);
    gtk_container_add (GTK_CONTAINER (hbox), button);
    gtk_container_add (GTK_CONTAINER (vbox), hbox);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_container_add (GTK_CONTAINER (dp->vbox), frame);

    background = get_preference (game->prefs, background_key);
    if (background != NULL)
	gtk_entry_set_text (GTK_ENTRY (args->image_entry), background);
    background_type = get_int_preference (game->prefs, background_type_key, 0);
    switch (background_type) {
    case PLAYAREA_TILE:
	toggle_button_set_active (args->tile_button, TRUE);
	break;
    case PLAYAREA_CENTER:
	toggle_button_set_active (args->center_button, TRUE);
	break;
    }

    button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_ok), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);
    button = gtk_button_new_with_label ("Apply");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_apply), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_cancel), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);

    gtk_widget_show_all (args->dialog);
}

static void preferences_apply (GtkWidget *w, struct preferences_args *args)
{
    struct game *game = args->game;
    int option;
    const gchar *text;
    float ratio;

    option = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					   (args->download_button));
    set_int_preference (game->prefs, download_cardart_key, option);
    cardart_initialize (game->prefs);

    option = -1;
    if (is_button_active (args->picture_button))
	option = PLAYAREA_ART_ONLY;
    else if (is_button_active (args->name_button))
	option = PLAYAREA_NAME_ONLY;
    else if (is_button_active (args->both_button))
	option = PLAYAREA_ART_AND_NAME;
    if (option >= 0)
	set_int_preference (game->prefs, card_pic_type_key, option);

    text = gtk_entry_get_text (GTK_ENTRY (args->zoom_entry));
    ratio = strtod (text, NULL);
    if (ratio > 0)
	set_float_preference (game->prefs, zoom_ratio_key, ratio);

    text = gtk_entry_get_text (GTK_ENTRY (args->image_entry));
    if ((text == NULL) || (*text == 0)) {
	clear_preference (game->prefs, background_key);
    } else {
	set_preference (game->prefs, background_key, text);
    }
    option = -1;
    if (is_button_active (args->tile_button))
	option = PLAYAREA_TILE;
    else if (is_button_active (args->center_button))
	option = PLAYAREA_CENTER;
    if (option >= 0)
	set_int_preference (game->prefs, background_type_key, option);

    playarea_refresh_prefs (game->playarea);
    playarea_repaint (game->playarea);
}

static void preferences_ok (GtkWidget *w, struct preferences_args *args)
{
    preferences_apply (w, args);
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void preferences_cancel (GtkWidget *w, struct preferences_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void image_browse (GtkWidget *w, struct preferences_args *args)
{
    GtkWidget *filew;
    GtkFileSelection *fsp;

    filew = gtk_file_selection_new ("Background Image");
    gtk_object_set_user_data (GTK_OBJECT (filew), args);
    fsp = GTK_FILE_SELECTION (filew);
    gtk_signal_connect_object (GTK_OBJECT (fsp->ok_button), "clicked",
			       GTK_SIGNAL_FUNC (image_browse_ok),
			       GTK_OBJECT (filew));
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (filew));
    gtk_widget_show (filew);
}

static void image_browse_ok (GtkWidget *dialog, gpointer data)
{
    struct preferences_args *args;
    const gchar *filename;

    args = gtk_object_get_user_data (GTK_OBJECT (dialog));
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename == NULL)
	return;
    gtk_entry_set_text (GTK_ENTRY (args->image_entry), filename);
    gtk_widget_destroy (dialog);
}

/* ----------------------------------------------------------------- */

struct sblist_line {
    int qty;
    int globalid;
};

struct sb_list {
    struct game *game;
    GtkWidget *scrollbox;
    GtkWidget *clist;
    GtkWidget *button;
    GtkWidget *card_count;
    int num_lines;
    int num_cards;
    struct sblist_line *lineinfo;
};

struct sideboard_args {
    struct game *game;
    GtkWidget *dialog;
    struct sb_list deck_list;
    struct sb_list sb_list;
    int changed;
};

static void create_sb_list (struct sb_list *, int, int *, struct game *);
static void sideboard_select_row (GtkCList *, gint, gint, GdkEventButton *,
				  struct sb_list *);
static void sideboard_trade (GtkWidget *, struct sideboard_args *);
static void sideboard_add_to_deck (GtkWidget *, struct sideboard_args *);
static void sideboard_remove_from_deck (GtkWidget *, struct sideboard_args *);
static void sideboard_save_deck (GtkWidget *, struct sideboard_args *);
static void decrement_line (struct sb_list *listp, int idx);
static void increment_globalid (struct sb_list *, int, struct cardbase *);
static void sideboard_dialog_ok (GtkWidget *, struct sideboard_args *);
struct deck *sideboard_args_to_deck (struct sideboard_args *args);
static void sideboard_dialog_cancel (GtkWidget *, struct sideboard_args *);

/**
 * Creating is true if we are creating the deck (that is, we can freely
 * add/remove cards from it).
 */
void
do_sideboard_dialog (struct game *game, int creating)
{
    struct deck *deck;
    struct sideboard_args *args;
    GtkWidget *hbox, *vbox, *button, *add_button, *remove_button, *card_count;
    char *title;

    deck = game->player[game->local_player]->deck;
    if (deck == NULL)
	return;
    args = malloc (sizeof (struct sideboard_args));
    args->game = game;
    args->dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_transient_for (GTK_WINDOW (args->dialog),
				  GTK_WINDOW (game->new_game_dialog));

    title = (creating ? "Create Deck" : "Sideboard");
    gtk_window_set_title (GTK_WINDOW (args->dialog), title);

    gtk_container_set_border_width (GTK_CONTAINER (args->dialog), 10);
    hbox = gtk_hbox_new (FALSE, 10);
    create_sb_list (&args->deck_list, deck->size, deck->cardlist, game);
    gtk_box_pack_start (GTK_BOX (hbox), args->deck_list.scrollbox,
			FALSE, FALSE, 0);
    vbox = gtk_vbox_new (FALSE, 10);
    button = new_wide_button ("Trade", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_trade), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, FALSE, 0);
    card_count = add_button = remove_button = NULL;
    if (creating)
    {
	card_count = gtk_label_new ("Deck:\n0 cards");
	gtk_box_pack_start (GTK_BOX (vbox), card_count, TRUE, FALSE, 0);
	add_button = new_wide_button ("<<", 6);
	gtk_signal_connect (GTK_OBJECT (add_button), "clicked",
			    GTK_SIGNAL_FUNC (sideboard_add_to_deck),
			    args);
	gtk_box_pack_start (GTK_BOX (vbox), add_button, TRUE, FALSE, 0);
	remove_button = new_wide_button (">>", 6);
	gtk_signal_connect (GTK_OBJECT (remove_button), "clicked",
			    GTK_SIGNAL_FUNC (sideboard_remove_from_deck),
			    args);
	gtk_box_pack_start (GTK_BOX (vbox), remove_button, TRUE, FALSE, 0);
	button = new_wide_button ("Save deck", 6);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (sideboard_save_deck), args);
	gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, FALSE, 0);
    }
    button = new_wide_button ("Ok", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_dialog_ok), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
    button = new_wide_button ("Cancel", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_dialog_cancel), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
    create_sb_list (&args->sb_list, deck->sb_size, deck->sb_cardlist, game);
    gtk_box_pack_start (GTK_BOX (hbox), args->sb_list.scrollbox,
			FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (args->dialog), hbox);
    args->changed = FALSE;
    args->deck_list.card_count = card_count;
    args->deck_list.button = remove_button;
    args->sb_list.button = add_button;
    gtk_widget_show_all (args->dialog);
    if (creating)
    {
	display_message (game, "* %s drafted a Sealed Deck game.",
			 game->player[game->local_player]->name);
	send_drafted_sealed (game);
    }
    else
    {
	display_message (game, "+ %s is sideboarding.",
			 game->player[game->local_player]->name);
	send_inform_sideboard (game);
    }
}

static void
create_sb_list (struct sb_list *listp, int size, int *cardlist,
		struct game *game)
{
    int globalid, count;
    char buf[24], *sarr[2];
    struct cardinfo *info;

    listp->game = game;
    listp->scrollbox = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (listp->scrollbox),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize (GTK_WIDGET (listp->scrollbox), 210, 250);
    listp->clist = gtk_clist_new (2);
    gtk_container_add (GTK_CONTAINER (listp->scrollbox), listp->clist);

    gtk_clist_set_selection_mode (GTK_CLIST (listp->clist),
				  GTK_SELECTION_SINGLE);
    gtk_clist_set_column_width (GTK_CLIST (listp->clist), 0, 15);
    gtk_clist_set_column_width (GTK_CLIST (listp->clist), 1, 0);
    gtk_clist_set_button_actions (GTK_CLIST (listp->clist),
				  2, GTK_BUTTON_SELECTS);

    listp->button = NULL;
    listp->card_count = NULL;
    listp->num_lines = 0;
    listp->num_cards = 0;
    listp->lineinfo = malloc (sizeof (struct sblist_line) * size);
    while (listp->num_cards < size) {
	globalid = cardlist[listp->num_cards];
	count = 1;
	while (cardlist[listp->num_cards+count] == globalid)
	    count++;
	info = cardbase_get_card (game->cardbase, globalid);
	sprintf (buf, "%d", count);
	sarr[0] = buf;
	sarr[1] = (info == NULL ? "" : info->name);
	gtk_clist_append (GTK_CLIST (listp->clist), sarr);
	listp->lineinfo[listp->num_lines].qty = count;
	listp->lineinfo[listp->num_lines].globalid = globalid;
	listp->num_lines++;
	listp->num_cards += count;
    }

    gtk_signal_connect (GTK_OBJECT (listp->clist), "select-row",
			GTK_SIGNAL_FUNC (sideboard_select_row), listp);
    gtk_signal_connect (GTK_OBJECT (listp->clist), "unselect-row",
			GTK_SIGNAL_FUNC (sideboard_select_row), listp);
}

static void
sideboard_select_row (GtkCList *clist, gint row, gint column,
		      GdkEventButton *event, struct sb_list *listp)
{
    struct game *game = listp->game;
    int rnum, cnum, globalid;
    struct cardinfo *info;

    if ((event == NULL) ||
	(!gtk_clist_get_selection_info (clist, event->x, event->y,
					&rnum, &cnum)) ||
	(rnum != row))
	return;
    if (event->type == GDK_2BUTTON_PRESS && event->button == 1)
    {
	if (listp->button != NULL)
	    gtk_button_clicked (GTK_BUTTON (listp->button));
	return;
    }
    if (event->type == GDK_BUTTON_RELEASE && event->button == 3)
    {
	globalid = listp->lineinfo[row].globalid;
	info = cardbase_get_card (game->cardbase, globalid);
	do_view_card_dialog_from_cardinfo (game, info);
    }
}

static void
sideboard_trade (GtkWidget *w, struct sideboard_args *args)
{
    int idx, sb_idx, left_gid, right_gid;

    idx = gtk_clist_get_selected_row (GTK_CLIST (args->deck_list.clist));
    if (idx < 0)
	return;
    sb_idx = gtk_clist_get_selected_row (GTK_CLIST (args->sb_list.clist));
    if (sb_idx < 0)
	return;
    left_gid = args->deck_list.lineinfo[idx].globalid;
    right_gid = args->sb_list.lineinfo[sb_idx].globalid;
    decrement_line (&args->deck_list, idx);
    decrement_line (&args->sb_list, sb_idx);
    increment_globalid (&args->deck_list, right_gid,
			args->game->cardbase);
    increment_globalid (&args->sb_list, left_gid,
			args->game->cardbase);
    args->changed = TRUE;
}

static void
sideboard_add_to_deck (GtkWidget *w, struct sideboard_args *args)
{
    int sb_idx, right_gid;
    sb_idx = gtk_clist_get_selected_row (GTK_CLIST (args->sb_list.clist));
    if (sb_idx < 0)
	return;
    right_gid = args->sb_list.lineinfo[sb_idx].globalid;
    decrement_line (&args->sb_list, sb_idx);
    increment_globalid (&args->deck_list, right_gid,
			args->game->cardbase);
    args->changed = TRUE;
}

static void
sideboard_remove_from_deck (GtkWidget *w, struct sideboard_args *args)
{
    int idx, left_gid;
    idx = gtk_clist_get_selected_row (GTK_CLIST (args->deck_list.clist));
    if (idx < 0)
	return;
    left_gid = args->deck_list.lineinfo[idx].globalid;
    decrement_line (&args->deck_list, idx);
    increment_globalid (&args->sb_list, left_gid,
			args->game->cardbase);
    args->changed = TRUE;
}

static void
decrement_line (struct sb_list *listp, int idx)
{
    char buf[24];
    int i;

    listp->lineinfo[idx].qty--;
    listp->num_cards--;
    if (listp->card_count != NULL) {
	sprintf (buf, "Deck:\n%d cards", listp->num_cards);
	gtk_label_set_text (GTK_LABEL(listp->card_count), buf);
    }
    if (listp->lineinfo[idx].qty > 0) {
	sprintf (buf, "%d", listp->lineinfo[idx].qty);
	gtk_clist_set_text (GTK_CLIST (listp->clist), idx, 0, buf);
	return;
    }
    for (i = idx; i < listp->num_lines; i++)
	listp->lineinfo[i] = listp->lineinfo[i+1];
    listp->num_lines--;
    gtk_clist_remove (GTK_CLIST (listp->clist), idx);
}

static void
increment_globalid (struct sb_list *listp, int globalid,
		    struct cardbase *cardbase)
{
    int idx;
    char buf[24], *sarr[2];
    struct cardinfo *info;

    listp->num_cards++;
    if (listp->card_count != NULL) {
	sprintf (buf, "Deck:\n%d cards", listp->num_cards);
	gtk_label_set_text (GTK_LABEL(listp->card_count), buf);
    }
    for (idx = 0; idx < listp->num_lines; idx++) {
	if (listp->lineinfo[idx].globalid == globalid) {
	    listp->lineinfo[idx].qty++;
	    sprintf (buf, "%d", listp->lineinfo[idx].qty);
	    gtk_clist_set_text (GTK_CLIST (listp->clist), idx, 0, buf);
	    return;
	}
    }
    /* Add a new line to the list. */
    idx = listp->num_lines;
    listp->num_lines++;
    listp->lineinfo = realloc (listp->lineinfo, sizeof (struct sblist_line) *
			       listp->num_lines);
    listp->lineinfo[idx].qty = 1;
    listp->lineinfo[idx].globalid = globalid;
    info = cardbase_get_card (cardbase, globalid);
    sprintf (buf, "%d", listp->lineinfo[idx].qty);
    sarr[0] = buf;
    sarr[1] = (info == NULL ? "" : info->name);
    gtk_clist_append (GTK_CLIST (listp->clist), sarr);
}

#if (GTK_MAJOR_VERSION>2) || (GTK_MAJOR_VERSION==2&&GTK_MINOR_VERSION>2)
static void do_save_deck (GtkDialog *, gint, struct sideboard_args *);

static void
sideboard_save_deck (GtkWidget *w, struct sideboard_args *args)
{
    GtkWidget *dialog;
    GtkFileFilter *filter;

    dialog = gtk_file_chooser_dialog_new ("Save Deck",
			GTK_WINDOW (args->dialog),
			GTK_FILE_CHOOSER_ACTION_SAVE,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT,
			NULL);
    gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    filter = gtk_file_filter_new ();
    gtk_file_filter_add_pattern (filter, "*.dec");
    gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
    gtk_signal_connect (GTK_OBJECT (dialog), "response",
			GTK_SIGNAL_FUNC (do_save_deck), args);
    gtk_widget_show_all (dialog);
}

static void
do_save_deck (GtkDialog *dialog, gint arg, struct sideboard_args *args)
{
    char *filename;
    struct deck *deck;

    switch (arg) {
    case GTK_RESPONSE_ACCEPT:
	filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	deck = sideboard_args_to_deck (args);
	deck_write_file (filename, deck, args->game->cardbase);
	deck_destroy (deck);
	g_free (filename);
	/* fallthrough */
    case GTK_RESPONSE_CANCEL:
	gtk_widget_destroy (GTK_WIDGET (dialog));
	break;
    }
}
#else
static void do_save_deck (GtkWidget *w, GtkWidget *dialog);

static void
sideboard_save_deck (GtkWidget *w, struct sideboard_args *args)
{
    GtkWidget *dialog;
    GtkFileSelection *fsp;

    dialog = gtk_file_selection_new ("Save Deck");
    gtk_window_set_transient_for (GTK_WINDOW (dialog),
				  GTK_WINDOW (args->dialog));
    gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    gtk_object_set_user_data (GTK_OBJECT (dialog), args);
    fsp = GTK_FILE_SELECTION (dialog);
    gtk_signal_connect (GTK_OBJECT (fsp->ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_save_deck), dialog);
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_widget_show_all (dialog);
}

static void
do_save_deck (GtkWidget *w, GtkWidget *dialog)
{
    const gchar *filename;
    struct sideboard_args *args;
    struct deck *deck;

    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename == NULL)
	return;
    args = gtk_object_get_user_data (GTK_OBJECT (dialog));
    deck = sideboard_args_to_deck (args);
    deck_write_file (filename, deck, args->game->cardbase);
    deck_destroy (deck);
    gtk_widget_destroy (dialog);
}
#endif

static void
sideboard_dialog_ok (GtkWidget *w, struct sideboard_args *args)
{
    struct game *game;
    struct player *me;

    if (args->changed) {
	game = args->game;
	if (game->new_game_dialog == NULL) {
	    do_message_dialog ("You may not modify your deck "
			       "during a game.");
	    return;
	}
	me = game->player[game->local_player];
	if (me->deck != NULL)
	    deck_destroy (me->deck);
	me->deck = sideboard_args_to_deck (args);
    }
    sideboard_dialog_cancel (w, args);
}

struct deck *
sideboard_args_to_deck (struct sideboard_args *args)
{
    struct deck *deck;
    int idx;

    deck = deck_new ();
    for (idx = 0; idx < args->deck_list.num_lines; idx++) {
	deck_set_card (deck, deck->size,
		       args->deck_list.lineinfo[idx].qty,
		       args->deck_list.lineinfo[idx].
		       globalid);
    }
    for (idx = 0; idx < args->sb_list.num_lines; idx++) {
	deck_set_sb_card (deck, deck->sb_size,
			  args->sb_list.lineinfo[idx].qty,
			  args->sb_list.lineinfo[idx].
			  globalid);
    }
    return (deck);
}

static void
sideboard_dialog_cancel (GtkWidget *w, struct sideboard_args *args)
{
    free (args->deck_list.lineinfo);
    free (args->sb_list.lineinfo);
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static void finish_find_cardinfo (GtkWidget *, const char *, struct game *);
static int is_ascii_file (const char *filename);

#if (GTK_MAJOR_VERSION>2) || (GTK_MAJOR_VERSION==2&&GTK_MINOR_VERSION>2)
static void
find_cardinfo_response (GtkDialog *dialog, gint arg, struct game *game)
{
    const char *filename;
    switch (arg) {
    case GTK_RESPONSE_ACCEPT:
	filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	finish_find_cardinfo (GTK_WIDGET (dialog), filename, game);
	break;
    case GTK_RESPONSE_CANCEL:
	gtk_widget_destroy (GTK_WIDGET (dialog));
	break;
    }
}

void
do_find_cardinfo_dialog (struct game *game)
{
    GtkWidget *dialog;

    dialog = gtk_file_chooser_dialog_new ("Load Magic Card Database",
			GTK_WINDOW (game->main_window),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);
    gtk_signal_connect (GTK_OBJECT (dialog), "response",
			GTK_SIGNAL_FUNC (find_cardinfo_response), game);
    gtk_widget_show_all (dialog);
}
#else
static void
find_cardinfo_ok (GtkWidget *w, GtkWidget *dialog)
{
    const char *filename;
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename != NULL)
	finish_find_cardinfo (dialog, filename,
			  gtk_object_get_user_data (GTK_OBJECT (dialog)));
}

void
do_find_cardinfo_dialog (struct game *game)
{
    GtkWidget *dialog;
    GtkFileSelection *fsp;

    dialog = gtk_file_selection_new ("Load Magic Card Database");
    game_make_transient (game, dialog);
    gtk_object_set_user_data (GTK_OBJECT (dialog), game);
    fsp = GTK_FILE_SELECTION (dialog);
    gtk_signal_connect (GTK_OBJECT (fsp->ok_button), "clicked",
			GTK_SIGNAL_FUNC (find_cardinfo_ok), dialog);
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_widget_show (dialog);
}
#endif

static void
finish_find_cardinfo (GtkWidget *dialog, const char *filename,
		      struct game *game)
{
    char *text;
    struct cardbase *cardbase;
    int pid;
    struct player *me;

    if (game->opponent != NULL) {
	text = "Cannot load new card data while game is in progress.";
	do_message_dialog (text);
	return;
    }
    if (filename == NULL)
	return;
    cardbase = NULL;
    if (is_ascii_file (filename)) {
	cardbase = cardbase_new_from_ort (filename);
	if (cardbase == NULL) {
	    display_message (game, "%s: Failed to read oracle card database.",
			     filename);
	    return;
	}
	set_preference (game->prefs, oracle_name_key, filename);
	clear_preference (game->prefs, cardinfo_name_key);
    }
    else {
	cardbase = cardbase_new_from_dat (filename);
	if (cardbase == NULL) {
	    display_message (game, "%s: Failed to read cardinfo database.",
			     filename);
	    return;
	}
	set_preference (game->prefs, cardinfo_name_key, filename);
	clear_preference (game->prefs, oracle_name_key);
    }
    gtk_widget_destroy (dialog);

    if (game->cardbase != NULL) {
	game_empty_all_zones (game);
	for (pid = 0; pid < game->num_players; pid++) {
	    me = game->player[pid];
	    if (me->deck != NULL) {
		display_message (game, "Unloading %s's deck.", me->name);
		deck_destroy (me->deck);
		me->deck = NULL;
	    }
	}
	if (game->expansions != NULL) {
	    display_message (game, "Unloading expansions data.");
	    expansions_destroy (game->expansions);
	    game->expansions = NULL;
	}
	cardbase_destroy (game->cardbase);
    }
    game->cardbase = cardbase;
}

/**
 *  Return true if the first line of the given file contains all plain-text
 *  characters.
 */
static int is_ascii_file (const char *filename)
{
    FILE *fp;
    char buf[256];
    int i;

    fp = fopen (filename, "r");
    if (fp == NULL)
	return (FALSE);
    fgets (buf, sizeof (buf), fp);
    fclose (fp);
    for (i = 0; i < sizeof (buf); i++) {
	if (buf[i] == '\r')
	    return ((buf[i+1] == '\n') && (buf[i+2] == 0));
	if (buf[i] == '\n')
	    return (buf[i+1] == 0);
	if (!isascii (buf[i]))
	    return (FALSE);
    }
    return (FALSE);
}
