/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  mindless.c -- Create the main window and all its widgets.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "game.h"
#include "playarea.h"
#include "zone.h"
#include "deck.h"
#include "opponent.h"
#include "dialogs.h"
#include "prefs.h"
#include "random.h"
#include "version.h"
#include "images.h"

static void add_tools (GtkToolbar *toolbar);
static void add_phase_button (GtkToolbar *, char *, char *, int, const char *);
static void add_drop_tools (GtkToolbar *toolbar);
static void chdir_of_file (char *filename);
static void do_load_cardbase (GtkWidget *w, gpointer data);
static void do_new_game (GtkWidget *w, gpointer data);
static void do_disconnect (GtkWidget *w, gpointer data);
static void do_exit (GtkWidget *w, gpointer data);
static void do_draw_card (GtkWidget *w, gpointer data);
static void do_untap_cards (GtkWidget *w, gpointer data);
static void do_create_card (GtkWidget *w, gpointer data);
static void do_flip_coin (GtkWidget *w, gpointer data);
static void do_roll_die (GtkWidget *w, gpointer data);
static int do_roll_die_cb (struct game *game, gpointer p, const char *val);
static void do_shuffle (GtkWidget *w, gpointer data);
static void do_mulligan (GtkWidget *w, gpointer data);
static void do_grave_to_lib (GtkWidget *w, gpointer data);
static void do_grave_to_rem (GtkWidget *w, gpointer data);
static void do_toplib_to_grave (GtkWidget *w, gpointer data);
static void move_cards_in_zone (struct game *game, int src, int dest);
static void do_set_phase (GtkWidget *w, gpointer data);
static void do_next_turn (GtkWidget *w, gpointer data);
static void do_prev_turn (GtkWidget *w, gpointer data);
static void do_set_life (GtkWidget *w, gpointer data);
static void do_next_phase (GtkWidget *w, gpointer data);
static void do_ping_opponent (GtkWidget *w, gpointer data);
static void do_library (GtkWidget *w, gpointer data);
static void do_view_library (GtkWidget *w, gpointer data);
static void do_view_libtop (GtkWidget *w, gpointer data);
static int do_view_libtop_cb (struct game *game, gpointer p, const char *val);
static void do_graveyard (GtkWidget *w, gpointer data);
static void do_opp_graveyard (GtkWidget *w, gpointer data);
static void do_removed (GtkWidget *w, gpointer data);
static void do_opp_removed (GtkWidget *w, gpointer data);
static void do_sideboard (GtkWidget *w, gpointer data);
static void do_show_hand (GtkWidget *w, gpointer data);
static void do_show_library (GtkWidget *w, gpointer data);
static void do_show_random (GtkWidget *w, gpointer data);
static void do_show_libtop (GtkWidget *w, gpointer data);
static int do_show_libtop_cb (struct game *game, gpointer zp, const char *val);
static void do_show_zone (struct game *game, int znum, int quantity);
static void do_preferences (GtkWidget *w, gpointer data);
static void do_set_name (GtkWidget *w, gpointer data);
static void do_about_box (GtkWidget *w, gpointer data);
static int set_name_callback (struct game *game, gpointer data, const char *text);
static void do_view_card (GtkWidget *w, struct game *game);
static int entry_box_key_pressed (GtkWidget *, GdkEventKey *, gpointer);
static void set_main_window_size (struct game *game, GtkWidget *);

#if GTK_MAJOR_VERSION >= 2
#define MICON(x) "<StockItem>", x
#define KEY_EVENT "key_release_event"
#else
#define MICON(x) NULL
#define KEY_EVENT "key_press_event"
#endif
#ifndef GTK_STOCK_ABOUT
#define GTK_STOCK_ABOUT GTK_STOCK_DIALOG_INFO
#endif

/*
 *  When I get the time, I'll make this global variable go away.
 */
struct game *game;

static GtkItemFactoryEntry menu_items[] = {
    { "/_File", NULL, NULL, 0, "<Branch>" },
    { "/File/_New Game...", "", do_new_game, 0, MICON(M_NEW_ICON) },
    { "/File/_Load Card Data...", "", do_load_cardbase, 0, MICON(GTK_STOCK_OPEN) },
    { "/File/sep1", NULL, NULL, 0, "<Separator>" },
    { "/File/_Disconnect", "", do_disconnect, 0, MICON(GTK_STOCK_STOP) },
    { "/File/sep2", NULL, NULL, 0, "<Separator>" },
    { "/File/_Quit", "", do_exit, 0, MICON(GTK_STOCK_QUIT) },

    { "/_Game", NULL, NULL, 0, "<Branch>" },
    { "/Game/Set _Life...", "<control>L", do_set_life, SET_LIFE, MICON(M_LIFESET_ICON) },
    { "/Game/Next _Phase", "<control>P", do_next_phase, 0, MICON(GTK_STOCK_GO_FORWARD) },
    { "/Game/sep1", NULL, NULL, 0, "<Separator>" },
    { "/Game/Increment Life", "F12", do_set_life, INCREMENT_LIFE, MICON(M_LIFEPLUS_ICON) },
    { "/Game/Decrement Life", "F11", do_set_life, DECREMENT_LIFE, MICON(M_LIFEMINUS_ICON) },
    { "/Game/sep2", NULL, NULL, 0, "<Separator>" },
    { "/Game/Increment Turn", "<control>T", do_next_turn, 0, MICON(GTK_STOCK_GOTO_LAST) },
    { "/Game/Decrement Turn", "", do_prev_turn, 0, MICON(GTK_STOCK_GOTO_FIRST) },

    { "/_Action", NULL, NULL, 0, "<Branch>" },
    { "/Action/_Draw Card", "<control>D", do_draw_card, 0, MICON(M_DRAW_ICON) },
    { "/Action/_Untap My Permanents", "<control>U", do_untap_cards, 0, MICON(M_UNTAP_ICON) },
    { "/Action/sep1", NULL, NULL, 0, "<Separator>" },
    { "/Action/_Create Card...", "", do_create_card, 0, MICON(M_CREATE_ICON) },
    { "/Action/sep2", NULL, NULL, 0, "<Separator>" },
    { "/Action/_Flip a Coin", "<control>F", do_flip_coin, 0, MICON(M_COIN_ICON) },
    { "/Action/Roll a _Die...", "<control>I", do_roll_die, 0, MICON(M_DIE_ICON) },
    { "/Action/sep3", NULL, NULL, 0, "<Separator>" },
    { "/Action/_Shuffle Library", "<control>S", do_shuffle, 0, MICON(M_SHUFFLE_ICON) },
    { "/Action/sep4", NULL, NULL, 0, "<Separator>" },
    { "/Action/Move Hand to Library", "", do_mulligan, 0, NULL },
    { "/Action/Move Graveyard to Library", "", do_grave_to_lib, 0, NULL },
    { "/Action/Move Graveyard to Removed", "", do_grave_to_rem, 0, NULL },
    { "/Action/Move Top of Library to Graveyard", "<control>M", do_toplib_to_grave, 0, NULL },

    { "/_View", NULL, NULL, 0, "<Branch>" },
    { "/View/My _Graveyard...", "", do_graveyard, 0, MICON(GTK_STOCK_ZOOM_IN) },
    { "/View/Opponent's Graveyard...", "", do_opp_graveyard, 0, MICON(GTK_STOCK_ZOOM_FIT) },
    { "/View/sep1", NULL, NULL, 0, "<Separator>" },
    { "/View/My _Removed Cards...", "", do_removed, 0, MICON(GTK_STOCK_ZOOM_IN) },
    { "/View/Opponent's Removed Cards...", "", do_opp_removed, 0, MICON(GTK_STOCK_ZOOM_FIT) },
    { "/View/sep2", NULL, NULL, 0, "<Separator>" },
    { "/View/My _Library...", "", do_view_library, 0, MICON(GTK_STOCK_ZOOM_IN) },
    { "/View/_Top Cards of My Library...", "", do_view_libtop, 0, MICON(GTK_STOCK_ZOOM_FIT) },
    { "/View/sep3", NULL, NULL, 0, "<Separator>" },
    { "/View/My Sideboard", "", do_sideboard, 0, MICON(M_SIDEBOARD_ICON) },

    { "/_Show", NULL, NULL, 0, "<Branch>" },
    { "/Show/_My Hand", "", do_show_hand, 0, MICON(M_HAND_ICON) },
    { "/Show/My Entire Library", "", do_show_library, 0, MICON(M_DECK_ICON) },
    { "/Show/sep1", NULL, NULL, 0, "<Separator>" },
    { "/Show/_Random Card From Hand", "", do_show_random, 0, MICON(GTK_STOCK_DIALOG_QUESTION) },
    { "/Show/sep2", NULL, NULL, 0, "<Separator>" },
    { "/Show/_Top Cards of My Library...", "", do_show_libtop, 0, MICON(M_LIBTOP_ICON) },

    { "/_Options", NULL, NULL, 0, "<Branch>" },
    { "/Options/_Preferences...", "", do_preferences, 0, MICON(GTK_STOCK_PREFERENCES) },
    { "/Options/_Set Name...", "", do_set_name, 0, MICON(GTK_STOCK_PROPERTIES) },

    { "/_Help", NULL, NULL, 0, "<Branch>" },
    { "/Help/_About...", "", do_about_box, 0, MICON(GTK_STOCK_ABOUT) },
};

int main (int argc, char *argv[])
{
    GtkWidget *wp, *vbox, *menubar, *toolbar, *hand_scroll, *hand_list;
    GtkWidget *middle_box, *left_box, *right_box, *dropbar, *tabletop;
    GtkWidget *mscroll, *message_box, *entry_box;
    GtkWidget *button, *frame, *vpane, *player_vbox, *align;
    GtkItemFactory *item_factory;
    GtkAccelGroup *accel_group;
    int nmenu_items, offset, life;
    struct player *me;
    char *filename;
    static gchar *hand_titles[] = { "Cost", "Name" };

    game = game_new ();
    if (game == NULL)
	return (-1);

    initialize_random_numbers ();
    gtk_set_locale ();
    gtk_init (&argc, &argv);
    gdk_rgb_init ();
    wp = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    add_stock_images (wp);
    gtk_window_set_title (GTK_WINDOW (wp), "Mindless Automaton");
    gtk_signal_connect (GTK_OBJECT (wp), "delete_event",
			GTK_SIGNAL_FUNC (do_exit), NULL);
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (wp), vbox);

    /*
     *  Create the menubar.
     */
    nmenu_items = sizeof (menu_items) / sizeof (menu_items[0]);
    accel_group = gtk_accel_group_new ();
    item_factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>",
					 accel_group);
    gtk_item_factory_create_items (item_factory, nmenu_items, menu_items,
				   NULL);
#if GTK_MAJOR_VERSION >= 2
    gtk_window_add_accel_group( GTK_WINDOW(wp), accel_group);
#else
    gtk_accel_group_attach (accel_group, GTK_OBJECT (wp));
#endif
    menubar = gtk_item_factory_get_widget (item_factory, "<main>");
    gtk_box_pack_start (GTK_BOX (vbox), menubar, FALSE, FALSE, 0);

    /*
     *  Create the toolbar.
     */
#if GTK_MAJOR_VERSION >= 2
    toolbar = gtk_toolbar_new ();
    gtk_toolbar_set_orientation(GTK_TOOLBAR(toolbar),
				GTK_ORIENTATION_HORIZONTAL);
    add_tools (GTK_TOOLBAR (toolbar));
    gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
#else
    toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
			       GTK_TOOLBAR_TEXT);
    gtk_toolbar_set_button_relief (GTK_TOOLBAR (toolbar), GTK_RELIEF_NONE);
    add_tools (GTK_TOOLBAR (toolbar));
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), toolbar);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
#endif

    middle_box = gtk_hbox_new (FALSE, 10);
    left_box = gtk_vbox_new (FALSE, 5);
    right_box = gtk_vbox_new (FALSE, 0);

    /*
     *  Create the left side.
     */
#if GTK_MAJOR_VERSION >= 2
    dropbar = gtk_toolbar_new ();
    gtk_toolbar_set_orientation(GTK_TOOLBAR(dropbar),
				GTK_ORIENTATION_HORIZONTAL);
    add_drop_tools (GTK_TOOLBAR (dropbar));
    gtk_box_pack_start (GTK_BOX (left_box), dropbar, FALSE, FALSE, 0);
#else
    dropbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL,
			       GTK_TOOLBAR_ICONS);
    gtk_toolbar_set_button_relief (GTK_TOOLBAR (dropbar), GTK_RELIEF_NONE);
    add_drop_tools (GTK_TOOLBAR (dropbar));
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), dropbar);
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_OUT);
    gtk_box_pack_start (GTK_BOX (left_box), frame, FALSE, FALSE, 0);
#endif
    hand_scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (hand_scroll),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    hand_list = gtk_clist_new_with_titles (2, hand_titles);
    gtk_container_add (GTK_CONTAINER (hand_scroll), hand_list);
    gtk_clist_set_selection_mode (GTK_CLIST (hand_list),
				  GTK_SELECTION_SINGLE);
    gtk_clist_column_titles_passive (GTK_CLIST (hand_list));
    gtk_clist_set_column_width (GTK_CLIST (hand_list), 0, 35);
    gtk_box_pack_start (GTK_BOX (left_box), hand_scroll, TRUE, TRUE, 0);
    align = gtk_alignment_new (.5, .5, .5, 0);
    button = gtk_button_new_with_label ("View Card");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_view_card), game);
    gtk_container_add (GTK_CONTAINER (align), button);
    gtk_box_pack_start (GTK_BOX (left_box), align, FALSE, FALSE, 0);
    player_vbox = gtk_vbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (left_box), player_vbox, FALSE, FALSE, 0);

    /*
     *  Create the right side.
     */
    vpane = gtk_vpaned_new ();
    gtk_paned_set_gutter_size (GTK_PANED (vpane), 19);
    game->playarea = playarea_new (game);
    tabletop = game->playarea->darea;
    // gtk_drawing_area_size (GTK_DRAWING_AREA (tabletop), 200, 200);
    gtk_paned_pack1 (GTK_PANED (vpane), tabletop, TRUE, TRUE);
    mscroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (mscroll),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize (mscroll, 0, 80);
#if GTK_MAJOR_VERSION >= 2
    message_box = gtk_text_view_new ();
    gtk_text_view_set_editable (GTK_TEXT_VIEW (message_box), FALSE);
    gtk_container_add (GTK_CONTAINER (mscroll), message_box);
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), mscroll);
    gtk_paned_pack2 (GTK_PANED (vpane), frame, FALSE, TRUE);
#else
    message_box = gtk_text_new (NULL, NULL);
    gtk_text_set_editable (GTK_TEXT (message_box), FALSE);
    gtk_container_add (GTK_CONTAINER (mscroll), message_box);
    gtk_paned_pack2 (GTK_PANED (vpane), mscroll, FALSE, TRUE);
#endif
    gtk_box_pack_start (GTK_BOX (right_box), vpane, TRUE, TRUE, 0);
    entry_box = gtk_entry_new ();
    gtk_signal_connect (GTK_OBJECT (entry_box), KEY_EVENT,
			GTK_SIGNAL_FUNC (entry_box_key_pressed), NULL);
    gtk_box_pack_start (GTK_BOX (right_box), entry_box, FALSE, FALSE, 0);

    gtk_box_pack_start (GTK_BOX (middle_box), left_box, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (middle_box), right_box, TRUE, TRUE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (middle_box), 10);
    gtk_box_pack_start (GTK_BOX (vbox), middle_box, TRUE, TRUE, 0);

    game->main_window = wp;
    game->message_box = message_box;
    game->entry_box = entry_box;
    game->player_vbox = player_vbox;
    me = game->player[game->local_player];
    zone_set_list (me->zone[HAND_ZONE], hand_list);
    player_make_labels (me, player_vbox);
    life = get_int_preference (game->prefs, starting_life_key, 20);
    player_set_life (me, life);

    filename = get_preference (game->prefs, deck_name_key);
    if (filename != NULL)
	chdir_of_file (filename);
    if ((filename != NULL) && (game->cardbase != NULL)) {
	me = game->player[game->local_player];
	me->deck = deck_read_file (filename, game->cardbase);
	if (me->deck != NULL) {
	    offset = player_deck_offset (game->local_player);
	    zone_add_deck (me->zone[LIBRARY_ZONE], offset, me->deck->size);
	    offset = player_sideboard_offset (game, game->local_player);
	    zone_add_deck (me->zone[SIDEBOARD_ZONE], offset,
			   me->deck->sb_size);
	}
    }
    set_game_turn (game, 1);
    set_game_phase (game, UNTAP_PHASE);

    init_opponent_signals ();
    set_main_window_size (game, vpane);
    gtk_widget_show_all (wp);
    gtk_widget_grab_focus (game->entry_box);

    if (game->cardbase == NULL)
	do_find_cardinfo_dialog (game);

    gtk_main ();
    return (0);
}

static void add_tools (GtkToolbar *toolbar)
{
    GtkIconSize size = gtk_toolbar_get_icon_size (toolbar);
    GtkWidget *image;

    image = gtk_image_new_from_stock (M_NEW_ICON, size);
    gtk_toolbar_append_item (toolbar, "New", "New Game", NULL, image,
			     GTK_SIGNAL_FUNC (do_new_game), game);
    image = gtk_image_new_from_stock (M_UNTAP_ICON, size);
    gtk_toolbar_append_item (toolbar, "Untap", "Untap Cards", NULL, image,
			     GTK_SIGNAL_FUNC (do_untap_cards), game);
    image = gtk_image_new_from_stock (M_DRAW_ICON, size);
    gtk_toolbar_append_item (toolbar, "Draw", "Draw Card", NULL, image,
			     GTK_SIGNAL_FUNC (do_draw_card), game);

    gtk_toolbar_append_space (toolbar);

    add_phase_button (toolbar, "Untap", "Untap Phase", UNTAP_PHASE,
		      M_UNTAP_PHASE_ICON);
    add_phase_button (toolbar, "Upkeep", "Upkeep Phase", UPKEEP_PHASE,
		      M_UPKEEP_PHASE_ICON);
    add_phase_button (toolbar, "Draw", "Draw Step", DRAW_PHASE,
		      M_DRAW_PHASE_ICON);
    add_phase_button (toolbar, "Main", "Main Phase", MAIN_PHASE,
		      M_MAIN_PHASE_ICON);
    add_phase_button (toolbar, "Combat", "Combat Phase", COMBAT_PHASE,
		      M_COMBAT_PHASE_ICON);
    add_phase_button (toolbar, "Cleanup", "Cleanup Phase", CLEANUP_PHASE,
		      M_CLEANUP_PHASE_ICON);

    gtk_toolbar_append_space (toolbar);

    image = gtk_image_new_from_stock (GTK_STOCK_GOTO_LAST, size);
    gtk_toolbar_append_item (toolbar, "Next", "Next Turn", NULL, image,
			     GTK_SIGNAL_FUNC (do_next_turn), game);
    image = gtk_image_new_from_stock (M_LIFESET_ICON, size);
    gtk_toolbar_append_item (toolbar, "Life", "Set Life", NULL, image,
			     GTK_SIGNAL_FUNC (do_set_life),
			     GINT_TO_POINTER (SET_LIFE));
    image = gtk_image_new_from_stock (M_LIFEMINUS_ICON, size);
    gtk_toolbar_append_item (toolbar, "-", "Decrement Life", NULL, image,
			     GTK_SIGNAL_FUNC (do_set_life),
			     GINT_TO_POINTER (DECREMENT_LIFE));
    image = gtk_image_new_from_stock (M_LIFEPLUS_ICON, size);
    gtk_toolbar_append_item (toolbar, "+", "Increment Life", NULL, image,
			     GTK_SIGNAL_FUNC (do_set_life),
			     GINT_TO_POINTER (INCREMENT_LIFE));
    image = gtk_image_new_from_stock (M_PING_ICON, size);
    gtk_toolbar_append_item (toolbar, "Ping", "Ping Opponent", NULL, image,
			     GTK_SIGNAL_FUNC (do_ping_opponent), game);
}

static void add_phase_button (GtkToolbar *toolbar, char *text, char *tip,
			      int phase, const char *icon)
{
    GtkIconSize size = gtk_toolbar_get_icon_size (toolbar);
    GtkWidget *image;

    image = gtk_image_new_from_stock (icon, size);
    game->phase_button[phase] =
	gtk_toolbar_append_element (toolbar, GTK_TOOLBAR_CHILD_TOGGLEBUTTON,
				    NULL, text, tip, NULL, image,
				    GTK_SIGNAL_FUNC(do_set_phase),
				    GINT_TO_POINTER(phase));
}

static void add_drop_tools (GtkToolbar *toolbar)
{
    GtkWidget *image, *t1, *t2, *t3;
    GtkIconSize size = gtk_toolbar_get_icon_size (toolbar);
    struct player *me;

    image = gtk_image_new_from_stock (M_DECK_ICON, size);
    t1 = gtk_toolbar_append_item (toolbar, "Library", "Library",
				  NULL, image,
				  GTK_SIGNAL_FUNC (do_library), game);
    image = gtk_image_new_from_stock (M_GRAVEYARD_ICON, size);
    t2 = gtk_toolbar_append_item (toolbar, "Graveyard", "Graveyard",
				  NULL, image,
				  GTK_SIGNAL_FUNC (do_graveyard), game);
    image = gtk_image_new_from_stock (M_REMOVED_ICON, size);
    t3 = gtk_toolbar_append_item (toolbar, "Removed", "Remove From Game",
				  NULL, image,
				  GTK_SIGNAL_FUNC (do_removed), game);

    me = game->player[game->local_player];
    zone_set_box (me->zone[LIBRARY_ZONE], t1, TRUE);
    zone_set_box (me->zone[GRAVEYARD_ZONE], t2, FALSE);
    zone_set_box (me->zone[REMOVED_ZONE], t3, FALSE);
}

static void chdir_of_file (char *filename)
{
    char *directory, *cp;

    directory = strdup (filename);
    cp = strrchr (directory, '/');
    if (cp != NULL) {
	*cp = 0;
	chdir (directory);
    }
    free (directory);
}

static void do_load_cardbase (GtkWidget *w, gpointer data)
{
    char *text;

    if (game->opponent != NULL) {
	text = "Cannot load new card data while game is in progress.";
	do_message_dialog (text);
    } else {
	do_find_cardinfo_dialog (game);
    }
}

static void do_new_game (GtkWidget *w, gpointer data)
{
    if (game->cardbase == NULL) {
	do_find_cardinfo_dialog (game);
	return;
    }
    new_game_dialog (game);
}

static void do_disconnect (GtkWidget *w, gpointer data)
{
    int pid;
    char *pname;

    if (game->opponent != NULL) {
	pid = pid_of_single_opponent (game);
	pname = (pid >= 0 ? game->player[pid]->name : "Server");
	opponent_destroy (game->opponent);
	game->opponent = NULL;
	display_message (game, "You have disconnected from %s.", pname);
    }
}

static void do_exit (GtkWidget *w, gpointer data)
{
    int width, height;

    gdk_window_get_size (game->main_window->window, &width, &height);
    set_int_preference (game->prefs, "window_width", width);
    set_int_preference (game->prefs, "window_height", height);
    gdk_window_get_size (game->playarea->darea->window, &width, &height);
    set_int_preference (game->prefs, "playarea_width", width);
    set_int_preference (game->prefs, "playarea_height", height);
    save_prefs (game->prefs);
    gtk_main_quit ();
}

static void do_draw_card (GtkWidget *w, gpointer data)
{
    struct player *me;

    me = game->player[game->local_player];
    if (me->zone[LIBRARY_ZONE]->size <= 0) {
	display_message (game, "No cards left in library.");
	return;
    }
    game_draw_cards (game, game->local_player, 1);
    send_draw_cards (game, 1);
}

static void do_untap_cards (GtkWidget *w, gpointer data)
{
    int i;
    struct table_card *tcard;

    for (i = 0; i < game->playarea->size; i++) {
	tcard = game->playarea->cardlist[i];
	if ((tcard->controller == game->local_player) &&
	    ((tcard->flags & CARD_TAPPED_FLAG) != 0) &&
	    ((tcard->flags & CARD_NO_UNTAP_FLAG) == 0)) {
	    set_table_card_flag (game, tcard, CARD_TAPPED_FLAG, FALSE,
				 game->local_player);
	    send_card_flag (game, tcard->cid, CARD_TAPPED_FLAG, FALSE);
	}
    }
}

static void do_create_card (GtkWidget *w, gpointer data)
{
    do_create_card_dialog (game);
}

static void do_flip_coin (GtkWidget *w, gpointer data)
{
    int res;

    res = random_number (2);
    game_flip_coin (game, game->local_player, res);
    send_flip_coin (game, res);
}

static void do_roll_die (GtkWidget *w, gpointer data)
{
    do_input_dialog (game, "Roll Dice", "Number of sides?", "20",
		     do_roll_die_cb, NULL);
}

static int do_roll_die_cb (struct game *game, gpointer p, const char *val)
{
    int size, res;

    size = atoi (val);
    if (size > 0) {
	res = random_number (size) + 1;
	game_roll_die (game, game->local_player, res, size);
	send_roll_die (game, res, size);
    }
    return (TRUE);
}

static void do_shuffle (GtkWidget *w, gpointer data)
{
    local_player_shuffle (game);
}

void local_player_shuffle (struct game *game)
{
    struct player *me;
    struct zone *zone;

    me = game->player[game->local_player];
    display_message (game, "%s is shuffling library...", me->name);
    zone = me->zone[LIBRARY_ZONE];
    if ((zone == NULL) || (zone->size == 0))
	return;
    zone_shuffle (zone);
    if (zone->list != NULL)
	zone_refresh_list_display (zone);
    send_shuffle (game);
}

static void do_mulligan (GtkWidget *w, gpointer data)
{
    move_cards_in_zone (game, HAND_ZONE, LIBRARY_ZONE);
}

static void do_grave_to_lib (GtkWidget *w, gpointer data)
{
    move_cards_in_zone (game, GRAVEYARD_ZONE, LIBRARY_ZONE);
}

static void do_grave_to_rem (GtkWidget *w, gpointer data)
{
    move_cards_in_zone (game, GRAVEYARD_ZONE, REMOVED_ZONE);
}

static void do_toplib_to_grave (GtkWidget *w, gpointer data)
{
    struct card_move m;

    m.pid = game->local_player;
    m.srcznum = LIBRARY_ZONE;
    m.idx = -1;
    m.dstznum = GRAVEYARD_ZONE;
    m.bottom = FALSE;
    m.cid = 0;
    game_move_card (game, &m);
    if (m.cid > 0) {
	game_move_card_message (game, &m);
	send_card_zone (game, &m);
    }
}

static void move_cards_in_zone (struct game *game, int srcznum, int dstznum)
{
    struct card_move m;

    m.pid = game->local_player;
    m.srcznum = srcznum;
    m.idx = -1;
    m.dstznum = dstznum;
    m.bottom = FALSE;
    while (TRUE) {
	m.cid = 0;
	game_move_card (game, &m);
	if (m.cid <= 0)
	    break;
	game_move_card_message (game, &m);
	send_card_zone (game, &m);
    }
}

static void do_set_phase (GtkWidget *widget, gpointer data)
{
    int new_phase = GPOINTER_TO_INT (data);
    GtkToggleButton *button;

    if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
	return;
    if (game->phase_number == new_phase)
	return;
    if (game->current_player != game->local_player) {
	button = GTK_TOGGLE_BUTTON (game->phase_button[new_phase]);
	gtk_toggle_button_set_active (button, FALSE);
	button = GTK_TOGGLE_BUTTON (game->phase_button[game->phase_number]);
	gtk_toggle_button_set_active (button, TRUE);
	display_message (game, "You may only change phase during your turn.");
	return;
    }
    set_game_phase (game, new_phase);
    send_game_phase (game);
}

static void do_next_turn (GtkWidget *w, gpointer data)
{
    set_game_turn (game, game->turn_number+1);
    send_game_turn (game);
    set_game_phase (game, UNTAP_PHASE);
    send_game_phase (game);
}

static void do_prev_turn (GtkWidget *w, gpointer data)
{
    set_game_turn (game, game->turn_number-1);
    send_game_turn (game);
    set_game_phase (game, UNTAP_PHASE);
    send_game_phase (game);
}

static void do_set_life (GtkWidget *w, gpointer data)
{
    int todo, offset;
    struct player *me;

    todo = GPOINTER_TO_INT (data);
    offset = 0;

    switch (todo) {
    case SET_LIFE:
	do_life_dialog (game, todo);
	return;
    case INCREMENT_LIFE:
	if (is_shift_down ()) {
	    do_life_dialog (game, GAIN_LIFE);
	} else {
	    offset = +1;
	}
	break;
    case DECREMENT_LIFE:
	if (is_shift_down ()) {
	    do_life_dialog (game, LOSE_LIFE);
	} else {
	    offset = -1;
	}
	break;
    }
    me = game->player[game->local_player];
    set_player_life (game, game->local_player, me->life+offset);
    send_player_life (game);
}

static void do_next_phase (GtkWidget *w, gpointer data)
{
    int phase;

    phase = game->phase_number + 1;
    if (phase < NUM_PHASES) {
	set_game_phase (game, phase);
	send_game_phase (game);
    } else {
	do_next_turn (w, data);
    }
}

static void do_ping_opponent (GtkWidget *w, gpointer data)
{
    send_ping (game);
}

static void do_library (GtkWidget *w, gpointer data)
{
    do_library_dialog (game);
}

static void do_view_library (GtkWidget *w, gpointer data)
{
    do_zone_dialog (game, game->local_player, LIBRARY_ZONE, 0);
}

static void do_view_libtop (GtkWidget *w, gpointer data)
{
    do_input_dialog (game, "", "Enter the number of cards to view.", NULL,
		     do_view_libtop_cb, NULL);
}

static int do_view_libtop_cb (struct game *game, gpointer p, const char *val)
{
    int size = atoi (val);
    if (size > 0)
	do_zone_dialog (game, game->local_player, LIBRARY_ZONE, size);
    return (TRUE);
}

static void do_graveyard (GtkWidget *w, gpointer data)
{
    do_zone_dialog (game, game->local_player, GRAVEYARD_ZONE, 0);
}

static void do_opp_graveyard (GtkWidget *w, gpointer data)
{
    int pid;

    pid = pid_of_single_opponent (game);
    if (pid >= 0)
	do_zone_dialog (game, pid, GRAVEYARD_ZONE, 0);
}

static void do_removed (GtkWidget *w, gpointer data)
{
    do_zone_dialog (game, game->local_player, REMOVED_ZONE, 0);
}

static void do_opp_removed (GtkWidget *w, gpointer data)
{
    int pid;

    pid = pid_of_single_opponent (game);
    if (pid >= 0)
	do_zone_dialog (game, pid, REMOVED_ZONE, 0);
}

static void do_sideboard (GtkWidget *w, gpointer data)
{
    if (game->no_view_sideboard) {
	do_message_dialog ("View Sideboard disabled against Apprentice.");
    } else {
	do_zone_dialog (game, game->local_player, SIDEBOARD_ZONE, 0);
    }
}

static void do_show_hand (GtkWidget *w, gpointer data)
{
    do_show_zone (game, HAND_ZONE, 0);
}

static void do_show_library (GtkWidget *w, gpointer data)
{
    do_show_zone (game, LIBRARY_ZONE, 0);
}

static void do_show_random (GtkWidget *w, gpointer data)
{
    int znum, idx, cid;
    struct zone *zone;

    znum = HAND_ZONE;
    zone = game->player[game->local_player]->zone[znum];
    if (zone->size == 0)
	return;
    idx = random_number (zone->size);
    cid = zone->cardlist[idx];
    game_show_random (game, game->local_player, cid);
    send_random_card (game, cid, znum);
}

static void do_show_libtop (GtkWidget *w, gpointer data)
{
    do_input_dialog (game, "", "Enter the number of cards to show.", NULL,
		     do_show_libtop_cb, NULL);
}

static int do_show_libtop_cb (struct game *game, gpointer p, const char *val)
{
    int size = atoi (val);
    if (size > 0)
	do_show_zone (game, LIBRARY_ZONE, size);
    return (TRUE);
}

static void do_show_zone (struct game *game, int znum, int quantity)
{
#ifndef APPRENTICE
    int pid;
    char *oname, *pname, *zname;

    /*
     *  When we allow multiple players, we'll have to ask which opponent
     *  to show the library, and send the show-lib message only to him.
     */
    pid = pid_of_single_opponent (game);
    if (pid < 0)
	return;
    pname = game->player[game->local_player]->name;
    oname = game->player[pid]->name;
    zname = name_of_zone (znum);
    if (quantity > 0) {
	display_message (game, "%s is looking through the top %d cards "
			 "of %s's %s...", oname, quantity, pname, zname);
    } else {
	display_message (game, "%s is looking through %s's %s...",
			 oname, pname, zname);
    }
#endif
    send_show_zone (game, znum, quantity);
}

static void do_preferences (GtkWidget *w, gpointer data)
{
    do_preferences_dialog (game);
}

static void do_set_name (GtkWidget *w, gpointer data)
{
    char *title, *prompt;

    title = "Set Name";
    prompt = "Enter your name.";
    do_input_dialog (game, title, prompt, NULL, set_name_callback, NULL);
}

static void do_about_box (GtkWidget *w, gpointer data)
{
    char buf[255];

    sprintf (buf, "Mindless Automaton CCG Simulator\n"
	     "Version %s\n"
	     "by Salvatore Valente <svalente@mit.edu>\n\n"
	     "Not affiliated with or endorsed by\n"
	     "Wizards of the Coast", version_string);
    do_message_dialog (buf);
}

static int
set_name_callback (struct game *game, gpointer data, const char *text)
{
    if (*text != 0) {
	set_player_name (game, game->local_player, text);
	send_player_name (game);
	set_preference (game->prefs, player_name_key, text);
    }
    return (TRUE);
}

static void do_view_card (GtkWidget *w, struct game *game)
{
    view_card_selected_in_zone_list (game, game->local_player, HAND_ZONE);
}

static int
entry_box_key_pressed (GtkWidget *entry, GdkEventKey *event, gpointer data)
{
    char *message, *cp, *pname;

#if GTK_MAJOR_VERSION < 2
    /*
     *  Control-D means draw card over delete forward character.
     *  This should be a configuration option.
     */
    if (gtk_accel_groups_activate (GTK_OBJECT (game->main_window),
				   event->keyval, event->state)) {
	GtkType otype = gtk_signal_lookup (KEY_EVENT, GTK_OBJECT_TYPE (entry));
	gtk_signal_emit_stop (GTK_OBJECT (entry), otype);
	return (TRUE);
    }
#endif

    switch (event->keyval) {
    case GDK_KP_Enter:
    case GDK_Return:
	message = gtk_editable_get_chars (GTK_EDITABLE (game->entry_box),
					  0, -1);
	if ((message == NULL) || (*message == 0))
	    break;
	for (cp = message; *cp != 0; cp++) {
	    if (*cp == '"')
		*cp = '\'';
	}
	pname = game->player[game->local_player]->name;
	display_message (game, "%s says: '%s'", pname, message);
	send_message (game, message);
	g_free (message);
	gtk_entry_set_text (GTK_ENTRY (game->entry_box), "");
	break;
    }
    return (TRUE);
}

/*
 *  Create the main window.  Then calculate the difference between the
 *  current playarea/console size and the preferred playarea/console size.
 *  Then reset the main window size, and show the main window.
 */
static void set_main_window_size (struct game *game, GtkWidget *vpane)
{
    GtkWidget *wp = game->main_window;
    int fw, fh, ph;

    fw = get_int_preference (game->prefs, "window_width", 858);
    fh = get_int_preference (game->prefs, "window_height", 650);
    gtk_window_set_default_size (GTK_WINDOW (wp), fw, fh);
    ph = get_int_preference (game->prefs, "playarea_height", 579);
    gtk_paned_set_position (GTK_PANED (vpane), ph);
}
