/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  opponent.c -- An opponent, like the local player, issues simple commands
 *		  to the game manager.
 *
 *  An opponent is represented by a network socket.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#include <gtk/gtk.h>
#include "game.h"
#include "opponent.h"

#define INVALID_SOCKET -1
#define SOCKET_ERROR -1
#ifndef INADDR_NONE
#define INADDR_NONE ((unsigned long) -1)
#endif

/* #define OUTPUT_SEND 1 */
/* #define OUTPUT_RECV 1 */

static int got_sigpipe;
static void handle_sigpipe (int sig);
static void opp_has_disconnected (struct game *game);

/*
 *  Generic networking functions.
 */
char *get_ip_address_text (void)
{
    static char *ip_address_text = NULL;
    char hostname[256];
    struct hostent *hp;
    struct in_addr inaddr;

    if (ip_address_text == NULL) {
	if (gethostname (hostname, sizeof (hostname)) == 0) {
	    hp = gethostbyname (hostname);
	    if (hp != NULL) {
		inaddr = *(struct in_addr *)(hp->h_addr);
		ip_address_text = inet_ntoa (inaddr);
	    }
	}
    }
    return (ip_address_text);
}

void init_opponent_signals (void)
{
    struct sigaction act;

    bzero (&act, sizeof (struct sigaction));
    act.sa_handler = handle_sigpipe;
    sigemptyset (&act.sa_mask);
    sigaction (SIGPIPE, &act, NULL);
}

SOCKET create_socket_and_listen (int port_num)
{
    SOCKET s;
    struct sockaddr_in sai;
    int on;

    s = socket (PF_INET, SOCK_STREAM, 0);
    if (s == INVALID_SOCKET) {
	perror ("socket");
	return (s);
    }
    on = 1;
    setsockopt (s, SOL_SOCKET, SO_REUSEADDR, (void *) &on, sizeof (on));
    memset (&sai, 0, sizeof (struct sockaddr_in));
    sai.sin_family = PF_INET;
    sai.sin_addr.s_addr = INADDR_ANY;
    sai.sin_port = htons (port_num);
    if (bind (s, (struct sockaddr *) &sai, sizeof (sai)) == SOCKET_ERROR) {
	perror ("bind");
	return (INVALID_SOCKET);
    }
    listen (s, SOMAXCONN);
    return (s);
}

struct opponent *accept_new_opponent (SOCKET s)
{
    SOCKET oppsock;
    struct sockaddr_in sai;
    int len;
    struct opponent *opponent;

    len = sizeof (sai);
    oppsock = accept (s, (struct sockaddr *) &sai, &len);
    if (oppsock == INVALID_SOCKET) {
	perror ("accept");
	return (NULL);
    }
    opponent = calloc (1, sizeof (struct opponent));
    opponent->socket = oppsock;
    return (opponent);
}

struct opponent *call_new_opponent (char *hostname, int port_num)
{
    struct sockaddr_in address;
    struct hostent *hp;
    unsigned long ipaddr;
    SOCKET oppsock;
    struct opponent *opponent;

    /* Look up the hostname. */
    memset (&address, 0, sizeof (struct sockaddr_in));
    if (isdigit (*hostname) &&
	(ipaddr = inet_addr (hostname)) != INADDR_NONE) {
	memcpy (&(address.sin_addr), (char *)(&ipaddr), sizeof (ipaddr));
	address.sin_family = PF_INET;
    } else {
	hp = gethostbyname (hostname);
	if (!hp) {
	    perror (hostname);
	    return (NULL);
	}
	/* fill in the address */
	memcpy (&(address.sin_addr), hp->h_addr, hp->h_length);
	address.sin_family = hp->h_addrtype;
    }
    address.sin_port = htons (port_num);

    /* Open the socket. */
    oppsock = socket (PF_INET, SOCK_STREAM, 0);
    if (oppsock == INVALID_SOCKET) {
	perror ("socket");
	return (NULL);
    }
    /* Connect to the server. */
    if (connect (oppsock, (struct sockaddr *) &address,
		 sizeof (address)) == SOCKET_ERROR) {
	perror ("connect");
	return (NULL);
    }
    opponent = calloc (1, sizeof (struct opponent));
    opponent->socket = oppsock;
    return (opponent);
}

void opponent_destroy (struct opponent *opponent)
{
    if (opponent->socket != INVALID_SOCKET)
	close_socket (opponent->socket);
    if (opponent->tag > 0)
	gdk_input_remove (opponent->tag);
    if (opponent->text != NULL)
	free (opponent->text);
    free (opponent);
}

void close_socket (SOCKET s)
{
    close (s);
}

/* ----------------------------------------------------------------- */

void send_opponents (struct game *game, char *fmt, ...)
{
    char buf[512];
    va_list ap;
    int len;

    if (game->opponent == NULL)
	return;
    va_start (ap, fmt);
    vsnprintf (buf, sizeof (buf)-1, fmt, ap);
    va_end (ap);
    len = strlen (buf);
    buf[len++] = '\r';
    buf[len++] = '\n';
    buf[len] = 0;
#if OUTPUT_SEND
    fputs (buf, stdout);
#endif
    got_sigpipe = FALSE;
    send (game->opponent->socket, buf, len, 0);
    if (got_sigpipe) {
	opp_has_disconnected (game);
	got_sigpipe = FALSE;
    }
}

static void handle_sigpipe (int sig)
{
    got_sigpipe = TRUE;
}

static void opp_has_disconnected (struct game *game)
{
    int pid;
    char *pname;

    pid = pid_of_single_opponent (game);
    pname = (pid >= 0 ? game->player[pid]->name : "Server");
    opponent_destroy (game->opponent);
    game->opponent = NULL;
    display_message (game, "%s has disconnected.", pname);
}

/* ----------------------------------------------------------------- */

static void sync_opp_read_data (gpointer, gint, GdkInputCondition);
static void opp_read_data (gpointer, gint, GdkInputCondition);
static void run_command (struct game *, char *);
static void add_arg (struct rcmd_args *, char *, char *);

/*
 *  opponent_listen () -- Call this after a game has been reset with
 *	this opponent as one of the players.
 *	This begins taking commands from the opponent.
 */
void opponent_listen (struct game *game)
{
    struct opponent *opponent = game->opponent;
    int condition = GDK_INPUT_READ | GDK_INPUT_EXCEPTION;
    opponent->tag = gdk_input_add (opponent->socket, condition,
				   sync_opp_read_data, game);
}

static void
sync_opp_read_data (gpointer data, gint source, GdkInputCondition condition)
{
    static int lock;

    if (lock > 0)
	return;
    lock++;
    opp_read_data (data, source, condition);
    lock--;
}

static void
opp_read_data (gpointer data, gint source, GdkInputCondition condition)
{
    SOCKET oppsock = (SOCKET) source;
    struct game *game = (struct game *) data;
    int len, i, start, do_free;
    struct opponent *opponent;
    char buf[256], *line;

    len = recv (oppsock, buf, sizeof (buf) - 1, 0);
    if (len <= 0) {
	opp_has_disconnected (game);
	return;
    }
    if (len == 0) {
	return;
    }
#if OUTPUT_RECV
    printf("%c[31m", 0x1b);
    fwrite (buf, 1, len, stdout);
    printf("%c[0m", 0x1b);
#endif
    /* Make sure we can treat the block of data as a C string. */
    for (i = 0; i < len; i++)
	if (buf[i] == 0)
	    buf[i] = '0';
    buf[len] = 0;

    /* Run each command */
    opponent = game->opponent;
    start = 0;
    for (i = 0; i < len; i++) {
	if (!isnewline (buf[i]))
	    continue;
	buf[i++] = 0;
	while (isnewline (buf[i]))
	    i++;
	if (opponent->text != NULL) {
	    line = realloc (opponent->text, strlen (opponent->text)+i+1);
	    opponent->text = NULL;
	    strcat (line, buf+start);
	    do_free = TRUE;
	} else {
	    line = buf+start;
	    do_free = FALSE;
	}
	run_command (game, line);
	if (do_free)
	    free (line);
	start = i;
    }
    /* Remember the leftovers. */
    if (start < len) {
	if (opponent->text == NULL) {
	    opponent->text = strdup (buf+start);
	} else {
	    opponent->text = realloc (opponent->text, strlen (opponent->text)+
				      len-start+2);
	    strcat (opponent->text, buf+start);
	}
    }
}

static void run_command (struct game *game, char *cp)
{
    struct rcmd_args args;
    char *cmd, *arg, *val;

    bzero (&args, sizeof (struct rcmd_args));
    while (isspace (*cp))
	cp++;
    cmd = cp;
    while (!(isspace (*cp) || (*cp == 0)))
	cp++;
    if (*cp != 0) {
	*cp = 0;
	cp++;
    }
    while (*cp != 0) {
	while (isspace (*cp))
	    cp++;
	if (*cp == 0)
	    break;
	arg = cp;
	while ((*cp != ':') && (*cp != 0))
	    cp++;
	if (*cp == 0)
	    break;
	*cp = 0;
	cp++;
	while (isspace (*cp))
	    cp++;
	if (*cp == 0)
	    break;
	val = cp;
	if (*val == '"') {
	    val++;
	    cp++;
	    while ((*cp != '"') && (*cp != 0))
		cp++;
	    if (*cp == 0)
		break;
	    *cp = 0;
	    cp++;
	} else {
	    while (!(isspace (*cp) || (*cp == 0)))
		cp++;
	    if (*cp != 0) {
		*cp = 0;
		cp++;
	    }
	}
	add_arg (&args, arg, val);
    }
    run_parsed_command (game, cmd, &args);
}

static void add_arg (struct rcmd_args *args, char *arg, char *val)
{
    if (strcmp (arg, "pid") == 0)
	args->pid = atoi (val);
    else if (strcmp (arg, "cid") == 0)
	args->cid = atoi (val);
    else if (strcmp (arg, "lid") == 0)
	args->lid = atoi (val);
    else if (strcmp (arg, "value") == 0)
	args->value = atoi (val);
    else if (strcmp (arg, "name") == 0)
	args->name = val;
    else if (strcmp (arg, "text") == 0)
	args->text = val;
    else if (strcmp (arg, "startpos") == 0)
	args->startpos = atoi (val);
    else if (strcmp (arg, "qty") == 0)
	args->qty = atoi (val);
    else if (strcmp (arg, "flag") == 0)
	args->flag = val;
    else if (strcmp (arg, "x") == 0)
	args->x = atoi (val);
    else if (strcmp (arg, "y") == 0)
	args->y = atoi (val);
    else if (strcmp (arg, "width") == 0)
	args->width = atoi (val);
    else if (strcmp (arg, "height") == 0)
	args->height = atoi (val);
    else if (strcmp (arg, "face") == 0)
	args->face = atoi (val);
    else if (strcmp (arg, "border") == 0)
	args->border = atoi (val);

    /* Redundant args used by Apprentice. */
    else if (strcmp (arg, "startcid") == 0)
	args->cid = atoi (val);
    else if (strcmp (arg, "destlid") == 0)
	args->lid = atoi (val);
    else if ((strcmp (arg, "msg") == 0) || (strcmp (arg, "order") == 0) ||
	     (strcmp (arg, "powtuf") == 0) || (strcmp (arg, "ver") == 0))
	args->text = val;
    else if ((strcmp (arg, "res") == 0) || (strcmp (arg, "phase") == 0) ||
	     (strcmp (arg, "turn") == 0) || (strcmp (arg, "life") == 0) ||
	     (strcmp (arg, "counters") == 0) || (strcmp (arg, "set") == 0) ||
	     (strcmp (arg, "take") == 0) || (strcmp (arg, "color") == 0))
	args->value = atoi (val);
}

void process_send_ping (struct game *game, int pid)
{
    struct opponent *opponent;

    opponent = game->opponent;
    if (opponent->last_ping == NULL)
	opponent->last_ping = malloc (sizeof (struct timeval));
    gettimeofday (opponent->last_ping, NULL);
}

void process_return_ping (struct game *game, int pid)
{
    struct opponent *opponent;
    struct timeval tv;
    int seconds;

    opponent = game->opponent;
    if ((opponent == NULL) || (opponent->last_ping == NULL))
	return;
    gettimeofday (&tv, NULL);
    if (tv.tv_usec < opponent->last_ping->tv_usec) {
	tv.tv_usec += 1000000;
	tv.tv_sec--;
    }
    seconds = tv.tv_sec - opponent->last_ping->tv_sec;
    /*  Round up.  */
    if (tv.tv_usec - opponent->last_ping->tv_usec >= 500000)
	seconds++;
    display_message (game, "%s returned ping in %d seconds.",
		     game->player[pid]->name, seconds);
}
