
/* $Id: cipher.c,v 1.7 2005/03/11 23:16:02 alien-science Exp $ */

#include <string.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <openssl/err.h>
#include <unistd.h>
#include <termios.h>
#include <stdio.h>

#include "common.h"
#include "cipher.h"
#include "prog_config.h"

/*
 * Initialises a cipher structure
 */
void
cipher_new(struct cipher *cipher, const EVP_CIPHER *type)
{
   /* Get access to crypto library error strings */
   ERR_load_crypto_strings();

   /* Set the cipher type */
   cipher->type = type;

   /* Get the key and iv lengths */
   cipher->keylen = EVP_CIPHER_key_length(type);
   cipher->ivlen  = EVP_CIPHER_iv_length(type);

   /* Allocate key and iv buffers */
   cipher->key = XMALLOC(char, cipher->keylen);
   cipher->iv  = XMALLOC(char, cipher->ivlen);

   /* Mark them for clearing on exit */
   for_burning((void **)&cipher->key, cipher->keylen);
   for_burning((void **)&cipher->iv,  cipher->ivlen);

   /* Create a ctx structure */
   cipher->ctx = XMALLOC(EVP_CIPHER_CTX, 1);

   /* Initialise the ctx structure */
   EVP_CIPHER_CTX_init(cipher->ctx);
   
}


/*
 * Frees up a cipher structure
 */
void
cipher_delete(struct cipher *cipher)
{

   if (0 == cipher) return; 

   /* Free up memory */
   XBURN(cipher->key, cipher->keylen);
   XBURN(cipher->iv, cipher->ivlen);
   EVP_CIPHER_CTX_cleanup(cipher->ctx);
   XFREE(cipher->ctx);
   XFREE(cipher->name);
   XFREE(cipher->digest_name);

   /* Free loaded error strings */
   ERR_free_strings();

   /* Free cipher names */
   EVP_cleanup();

}


/*
 * Returns the type of cipher based on a user supplied string
 * Overwrites the given string with the actual openssl ciphername
 */
const EVP_CIPHER *
cipher_type(char *type)
{

   char *name = type;

   /* We will be accessing ciphers by name */
   OpenSSL_add_all_ciphers();

   /* 
    * Allow common names and various shortcuts
    * Default to cipher block chaining mode (man openssl for more info)  
    */
   if (0 == strncmp(type, "bfish", 6) ||
       0 == strncmp(type, "blowfish", 10)) { 
      strcpy(name,"bf");
      }

   /* Allow a null cipher for debugging */
   if (0 == strncmp(type, "null", 4)) {
      return EVP_enc_null();
      }

   /* Assume CBC mode but maybe make an option later */
   if (0 == strstr(name, "-cbc")) {
      strcat(name, "-cbc");
      }
   
   /* Get the cipher type */
   return EVP_get_cipherbyname(name);
}

/*
 * Returns the type of digest based on a user supplied string
 */
const EVP_MD *
digest_type(char *type)
{

   char *name = type;

   /* We will be accessing digests by name */
   OpenSSL_add_all_digests();

   /* Get the digest type */
   return EVP_get_digestbyname(name);
}


/**
  Generate a key and iv from a user supplied password/phrase
  TODO: upgrade to PKCS#5 v2.0 when it gets documented
**/
int 
cipher_key_iv(struct config *conf)
{
   char *password;
   struct termios t_old, 
                  t_new;
   int rc;

   /* Prompt for the password */
   password = XMALLOC(char, 81);
   printf("password: ");

   /* Echo off */
   tcgetattr(STDIN_FILENO, &t_old);
   t_new = t_old;
   t_new.c_lflag &= ~ECHO;
   tcsetattr(STDIN_FILENO, TCSAFLUSH, &t_new);

   /* Read the password */
   rc = scanf("%80[^\n]",password);
 
   /* Echo on */
   tcsetattr(STDIN_FILENO, TCSAFLUSH, &t_old);

   /* Clean read buffer, screen and string */
   getchar();
   printf("\n");
   password[80] = '\0';

   if (1 != rc) {
      if (conf->is_encrypting) {
         WARNING("Using empty password");
         }

      strcpy(password, "empty password");
      }

   /* Generate a key and iv from the password */
   EVP_BytesToKey (conf->cipher->type, EVP_md5(), 
                   conf->salt,
                   password, strlen(password), 
                   2, 
                   conf->cipher->key, conf->cipher->iv);

   /* Finished with the password */
   XBURN(password, 80);

   OK;
}

/**
   Generate a salt and checksum from a user supplied keyword
   This is a way of getting, an easy to remember, keyword to provide
   some repeatable encryption parameters 
**/
int cipher_keyword_conf(struct config *conf)
{
   char *fake_key, 
        *fake_iv;
   size_t keyword_len;
   const EVP_CIPHER *type;
   size_t keylen,
          ivlen;

   keyword_len = strlen(conf->keyword);

   /* Try DES as a valid cipher to work with -- default values if not */
   type = EVP_des_cbc();
   keylen = EVP_CIPHER_key_length(type);
   ivlen  = EVP_CIPHER_iv_length(type);

   if (8 > keylen) {
      conf->magic_sum = 0xffff;
      strcpy(conf->salt,"12345678");
      if (conf->is_encrypting) {
         WARNING("Using default salt");
         }
      OK;
      }

   fake_key = XMALLOC(char, keylen);
   fake_iv  = XMALLOC(char, ivlen);

   /* Abuse an OpenSSL call to provide some data from the keyword */
   EVP_BytesToKey (type, EVP_md5(), 
                   NULL,              /* Salt not used at this stage */
                   conf->keyword, keyword_len, 
                   1, fake_key, fake_iv);

   /* Use part of the key as the salt and part of the iv as 
      the magic checksum */
   memcpy(conf->salt, fake_key, 8);
   conf->salt[8]   = '\0';
   conf->magic_sum = *((unsigned *) fake_iv);

   VERBOSE_2("Salt 0x%x%x", *((unsigned *)conf->salt), 
                            *((unsigned *)(conf->salt + 4)));
   VERBOSE_1("Magic sum 0x%x", conf->magic_sum & 0xffff);

   XFREE(fake_iv);
   XFREE(fake_key);

   OK;
}


