
/* $Id: compress.c,v 1.4 2004/12/05 16:03:07 alien-science Exp $ */

#include <string.h>
#include <bzlib.h>

#include "common.h"


/*
 * Currently bzip2
 */
int
compress_init(void **stash)
{
   int res;
   bz_stream *strm;

   strm = XMALLOC(bz_stream, 1);

   /* Initialise the bz stream to defaults */
   strm->bzalloc = NULL;
   strm->bzfree  = NULL;
   strm->opaque  = NULL;

   res = BZ2_bzCompressInit(strm, 1, 0, 0);

   if (BZ_OK != res) {
      ERROR_1("Failed to init libbzip2, code: %d", res);
      }

   *stash = strm;

   OK;
}


/*
 * Called when all available data in a block has been compressed
 */
int
compress_end(void *stash, char *pout, size_t out_len, size_t *written)
{
   int res;
   bz_stream *strm  = stash;
   
   int ret = 0;
   
   /* Set the stream structure flush output data with no more input */
   strm->avail_in  = 0;
   strm->next_out  = pout;
   strm->avail_out = out_len;

  
   while (1) {

      res = BZ2_bzCompress(strm, BZ_FINISH);

      if (BZ_STREAM_END == res) break;

      if (!(BZ_FINISH_OK == res || BZ_STREAM_END == res)) {
         LERROR_1("Failure during end of compression, code: %d", res);
         }
      }

   *written += strm->total_out_lo32;

   BZ2_bzCompressEnd(strm); 
   XFREE(strm);

   OK;
} 



/*
 * Currently bzip2
 */
int
compress(void *stash, char *pin, size_t len, 
                      char *pout, size_t out_len, size_t *written)
{
   int res;
   bz_stream *strm  = stash;
   
   /* Set the stream structure to do incremental compression */
   strm->next_in   = pin;
   strm->avail_in  = len;
   strm->next_out  = pout;
   strm->avail_out = out_len;
    
   /* Compress incrementally */
   res = BZ2_bzCompress(strm, BZ_RUN);

   if (BZ_RUN_OK  != res) {
      ERROR_1("Failure returned during compression, code: %d", res);
      }

   /* Did this do all the data in one go ? */
   if (0 != strm->avail_in) {
      ERROR("Compression did not happen in a single call as expected\n");
      }

   /* Did this almost fill the destination buffer if so -- 
      compression has caused a large growth in data size */
   if (strm->total_out_lo32 >= out_len - 32) {
      ERROR("Compression of block too close to available buffer\n");
      }

   *written += strm->total_out_lo32;
   OK;
}


/*
 * Decompress
 */
int
uncompress(char *pin, size_t len, 
           char *pout, size_t out_len, size_t *written)
{
   bz_stream *strm;
   int res;
   int ret = 0;

   strm = XMALLOC(bz_stream, 1);

   /* Initialise the bz stream to defaults */
   strm->bzalloc = NULL;
   strm->bzfree  = NULL;
   strm->opaque  = NULL;

   /* Init */
   res =  BZ2_bzDecompressInit(strm, 0, 0);

   if (BZ_OK != res) {
      XFREE(strm);
      ERROR_1("Failed to init libbzip2, code: %d", res);
      }

   /* Prepare for one shot decompress */
   strm->next_in   = pin;
   strm->avail_in  = len;
   strm->next_out  = pout;
   strm->avail_out = out_len;
 
   /* Decompress */
   do {
      res = BZ2_bzDecompress(strm);

      if (BZ_STREAM_END != res &&
          BZ_OK         != res) {
         LERROR_1("Error returned during decompression, code: %d", res);
         }

      if (strm->avail_in > strm->avail_out) {
         LERROR("Run out of buffer space during decompression");
         }
      } while (BZ_STREAM_END != res && strm->avail_in > 0 );


   *written += strm->total_out_lo32;
   BZ2_bzDecompressEnd(strm);
   XFREE(strm);

   return ret;
}

