
/*  $Id: crypt.c,v 1.5 2005/03/29 19:25:40 alien-science Exp $ */

#include <openssl/evp.h>
#include <openssl/err.h>
#include <string.h>

#include "common.h"
#include "read_buf.h"
#include "checksum.h"
#include "cipher.h"

/*
 * Cleans up after a session
 */
void
crypt_endsession(struct cipher *cipher)
{
   EVP_cleanup();
}

      
/*
 * Crypts any remaining data and cleans up after a block
 */
int
crypt_endblock (struct cipher *cipher, char *pout, size_t *written)
{
   unsigned long err_code;
   size_t enc_len = 0;

   /* Get rid of any data that is sitting in the encryption engine */
   if (0 == EVP_CipherFinal_ex(cipher->ctx, pout, (int *) &enc_len)) {

      /* If this point has been reached -- an error occured */
      err_code = ERR_get_error();

      if (0x6065064 == err_code) {
         ERROR("Encryption library spotted a bad endblock -- "
               " the password or keyword could be wrong");
         } 
      else {
         ERROR_1("Encryption library returned error code on "
                 "EVP_CipherFinal_ex...\n%s",
                   ERR_error_string(err_code, NULL));
         }
      }

   *written += enc_len;

   OK;
}


/*
 * Resets the encryption engine using the supplied data as the iv
 */
int
encrypt_init(struct cipher *cipher, const char *iv_data, 
             size_t iv_data_len, char *actual_iv)
{
   int res;
   static int have_warned = 0;

   /* Construct an iv from the given data */
   if (iv_data_len < cipher->ivlen) {
      if (0 == have_warned) {
         WARNING("IV data doesn't fully fill iv -- "
                 "padding out with iv generated from password");
         have_warned = 1;
         }
      memcpy(actual_iv, iv_data, iv_data_len);
      memcpy(actual_iv + iv_data_len, cipher->iv, cipher->ivlen - iv_data_len);
      }
   else {
      memcpy(actual_iv, iv_data, cipher->ivlen);
      }

   /* Initialise the encryption session */
   res = EVP_CipherInit_ex(cipher->ctx,  cipher->type, 
                           NULL, cipher->key, actual_iv,  1);

   if (0 == res) {
      ERROR_1("Encryption library returned error code on " 
               "EVP_CipherInit_ex (encryption)...\n%s",
                  ERR_error_string(ERR_get_error(), NULL));   
      }
 
   OK;
}


/*
 * Runs the encryption/decryption engine on a block of memory
 */
int
crypt_mem(struct cipher *cipher, char *from, size_t len, 
          char *to, size_t *written)
{

   size_t crypt_len = 0;

   if (0 == len)  OK;

   if (0 == EVP_CipherUpdate(cipher->ctx, to, 
                             (int *) &crypt_len, from, len)) {
      ERROR_1("Encryption library returned error code on " 
               "EVP_CipherUpdate...\n%s",
                  ERR_error_string(ERR_get_error(), NULL));   
      }

   *written += crypt_len;

   OK;
}


/*
 * Resets the decryption engine
 */
int
decrypt_init(struct cipher *cipher, const char *iv)
{

  if (0 == EVP_CipherInit_ex(cipher->ctx,  cipher->type, 
                             NULL, cipher->key, iv,  0)) {
      ERROR_1("Encryption library returned error code on " 
               "EVP_CipherInit_ex (decryption)...\n%s",
                  ERR_error_string(ERR_get_error(), NULL));   
      }
  
  OK;
}


/*
 * Takes the message digest of the given buffer 
 */
int
crypt_digest(struct config *conf, char *buf, size_t buf_len, struct digest *d)
{
   EVP_MD_CTX mdctx;
   int res;

   int ret = 0;

   EVP_MD_CTX_init(&mdctx);

   do {
      res = EVP_DigestInit_ex(&mdctx, conf->cipher->digest_type, NULL);
      if (1 != res) { LERROR("Could not initialise digest"); }

      res = EVP_DigestUpdate(&mdctx, buf, buf_len);
      if (1 != res) { LERROR("Could not calculate digest"); }

      res = EVP_DigestFinal_ex(&mdctx, d->value, &d->len);
      if (1 != res) { LERROR("Could not finish digest"); }

   } while (0);

   EVP_MD_CTX_cleanup(&mdctx);

   return ret;
}


