
/*  $Id: header.c,v 1.7 2005/03/11 23:16:02 alien-science Exp $ */

#include <sys/types.h>
#include <unistd.h>
#include <string.h>

#include "prog_config.h"
#include "file.h"
#include "common.h"
#include "header.h"
#include "cipher.h"
#include "compress.h"

static unsigned char compress_header[] = COMPRESS_HEADER;

/* 
 * Uses the given config to write a minimal header out for
 * convenience when decrypting
 */
int
header_write(struct config *conf, int fh)
{
   char header[20 + (CIPHER_TYPE_MAXLEN * 2) + COMPRESS_HEADER_LEN];
   unsigned options;
   size_t to_write;
   size_t c_header_len;
   char *end;

   /* Build the options */
   options = 0;
   if (conf->is_compressing) {
      options     |= WITH_COMPRESSION;
      c_header_len = COMPRESS_HEADER_LEN;
      }
   else {
      c_header_len = 0;
      }

   /* Build the header */
   to_write = sprintf(header, "%s/%02x/%04x/%s/%s/%02x/", 
                      PREAMBLE, options, 
                      conf->max_block_size / 1024, 
                      conf->cipher->name,
                      conf->cipher->digest_name,
                      c_header_len );

   /* Add the compression header if needed */
   if (conf->is_compressing) {
      end = header + to_write;
      memcpy(end, compress_header, COMPRESS_HEADER_LEN);
      to_write += COMPRESS_HEADER_LEN;
      }

   ASSERT( write_block(fh, header, to_write));
   OK;
}


/*
 * Updates the given config with header information from the file
 * being decrypted
 */
int
header_read(struct config *conf, char *buf, size_t len)
{
   int rc;
   char preamble[7];
   unsigned options;
   
   /* These should be size_t but have had problems using sscanf with
    * size_t on some 64 bit platforms (which don't have the Z conversion) */
   unsigned c_header_len;
   unsigned block_size;

   char cipher_name[64];
   char digest_name[64];
   char *end;

   /* Read the first part of the header */
   rc = sscanf(buf, "%6s/%2x/%4x/%63[^/]/%63[^/]/%2x/", 
               preamble, &options, &block_size, 
               cipher_name, digest_name, &c_header_len);

   if (6 != rc || 0 != strncmp(preamble, PREAMBLE, 6)){
      ERROR("Type of file not recognised");
      }

   /* Check the compression header isn't oversize */
   if (c_header_len > COMPRESS_HEADER_MAXLEN - 1) {
      ERROR("Compression header is too big");
      }

   /* Check the blocksize is manageable */
   if (block_size > BLOCKSIZE_MAXKB) {
      ERROR("Blocksize specified in header is too big");
      }

   /* Read the compression header */
   if (c_header_len > 0) {
      end = buf + (6 + 1 + 2 + 1 + 4 + 1 + 
                   strlen(cipher_name) + 1 + strlen(digest_name) 
                   + 1 + 2 + 1);
      memcpy(conf->compress_header, end, c_header_len);
      }

   /* Add the details to the config */
   conf->is_compressing       = options & WITH_COMPRESSION;
   conf->compress_header_len  = c_header_len;   
   conf->max_block_size       = block_size * 1024;   
   ASSERT(conf_cipher(conf, cipher_name, digest_name));

   VERBOSE_1("Using %s for decryption", cipher_name);
   VERBOSE_1("IVs use digest %s", digest_name);
   VERBOSE_1("Max block size in header %#06x", conf->max_block_size);

   OK;
}

