
/* $Id: pipe.c,v 1.7 2005/03/11 23:16:02 alien-science Exp $ */

#include <string.h>

#include "common.h"
#include "pipe.h"
#include "crypt.h"
#include "compress.h"

/*
 * Initialises a new pipe
 */
void
pipe_alloc(struct pipe *pipe, size_t scratch_len, struct config *conf)
{
   /* Make note of the relevant config options */
   pipe->conf = conf;

   /* This encryption has not been initialised */
   pipe->done_init = 0;

   /* Compression/decompression requires an extra buffer */
   if (conf->is_compressing) {
      pipe->scratch_len = scratch_len;
      pipe->scratch_buf = XMALLOC(char, pipe->scratch_len);
      }
   else {
      pipe->scratch_buf = 0;
      }
}


/*
 * Finish with a pipe
 */
void
pipe_free(struct pipe *pipe)
{
   pipe->conf = 0;
   pipe->done_init = 0;
   XFREE(pipe->scratch_buf);
}


/*
 * Initialises the pipe for encryption using the given data
 * for the iv
 */
int
pipe_encrypt_init(struct pipe *pipe, char *iv_data, size_t iv_data_len, 
                  char *out_buf, size_t *enc_len)
{
   ASSERT( encrypt_init(pipe->conf->cipher, iv_data, iv_data_len, out_buf));

   /* The iv has been copied to the output buffer */
   *enc_len += pipe->conf->cipher->ivlen;

   if (pipe->conf->is_compressing) {
      ASSERT( compress_init(&(pipe->compressor)));
      }

   /* No data has been compressed yet -- this has to be taken note of so the
      compression header can be removed when it first appears */
   pipe->have_compressed = 0;

   OK;
}

/*
 * Sends data down the pipe to be encrypted
 */
int
pipe_encrypt(struct pipe *pipe, char *in, size_t len, 
                                char *out, size_t out_len, size_t *written)
{
   size_t compress_len = 0;

   /* Do the compression as required */
   if (pipe->conf->is_compressing) {

      ASSERT( compress(pipe->compressor, 
                       in, len, 
                       pipe->scratch_buf, pipe->scratch_len,
                       &compress_len));
     
      /* Compression doesn't spit anything out first time for small bits of data */          
      if (compress_len > 0 && pipe->have_compressed) {
         ASSERT( crypt_mem(pipe->conf->cipher, 
                           pipe->scratch_buf, 
                           compress_len - pipe->have_compressed,
                           out, written));
         }
      else if (compress_len > 0) {
         ASSERT( crypt_mem(pipe->conf->cipher, 
                           pipe->scratch_buf + COMPRESS_HEADER_LEN, 
                           compress_len      - COMPRESS_HEADER_LEN,
                           out, written));
         
         pipe->have_compressed = compress_len;
         }
      
      }
   else {
      ASSERT( crypt_mem(pipe->conf->cipher, in, len, out, written));
      }

   OK;
}



/*
 * Sends data down the pipe to be decrypted (one shot)
 */
int
pipe_decrypt(struct pipe *pipe, char *in, size_t len, 
                                char *out, size_t out_len, size_t *written)
{
   char *pread,
        *pwrite;
   size_t in_len;
   size_t compress_len;
   struct config *conf;

   conf = pipe->conf;

   /* Initialise -- the first few bytes of the input buffer are 
                    actually the iv */
   ASSERT( decrypt_init(conf->cipher, in));
   pread  = in + conf->cipher->ivlen;
   in_len = len - conf->cipher->ivlen;

   /* Uncompress as needed */
   if (conf->is_compressing) {

      /* Add the expected compression header */
      memcpy(pipe->scratch_buf, conf->compress_header, 
                                conf->compress_header_len);
      pwrite       = pipe->scratch_buf + conf->compress_header_len;
      compress_len = conf->compress_header_len;

      ASSERT( crypt_mem(conf->cipher, pread, in_len, 
                        pwrite, &compress_len));

      ASSERT( crypt_endblock(conf->cipher, 
                             pipe->scratch_buf + compress_len, 
                             &compress_len)); 

      ASSERT( uncompress(pipe->scratch_buf, compress_len, 
                         out, out_len, written));
      }
   else {
      ASSERT( crypt_mem(conf->cipher, pread, in_len, out, written));
      ASSERT( crypt_endblock(conf->cipher, out + *written, written)); 
      }

   /* The encryption lib will need re-initialising afterwards */
   crypt_endsession(pipe->conf->cipher);

   OK;

}


/*
 * Gets any remaining data from the encryption lib
 */
int
pipe_enc_end(struct pipe *pipe, char *out, size_t out_len, size_t *written)
{
   size_t compress_len, crypt_len;
   char *pdata;

   compress_len = crypt_len = 0;

   /* Flush the compression as required 
      (this is where most/all of the data comes out) */
   if (pipe->conf->is_compressing) {

      ASSERT( compress_end(pipe->compressor, 
                           pipe->scratch_buf, pipe->scratch_len, 
                           &compress_len));
      
      /* See if the known text of the compression header has to be removed */
      if (pipe->have_compressed) {
         pdata         = pipe->scratch_buf;
         }
      else {
         pdata         = pipe->scratch_buf + COMPRESS_HEADER_LEN;
         compress_len -= COMPRESS_HEADER_LEN;
         }

      ASSERT( crypt_mem(pipe->conf->cipher, 
                        pdata, compress_len - pipe->have_compressed,
                        out, &crypt_len));
      }
      
   /* Flush the encryption */
   ASSERT(crypt_endblock(pipe->conf->cipher, out + crypt_len, &crypt_len)); 

   /* The encryption lib will need re-initialising afterwards */
   crypt_endsession(pipe->conf->cipher);
   pipe->done_init = 0;

   /* Add the bytes now written */
   *written += crypt_len;
   OK;
}



/*
 * Gets any remaining data from the encryption lib
 */
int
pipe_dec_end(struct pipe *pipe, char *out, size_t out_len, size_t *written)
{
   /* Flush the encryption */
   ASSERT( crypt_endblock(pipe->conf->cipher, out, written)); 

   /* The encryption lib will need re-initialising afterwards */
   crypt_endsession(pipe->conf->cipher);
   pipe->done_init = 0;

   OK;
}


