/***************************************************************************
 Mutella - A commandline/HTTP client for the Gnutella filesharing network.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 gnucache.cpp  -  interface for the node cache.

 the original version of this file was taken from Gnucleus (http://gnucleus.sourceforge.net)

    begin                : Tue May 29 2001
    copyright            : (C) 2001 by
    email                : maksik@gmx.co.uk
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mutella.h"
#include "structures.h"

#include "gnucache.h"
#include "asyncsocket.h"
#include "packet.h"

#include "gnunode.h"
#include "gnudirector.h"
#include "common.h"
#include "conversions.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

MGnuCache::MGnuCache(MGnuDirector* pCntr)
{
	m_pDirector = pCntr;
}

MGnuCache::~MGnuCache()
{
}

void MGnuCache::UpdateCache(const Node& node)
{
	if( node.Port > 0 )
	{
		MLock lock(m_mutex);
		if (CheckIfNotRecent(node.Host) &&
			m_pDirector->IsOkForDirectConnect(node.Host))
		{
			if(m_listCache.size() >= NODECACHE_SIZE)
				m_listCache.pop_front();
			m_listCache.push_back(node);
		}
	}
}

bool MGnuCache::CheckIfNotRecent(const IP& ip)
{
	ASSERT(m_mutex.locked());
	std::deque<IP>::iterator itIP;
	for(itIP = m_queueRecentIPs.begin(); itIP != m_queueRecentIPs.end(); itIP++)
		if( (*itIP).S_addr == ip.S_addr)
		{
			m_queueRecentIPs.erase(itIP);
			m_queueRecentIPs.push_back(ip);
			return false;
		}
	m_queueRecentIPs.push_back(ip);
	if(m_queueRecentIPs.size() > 10)
		m_queueRecentIPs.pop_front();
	return true;
}

void MGnuCache::GetNodeList(list<Node>& list)
{
	MLock lock(m_mutex);
	list.clear();
	list = m_listCache;
}

int MGnuCache::GetSize()
{
	MLock lock(m_mutex);
	return m_listCache.size();
}

bool MGnuCache::IsHalfEmpty()
{
	MLock lock(m_mutex);
	return m_listCache.size() < NODECACHE_SIZE/2;
}

void MGnuCache::SendCache(MGnuNode* pNode, int nHosts)
{
	MLock lock(m_mutex);
	//
	int nHostsSent = 0, nPos;	
	while(nHostsSent < nHosts)
	{
		// don't send anything if the cache is half-empty
		if( m_listCache.size() < NODECACHE_SIZE/2 )
			return;
		nPos = rand() % m_listCache.size();
		for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		{
			if(!nPos)
			{
				pNode->Send_Host(*itNode);
				break;
			}
			nPos--;
		}
		nHostsSent++;
	}
}

CString MGnuCache::GetNodeList(int nHosts, bool bRemove /*=true*/ )
{
	CString sHosts;
	MLock lock(m_mutex);
	//
	int nHostsAdded = 0, nPos;
	while(nHostsAdded < nHosts && m_listCache.size())
	{
		// don't remove hosts if the cache is half-empty
		if( m_listCache.size() < NODECACHE_SIZE/2 )
			bRemove = false;
		nPos = rand() % m_listCache.size();
		for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		{
			if(!nPos)
			{
				if (nHostsAdded)
					sHosts += ',';
				sHosts += Ip2Str(itNode->Host)+':'+DWrdtoStr(itNode->Port);
				if (bRemove)
					m_listCache.erase(itNode);
				break;
			}
			nPos--;
		}
		nHostsAdded++;
	}
	return sHosts;
}

bool MGnuCache::GetRandomNode(Node& rNode, bool bRemove /*=true*/)
{
	MLock lock(m_mutex);
	if(m_listCache.size())
	{
		int nPos = rand() % m_listCache.size();
		for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		{
			if(!nPos)
			{
				rNode = (*itNode);
				m_listCache.erase(itNode);
				return true;
			}
			nPos--;
		}
	}
	return false;
}

void MGnuCache::Add(const Node& node)
{
	MLock lock(m_mutex);
	for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		if((*itNode).Host.S_addr == node.Host.S_addr && (*itNode).Port == node.Port)
		{
			m_listCache.erase(itNode);
			break;
		}
	if(m_listCache.size() >= NODECACHE_SIZE)
		m_listCache.pop_front();
	m_listCache.push_back(node);
}

void MGnuCache::Add(const IP& ip, int nPort)
{
	Node newNode;
	newNode.Ping	   = 0;
	newNode.Speed      = 0;
	newNode.ShareSize  = 0;
	newNode.ShareCount = 0;
	newNode.Distance   = 0;
	newNode.Host       = ip;
	newNode.Port       = nPort;
	Add(newNode);
}

void MGnuCache::Remove(const IP& ip)
{
	MLock lock(m_mutex);
	for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		if( (*itNode).Host.S_addr == ip.S_addr )
		{
			m_listCache.erase(itNode);
			return;
		}
}

bool MGnuCache::LoadNodes(const CString& sPath)
{
	MLock lock(m_mutex);
	FILE * pFile;
	if ( (pFile = fopen(ExpandPath(sPath).c_str(), "r")) )
	{
		CString sNode;
		Node	newNode;
		newNode.Ping	   = 0;
		newNode.Speed      = 0;
		newNode.ShareSize  = 0;
		newNode.ShareCount = 0;
		newNode.Distance   = 0;
		char buf[1024];
		while ( !feof(pFile) )
		{
			sNode = fgets(buf,1024,pFile);
			if ( sNode.length()==0 || buf[0]=='#' )
				continue;
			// Address
			int nPos  = sNode.find(":", 0);
			if (!Str2Ip(sNode.substr(0, nPos), newNode.Host))
				continue;
			// Port
			newNode.Port = atoi(sNode.substr(nPos + 1).c_str());
			// this check also iliminates file read/parse errors
			if(m_pDirector->IsOkForDirectConnect(newNode.Host))
			{
				bool bDuplicate = false;
				for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
					if((*itNode).Host.S_addr == newNode.Host.S_addr && (*itNode).Port == newNode.Port)
						bDuplicate = true;
				if(!bDuplicate)
					m_listCache.push_back(newNode);
			}
		}
		TRACE3("MGnuCache::LoadNodes(): ", m_listCache.size(), " host(s) read");
		fclose(pFile);
	}
	return NULL!=pFile;
}

bool MGnuCache::SaveNodes(const CString& sPath)
{
	MLock lock(m_mutex);
	if( !m_listCache.size() )
		return true; // all OK, we just have nothing to write
	CString strNode;
	FILE * pFile;
	if ( pFile = fopen(ExpandPath(sPath).c_str(), "w") )
	{
		fprintf(pFile, "# this is mutella's host cache file\n");
		fprintf(pFile, "# it is created automatically at the end\n");
		fprintf(pFile, "# of mutella session and updated periodically\n");
		fprintf(pFile, "# during the session\n");
		fprintf(pFile, "# \n");
		for(list<Node>::iterator itNode = m_listCache.begin(); itNode != m_listCache.end(); itNode++)
		{
			strNode  = Ip2Str((*itNode).Host)    + ":";
			strNode += DWrdtoStr((*itNode).Port) + "\n";
			fputs(strNode.c_str(), pFile);
		}
		fclose(pFile);
	}
	return NULL!=pFile;
}

