#line 9 "../../libsrc/putget.m4"
/* Do not edit this file. It is produced from the corresponding .m4 source */
#line 11
/*
 *	Copyright 1996, University Corporation for Atmospheric Research
 *      See netcdf/COPYRIGHT file for copying and redistribution conditions.
 */
/* $Id: putget.m4,v 2.79 2010/05/29 22:25:01 russ Exp $ */

#include "config.h"
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include "netcdf.h"
#include "nc3internal.h"
#include "ncx.h"
#include "fbits.h"
#include "onstack.h"
#ifdef LOCKNUMREC
#  include <mpp/shmem.h>	/* for SGI/Cray SHMEM routines */
#  ifdef LN_TEST
#    include <stdio.h>
#  endif
#endif
#include "nc3dispatch.h"


#undef MIN  /* system may define MIN somewhere and complain */
#define MIN(mm,nn) (((mm) < (nn)) ? (mm) : (nn))

static int
readNCv(const NC3_INFO* ncp, const NC_var* varp, const size_t* start,
        const size_t nelems, void* value, const nc_type memtype);
static int
writeNCv(NC3_INFO* ncp, const NC_var* varp, const size_t* start,
         const size_t nelems, const void* value, const nc_type memtype);


/* #define ODEBUG 1 */

#if ODEBUG
#include <stdio.h>
/*
 * Print the values of an array of size_t
 */
void
arrayp(const char *label, size_t count, const size_t *array)
{
	(void) fprintf(stderr, "%s", label);
	(void) fputc('\t',stderr);	
	for(; count > 0; count--, array++)
		(void) fprintf(stderr," %lu", (unsigned long)*array);
	(void) fputc('\n',stderr);	
}
#endif /* ODEBUG */


/* Begin fill */
/*
 * This is tunable parameter.
 * It essentially controls the tradeoff between the number of times
 * memcpy() gets called to copy the external data to fill 
 * a large buffer vs the number of times its called to
 * prepare the external data.
 */
#if	_SX
/* NEC SX specific optimization */
#define	NFILL	2048
#else
#define	NFILL	16
#endif


#line 107

/*
 * Next 6 type specific functions
 * Fill a some memory with the default special value.
 * Formerly
NC_arrayfill()
 */
static int
#line 114
NC_fill_schar(
#line 114
	void **xpp,
#line 114
	size_t nelems)	/* how many */
#line 114
{
#line 114
	schar fillp[NFILL * sizeof(double)/X_SIZEOF_CHAR];
#line 114

#line 114
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 114

#line 114
	{
#line 114
		schar *vp = fillp;	/* lower bound of area to be filled */
#line 114
		const schar *const end = vp + nelems;
#line 114
		while(vp < end)
#line 114
		{
#line 114
			*vp++ = NC_FILL_BYTE;
#line 114
		}
#line 114
	}
#line 114
	return ncx_putn_schar_schar(xpp, nelems, fillp);
#line 114
}
#line 114

static int
#line 115
NC_fill_char(
#line 115
	void **xpp,
#line 115
	size_t nelems)	/* how many */
#line 115
{
#line 115
	char fillp[NFILL * sizeof(double)/X_SIZEOF_CHAR];
#line 115

#line 115
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 115

#line 115
	{
#line 115
		char *vp = fillp;	/* lower bound of area to be filled */
#line 115
		const char *const end = vp + nelems;
#line 115
		while(vp < end)
#line 115
		{
#line 115
			*vp++ = NC_FILL_CHAR;
#line 115
		}
#line 115
	}
#line 115
	return ncx_putn_char_char(xpp, nelems, fillp);
#line 115
}
#line 115

static int
#line 116
NC_fill_short(
#line 116
	void **xpp,
#line 116
	size_t nelems)	/* how many */
#line 116
{
#line 116
	short fillp[NFILL * sizeof(double)/X_SIZEOF_SHORT];
#line 116

#line 116
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 116

#line 116
	{
#line 116
		short *vp = fillp;	/* lower bound of area to be filled */
#line 116
		const short *const end = vp + nelems;
#line 116
		while(vp < end)
#line 116
		{
#line 116
			*vp++ = NC_FILL_SHORT;
#line 116
		}
#line 116
	}
#line 116
	return ncx_putn_short_short(xpp, nelems, fillp);
#line 116
}
#line 116


#if (SIZEOF_INT >= X_SIZEOF_INT)
static int
#line 119
NC_fill_int(
#line 119
	void **xpp,
#line 119
	size_t nelems)	/* how many */
#line 119
{
#line 119
	int fillp[NFILL * sizeof(double)/X_SIZEOF_INT];
#line 119

#line 119
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 119

#line 119
	{
#line 119
		int *vp = fillp;	/* lower bound of area to be filled */
#line 119
		const int *const end = vp + nelems;
#line 119
		while(vp < end)
#line 119
		{
#line 119
			*vp++ = NC_FILL_INT;
#line 119
		}
#line 119
	}
#line 119
	return ncx_putn_int_int(xpp, nelems, fillp);
#line 119
}
#line 119

#elif SIZEOF_LONG == X_SIZEOF_INT
static int
#line 121
NC_fill_int(
#line 121
	void **xpp,
#line 121
	size_t nelems)	/* how many */
#line 121
{
#line 121
	long fillp[NFILL * sizeof(double)/X_SIZEOF_INT];
#line 121

#line 121
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 121

#line 121
	{
#line 121
		long *vp = fillp;	/* lower bound of area to be filled */
#line 121
		const long *const end = vp + nelems;
#line 121
		while(vp < end)
#line 121
		{
#line 121
			*vp++ = NC_FILL_INT;
#line 121
		}
#line 121
	}
#line 121
	return ncx_putn_int_long(xpp, nelems, fillp);
#line 121
}
#line 121

#else
#error "NC_fill_int implementation"
#endif

static int
#line 126
NC_fill_float(
#line 126
	void **xpp,
#line 126
	size_t nelems)	/* how many */
#line 126
{
#line 126
	float fillp[NFILL * sizeof(double)/X_SIZEOF_FLOAT];
#line 126

#line 126
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 126

#line 126
	{
#line 126
		float *vp = fillp;	/* lower bound of area to be filled */
#line 126
		const float *const end = vp + nelems;
#line 126
		while(vp < end)
#line 126
		{
#line 126
			*vp++ = NC_FILL_FLOAT;
#line 126
		}
#line 126
	}
#line 126
	return ncx_putn_float_float(xpp, nelems, fillp);
#line 126
}
#line 126

static int
#line 127
NC_fill_double(
#line 127
	void **xpp,
#line 127
	size_t nelems)	/* how many */
#line 127
{
#line 127
	double fillp[NFILL * sizeof(double)/X_SIZEOF_DOUBLE];
#line 127

#line 127
	assert(nelems <= sizeof(fillp)/sizeof(fillp[0]));
#line 127

#line 127
	{
#line 127
		double *vp = fillp;	/* lower bound of area to be filled */
#line 127
		const double *const end = vp + nelems;
#line 127
		while(vp < end)
#line 127
		{
#line 127
			*vp++ = NC_FILL_DOUBLE;
#line 127
		}
#line 127
	}
#line 127
	return ncx_putn_double_double(xpp, nelems, fillp);
#line 127
}
#line 127





/* 
 * Fill the external space for variable 'varp' values at 'recno' with
 * the appropriate value. If 'varp' is not a record variable, fill the
 * whole thing.  For the special case when 'varp' is the only record
 * variable and it is of type byte, char, or short, varsize should be
 * ncp->recsize, otherwise it should be varp->len.
 * Formerly
xdr_NC_fill()
 */
int
fill_NC_var(NC3_INFO* ncp, const NC_var *varp, size_t varsize, size_t recno)
{
	char xfillp[NFILL * X_SIZEOF_DOUBLE];
	const size_t step = varp->xsz;
	const size_t nelems = sizeof(xfillp)/step;
	const size_t xsz = varp->xsz * nelems;
	NC_attr **attrpp = NULL;
	off_t offset;
	size_t remaining = varsize;

	void *xp;
	int status = NC_NOERR;

	/*
	 * Set up fill value
	 */
	attrpp = NC_findattr(&varp->attrs, _FillValue);
	if( attrpp != NULL )
	{
		/* User defined fill value */
		if( (*attrpp)->type != varp->type || (*attrpp)->nelems != 1 )
		{
			return NC_EBADTYPE;
		}
		else
		{
			/* Use the user defined value */
			char *cp = xfillp;
			const char *const end = &xfillp[sizeof(xfillp)];

			assert(step <= (*attrpp)->xsz);

			for( /*NADA*/; cp < end; cp += step)
			{
				(void) memcpy(cp, (*attrpp)->xvalue, step);
			}
		}
	}
	else
	{
		/* use the default */
		
		assert(xsz % X_ALIGN == 0);
		assert(xsz <= sizeof(xfillp));
	
		xp = xfillp;
	
		switch(varp->type){
		case NC_BYTE :
			status = NC_fill_schar(&xp, nelems);
			break;
		case NC_CHAR :
			status = NC_fill_char(&xp, nelems);
			break;
		case NC_SHORT :
			status = NC_fill_short(&xp, nelems);
			break;
		case NC_INT :
			status = NC_fill_int(&xp, nelems);
			break;
		case NC_FLOAT :
			status = NC_fill_float(&xp, nelems);
			break;
		case NC_DOUBLE : 
			status = NC_fill_double(&xp, nelems);
			break;
		default :
			assert("fill_NC_var invalid type" == 0);
			status = NC_EBADTYPE;
			break;
		}
		if(status != NC_NOERR)
			return status;
	
		assert(xp == xfillp + xsz);
	}

	/*
	 * copyout:
	 * xfillp now contains 'nelems' elements of the fill value
	 * in external representation.
	 */

	/*
	 * Copy it out.
	 */

	offset = varp->begin;
	if(IS_RECVAR(varp))
	{
		offset += (off_t)ncp->recsize * recno;
	}

	assert(remaining > 0);
	for(;;)
	{
		const size_t chunksz = MIN(remaining, ncp->chunk);
		size_t ii;

		status = ncio_get(ncp->nciop, offset, chunksz,
				 RGN_WRITE, &xp);	
		if(status != NC_NOERR)
		{
			return status;
		}

		/*
		 * fill the chunksz buffer in units  of xsz
		 */
		for(ii = 0; ii < chunksz/xsz; ii++)
		{
			(void) memcpy(xp, xfillp, xsz);
			xp = (char *)xp + xsz;
		}
		/*
		 * Deal with any remainder
		 */
		{
			const size_t rem = chunksz % xsz;
			if(rem != 0)
			{
				(void) memcpy(xp, xfillp, rem);
				/* xp = (char *)xp + xsz; */
			}

		}

		status = ncio_rel(ncp->nciop, offset, RGN_MODIFIED);

		if(status != NC_NOERR)
		{
			break;
		}

		remaining -= chunksz;
		if(remaining == 0)
			break;	/* normal loop exit */
		offset += chunksz;

	}

	return status;
}
/* End fill */


/*
 * Add a record containing the fill values.
 */
static int
NCfillrecord(NC3_INFO* ncp, const NC_var *const *varpp, size_t recno)
{
	size_t ii = 0;
	for(; ii < ncp->vars.nelems; ii++, varpp++)
	{
		if( !IS_RECVAR(*varpp) )
		{
			continue;	/* skip non-record variables */
		}
		{
		const int status = fill_NC_var(ncp, *varpp, (*varpp)->len, recno);
		if(status != NC_NOERR)
			return status;
		}
	}
	return NC_NOERR;
}


/*
 * Add a record containing the fill values in the special case when
 * there is exactly one record variable, where we don't require each
 * record to be four-byte aligned (no record padding).
 */
static int
NCfillspecialrecord(NC3_INFO* ncp, const NC_var *varp, size_t recno)
{
    int status;
    assert(IS_RECVAR(varp));
    status = fill_NC_var(ncp, varp, ncp->recsize, recno);
    if(status != NC_NOERR)
	return status;
    return NC_NOERR;
}


/*
 * It is advantageous to
 * #define TOUCH_LAST
 * when using memory mapped io.
 */
#if TOUCH_LAST
/*
 * Grow the file to a size which can contain recno
 */
static int
NCtouchlast(NC3_INFO* ncp, const NC_var *const *varpp, size_t recno)
{
	int status = NC_NOERR;
	const NC_var *varp = NULL;
	
	{
	size_t ii = 0;
	for(; ii < ncp->vars.nelems; ii++, varpp++)
	{
		if( !IS_RECVAR(*varpp) )
		{
			continue;	/* skip non-record variables */
		}
		varp = *varpp;
	}
	}
	assert(varp != NULL);
	assert( IS_RECVAR(varp) );
	{
		const off_t offset = varp->begin
				+ (off_t)(recno-1) * (off_t)ncp->recsize
				+ (off_t)(varp->len - varp->xsz);
		void *xp;


		status = ncio_get(ncp->nciop, offset, varp->xsz,
				 RGN_WRITE, &xp);	
		if(status != NC_NOERR)
			return status;
		(void)memset(xp, 0, varp->xsz);
		status = ncio_rel(ncp->nciop, offset, RGN_MODIFIED);
	}
	return status;
}
#endif /* TOUCH_LAST */


/*
 * Ensure that the netcdf file has 'numrecs' records,
 * add records and fill as neccessary.
 */
static int
NCvnrecs(NC3_INFO* ncp, size_t numrecs)
{
	int status = NC_NOERR;
#ifdef LOCKNUMREC
	ushmem_t myticket = 0, nowserving = 0;
	ushmem_t numpe = (ushmem_t) _num_pes();

	/* get ticket and wait */
	myticket = shmem_short_finc((shmem_t *) ncp->lock + LOCKNUMREC_LOCK,
		ncp->lock[LOCKNUMREC_BASEPE]);
#ifdef LN_TEST
		fprintf(stderr,"%d of %d : ticket = %hu\n",
			_my_pe(), _num_pes(), myticket);
#endif
	do {
		shmem_short_get((shmem_t *) &nowserving,
			(shmem_t *) ncp->lock + LOCKNUMREC_SERVING, 1,
			ncp->lock[LOCKNUMREC_BASEPE]);
#ifdef LN_TEST
		fprintf(stderr,"%d of %d : serving = %hu\n",
			_my_pe(), _num_pes(), nowserving);
#endif
		/* work-around for non-unique tickets */
		if (nowserving > myticket && nowserving < myticket + numpe ) {
			/* get a new ticket ... you've been bypassed */ 
			/* and handle the unlikely wrap-around effect */
			myticket = shmem_short_finc(
				(shmem_t *) ncp->lock + LOCKNUMREC_LOCK,
				ncp->lock[LOCKNUMREC_BASEPE]);
#ifdef LN_TEST
				fprintf(stderr,"%d of %d : new ticket = %hu\n",
					_my_pe(), _num_pes(), myticket);
#endif
		}
	} while(nowserving != myticket);
	/* now our turn to check & update value */
#endif

	if(numrecs > NC_get_numrecs(ncp))
	{


#if TOUCH_LAST
		status = NCtouchlast(ncp,
			(const NC_var *const*)ncp->vars.value,
			numrecs);
		if(status != NC_NOERR)
			goto common_return;
#endif /* TOUCH_LAST */

		set_NC_ndirty(ncp);

		if(!NC_dofill(ncp))
		{
			/* Simply set the new numrecs value */
			NC_set_numrecs(ncp, numrecs);
		}
		else
		{
		    /* Treat two cases differently: 
		        - exactly one record variable (no padding)
                        - multiple record variables (each record padded 
                          to 4-byte alignment)
		    */
		    NC_var **vpp = (NC_var **)ncp->vars.value;
		    NC_var *const *const end = &vpp[ncp->vars.nelems];
		    NC_var *recvarp = NULL;	/* last record var */
		    int numrecvars = 0;
		    size_t cur_nrecs;
		    
		    /* determine how many record variables */
		    for( /*NADA*/; vpp < end; vpp++) {
			if(IS_RECVAR(*vpp)) {
			    recvarp = *vpp;
			    numrecvars++;
			}
		    }
		    
		    if (numrecvars != 1) { /* usual case */
			/* Fill each record out to numrecs */
			while((cur_nrecs = NC_get_numrecs(ncp)) < numrecs)
			    {
				status = NCfillrecord(ncp,
					(const NC_var *const*)ncp->vars.value,
					cur_nrecs);
				if(status != NC_NOERR)
				{
					break;
				}
				NC_increase_numrecs(ncp, cur_nrecs +1);
			}
			if(status != NC_NOERR)
				goto common_return;
		    } else {	/* special case */
			/* Fill each record out to numrecs */
			while((cur_nrecs = NC_get_numrecs(ncp)) < numrecs)
			    {
				status = NCfillspecialrecord(ncp,
					recvarp,
					cur_nrecs);
				if(status != NC_NOERR)
				{
					break;
				}
				NC_increase_numrecs(ncp, cur_nrecs +1);
			}
			if(status != NC_NOERR)
				goto common_return;
			
		    }
		}

		if(NC_doNsync(ncp))
		{
			status = write_numrecs(ncp);
		}

	}
common_return:
#ifdef LOCKNUMREC
	/* finished with our lock - increment serving number */
	(void) shmem_short_finc((shmem_t *) ncp->lock + LOCKNUMREC_SERVING,
		ncp->lock[LOCKNUMREC_BASEPE]);
#endif
	return status;
}


/* 
 * Check whether 'coord' values are valid for the variable.
 */
static int
NCcoordck(NC3_INFO* ncp, const NC_var *varp, const size_t *coord)
{
	const size_t *ip;
	size_t *up;

	if(varp->ndims == 0)
		return NC_NOERR;	/* 'scalar' variable */

	if(IS_RECVAR(varp))
	{
		if(*coord > X_UINT_MAX) /* rkr: bug fix from previous X_INT_MAX */
			return NC_EINVALCOORDS; /* sanity check */
		if(NC_readonly(ncp) && *coord >= NC_get_numrecs(ncp))
		{
			if(!NC_doNsync(ncp))
				return NC_EINVALCOORDS;
			/* else */
			{
				/* Update from disk and check again */
				const int status = read_numrecs(ncp);
				if(status != NC_NOERR)
					return status;
				if(*coord >= NC_get_numrecs(ncp))
					return NC_EINVALCOORDS;
			}
		}
		ip = coord + 1;
		up = varp->shape + 1;
	}
	else
	{
		ip = coord;
		up = varp->shape;
	}
	
#ifdef CDEBUG
fprintf(stderr,"	NCcoordck: coord %ld, count %d, ip %ld\n",
		coord, varp->ndims, ip );
#endif /* CDEBUG */

	for(; ip < coord + varp->ndims; ip++, up++)
	{

#ifdef CDEBUG
fprintf(stderr,"	NCcoordck: ip %p, *ip %ld, up %p, *up %lu\n",
			ip, *ip, up, *up );
#endif /* CDEBUG */

		/* cast needed for braindead systems with signed size_t */
		if((unsigned long) *ip >= (unsigned long) *up )
			return NC_EINVALCOORDS;
	}

	return NC_NOERR;
}


/* 
 * Check whether 'edges' are valid for the variable and 'start'
 */
/*ARGSUSED*/
static int
NCedgeck(const NC3_INFO* ncp, const NC_var *varp,
	 const size_t *start, const size_t *edges)
{
	const size_t *const end = start + varp->ndims;
	const size_t *shp = varp->shape;

	if(varp->ndims == 0)
		return NC_NOERR;	/* 'scalar' variable */

	if(IS_RECVAR(varp))
	{
		start++;
		edges++;
		shp++;
	}

	for(; start < end; start++, edges++, shp++)
	{
		/* cast needed for braindead systems with signed size_t */
		if((unsigned long) *edges > *shp ||
			(unsigned long) *start + (unsigned long) *edges > *shp)
		{
			return(NC_EEDGE);
		}
	}
	return NC_NOERR;
}


/* 
 * Translate the (variable, coord) pair into a seek index
 */
static off_t
NC_varoffset(const NC3_INFO* ncp, const NC_var *varp, const size_t *coord)
{
	if(varp->ndims == 0) /* 'scalar' variable */
		return varp->begin;

	if(varp->ndims == 1)
	{
		if(IS_RECVAR(varp))
			return varp->begin +
				 (off_t)(*coord) * (off_t)ncp->recsize;
		/* else */
		return varp->begin + (off_t)(*coord) * (off_t)varp->xsz;
	}
	/* else */
	{
		off_t lcoord = (off_t)coord[varp->ndims -1];

		off_t *up = varp->dsizes +1;
		const size_t *ip = coord;
		const off_t *const end = varp->dsizes + varp->ndims;
		
		if(IS_RECVAR(varp))
			up++, ip++;

		for(; up < end; up++, ip++)
			lcoord += (off_t)(*up) * (off_t)(*ip);

		lcoord *= varp->xsz;
		
		if(IS_RECVAR(varp))
			lcoord += (off_t)(*coord) * ncp->recsize;
		
		lcoord += varp->begin;
		return lcoord;
	}
}


#line 700

static int
#line 701
putNCvx_char_char(NC3_INFO* ncp, const NC_var *varp,
#line 701
		 const size_t *start, size_t nelems, const char *value)
#line 701
{
#line 701
	off_t offset = NC_varoffset(ncp, varp, start);
#line 701
	size_t remaining = varp->xsz * nelems;
#line 701
	int status = NC_NOERR;
#line 701
	void *xp;
#line 701

#line 701
	if(nelems == 0)
#line 701
		return NC_NOERR;
#line 701

#line 701
	assert(value != NULL);
#line 701

#line 701
	for(;;)
#line 701
	{
#line 701
		size_t extent = MIN(remaining, ncp->chunk);
#line 701
		size_t nput = ncx_howmany(varp->type, extent);
#line 701

#line 701
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 701
				 RGN_WRITE, &xp);	
#line 701
		if(lstatus != NC_NOERR)
#line 701
			return lstatus;
#line 701
		
#line 701
		lstatus = ncx_putn_char_char(&xp, nput, value);
#line 701
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 701
		{
#line 701
			/* not fatal to the loop */
#line 701
			status = lstatus;
#line 701
		}
#line 701

#line 701
		(void) ncio_rel(ncp->nciop, offset,
#line 701
				 RGN_MODIFIED);	
#line 701

#line 701
		remaining -= extent;
#line 701
		if(remaining == 0)
#line 701
			break; /* normal loop exit */
#line 701
		offset += extent;
#line 701
		value += nput;
#line 701

#line 701
	}
#line 701

#line 701
	return status;
#line 701
}
#line 701


static int
#line 703
putNCvx_schar_schar(NC3_INFO* ncp, const NC_var *varp,
#line 703
		 const size_t *start, size_t nelems, const schar *value)
#line 703
{
#line 703
	off_t offset = NC_varoffset(ncp, varp, start);
#line 703
	size_t remaining = varp->xsz * nelems;
#line 703
	int status = NC_NOERR;
#line 703
	void *xp;
#line 703

#line 703
	if(nelems == 0)
#line 703
		return NC_NOERR;
#line 703

#line 703
	assert(value != NULL);
#line 703

#line 703
	for(;;)
#line 703
	{
#line 703
		size_t extent = MIN(remaining, ncp->chunk);
#line 703
		size_t nput = ncx_howmany(varp->type, extent);
#line 703

#line 703
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 703
				 RGN_WRITE, &xp);	
#line 703
		if(lstatus != NC_NOERR)
#line 703
			return lstatus;
#line 703
		
#line 703
		lstatus = ncx_putn_schar_schar(&xp, nput, value);
#line 703
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 703
		{
#line 703
			/* not fatal to the loop */
#line 703
			status = lstatus;
#line 703
		}
#line 703

#line 703
		(void) ncio_rel(ncp->nciop, offset,
#line 703
				 RGN_MODIFIED);	
#line 703

#line 703
		remaining -= extent;
#line 703
		if(remaining == 0)
#line 703
			break; /* normal loop exit */
#line 703
		offset += extent;
#line 703
		value += nput;
#line 703

#line 703
	}
#line 703

#line 703
	return status;
#line 703
}
#line 703

static int
#line 704
putNCvx_schar_uchar(NC3_INFO* ncp, const NC_var *varp,
#line 704
		 const size_t *start, size_t nelems, const uchar *value)
#line 704
{
#line 704
	off_t offset = NC_varoffset(ncp, varp, start);
#line 704
	size_t remaining = varp->xsz * nelems;
#line 704
	int status = NC_NOERR;
#line 704
	void *xp;
#line 704

#line 704
	if(nelems == 0)
#line 704
		return NC_NOERR;
#line 704

#line 704
	assert(value != NULL);
#line 704

#line 704
	for(;;)
#line 704
	{
#line 704
		size_t extent = MIN(remaining, ncp->chunk);
#line 704
		size_t nput = ncx_howmany(varp->type, extent);
#line 704

#line 704
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 704
				 RGN_WRITE, &xp);	
#line 704
		if(lstatus != NC_NOERR)
#line 704
			return lstatus;
#line 704
		
#line 704
		lstatus = ncx_putn_schar_uchar(&xp, nput, value);
#line 704
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 704
		{
#line 704
			/* not fatal to the loop */
#line 704
			status = lstatus;
#line 704
		}
#line 704

#line 704
		(void) ncio_rel(ncp->nciop, offset,
#line 704
				 RGN_MODIFIED);	
#line 704

#line 704
		remaining -= extent;
#line 704
		if(remaining == 0)
#line 704
			break; /* normal loop exit */
#line 704
		offset += extent;
#line 704
		value += nput;
#line 704

#line 704
	}
#line 704

#line 704
	return status;
#line 704
}
#line 704

static int
#line 705
putNCvx_schar_short(NC3_INFO* ncp, const NC_var *varp,
#line 705
		 const size_t *start, size_t nelems, const short *value)
#line 705
{
#line 705
	off_t offset = NC_varoffset(ncp, varp, start);
#line 705
	size_t remaining = varp->xsz * nelems;
#line 705
	int status = NC_NOERR;
#line 705
	void *xp;
#line 705

#line 705
	if(nelems == 0)
#line 705
		return NC_NOERR;
#line 705

#line 705
	assert(value != NULL);
#line 705

#line 705
	for(;;)
#line 705
	{
#line 705
		size_t extent = MIN(remaining, ncp->chunk);
#line 705
		size_t nput = ncx_howmany(varp->type, extent);
#line 705

#line 705
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 705
				 RGN_WRITE, &xp);	
#line 705
		if(lstatus != NC_NOERR)
#line 705
			return lstatus;
#line 705
		
#line 705
		lstatus = ncx_putn_schar_short(&xp, nput, value);
#line 705
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 705
		{
#line 705
			/* not fatal to the loop */
#line 705
			status = lstatus;
#line 705
		}
#line 705

#line 705
		(void) ncio_rel(ncp->nciop, offset,
#line 705
				 RGN_MODIFIED);	
#line 705

#line 705
		remaining -= extent;
#line 705
		if(remaining == 0)
#line 705
			break; /* normal loop exit */
#line 705
		offset += extent;
#line 705
		value += nput;
#line 705

#line 705
	}
#line 705

#line 705
	return status;
#line 705
}
#line 705

static int
#line 706
putNCvx_schar_int(NC3_INFO* ncp, const NC_var *varp,
#line 706
		 const size_t *start, size_t nelems, const int *value)
#line 706
{
#line 706
	off_t offset = NC_varoffset(ncp, varp, start);
#line 706
	size_t remaining = varp->xsz * nelems;
#line 706
	int status = NC_NOERR;
#line 706
	void *xp;
#line 706

#line 706
	if(nelems == 0)
#line 706
		return NC_NOERR;
#line 706

#line 706
	assert(value != NULL);
#line 706

#line 706
	for(;;)
#line 706
	{
#line 706
		size_t extent = MIN(remaining, ncp->chunk);
#line 706
		size_t nput = ncx_howmany(varp->type, extent);
#line 706

#line 706
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 706
				 RGN_WRITE, &xp);	
#line 706
		if(lstatus != NC_NOERR)
#line 706
			return lstatus;
#line 706
		
#line 706
		lstatus = ncx_putn_schar_int(&xp, nput, value);
#line 706
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 706
		{
#line 706
			/* not fatal to the loop */
#line 706
			status = lstatus;
#line 706
		}
#line 706

#line 706
		(void) ncio_rel(ncp->nciop, offset,
#line 706
				 RGN_MODIFIED);	
#line 706

#line 706
		remaining -= extent;
#line 706
		if(remaining == 0)
#line 706
			break; /* normal loop exit */
#line 706
		offset += extent;
#line 706
		value += nput;
#line 706

#line 706
	}
#line 706

#line 706
	return status;
#line 706
}
#line 706

static int
#line 707
putNCvx_schar_float(NC3_INFO* ncp, const NC_var *varp,
#line 707
		 const size_t *start, size_t nelems, const float *value)
#line 707
{
#line 707
	off_t offset = NC_varoffset(ncp, varp, start);
#line 707
	size_t remaining = varp->xsz * nelems;
#line 707
	int status = NC_NOERR;
#line 707
	void *xp;
#line 707

#line 707
	if(nelems == 0)
#line 707
		return NC_NOERR;
#line 707

#line 707
	assert(value != NULL);
#line 707

#line 707
	for(;;)
#line 707
	{
#line 707
		size_t extent = MIN(remaining, ncp->chunk);
#line 707
		size_t nput = ncx_howmany(varp->type, extent);
#line 707

#line 707
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 707
				 RGN_WRITE, &xp);	
#line 707
		if(lstatus != NC_NOERR)
#line 707
			return lstatus;
#line 707
		
#line 707
		lstatus = ncx_putn_schar_float(&xp, nput, value);
#line 707
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 707
		{
#line 707
			/* not fatal to the loop */
#line 707
			status = lstatus;
#line 707
		}
#line 707

#line 707
		(void) ncio_rel(ncp->nciop, offset,
#line 707
				 RGN_MODIFIED);	
#line 707

#line 707
		remaining -= extent;
#line 707
		if(remaining == 0)
#line 707
			break; /* normal loop exit */
#line 707
		offset += extent;
#line 707
		value += nput;
#line 707

#line 707
	}
#line 707

#line 707
	return status;
#line 707
}
#line 707

static int
#line 708
putNCvx_schar_double(NC3_INFO* ncp, const NC_var *varp,
#line 708
		 const size_t *start, size_t nelems, const double *value)
#line 708
{
#line 708
	off_t offset = NC_varoffset(ncp, varp, start);
#line 708
	size_t remaining = varp->xsz * nelems;
#line 708
	int status = NC_NOERR;
#line 708
	void *xp;
#line 708

#line 708
	if(nelems == 0)
#line 708
		return NC_NOERR;
#line 708

#line 708
	assert(value != NULL);
#line 708

#line 708
	for(;;)
#line 708
	{
#line 708
		size_t extent = MIN(remaining, ncp->chunk);
#line 708
		size_t nput = ncx_howmany(varp->type, extent);
#line 708

#line 708
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 708
				 RGN_WRITE, &xp);	
#line 708
		if(lstatus != NC_NOERR)
#line 708
			return lstatus;
#line 708
		
#line 708
		lstatus = ncx_putn_schar_double(&xp, nput, value);
#line 708
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 708
		{
#line 708
			/* not fatal to the loop */
#line 708
			status = lstatus;
#line 708
		}
#line 708

#line 708
		(void) ncio_rel(ncp->nciop, offset,
#line 708
				 RGN_MODIFIED);	
#line 708

#line 708
		remaining -= extent;
#line 708
		if(remaining == 0)
#line 708
			break; /* normal loop exit */
#line 708
		offset += extent;
#line 708
		value += nput;
#line 708

#line 708
	}
#line 708

#line 708
	return status;
#line 708
}
#line 708

static int
#line 709
putNCvx_schar_longlong(NC3_INFO* ncp, const NC_var *varp,
#line 709
		 const size_t *start, size_t nelems, const longlong *value)
#line 709
{
#line 709
	off_t offset = NC_varoffset(ncp, varp, start);
#line 709
	size_t remaining = varp->xsz * nelems;
#line 709
	int status = NC_NOERR;
#line 709
	void *xp;
#line 709

#line 709
	if(nelems == 0)
#line 709
		return NC_NOERR;
#line 709

#line 709
	assert(value != NULL);
#line 709

#line 709
	for(;;)
#line 709
	{
#line 709
		size_t extent = MIN(remaining, ncp->chunk);
#line 709
		size_t nput = ncx_howmany(varp->type, extent);
#line 709

#line 709
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 709
				 RGN_WRITE, &xp);	
#line 709
		if(lstatus != NC_NOERR)
#line 709
			return lstatus;
#line 709
		
#line 709
		lstatus = ncx_putn_schar_longlong(&xp, nput, value);
#line 709
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 709
		{
#line 709
			/* not fatal to the loop */
#line 709
			status = lstatus;
#line 709
		}
#line 709

#line 709
		(void) ncio_rel(ncp->nciop, offset,
#line 709
				 RGN_MODIFIED);	
#line 709

#line 709
		remaining -= extent;
#line 709
		if(remaining == 0)
#line 709
			break; /* normal loop exit */
#line 709
		offset += extent;
#line 709
		value += nput;
#line 709

#line 709
	}
#line 709

#line 709
	return status;
#line 709
}
#line 709


static int
#line 711
putNCvx_short_schar(NC3_INFO* ncp, const NC_var *varp,
#line 711
		 const size_t *start, size_t nelems, const schar *value)
#line 711
{
#line 711
	off_t offset = NC_varoffset(ncp, varp, start);
#line 711
	size_t remaining = varp->xsz * nelems;
#line 711
	int status = NC_NOERR;
#line 711
	void *xp;
#line 711

#line 711
	if(nelems == 0)
#line 711
		return NC_NOERR;
#line 711

#line 711
	assert(value != NULL);
#line 711

#line 711
	for(;;)
#line 711
	{
#line 711
		size_t extent = MIN(remaining, ncp->chunk);
#line 711
		size_t nput = ncx_howmany(varp->type, extent);
#line 711

#line 711
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 711
				 RGN_WRITE, &xp);	
#line 711
		if(lstatus != NC_NOERR)
#line 711
			return lstatus;
#line 711
		
#line 711
		lstatus = ncx_putn_short_schar(&xp, nput, value);
#line 711
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 711
		{
#line 711
			/* not fatal to the loop */
#line 711
			status = lstatus;
#line 711
		}
#line 711

#line 711
		(void) ncio_rel(ncp->nciop, offset,
#line 711
				 RGN_MODIFIED);	
#line 711

#line 711
		remaining -= extent;
#line 711
		if(remaining == 0)
#line 711
			break; /* normal loop exit */
#line 711
		offset += extent;
#line 711
		value += nput;
#line 711

#line 711
	}
#line 711

#line 711
	return status;
#line 711
}
#line 711

static int
#line 712
putNCvx_short_uchar(NC3_INFO* ncp, const NC_var *varp,
#line 712
		 const size_t *start, size_t nelems, const uchar *value)
#line 712
{
#line 712
	off_t offset = NC_varoffset(ncp, varp, start);
#line 712
	size_t remaining = varp->xsz * nelems;
#line 712
	int status = NC_NOERR;
#line 712
	void *xp;
#line 712

#line 712
	if(nelems == 0)
#line 712
		return NC_NOERR;
#line 712

#line 712
	assert(value != NULL);
#line 712

#line 712
	for(;;)
#line 712
	{
#line 712
		size_t extent = MIN(remaining, ncp->chunk);
#line 712
		size_t nput = ncx_howmany(varp->type, extent);
#line 712

#line 712
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 712
				 RGN_WRITE, &xp);	
#line 712
		if(lstatus != NC_NOERR)
#line 712
			return lstatus;
#line 712
		
#line 712
		lstatus = ncx_putn_short_uchar(&xp, nput, value);
#line 712
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 712
		{
#line 712
			/* not fatal to the loop */
#line 712
			status = lstatus;
#line 712
		}
#line 712

#line 712
		(void) ncio_rel(ncp->nciop, offset,
#line 712
				 RGN_MODIFIED);	
#line 712

#line 712
		remaining -= extent;
#line 712
		if(remaining == 0)
#line 712
			break; /* normal loop exit */
#line 712
		offset += extent;
#line 712
		value += nput;
#line 712

#line 712
	}
#line 712

#line 712
	return status;
#line 712
}
#line 712

static int
#line 713
putNCvx_short_short(NC3_INFO* ncp, const NC_var *varp,
#line 713
		 const size_t *start, size_t nelems, const short *value)
#line 713
{
#line 713
	off_t offset = NC_varoffset(ncp, varp, start);
#line 713
	size_t remaining = varp->xsz * nelems;
#line 713
	int status = NC_NOERR;
#line 713
	void *xp;
#line 713

#line 713
	if(nelems == 0)
#line 713
		return NC_NOERR;
#line 713

#line 713
	assert(value != NULL);
#line 713

#line 713
	for(;;)
#line 713
	{
#line 713
		size_t extent = MIN(remaining, ncp->chunk);
#line 713
		size_t nput = ncx_howmany(varp->type, extent);
#line 713

#line 713
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 713
				 RGN_WRITE, &xp);	
#line 713
		if(lstatus != NC_NOERR)
#line 713
			return lstatus;
#line 713
		
#line 713
		lstatus = ncx_putn_short_short(&xp, nput, value);
#line 713
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 713
		{
#line 713
			/* not fatal to the loop */
#line 713
			status = lstatus;
#line 713
		}
#line 713

#line 713
		(void) ncio_rel(ncp->nciop, offset,
#line 713
				 RGN_MODIFIED);	
#line 713

#line 713
		remaining -= extent;
#line 713
		if(remaining == 0)
#line 713
			break; /* normal loop exit */
#line 713
		offset += extent;
#line 713
		value += nput;
#line 713

#line 713
	}
#line 713

#line 713
	return status;
#line 713
}
#line 713

static int
#line 714
putNCvx_short_int(NC3_INFO* ncp, const NC_var *varp,
#line 714
		 const size_t *start, size_t nelems, const int *value)
#line 714
{
#line 714
	off_t offset = NC_varoffset(ncp, varp, start);
#line 714
	size_t remaining = varp->xsz * nelems;
#line 714
	int status = NC_NOERR;
#line 714
	void *xp;
#line 714

#line 714
	if(nelems == 0)
#line 714
		return NC_NOERR;
#line 714

#line 714
	assert(value != NULL);
#line 714

#line 714
	for(;;)
#line 714
	{
#line 714
		size_t extent = MIN(remaining, ncp->chunk);
#line 714
		size_t nput = ncx_howmany(varp->type, extent);
#line 714

#line 714
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 714
				 RGN_WRITE, &xp);	
#line 714
		if(lstatus != NC_NOERR)
#line 714
			return lstatus;
#line 714
		
#line 714
		lstatus = ncx_putn_short_int(&xp, nput, value);
#line 714
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 714
		{
#line 714
			/* not fatal to the loop */
#line 714
			status = lstatus;
#line 714
		}
#line 714

#line 714
		(void) ncio_rel(ncp->nciop, offset,
#line 714
				 RGN_MODIFIED);	
#line 714

#line 714
		remaining -= extent;
#line 714
		if(remaining == 0)
#line 714
			break; /* normal loop exit */
#line 714
		offset += extent;
#line 714
		value += nput;
#line 714

#line 714
	}
#line 714

#line 714
	return status;
#line 714
}
#line 714

static int
#line 715
putNCvx_short_float(NC3_INFO* ncp, const NC_var *varp,
#line 715
		 const size_t *start, size_t nelems, const float *value)
#line 715
{
#line 715
	off_t offset = NC_varoffset(ncp, varp, start);
#line 715
	size_t remaining = varp->xsz * nelems;
#line 715
	int status = NC_NOERR;
#line 715
	void *xp;
#line 715

#line 715
	if(nelems == 0)
#line 715
		return NC_NOERR;
#line 715

#line 715
	assert(value != NULL);
#line 715

#line 715
	for(;;)
#line 715
	{
#line 715
		size_t extent = MIN(remaining, ncp->chunk);
#line 715
		size_t nput = ncx_howmany(varp->type, extent);
#line 715

#line 715
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 715
				 RGN_WRITE, &xp);	
#line 715
		if(lstatus != NC_NOERR)
#line 715
			return lstatus;
#line 715
		
#line 715
		lstatus = ncx_putn_short_float(&xp, nput, value);
#line 715
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 715
		{
#line 715
			/* not fatal to the loop */
#line 715
			status = lstatus;
#line 715
		}
#line 715

#line 715
		(void) ncio_rel(ncp->nciop, offset,
#line 715
				 RGN_MODIFIED);	
#line 715

#line 715
		remaining -= extent;
#line 715
		if(remaining == 0)
#line 715
			break; /* normal loop exit */
#line 715
		offset += extent;
#line 715
		value += nput;
#line 715

#line 715
	}
#line 715

#line 715
	return status;
#line 715
}
#line 715

static int
#line 716
putNCvx_short_double(NC3_INFO* ncp, const NC_var *varp,
#line 716
		 const size_t *start, size_t nelems, const double *value)
#line 716
{
#line 716
	off_t offset = NC_varoffset(ncp, varp, start);
#line 716
	size_t remaining = varp->xsz * nelems;
#line 716
	int status = NC_NOERR;
#line 716
	void *xp;
#line 716

#line 716
	if(nelems == 0)
#line 716
		return NC_NOERR;
#line 716

#line 716
	assert(value != NULL);
#line 716

#line 716
	for(;;)
#line 716
	{
#line 716
		size_t extent = MIN(remaining, ncp->chunk);
#line 716
		size_t nput = ncx_howmany(varp->type, extent);
#line 716

#line 716
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 716
				 RGN_WRITE, &xp);	
#line 716
		if(lstatus != NC_NOERR)
#line 716
			return lstatus;
#line 716
		
#line 716
		lstatus = ncx_putn_short_double(&xp, nput, value);
#line 716
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 716
		{
#line 716
			/* not fatal to the loop */
#line 716
			status = lstatus;
#line 716
		}
#line 716

#line 716
		(void) ncio_rel(ncp->nciop, offset,
#line 716
				 RGN_MODIFIED);	
#line 716

#line 716
		remaining -= extent;
#line 716
		if(remaining == 0)
#line 716
			break; /* normal loop exit */
#line 716
		offset += extent;
#line 716
		value += nput;
#line 716

#line 716
	}
#line 716

#line 716
	return status;
#line 716
}
#line 716

static int
#line 717
putNCvx_short_longlong(NC3_INFO* ncp, const NC_var *varp,
#line 717
		 const size_t *start, size_t nelems, const longlong *value)
#line 717
{
#line 717
	off_t offset = NC_varoffset(ncp, varp, start);
#line 717
	size_t remaining = varp->xsz * nelems;
#line 717
	int status = NC_NOERR;
#line 717
	void *xp;
#line 717

#line 717
	if(nelems == 0)
#line 717
		return NC_NOERR;
#line 717

#line 717
	assert(value != NULL);
#line 717

#line 717
	for(;;)
#line 717
	{
#line 717
		size_t extent = MIN(remaining, ncp->chunk);
#line 717
		size_t nput = ncx_howmany(varp->type, extent);
#line 717

#line 717
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 717
				 RGN_WRITE, &xp);	
#line 717
		if(lstatus != NC_NOERR)
#line 717
			return lstatus;
#line 717
		
#line 717
		lstatus = ncx_putn_short_longlong(&xp, nput, value);
#line 717
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 717
		{
#line 717
			/* not fatal to the loop */
#line 717
			status = lstatus;
#line 717
		}
#line 717

#line 717
		(void) ncio_rel(ncp->nciop, offset,
#line 717
				 RGN_MODIFIED);	
#line 717

#line 717
		remaining -= extent;
#line 717
		if(remaining == 0)
#line 717
			break; /* normal loop exit */
#line 717
		offset += extent;
#line 717
		value += nput;
#line 717

#line 717
	}
#line 717

#line 717
	return status;
#line 717
}
#line 717


static int
#line 719
putNCvx_int_schar(NC3_INFO* ncp, const NC_var *varp,
#line 719
		 const size_t *start, size_t nelems, const schar *value)
#line 719
{
#line 719
	off_t offset = NC_varoffset(ncp, varp, start);
#line 719
	size_t remaining = varp->xsz * nelems;
#line 719
	int status = NC_NOERR;
#line 719
	void *xp;
#line 719

#line 719
	if(nelems == 0)
#line 719
		return NC_NOERR;
#line 719

#line 719
	assert(value != NULL);
#line 719

#line 719
	for(;;)
#line 719
	{
#line 719
		size_t extent = MIN(remaining, ncp->chunk);
#line 719
		size_t nput = ncx_howmany(varp->type, extent);
#line 719

#line 719
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 719
				 RGN_WRITE, &xp);	
#line 719
		if(lstatus != NC_NOERR)
#line 719
			return lstatus;
#line 719
		
#line 719
		lstatus = ncx_putn_int_schar(&xp, nput, value);
#line 719
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 719
		{
#line 719
			/* not fatal to the loop */
#line 719
			status = lstatus;
#line 719
		}
#line 719

#line 719
		(void) ncio_rel(ncp->nciop, offset,
#line 719
				 RGN_MODIFIED);	
#line 719

#line 719
		remaining -= extent;
#line 719
		if(remaining == 0)
#line 719
			break; /* normal loop exit */
#line 719
		offset += extent;
#line 719
		value += nput;
#line 719

#line 719
	}
#line 719

#line 719
	return status;
#line 719
}
#line 719

static int
#line 720
putNCvx_int_uchar(NC3_INFO* ncp, const NC_var *varp,
#line 720
		 const size_t *start, size_t nelems, const uchar *value)
#line 720
{
#line 720
	off_t offset = NC_varoffset(ncp, varp, start);
#line 720
	size_t remaining = varp->xsz * nelems;
#line 720
	int status = NC_NOERR;
#line 720
	void *xp;
#line 720

#line 720
	if(nelems == 0)
#line 720
		return NC_NOERR;
#line 720

#line 720
	assert(value != NULL);
#line 720

#line 720
	for(;;)
#line 720
	{
#line 720
		size_t extent = MIN(remaining, ncp->chunk);
#line 720
		size_t nput = ncx_howmany(varp->type, extent);
#line 720

#line 720
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 720
				 RGN_WRITE, &xp);	
#line 720
		if(lstatus != NC_NOERR)
#line 720
			return lstatus;
#line 720
		
#line 720
		lstatus = ncx_putn_int_uchar(&xp, nput, value);
#line 720
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 720
		{
#line 720
			/* not fatal to the loop */
#line 720
			status = lstatus;
#line 720
		}
#line 720

#line 720
		(void) ncio_rel(ncp->nciop, offset,
#line 720
				 RGN_MODIFIED);	
#line 720

#line 720
		remaining -= extent;
#line 720
		if(remaining == 0)
#line 720
			break; /* normal loop exit */
#line 720
		offset += extent;
#line 720
		value += nput;
#line 720

#line 720
	}
#line 720

#line 720
	return status;
#line 720
}
#line 720

static int
#line 721
putNCvx_int_short(NC3_INFO* ncp, const NC_var *varp,
#line 721
		 const size_t *start, size_t nelems, const short *value)
#line 721
{
#line 721
	off_t offset = NC_varoffset(ncp, varp, start);
#line 721
	size_t remaining = varp->xsz * nelems;
#line 721
	int status = NC_NOERR;
#line 721
	void *xp;
#line 721

#line 721
	if(nelems == 0)
#line 721
		return NC_NOERR;
#line 721

#line 721
	assert(value != NULL);
#line 721

#line 721
	for(;;)
#line 721
	{
#line 721
		size_t extent = MIN(remaining, ncp->chunk);
#line 721
		size_t nput = ncx_howmany(varp->type, extent);
#line 721

#line 721
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 721
				 RGN_WRITE, &xp);	
#line 721
		if(lstatus != NC_NOERR)
#line 721
			return lstatus;
#line 721
		
#line 721
		lstatus = ncx_putn_int_short(&xp, nput, value);
#line 721
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 721
		{
#line 721
			/* not fatal to the loop */
#line 721
			status = lstatus;
#line 721
		}
#line 721

#line 721
		(void) ncio_rel(ncp->nciop, offset,
#line 721
				 RGN_MODIFIED);	
#line 721

#line 721
		remaining -= extent;
#line 721
		if(remaining == 0)
#line 721
			break; /* normal loop exit */
#line 721
		offset += extent;
#line 721
		value += nput;
#line 721

#line 721
	}
#line 721

#line 721
	return status;
#line 721
}
#line 721

static int
#line 722
putNCvx_int_int(NC3_INFO* ncp, const NC_var *varp,
#line 722
		 const size_t *start, size_t nelems, const int *value)
#line 722
{
#line 722
	off_t offset = NC_varoffset(ncp, varp, start);
#line 722
	size_t remaining = varp->xsz * nelems;
#line 722
	int status = NC_NOERR;
#line 722
	void *xp;
#line 722

#line 722
	if(nelems == 0)
#line 722
		return NC_NOERR;
#line 722

#line 722
	assert(value != NULL);
#line 722

#line 722
	for(;;)
#line 722
	{
#line 722
		size_t extent = MIN(remaining, ncp->chunk);
#line 722
		size_t nput = ncx_howmany(varp->type, extent);
#line 722

#line 722
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 722
				 RGN_WRITE, &xp);	
#line 722
		if(lstatus != NC_NOERR)
#line 722
			return lstatus;
#line 722
		
#line 722
		lstatus = ncx_putn_int_int(&xp, nput, value);
#line 722
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 722
		{
#line 722
			/* not fatal to the loop */
#line 722
			status = lstatus;
#line 722
		}
#line 722

#line 722
		(void) ncio_rel(ncp->nciop, offset,
#line 722
				 RGN_MODIFIED);	
#line 722

#line 722
		remaining -= extent;
#line 722
		if(remaining == 0)
#line 722
			break; /* normal loop exit */
#line 722
		offset += extent;
#line 722
		value += nput;
#line 722

#line 722
	}
#line 722

#line 722
	return status;
#line 722
}
#line 722

static int
#line 723
putNCvx_int_float(NC3_INFO* ncp, const NC_var *varp,
#line 723
		 const size_t *start, size_t nelems, const float *value)
#line 723
{
#line 723
	off_t offset = NC_varoffset(ncp, varp, start);
#line 723
	size_t remaining = varp->xsz * nelems;
#line 723
	int status = NC_NOERR;
#line 723
	void *xp;
#line 723

#line 723
	if(nelems == 0)
#line 723
		return NC_NOERR;
#line 723

#line 723
	assert(value != NULL);
#line 723

#line 723
	for(;;)
#line 723
	{
#line 723
		size_t extent = MIN(remaining, ncp->chunk);
#line 723
		size_t nput = ncx_howmany(varp->type, extent);
#line 723

#line 723
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 723
				 RGN_WRITE, &xp);	
#line 723
		if(lstatus != NC_NOERR)
#line 723
			return lstatus;
#line 723
		
#line 723
		lstatus = ncx_putn_int_float(&xp, nput, value);
#line 723
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 723
		{
#line 723
			/* not fatal to the loop */
#line 723
			status = lstatus;
#line 723
		}
#line 723

#line 723
		(void) ncio_rel(ncp->nciop, offset,
#line 723
				 RGN_MODIFIED);	
#line 723

#line 723
		remaining -= extent;
#line 723
		if(remaining == 0)
#line 723
			break; /* normal loop exit */
#line 723
		offset += extent;
#line 723
		value += nput;
#line 723

#line 723
	}
#line 723

#line 723
	return status;
#line 723
}
#line 723

static int
#line 724
putNCvx_int_double(NC3_INFO* ncp, const NC_var *varp,
#line 724
		 const size_t *start, size_t nelems, const double *value)
#line 724
{
#line 724
	off_t offset = NC_varoffset(ncp, varp, start);
#line 724
	size_t remaining = varp->xsz * nelems;
#line 724
	int status = NC_NOERR;
#line 724
	void *xp;
#line 724

#line 724
	if(nelems == 0)
#line 724
		return NC_NOERR;
#line 724

#line 724
	assert(value != NULL);
#line 724

#line 724
	for(;;)
#line 724
	{
#line 724
		size_t extent = MIN(remaining, ncp->chunk);
#line 724
		size_t nput = ncx_howmany(varp->type, extent);
#line 724

#line 724
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 724
				 RGN_WRITE, &xp);	
#line 724
		if(lstatus != NC_NOERR)
#line 724
			return lstatus;
#line 724
		
#line 724
		lstatus = ncx_putn_int_double(&xp, nput, value);
#line 724
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 724
		{
#line 724
			/* not fatal to the loop */
#line 724
			status = lstatus;
#line 724
		}
#line 724

#line 724
		(void) ncio_rel(ncp->nciop, offset,
#line 724
				 RGN_MODIFIED);	
#line 724

#line 724
		remaining -= extent;
#line 724
		if(remaining == 0)
#line 724
			break; /* normal loop exit */
#line 724
		offset += extent;
#line 724
		value += nput;
#line 724

#line 724
	}
#line 724

#line 724
	return status;
#line 724
}
#line 724

static int
#line 725
putNCvx_int_longlong(NC3_INFO* ncp, const NC_var *varp,
#line 725
		 const size_t *start, size_t nelems, const longlong *value)
#line 725
{
#line 725
	off_t offset = NC_varoffset(ncp, varp, start);
#line 725
	size_t remaining = varp->xsz * nelems;
#line 725
	int status = NC_NOERR;
#line 725
	void *xp;
#line 725

#line 725
	if(nelems == 0)
#line 725
		return NC_NOERR;
#line 725

#line 725
	assert(value != NULL);
#line 725

#line 725
	for(;;)
#line 725
	{
#line 725
		size_t extent = MIN(remaining, ncp->chunk);
#line 725
		size_t nput = ncx_howmany(varp->type, extent);
#line 725

#line 725
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 725
				 RGN_WRITE, &xp);	
#line 725
		if(lstatus != NC_NOERR)
#line 725
			return lstatus;
#line 725
		
#line 725
		lstatus = ncx_putn_int_longlong(&xp, nput, value);
#line 725
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 725
		{
#line 725
			/* not fatal to the loop */
#line 725
			status = lstatus;
#line 725
		}
#line 725

#line 725
		(void) ncio_rel(ncp->nciop, offset,
#line 725
				 RGN_MODIFIED);	
#line 725

#line 725
		remaining -= extent;
#line 725
		if(remaining == 0)
#line 725
			break; /* normal loop exit */
#line 725
		offset += extent;
#line 725
		value += nput;
#line 725

#line 725
	}
#line 725

#line 725
	return status;
#line 725
}
#line 725


static int
#line 727
putNCvx_float_schar(NC3_INFO* ncp, const NC_var *varp,
#line 727
		 const size_t *start, size_t nelems, const schar *value)
#line 727
{
#line 727
	off_t offset = NC_varoffset(ncp, varp, start);
#line 727
	size_t remaining = varp->xsz * nelems;
#line 727
	int status = NC_NOERR;
#line 727
	void *xp;
#line 727

#line 727
	if(nelems == 0)
#line 727
		return NC_NOERR;
#line 727

#line 727
	assert(value != NULL);
#line 727

#line 727
	for(;;)
#line 727
	{
#line 727
		size_t extent = MIN(remaining, ncp->chunk);
#line 727
		size_t nput = ncx_howmany(varp->type, extent);
#line 727

#line 727
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 727
				 RGN_WRITE, &xp);	
#line 727
		if(lstatus != NC_NOERR)
#line 727
			return lstatus;
#line 727
		
#line 727
		lstatus = ncx_putn_float_schar(&xp, nput, value);
#line 727
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 727
		{
#line 727
			/* not fatal to the loop */
#line 727
			status = lstatus;
#line 727
		}
#line 727

#line 727
		(void) ncio_rel(ncp->nciop, offset,
#line 727
				 RGN_MODIFIED);	
#line 727

#line 727
		remaining -= extent;
#line 727
		if(remaining == 0)
#line 727
			break; /* normal loop exit */
#line 727
		offset += extent;
#line 727
		value += nput;
#line 727

#line 727
	}
#line 727

#line 727
	return status;
#line 727
}
#line 727

static int
#line 728
putNCvx_float_uchar(NC3_INFO* ncp, const NC_var *varp,
#line 728
		 const size_t *start, size_t nelems, const uchar *value)
#line 728
{
#line 728
	off_t offset = NC_varoffset(ncp, varp, start);
#line 728
	size_t remaining = varp->xsz * nelems;
#line 728
	int status = NC_NOERR;
#line 728
	void *xp;
#line 728

#line 728
	if(nelems == 0)
#line 728
		return NC_NOERR;
#line 728

#line 728
	assert(value != NULL);
#line 728

#line 728
	for(;;)
#line 728
	{
#line 728
		size_t extent = MIN(remaining, ncp->chunk);
#line 728
		size_t nput = ncx_howmany(varp->type, extent);
#line 728

#line 728
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 728
				 RGN_WRITE, &xp);	
#line 728
		if(lstatus != NC_NOERR)
#line 728
			return lstatus;
#line 728
		
#line 728
		lstatus = ncx_putn_float_uchar(&xp, nput, value);
#line 728
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 728
		{
#line 728
			/* not fatal to the loop */
#line 728
			status = lstatus;
#line 728
		}
#line 728

#line 728
		(void) ncio_rel(ncp->nciop, offset,
#line 728
				 RGN_MODIFIED);	
#line 728

#line 728
		remaining -= extent;
#line 728
		if(remaining == 0)
#line 728
			break; /* normal loop exit */
#line 728
		offset += extent;
#line 728
		value += nput;
#line 728

#line 728
	}
#line 728

#line 728
	return status;
#line 728
}
#line 728

static int
#line 729
putNCvx_float_short(NC3_INFO* ncp, const NC_var *varp,
#line 729
		 const size_t *start, size_t nelems, const short *value)
#line 729
{
#line 729
	off_t offset = NC_varoffset(ncp, varp, start);
#line 729
	size_t remaining = varp->xsz * nelems;
#line 729
	int status = NC_NOERR;
#line 729
	void *xp;
#line 729

#line 729
	if(nelems == 0)
#line 729
		return NC_NOERR;
#line 729

#line 729
	assert(value != NULL);
#line 729

#line 729
	for(;;)
#line 729
	{
#line 729
		size_t extent = MIN(remaining, ncp->chunk);
#line 729
		size_t nput = ncx_howmany(varp->type, extent);
#line 729

#line 729
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 729
				 RGN_WRITE, &xp);	
#line 729
		if(lstatus != NC_NOERR)
#line 729
			return lstatus;
#line 729
		
#line 729
		lstatus = ncx_putn_float_short(&xp, nput, value);
#line 729
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 729
		{
#line 729
			/* not fatal to the loop */
#line 729
			status = lstatus;
#line 729
		}
#line 729

#line 729
		(void) ncio_rel(ncp->nciop, offset,
#line 729
				 RGN_MODIFIED);	
#line 729

#line 729
		remaining -= extent;
#line 729
		if(remaining == 0)
#line 729
			break; /* normal loop exit */
#line 729
		offset += extent;
#line 729
		value += nput;
#line 729

#line 729
	}
#line 729

#line 729
	return status;
#line 729
}
#line 729

static int
#line 730
putNCvx_float_int(NC3_INFO* ncp, const NC_var *varp,
#line 730
		 const size_t *start, size_t nelems, const int *value)
#line 730
{
#line 730
	off_t offset = NC_varoffset(ncp, varp, start);
#line 730
	size_t remaining = varp->xsz * nelems;
#line 730
	int status = NC_NOERR;
#line 730
	void *xp;
#line 730

#line 730
	if(nelems == 0)
#line 730
		return NC_NOERR;
#line 730

#line 730
	assert(value != NULL);
#line 730

#line 730
	for(;;)
#line 730
	{
#line 730
		size_t extent = MIN(remaining, ncp->chunk);
#line 730
		size_t nput = ncx_howmany(varp->type, extent);
#line 730

#line 730
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 730
				 RGN_WRITE, &xp);	
#line 730
		if(lstatus != NC_NOERR)
#line 730
			return lstatus;
#line 730
		
#line 730
		lstatus = ncx_putn_float_int(&xp, nput, value);
#line 730
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 730
		{
#line 730
			/* not fatal to the loop */
#line 730
			status = lstatus;
#line 730
		}
#line 730

#line 730
		(void) ncio_rel(ncp->nciop, offset,
#line 730
				 RGN_MODIFIED);	
#line 730

#line 730
		remaining -= extent;
#line 730
		if(remaining == 0)
#line 730
			break; /* normal loop exit */
#line 730
		offset += extent;
#line 730
		value += nput;
#line 730

#line 730
	}
#line 730

#line 730
	return status;
#line 730
}
#line 730

static int
#line 731
putNCvx_float_float(NC3_INFO* ncp, const NC_var *varp,
#line 731
		 const size_t *start, size_t nelems, const float *value)
#line 731
{
#line 731
	off_t offset = NC_varoffset(ncp, varp, start);
#line 731
	size_t remaining = varp->xsz * nelems;
#line 731
	int status = NC_NOERR;
#line 731
	void *xp;
#line 731

#line 731
	if(nelems == 0)
#line 731
		return NC_NOERR;
#line 731

#line 731
	assert(value != NULL);
#line 731

#line 731
	for(;;)
#line 731
	{
#line 731
		size_t extent = MIN(remaining, ncp->chunk);
#line 731
		size_t nput = ncx_howmany(varp->type, extent);
#line 731

#line 731
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 731
				 RGN_WRITE, &xp);	
#line 731
		if(lstatus != NC_NOERR)
#line 731
			return lstatus;
#line 731
		
#line 731
		lstatus = ncx_putn_float_float(&xp, nput, value);
#line 731
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 731
		{
#line 731
			/* not fatal to the loop */
#line 731
			status = lstatus;
#line 731
		}
#line 731

#line 731
		(void) ncio_rel(ncp->nciop, offset,
#line 731
				 RGN_MODIFIED);	
#line 731

#line 731
		remaining -= extent;
#line 731
		if(remaining == 0)
#line 731
			break; /* normal loop exit */
#line 731
		offset += extent;
#line 731
		value += nput;
#line 731

#line 731
	}
#line 731

#line 731
	return status;
#line 731
}
#line 731

static int
#line 732
putNCvx_float_double(NC3_INFO* ncp, const NC_var *varp,
#line 732
		 const size_t *start, size_t nelems, const double *value)
#line 732
{
#line 732
	off_t offset = NC_varoffset(ncp, varp, start);
#line 732
	size_t remaining = varp->xsz * nelems;
#line 732
	int status = NC_NOERR;
#line 732
	void *xp;
#line 732

#line 732
	if(nelems == 0)
#line 732
		return NC_NOERR;
#line 732

#line 732
	assert(value != NULL);
#line 732

#line 732
	for(;;)
#line 732
	{
#line 732
		size_t extent = MIN(remaining, ncp->chunk);
#line 732
		size_t nput = ncx_howmany(varp->type, extent);
#line 732

#line 732
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 732
				 RGN_WRITE, &xp);	
#line 732
		if(lstatus != NC_NOERR)
#line 732
			return lstatus;
#line 732
		
#line 732
		lstatus = ncx_putn_float_double(&xp, nput, value);
#line 732
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 732
		{
#line 732
			/* not fatal to the loop */
#line 732
			status = lstatus;
#line 732
		}
#line 732

#line 732
		(void) ncio_rel(ncp->nciop, offset,
#line 732
				 RGN_MODIFIED);	
#line 732

#line 732
		remaining -= extent;
#line 732
		if(remaining == 0)
#line 732
			break; /* normal loop exit */
#line 732
		offset += extent;
#line 732
		value += nput;
#line 732

#line 732
	}
#line 732

#line 732
	return status;
#line 732
}
#line 732

static int
#line 733
putNCvx_float_longlong(NC3_INFO* ncp, const NC_var *varp,
#line 733
		 const size_t *start, size_t nelems, const longlong *value)
#line 733
{
#line 733
	off_t offset = NC_varoffset(ncp, varp, start);
#line 733
	size_t remaining = varp->xsz * nelems;
#line 733
	int status = NC_NOERR;
#line 733
	void *xp;
#line 733

#line 733
	if(nelems == 0)
#line 733
		return NC_NOERR;
#line 733

#line 733
	assert(value != NULL);
#line 733

#line 733
	for(;;)
#line 733
	{
#line 733
		size_t extent = MIN(remaining, ncp->chunk);
#line 733
		size_t nput = ncx_howmany(varp->type, extent);
#line 733

#line 733
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 733
				 RGN_WRITE, &xp);	
#line 733
		if(lstatus != NC_NOERR)
#line 733
			return lstatus;
#line 733
		
#line 733
		lstatus = ncx_putn_float_longlong(&xp, nput, value);
#line 733
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 733
		{
#line 733
			/* not fatal to the loop */
#line 733
			status = lstatus;
#line 733
		}
#line 733

#line 733
		(void) ncio_rel(ncp->nciop, offset,
#line 733
				 RGN_MODIFIED);	
#line 733

#line 733
		remaining -= extent;
#line 733
		if(remaining == 0)
#line 733
			break; /* normal loop exit */
#line 733
		offset += extent;
#line 733
		value += nput;
#line 733

#line 733
	}
#line 733

#line 733
	return status;
#line 733
}
#line 733


static int
#line 735
putNCvx_double_schar(NC3_INFO* ncp, const NC_var *varp,
#line 735
		 const size_t *start, size_t nelems, const schar *value)
#line 735
{
#line 735
	off_t offset = NC_varoffset(ncp, varp, start);
#line 735
	size_t remaining = varp->xsz * nelems;
#line 735
	int status = NC_NOERR;
#line 735
	void *xp;
#line 735

#line 735
	if(nelems == 0)
#line 735
		return NC_NOERR;
#line 735

#line 735
	assert(value != NULL);
#line 735

#line 735
	for(;;)
#line 735
	{
#line 735
		size_t extent = MIN(remaining, ncp->chunk);
#line 735
		size_t nput = ncx_howmany(varp->type, extent);
#line 735

#line 735
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 735
				 RGN_WRITE, &xp);	
#line 735
		if(lstatus != NC_NOERR)
#line 735
			return lstatus;
#line 735
		
#line 735
		lstatus = ncx_putn_double_schar(&xp, nput, value);
#line 735
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 735
		{
#line 735
			/* not fatal to the loop */
#line 735
			status = lstatus;
#line 735
		}
#line 735

#line 735
		(void) ncio_rel(ncp->nciop, offset,
#line 735
				 RGN_MODIFIED);	
#line 735

#line 735
		remaining -= extent;
#line 735
		if(remaining == 0)
#line 735
			break; /* normal loop exit */
#line 735
		offset += extent;
#line 735
		value += nput;
#line 735

#line 735
	}
#line 735

#line 735
	return status;
#line 735
}
#line 735

static int
#line 736
putNCvx_double_uchar(NC3_INFO* ncp, const NC_var *varp,
#line 736
		 const size_t *start, size_t nelems, const uchar *value)
#line 736
{
#line 736
	off_t offset = NC_varoffset(ncp, varp, start);
#line 736
	size_t remaining = varp->xsz * nelems;
#line 736
	int status = NC_NOERR;
#line 736
	void *xp;
#line 736

#line 736
	if(nelems == 0)
#line 736
		return NC_NOERR;
#line 736

#line 736
	assert(value != NULL);
#line 736

#line 736
	for(;;)
#line 736
	{
#line 736
		size_t extent = MIN(remaining, ncp->chunk);
#line 736
		size_t nput = ncx_howmany(varp->type, extent);
#line 736

#line 736
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 736
				 RGN_WRITE, &xp);	
#line 736
		if(lstatus != NC_NOERR)
#line 736
			return lstatus;
#line 736
		
#line 736
		lstatus = ncx_putn_double_uchar(&xp, nput, value);
#line 736
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 736
		{
#line 736
			/* not fatal to the loop */
#line 736
			status = lstatus;
#line 736
		}
#line 736

#line 736
		(void) ncio_rel(ncp->nciop, offset,
#line 736
				 RGN_MODIFIED);	
#line 736

#line 736
		remaining -= extent;
#line 736
		if(remaining == 0)
#line 736
			break; /* normal loop exit */
#line 736
		offset += extent;
#line 736
		value += nput;
#line 736

#line 736
	}
#line 736

#line 736
	return status;
#line 736
}
#line 736

static int
#line 737
putNCvx_double_short(NC3_INFO* ncp, const NC_var *varp,
#line 737
		 const size_t *start, size_t nelems, const short *value)
#line 737
{
#line 737
	off_t offset = NC_varoffset(ncp, varp, start);
#line 737
	size_t remaining = varp->xsz * nelems;
#line 737
	int status = NC_NOERR;
#line 737
	void *xp;
#line 737

#line 737
	if(nelems == 0)
#line 737
		return NC_NOERR;
#line 737

#line 737
	assert(value != NULL);
#line 737

#line 737
	for(;;)
#line 737
	{
#line 737
		size_t extent = MIN(remaining, ncp->chunk);
#line 737
		size_t nput = ncx_howmany(varp->type, extent);
#line 737

#line 737
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 737
				 RGN_WRITE, &xp);	
#line 737
		if(lstatus != NC_NOERR)
#line 737
			return lstatus;
#line 737
		
#line 737
		lstatus = ncx_putn_double_short(&xp, nput, value);
#line 737
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 737
		{
#line 737
			/* not fatal to the loop */
#line 737
			status = lstatus;
#line 737
		}
#line 737

#line 737
		(void) ncio_rel(ncp->nciop, offset,
#line 737
				 RGN_MODIFIED);	
#line 737

#line 737
		remaining -= extent;
#line 737
		if(remaining == 0)
#line 737
			break; /* normal loop exit */
#line 737
		offset += extent;
#line 737
		value += nput;
#line 737

#line 737
	}
#line 737

#line 737
	return status;
#line 737
}
#line 737

static int
#line 738
putNCvx_double_int(NC3_INFO* ncp, const NC_var *varp,
#line 738
		 const size_t *start, size_t nelems, const int *value)
#line 738
{
#line 738
	off_t offset = NC_varoffset(ncp, varp, start);
#line 738
	size_t remaining = varp->xsz * nelems;
#line 738
	int status = NC_NOERR;
#line 738
	void *xp;
#line 738

#line 738
	if(nelems == 0)
#line 738
		return NC_NOERR;
#line 738

#line 738
	assert(value != NULL);
#line 738

#line 738
	for(;;)
#line 738
	{
#line 738
		size_t extent = MIN(remaining, ncp->chunk);
#line 738
		size_t nput = ncx_howmany(varp->type, extent);
#line 738

#line 738
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 738
				 RGN_WRITE, &xp);	
#line 738
		if(lstatus != NC_NOERR)
#line 738
			return lstatus;
#line 738
		
#line 738
		lstatus = ncx_putn_double_int(&xp, nput, value);
#line 738
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 738
		{
#line 738
			/* not fatal to the loop */
#line 738
			status = lstatus;
#line 738
		}
#line 738

#line 738
		(void) ncio_rel(ncp->nciop, offset,
#line 738
				 RGN_MODIFIED);	
#line 738

#line 738
		remaining -= extent;
#line 738
		if(remaining == 0)
#line 738
			break; /* normal loop exit */
#line 738
		offset += extent;
#line 738
		value += nput;
#line 738

#line 738
	}
#line 738

#line 738
	return status;
#line 738
}
#line 738

static int
#line 739
putNCvx_double_float(NC3_INFO* ncp, const NC_var *varp,
#line 739
		 const size_t *start, size_t nelems, const float *value)
#line 739
{
#line 739
	off_t offset = NC_varoffset(ncp, varp, start);
#line 739
	size_t remaining = varp->xsz * nelems;
#line 739
	int status = NC_NOERR;
#line 739
	void *xp;
#line 739

#line 739
	if(nelems == 0)
#line 739
		return NC_NOERR;
#line 739

#line 739
	assert(value != NULL);
#line 739

#line 739
	for(;;)
#line 739
	{
#line 739
		size_t extent = MIN(remaining, ncp->chunk);
#line 739
		size_t nput = ncx_howmany(varp->type, extent);
#line 739

#line 739
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 739
				 RGN_WRITE, &xp);	
#line 739
		if(lstatus != NC_NOERR)
#line 739
			return lstatus;
#line 739
		
#line 739
		lstatus = ncx_putn_double_float(&xp, nput, value);
#line 739
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 739
		{
#line 739
			/* not fatal to the loop */
#line 739
			status = lstatus;
#line 739
		}
#line 739

#line 739
		(void) ncio_rel(ncp->nciop, offset,
#line 739
				 RGN_MODIFIED);	
#line 739

#line 739
		remaining -= extent;
#line 739
		if(remaining == 0)
#line 739
			break; /* normal loop exit */
#line 739
		offset += extent;
#line 739
		value += nput;
#line 739

#line 739
	}
#line 739

#line 739
	return status;
#line 739
}
#line 739

static int
#line 740
putNCvx_double_double(NC3_INFO* ncp, const NC_var *varp,
#line 740
		 const size_t *start, size_t nelems, const double *value)
#line 740
{
#line 740
	off_t offset = NC_varoffset(ncp, varp, start);
#line 740
	size_t remaining = varp->xsz * nelems;
#line 740
	int status = NC_NOERR;
#line 740
	void *xp;
#line 740

#line 740
	if(nelems == 0)
#line 740
		return NC_NOERR;
#line 740

#line 740
	assert(value != NULL);
#line 740

#line 740
	for(;;)
#line 740
	{
#line 740
		size_t extent = MIN(remaining, ncp->chunk);
#line 740
		size_t nput = ncx_howmany(varp->type, extent);
#line 740

#line 740
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 740
				 RGN_WRITE, &xp);	
#line 740
		if(lstatus != NC_NOERR)
#line 740
			return lstatus;
#line 740
		
#line 740
		lstatus = ncx_putn_double_double(&xp, nput, value);
#line 740
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 740
		{
#line 740
			/* not fatal to the loop */
#line 740
			status = lstatus;
#line 740
		}
#line 740

#line 740
		(void) ncio_rel(ncp->nciop, offset,
#line 740
				 RGN_MODIFIED);	
#line 740

#line 740
		remaining -= extent;
#line 740
		if(remaining == 0)
#line 740
			break; /* normal loop exit */
#line 740
		offset += extent;
#line 740
		value += nput;
#line 740

#line 740
	}
#line 740

#line 740
	return status;
#line 740
}
#line 740

static int
#line 741
putNCvx_double_longlong(NC3_INFO* ncp, const NC_var *varp,
#line 741
		 const size_t *start, size_t nelems, const longlong *value)
#line 741
{
#line 741
	off_t offset = NC_varoffset(ncp, varp, start);
#line 741
	size_t remaining = varp->xsz * nelems;
#line 741
	int status = NC_NOERR;
#line 741
	void *xp;
#line 741

#line 741
	if(nelems == 0)
#line 741
		return NC_NOERR;
#line 741

#line 741
	assert(value != NULL);
#line 741

#line 741
	for(;;)
#line 741
	{
#line 741
		size_t extent = MIN(remaining, ncp->chunk);
#line 741
		size_t nput = ncx_howmany(varp->type, extent);
#line 741

#line 741
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 741
				 RGN_WRITE, &xp);	
#line 741
		if(lstatus != NC_NOERR)
#line 741
			return lstatus;
#line 741
		
#line 741
		lstatus = ncx_putn_double_longlong(&xp, nput, value);
#line 741
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 741
		{
#line 741
			/* not fatal to the loop */
#line 741
			status = lstatus;
#line 741
		}
#line 741

#line 741
		(void) ncio_rel(ncp->nciop, offset,
#line 741
				 RGN_MODIFIED);	
#line 741

#line 741
		remaining -= extent;
#line 741
		if(remaining == 0)
#line 741
			break; /* normal loop exit */
#line 741
		offset += extent;
#line 741
		value += nput;
#line 741

#line 741
	}
#line 741

#line 741
	return status;
#line 741
}
#line 741


#line 744
#ifdef NOTUSED
static int
#line 745
putNCvx_schar_uint(NC3_INFO* ncp, const NC_var *varp,
#line 745
		 const size_t *start, size_t nelems, const uint *value)
#line 745
{
#line 745
	off_t offset = NC_varoffset(ncp, varp, start);
#line 745
	size_t remaining = varp->xsz * nelems;
#line 745
	int status = NC_NOERR;
#line 745
	void *xp;
#line 745

#line 745
	if(nelems == 0)
#line 745
		return NC_NOERR;
#line 745

#line 745
	assert(value != NULL);
#line 745

#line 745
	for(;;)
#line 745
	{
#line 745
		size_t extent = MIN(remaining, ncp->chunk);
#line 745
		size_t nput = ncx_howmany(varp->type, extent);
#line 745

#line 745
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 745
				 RGN_WRITE, &xp);	
#line 745
		if(lstatus != NC_NOERR)
#line 745
			return lstatus;
#line 745
		
#line 745
		lstatus = ncx_putn_schar_uint(&xp, nput, value);
#line 745
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 745
		{
#line 745
			/* not fatal to the loop */
#line 745
			status = lstatus;
#line 745
		}
#line 745

#line 745
		(void) ncio_rel(ncp->nciop, offset,
#line 745
				 RGN_MODIFIED);	
#line 745

#line 745
		remaining -= extent;
#line 745
		if(remaining == 0)
#line 745
			break; /* normal loop exit */
#line 745
		offset += extent;
#line 745
		value += nput;
#line 745

#line 745
	}
#line 745

#line 745
	return status;
#line 745
}
#line 745

static int
#line 746
putNCvx_schar_ulonglong(NC3_INFO* ncp, const NC_var *varp,
#line 746
		 const size_t *start, size_t nelems, const ulonglong *value)
#line 746
{
#line 746
	off_t offset = NC_varoffset(ncp, varp, start);
#line 746
	size_t remaining = varp->xsz * nelems;
#line 746
	int status = NC_NOERR;
#line 746
	void *xp;
#line 746

#line 746
	if(nelems == 0)
#line 746
		return NC_NOERR;
#line 746

#line 746
	assert(value != NULL);
#line 746

#line 746
	for(;;)
#line 746
	{
#line 746
		size_t extent = MIN(remaining, ncp->chunk);
#line 746
		size_t nput = ncx_howmany(varp->type, extent);
#line 746

#line 746
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 746
				 RGN_WRITE, &xp);	
#line 746
		if(lstatus != NC_NOERR)
#line 746
			return lstatus;
#line 746
		
#line 746
		lstatus = ncx_putn_schar_ulonglong(&xp, nput, value);
#line 746
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 746
		{
#line 746
			/* not fatal to the loop */
#line 746
			status = lstatus;
#line 746
		}
#line 746

#line 746
		(void) ncio_rel(ncp->nciop, offset,
#line 746
				 RGN_MODIFIED);	
#line 746

#line 746
		remaining -= extent;
#line 746
		if(remaining == 0)
#line 746
			break; /* normal loop exit */
#line 746
		offset += extent;
#line 746
		value += nput;
#line 746

#line 746
	}
#line 746

#line 746
	return status;
#line 746
}
#line 746

static int
#line 747
putNCvx_short_uint(NC3_INFO* ncp, const NC_var *varp,
#line 747
		 const size_t *start, size_t nelems, const uint *value)
#line 747
{
#line 747
	off_t offset = NC_varoffset(ncp, varp, start);
#line 747
	size_t remaining = varp->xsz * nelems;
#line 747
	int status = NC_NOERR;
#line 747
	void *xp;
#line 747

#line 747
	if(nelems == 0)
#line 747
		return NC_NOERR;
#line 747

#line 747
	assert(value != NULL);
#line 747

#line 747
	for(;;)
#line 747
	{
#line 747
		size_t extent = MIN(remaining, ncp->chunk);
#line 747
		size_t nput = ncx_howmany(varp->type, extent);
#line 747

#line 747
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 747
				 RGN_WRITE, &xp);	
#line 747
		if(lstatus != NC_NOERR)
#line 747
			return lstatus;
#line 747
		
#line 747
		lstatus = ncx_putn_short_uint(&xp, nput, value);
#line 747
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 747
		{
#line 747
			/* not fatal to the loop */
#line 747
			status = lstatus;
#line 747
		}
#line 747

#line 747
		(void) ncio_rel(ncp->nciop, offset,
#line 747
				 RGN_MODIFIED);	
#line 747

#line 747
		remaining -= extent;
#line 747
		if(remaining == 0)
#line 747
			break; /* normal loop exit */
#line 747
		offset += extent;
#line 747
		value += nput;
#line 747

#line 747
	}
#line 747

#line 747
	return status;
#line 747
}
#line 747

static int
#line 748
putNCvx_short_ulonglong(NC3_INFO* ncp, const NC_var *varp,
#line 748
		 const size_t *start, size_t nelems, const ulonglong *value)
#line 748
{
#line 748
	off_t offset = NC_varoffset(ncp, varp, start);
#line 748
	size_t remaining = varp->xsz * nelems;
#line 748
	int status = NC_NOERR;
#line 748
	void *xp;
#line 748

#line 748
	if(nelems == 0)
#line 748
		return NC_NOERR;
#line 748

#line 748
	assert(value != NULL);
#line 748

#line 748
	for(;;)
#line 748
	{
#line 748
		size_t extent = MIN(remaining, ncp->chunk);
#line 748
		size_t nput = ncx_howmany(varp->type, extent);
#line 748

#line 748
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 748
				 RGN_WRITE, &xp);	
#line 748
		if(lstatus != NC_NOERR)
#line 748
			return lstatus;
#line 748
		
#line 748
		lstatus = ncx_putn_short_ulonglong(&xp, nput, value);
#line 748
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 748
		{
#line 748
			/* not fatal to the loop */
#line 748
			status = lstatus;
#line 748
		}
#line 748

#line 748
		(void) ncio_rel(ncp->nciop, offset,
#line 748
				 RGN_MODIFIED);	
#line 748

#line 748
		remaining -= extent;
#line 748
		if(remaining == 0)
#line 748
			break; /* normal loop exit */
#line 748
		offset += extent;
#line 748
		value += nput;
#line 748

#line 748
	}
#line 748

#line 748
	return status;
#line 748
}
#line 748

static int
#line 749
putNCvx_int_uint(NC3_INFO* ncp, const NC_var *varp,
#line 749
		 const size_t *start, size_t nelems, const uint *value)
#line 749
{
#line 749
	off_t offset = NC_varoffset(ncp, varp, start);
#line 749
	size_t remaining = varp->xsz * nelems;
#line 749
	int status = NC_NOERR;
#line 749
	void *xp;
#line 749

#line 749
	if(nelems == 0)
#line 749
		return NC_NOERR;
#line 749

#line 749
	assert(value != NULL);
#line 749

#line 749
	for(;;)
#line 749
	{
#line 749
		size_t extent = MIN(remaining, ncp->chunk);
#line 749
		size_t nput = ncx_howmany(varp->type, extent);
#line 749

#line 749
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 749
				 RGN_WRITE, &xp);	
#line 749
		if(lstatus != NC_NOERR)
#line 749
			return lstatus;
#line 749
		
#line 749
		lstatus = ncx_putn_int_uint(&xp, nput, value);
#line 749
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 749
		{
#line 749
			/* not fatal to the loop */
#line 749
			status = lstatus;
#line 749
		}
#line 749

#line 749
		(void) ncio_rel(ncp->nciop, offset,
#line 749
				 RGN_MODIFIED);	
#line 749

#line 749
		remaining -= extent;
#line 749
		if(remaining == 0)
#line 749
			break; /* normal loop exit */
#line 749
		offset += extent;
#line 749
		value += nput;
#line 749

#line 749
	}
#line 749

#line 749
	return status;
#line 749
}
#line 749

static int
#line 750
putNCvx_int_ulonglong(NC3_INFO* ncp, const NC_var *varp,
#line 750
		 const size_t *start, size_t nelems, const ulonglong *value)
#line 750
{
#line 750
	off_t offset = NC_varoffset(ncp, varp, start);
#line 750
	size_t remaining = varp->xsz * nelems;
#line 750
	int status = NC_NOERR;
#line 750
	void *xp;
#line 750

#line 750
	if(nelems == 0)
#line 750
		return NC_NOERR;
#line 750

#line 750
	assert(value != NULL);
#line 750

#line 750
	for(;;)
#line 750
	{
#line 750
		size_t extent = MIN(remaining, ncp->chunk);
#line 750
		size_t nput = ncx_howmany(varp->type, extent);
#line 750

#line 750
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 750
				 RGN_WRITE, &xp);	
#line 750
		if(lstatus != NC_NOERR)
#line 750
			return lstatus;
#line 750
		
#line 750
		lstatus = ncx_putn_int_ulonglong(&xp, nput, value);
#line 750
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 750
		{
#line 750
			/* not fatal to the loop */
#line 750
			status = lstatus;
#line 750
		}
#line 750

#line 750
		(void) ncio_rel(ncp->nciop, offset,
#line 750
				 RGN_MODIFIED);	
#line 750

#line 750
		remaining -= extent;
#line 750
		if(remaining == 0)
#line 750
			break; /* normal loop exit */
#line 750
		offset += extent;
#line 750
		value += nput;
#line 750

#line 750
	}
#line 750

#line 750
	return status;
#line 750
}
#line 750

static int
#line 751
putNCvx_float_uint(NC3_INFO* ncp, const NC_var *varp,
#line 751
		 const size_t *start, size_t nelems, const uint *value)
#line 751
{
#line 751
	off_t offset = NC_varoffset(ncp, varp, start);
#line 751
	size_t remaining = varp->xsz * nelems;
#line 751
	int status = NC_NOERR;
#line 751
	void *xp;
#line 751

#line 751
	if(nelems == 0)
#line 751
		return NC_NOERR;
#line 751

#line 751
	assert(value != NULL);
#line 751

#line 751
	for(;;)
#line 751
	{
#line 751
		size_t extent = MIN(remaining, ncp->chunk);
#line 751
		size_t nput = ncx_howmany(varp->type, extent);
#line 751

#line 751
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 751
				 RGN_WRITE, &xp);	
#line 751
		if(lstatus != NC_NOERR)
#line 751
			return lstatus;
#line 751
		
#line 751
		lstatus = ncx_putn_float_uint(&xp, nput, value);
#line 751
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 751
		{
#line 751
			/* not fatal to the loop */
#line 751
			status = lstatus;
#line 751
		}
#line 751

#line 751
		(void) ncio_rel(ncp->nciop, offset,
#line 751
				 RGN_MODIFIED);	
#line 751

#line 751
		remaining -= extent;
#line 751
		if(remaining == 0)
#line 751
			break; /* normal loop exit */
#line 751
		offset += extent;
#line 751
		value += nput;
#line 751

#line 751
	}
#line 751

#line 751
	return status;
#line 751
}
#line 751

static int
#line 752
putNCvx_float_ulonglong(NC3_INFO* ncp, const NC_var *varp,
#line 752
		 const size_t *start, size_t nelems, const ulonglong *value)
#line 752
{
#line 752
	off_t offset = NC_varoffset(ncp, varp, start);
#line 752
	size_t remaining = varp->xsz * nelems;
#line 752
	int status = NC_NOERR;
#line 752
	void *xp;
#line 752

#line 752
	if(nelems == 0)
#line 752
		return NC_NOERR;
#line 752

#line 752
	assert(value != NULL);
#line 752

#line 752
	for(;;)
#line 752
	{
#line 752
		size_t extent = MIN(remaining, ncp->chunk);
#line 752
		size_t nput = ncx_howmany(varp->type, extent);
#line 752

#line 752
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 752
				 RGN_WRITE, &xp);	
#line 752
		if(lstatus != NC_NOERR)
#line 752
			return lstatus;
#line 752
		
#line 752
		lstatus = ncx_putn_float_ulonglong(&xp, nput, value);
#line 752
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 752
		{
#line 752
			/* not fatal to the loop */
#line 752
			status = lstatus;
#line 752
		}
#line 752

#line 752
		(void) ncio_rel(ncp->nciop, offset,
#line 752
				 RGN_MODIFIED);	
#line 752

#line 752
		remaining -= extent;
#line 752
		if(remaining == 0)
#line 752
			break; /* normal loop exit */
#line 752
		offset += extent;
#line 752
		value += nput;
#line 752

#line 752
	}
#line 752

#line 752
	return status;
#line 752
}
#line 752

static int
#line 753
putNCvx_double_uint(NC3_INFO* ncp, const NC_var *varp,
#line 753
		 const size_t *start, size_t nelems, const uint *value)
#line 753
{
#line 753
	off_t offset = NC_varoffset(ncp, varp, start);
#line 753
	size_t remaining = varp->xsz * nelems;
#line 753
	int status = NC_NOERR;
#line 753
	void *xp;
#line 753

#line 753
	if(nelems == 0)
#line 753
		return NC_NOERR;
#line 753

#line 753
	assert(value != NULL);
#line 753

#line 753
	for(;;)
#line 753
	{
#line 753
		size_t extent = MIN(remaining, ncp->chunk);
#line 753
		size_t nput = ncx_howmany(varp->type, extent);
#line 753

#line 753
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 753
				 RGN_WRITE, &xp);	
#line 753
		if(lstatus != NC_NOERR)
#line 753
			return lstatus;
#line 753
		
#line 753
		lstatus = ncx_putn_double_uint(&xp, nput, value);
#line 753
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 753
		{
#line 753
			/* not fatal to the loop */
#line 753
			status = lstatus;
#line 753
		}
#line 753

#line 753
		(void) ncio_rel(ncp->nciop, offset,
#line 753
				 RGN_MODIFIED);	
#line 753

#line 753
		remaining -= extent;
#line 753
		if(remaining == 0)
#line 753
			break; /* normal loop exit */
#line 753
		offset += extent;
#line 753
		value += nput;
#line 753

#line 753
	}
#line 753

#line 753
	return status;
#line 753
}
#line 753

static int
#line 754
putNCvx_double_ulonglong(NC3_INFO* ncp, const NC_var *varp,
#line 754
		 const size_t *start, size_t nelems, const ulonglong *value)
#line 754
{
#line 754
	off_t offset = NC_varoffset(ncp, varp, start);
#line 754
	size_t remaining = varp->xsz * nelems;
#line 754
	int status = NC_NOERR;
#line 754
	void *xp;
#line 754

#line 754
	if(nelems == 0)
#line 754
		return NC_NOERR;
#line 754

#line 754
	assert(value != NULL);
#line 754

#line 754
	for(;;)
#line 754
	{
#line 754
		size_t extent = MIN(remaining, ncp->chunk);
#line 754
		size_t nput = ncx_howmany(varp->type, extent);
#line 754

#line 754
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 754
				 RGN_WRITE, &xp);	
#line 754
		if(lstatus != NC_NOERR)
#line 754
			return lstatus;
#line 754
		
#line 754
		lstatus = ncx_putn_double_ulonglong(&xp, nput, value);
#line 754
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 754
		{
#line 754
			/* not fatal to the loop */
#line 754
			status = lstatus;
#line 754
		}
#line 754

#line 754
		(void) ncio_rel(ncp->nciop, offset,
#line 754
				 RGN_MODIFIED);	
#line 754

#line 754
		remaining -= extent;
#line 754
		if(remaining == 0)
#line 754
			break; /* normal loop exit */
#line 754
		offset += extent;
#line 754
		value += nput;
#line 754

#line 754
	}
#line 754

#line 754
	return status;
#line 754
}
#line 754

#endif /*NOTUSED*/

#line 802

static int
#line 803
getNCvx_char_char(const NC3_INFO* ncp, const NC_var *varp,
#line 803
		 const size_t *start, size_t nelems, char *value)
#line 803
{
#line 803
	off_t offset = NC_varoffset(ncp, varp, start);
#line 803
	size_t remaining = varp->xsz * nelems;
#line 803
	int status = NC_NOERR;
#line 803
	const void *xp;
#line 803

#line 803
	if(nelems == 0)
#line 803
		return NC_NOERR;
#line 803

#line 803
	assert(value != NULL);
#line 803

#line 803
	for(;;)
#line 803
	{
#line 803
		size_t extent = MIN(remaining, ncp->chunk);
#line 803
		size_t nget = ncx_howmany(varp->type, extent);
#line 803

#line 803
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 803
				 0, (void **)&xp);	/* cast away const */
#line 803
		if(lstatus != NC_NOERR)
#line 803
			return lstatus;
#line 803
		
#line 803
		lstatus = ncx_getn_char_char(&xp, nget, value);
#line 803
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 803
			status = lstatus;
#line 803

#line 803
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 803

#line 803
		remaining -= extent;
#line 803
		if(remaining == 0)
#line 803
			break; /* normal loop exit */
#line 803
		offset += extent;
#line 803
		value += nget;
#line 803
	}
#line 803

#line 803
	return status;
#line 803
}
#line 803


static int
#line 805
getNCvx_schar_schar(const NC3_INFO* ncp, const NC_var *varp,
#line 805
		 const size_t *start, size_t nelems, schar *value)
#line 805
{
#line 805
	off_t offset = NC_varoffset(ncp, varp, start);
#line 805
	size_t remaining = varp->xsz * nelems;
#line 805
	int status = NC_NOERR;
#line 805
	const void *xp;
#line 805

#line 805
	if(nelems == 0)
#line 805
		return NC_NOERR;
#line 805

#line 805
	assert(value != NULL);
#line 805

#line 805
	for(;;)
#line 805
	{
#line 805
		size_t extent = MIN(remaining, ncp->chunk);
#line 805
		size_t nget = ncx_howmany(varp->type, extent);
#line 805

#line 805
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 805
				 0, (void **)&xp);	/* cast away const */
#line 805
		if(lstatus != NC_NOERR)
#line 805
			return lstatus;
#line 805
		
#line 805
		lstatus = ncx_getn_schar_schar(&xp, nget, value);
#line 805
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 805
			status = lstatus;
#line 805

#line 805
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 805

#line 805
		remaining -= extent;
#line 805
		if(remaining == 0)
#line 805
			break; /* normal loop exit */
#line 805
		offset += extent;
#line 805
		value += nget;
#line 805
	}
#line 805

#line 805
	return status;
#line 805
}
#line 805

static int
#line 806
getNCvx_schar_short(const NC3_INFO* ncp, const NC_var *varp,
#line 806
		 const size_t *start, size_t nelems, short *value)
#line 806
{
#line 806
	off_t offset = NC_varoffset(ncp, varp, start);
#line 806
	size_t remaining = varp->xsz * nelems;
#line 806
	int status = NC_NOERR;
#line 806
	const void *xp;
#line 806

#line 806
	if(nelems == 0)
#line 806
		return NC_NOERR;
#line 806

#line 806
	assert(value != NULL);
#line 806

#line 806
	for(;;)
#line 806
	{
#line 806
		size_t extent = MIN(remaining, ncp->chunk);
#line 806
		size_t nget = ncx_howmany(varp->type, extent);
#line 806

#line 806
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 806
				 0, (void **)&xp);	/* cast away const */
#line 806
		if(lstatus != NC_NOERR)
#line 806
			return lstatus;
#line 806
		
#line 806
		lstatus = ncx_getn_schar_short(&xp, nget, value);
#line 806
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 806
			status = lstatus;
#line 806

#line 806
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 806

#line 806
		remaining -= extent;
#line 806
		if(remaining == 0)
#line 806
			break; /* normal loop exit */
#line 806
		offset += extent;
#line 806
		value += nget;
#line 806
	}
#line 806

#line 806
	return status;
#line 806
}
#line 806

static int
#line 807
getNCvx_schar_int(const NC3_INFO* ncp, const NC_var *varp,
#line 807
		 const size_t *start, size_t nelems, int *value)
#line 807
{
#line 807
	off_t offset = NC_varoffset(ncp, varp, start);
#line 807
	size_t remaining = varp->xsz * nelems;
#line 807
	int status = NC_NOERR;
#line 807
	const void *xp;
#line 807

#line 807
	if(nelems == 0)
#line 807
		return NC_NOERR;
#line 807

#line 807
	assert(value != NULL);
#line 807

#line 807
	for(;;)
#line 807
	{
#line 807
		size_t extent = MIN(remaining, ncp->chunk);
#line 807
		size_t nget = ncx_howmany(varp->type, extent);
#line 807

#line 807
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 807
				 0, (void **)&xp);	/* cast away const */
#line 807
		if(lstatus != NC_NOERR)
#line 807
			return lstatus;
#line 807
		
#line 807
		lstatus = ncx_getn_schar_int(&xp, nget, value);
#line 807
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 807
			status = lstatus;
#line 807

#line 807
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 807

#line 807
		remaining -= extent;
#line 807
		if(remaining == 0)
#line 807
			break; /* normal loop exit */
#line 807
		offset += extent;
#line 807
		value += nget;
#line 807
	}
#line 807

#line 807
	return status;
#line 807
}
#line 807

static int
#line 808
getNCvx_schar_float(const NC3_INFO* ncp, const NC_var *varp,
#line 808
		 const size_t *start, size_t nelems, float *value)
#line 808
{
#line 808
	off_t offset = NC_varoffset(ncp, varp, start);
#line 808
	size_t remaining = varp->xsz * nelems;
#line 808
	int status = NC_NOERR;
#line 808
	const void *xp;
#line 808

#line 808
	if(nelems == 0)
#line 808
		return NC_NOERR;
#line 808

#line 808
	assert(value != NULL);
#line 808

#line 808
	for(;;)
#line 808
	{
#line 808
		size_t extent = MIN(remaining, ncp->chunk);
#line 808
		size_t nget = ncx_howmany(varp->type, extent);
#line 808

#line 808
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 808
				 0, (void **)&xp);	/* cast away const */
#line 808
		if(lstatus != NC_NOERR)
#line 808
			return lstatus;
#line 808
		
#line 808
		lstatus = ncx_getn_schar_float(&xp, nget, value);
#line 808
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 808
			status = lstatus;
#line 808

#line 808
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 808

#line 808
		remaining -= extent;
#line 808
		if(remaining == 0)
#line 808
			break; /* normal loop exit */
#line 808
		offset += extent;
#line 808
		value += nget;
#line 808
	}
#line 808

#line 808
	return status;
#line 808
}
#line 808

static int
#line 809
getNCvx_schar_double(const NC3_INFO* ncp, const NC_var *varp,
#line 809
		 const size_t *start, size_t nelems, double *value)
#line 809
{
#line 809
	off_t offset = NC_varoffset(ncp, varp, start);
#line 809
	size_t remaining = varp->xsz * nelems;
#line 809
	int status = NC_NOERR;
#line 809
	const void *xp;
#line 809

#line 809
	if(nelems == 0)
#line 809
		return NC_NOERR;
#line 809

#line 809
	assert(value != NULL);
#line 809

#line 809
	for(;;)
#line 809
	{
#line 809
		size_t extent = MIN(remaining, ncp->chunk);
#line 809
		size_t nget = ncx_howmany(varp->type, extent);
#line 809

#line 809
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 809
				 0, (void **)&xp);	/* cast away const */
#line 809
		if(lstatus != NC_NOERR)
#line 809
			return lstatus;
#line 809
		
#line 809
		lstatus = ncx_getn_schar_double(&xp, nget, value);
#line 809
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 809
			status = lstatus;
#line 809

#line 809
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 809

#line 809
		remaining -= extent;
#line 809
		if(remaining == 0)
#line 809
			break; /* normal loop exit */
#line 809
		offset += extent;
#line 809
		value += nget;
#line 809
	}
#line 809

#line 809
	return status;
#line 809
}
#line 809

static int
#line 810
getNCvx_schar_longlong(const NC3_INFO* ncp, const NC_var *varp,
#line 810
		 const size_t *start, size_t nelems, longlong *value)
#line 810
{
#line 810
	off_t offset = NC_varoffset(ncp, varp, start);
#line 810
	size_t remaining = varp->xsz * nelems;
#line 810
	int status = NC_NOERR;
#line 810
	const void *xp;
#line 810

#line 810
	if(nelems == 0)
#line 810
		return NC_NOERR;
#line 810

#line 810
	assert(value != NULL);
#line 810

#line 810
	for(;;)
#line 810
	{
#line 810
		size_t extent = MIN(remaining, ncp->chunk);
#line 810
		size_t nget = ncx_howmany(varp->type, extent);
#line 810

#line 810
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 810
				 0, (void **)&xp);	/* cast away const */
#line 810
		if(lstatus != NC_NOERR)
#line 810
			return lstatus;
#line 810
		
#line 810
		lstatus = ncx_getn_schar_longlong(&xp, nget, value);
#line 810
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 810
			status = lstatus;
#line 810

#line 810
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 810

#line 810
		remaining -= extent;
#line 810
		if(remaining == 0)
#line 810
			break; /* normal loop exit */
#line 810
		offset += extent;
#line 810
		value += nget;
#line 810
	}
#line 810

#line 810
	return status;
#line 810
}
#line 810

static int
#line 811
getNCvx_schar_uint(const NC3_INFO* ncp, const NC_var *varp,
#line 811
		 const size_t *start, size_t nelems, uint *value)
#line 811
{
#line 811
	off_t offset = NC_varoffset(ncp, varp, start);
#line 811
	size_t remaining = varp->xsz * nelems;
#line 811
	int status = NC_NOERR;
#line 811
	const void *xp;
#line 811

#line 811
	if(nelems == 0)
#line 811
		return NC_NOERR;
#line 811

#line 811
	assert(value != NULL);
#line 811

#line 811
	for(;;)
#line 811
	{
#line 811
		size_t extent = MIN(remaining, ncp->chunk);
#line 811
		size_t nget = ncx_howmany(varp->type, extent);
#line 811

#line 811
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 811
				 0, (void **)&xp);	/* cast away const */
#line 811
		if(lstatus != NC_NOERR)
#line 811
			return lstatus;
#line 811
		
#line 811
		lstatus = ncx_getn_schar_uint(&xp, nget, value);
#line 811
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 811
			status = lstatus;
#line 811

#line 811
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 811

#line 811
		remaining -= extent;
#line 811
		if(remaining == 0)
#line 811
			break; /* normal loop exit */
#line 811
		offset += extent;
#line 811
		value += nget;
#line 811
	}
#line 811

#line 811
	return status;
#line 811
}
#line 811

static int
#line 812
getNCvx_schar_ulonglong(const NC3_INFO* ncp, const NC_var *varp,
#line 812
		 const size_t *start, size_t nelems, ulonglong *value)
#line 812
{
#line 812
	off_t offset = NC_varoffset(ncp, varp, start);
#line 812
	size_t remaining = varp->xsz * nelems;
#line 812
	int status = NC_NOERR;
#line 812
	const void *xp;
#line 812

#line 812
	if(nelems == 0)
#line 812
		return NC_NOERR;
#line 812

#line 812
	assert(value != NULL);
#line 812

#line 812
	for(;;)
#line 812
	{
#line 812
		size_t extent = MIN(remaining, ncp->chunk);
#line 812
		size_t nget = ncx_howmany(varp->type, extent);
#line 812

#line 812
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 812
				 0, (void **)&xp);	/* cast away const */
#line 812
		if(lstatus != NC_NOERR)
#line 812
			return lstatus;
#line 812
		
#line 812
		lstatus = ncx_getn_schar_ulonglong(&xp, nget, value);
#line 812
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 812
			status = lstatus;
#line 812

#line 812
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 812

#line 812
		remaining -= extent;
#line 812
		if(remaining == 0)
#line 812
			break; /* normal loop exit */
#line 812
		offset += extent;
#line 812
		value += nget;
#line 812
	}
#line 812

#line 812
	return status;
#line 812
}
#line 812


static int
#line 814
getNCvx_short_schar(const NC3_INFO* ncp, const NC_var *varp,
#line 814
		 const size_t *start, size_t nelems, schar *value)
#line 814
{
#line 814
	off_t offset = NC_varoffset(ncp, varp, start);
#line 814
	size_t remaining = varp->xsz * nelems;
#line 814
	int status = NC_NOERR;
#line 814
	const void *xp;
#line 814

#line 814
	if(nelems == 0)
#line 814
		return NC_NOERR;
#line 814

#line 814
	assert(value != NULL);
#line 814

#line 814
	for(;;)
#line 814
	{
#line 814
		size_t extent = MIN(remaining, ncp->chunk);
#line 814
		size_t nget = ncx_howmany(varp->type, extent);
#line 814

#line 814
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 814
				 0, (void **)&xp);	/* cast away const */
#line 814
		if(lstatus != NC_NOERR)
#line 814
			return lstatus;
#line 814
		
#line 814
		lstatus = ncx_getn_short_schar(&xp, nget, value);
#line 814
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 814
			status = lstatus;
#line 814

#line 814
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 814

#line 814
		remaining -= extent;
#line 814
		if(remaining == 0)
#line 814
			break; /* normal loop exit */
#line 814
		offset += extent;
#line 814
		value += nget;
#line 814
	}
#line 814

#line 814
	return status;
#line 814
}
#line 814

static int
#line 815
getNCvx_short_uchar(const NC3_INFO* ncp, const NC_var *varp,
#line 815
		 const size_t *start, size_t nelems, uchar *value)
#line 815
{
#line 815
	off_t offset = NC_varoffset(ncp, varp, start);
#line 815
	size_t remaining = varp->xsz * nelems;
#line 815
	int status = NC_NOERR;
#line 815
	const void *xp;
#line 815

#line 815
	if(nelems == 0)
#line 815
		return NC_NOERR;
#line 815

#line 815
	assert(value != NULL);
#line 815

#line 815
	for(;;)
#line 815
	{
#line 815
		size_t extent = MIN(remaining, ncp->chunk);
#line 815
		size_t nget = ncx_howmany(varp->type, extent);
#line 815

#line 815
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 815
				 0, (void **)&xp);	/* cast away const */
#line 815
		if(lstatus != NC_NOERR)
#line 815
			return lstatus;
#line 815
		
#line 815
		lstatus = ncx_getn_short_uchar(&xp, nget, value);
#line 815
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 815
			status = lstatus;
#line 815

#line 815
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 815

#line 815
		remaining -= extent;
#line 815
		if(remaining == 0)
#line 815
			break; /* normal loop exit */
#line 815
		offset += extent;
#line 815
		value += nget;
#line 815
	}
#line 815

#line 815
	return status;
#line 815
}
#line 815

static int
#line 816
getNCvx_short_short(const NC3_INFO* ncp, const NC_var *varp,
#line 816
		 const size_t *start, size_t nelems, short *value)
#line 816
{
#line 816
	off_t offset = NC_varoffset(ncp, varp, start);
#line 816
	size_t remaining = varp->xsz * nelems;
#line 816
	int status = NC_NOERR;
#line 816
	const void *xp;
#line 816

#line 816
	if(nelems == 0)
#line 816
		return NC_NOERR;
#line 816

#line 816
	assert(value != NULL);
#line 816

#line 816
	for(;;)
#line 816
	{
#line 816
		size_t extent = MIN(remaining, ncp->chunk);
#line 816
		size_t nget = ncx_howmany(varp->type, extent);
#line 816

#line 816
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 816
				 0, (void **)&xp);	/* cast away const */
#line 816
		if(lstatus != NC_NOERR)
#line 816
			return lstatus;
#line 816
		
#line 816
		lstatus = ncx_getn_short_short(&xp, nget, value);
#line 816
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 816
			status = lstatus;
#line 816

#line 816
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 816

#line 816
		remaining -= extent;
#line 816
		if(remaining == 0)
#line 816
			break; /* normal loop exit */
#line 816
		offset += extent;
#line 816
		value += nget;
#line 816
	}
#line 816

#line 816
	return status;
#line 816
}
#line 816

static int
#line 817
getNCvx_short_int(const NC3_INFO* ncp, const NC_var *varp,
#line 817
		 const size_t *start, size_t nelems, int *value)
#line 817
{
#line 817
	off_t offset = NC_varoffset(ncp, varp, start);
#line 817
	size_t remaining = varp->xsz * nelems;
#line 817
	int status = NC_NOERR;
#line 817
	const void *xp;
#line 817

#line 817
	if(nelems == 0)
#line 817
		return NC_NOERR;
#line 817

#line 817
	assert(value != NULL);
#line 817

#line 817
	for(;;)
#line 817
	{
#line 817
		size_t extent = MIN(remaining, ncp->chunk);
#line 817
		size_t nget = ncx_howmany(varp->type, extent);
#line 817

#line 817
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 817
				 0, (void **)&xp);	/* cast away const */
#line 817
		if(lstatus != NC_NOERR)
#line 817
			return lstatus;
#line 817
		
#line 817
		lstatus = ncx_getn_short_int(&xp, nget, value);
#line 817
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 817
			status = lstatus;
#line 817

#line 817
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 817

#line 817
		remaining -= extent;
#line 817
		if(remaining == 0)
#line 817
			break; /* normal loop exit */
#line 817
		offset += extent;
#line 817
		value += nget;
#line 817
	}
#line 817

#line 817
	return status;
#line 817
}
#line 817

static int
#line 818
getNCvx_short_float(const NC3_INFO* ncp, const NC_var *varp,
#line 818
		 const size_t *start, size_t nelems, float *value)
#line 818
{
#line 818
	off_t offset = NC_varoffset(ncp, varp, start);
#line 818
	size_t remaining = varp->xsz * nelems;
#line 818
	int status = NC_NOERR;
#line 818
	const void *xp;
#line 818

#line 818
	if(nelems == 0)
#line 818
		return NC_NOERR;
#line 818

#line 818
	assert(value != NULL);
#line 818

#line 818
	for(;;)
#line 818
	{
#line 818
		size_t extent = MIN(remaining, ncp->chunk);
#line 818
		size_t nget = ncx_howmany(varp->type, extent);
#line 818

#line 818
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 818
				 0, (void **)&xp);	/* cast away const */
#line 818
		if(lstatus != NC_NOERR)
#line 818
			return lstatus;
#line 818
		
#line 818
		lstatus = ncx_getn_short_float(&xp, nget, value);
#line 818
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 818
			status = lstatus;
#line 818

#line 818
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 818

#line 818
		remaining -= extent;
#line 818
		if(remaining == 0)
#line 818
			break; /* normal loop exit */
#line 818
		offset += extent;
#line 818
		value += nget;
#line 818
	}
#line 818

#line 818
	return status;
#line 818
}
#line 818

static int
#line 819
getNCvx_short_double(const NC3_INFO* ncp, const NC_var *varp,
#line 819
		 const size_t *start, size_t nelems, double *value)
#line 819
{
#line 819
	off_t offset = NC_varoffset(ncp, varp, start);
#line 819
	size_t remaining = varp->xsz * nelems;
#line 819
	int status = NC_NOERR;
#line 819
	const void *xp;
#line 819

#line 819
	if(nelems == 0)
#line 819
		return NC_NOERR;
#line 819

#line 819
	assert(value != NULL);
#line 819

#line 819
	for(;;)
#line 819
	{
#line 819
		size_t extent = MIN(remaining, ncp->chunk);
#line 819
		size_t nget = ncx_howmany(varp->type, extent);
#line 819

#line 819
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 819
				 0, (void **)&xp);	/* cast away const */
#line 819
		if(lstatus != NC_NOERR)
#line 819
			return lstatus;
#line 819
		
#line 819
		lstatus = ncx_getn_short_double(&xp, nget, value);
#line 819
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 819
			status = lstatus;
#line 819

#line 819
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 819

#line 819
		remaining -= extent;
#line 819
		if(remaining == 0)
#line 819
			break; /* normal loop exit */
#line 819
		offset += extent;
#line 819
		value += nget;
#line 819
	}
#line 819

#line 819
	return status;
#line 819
}
#line 819

static int
#line 820
getNCvx_short_longlong(const NC3_INFO* ncp, const NC_var *varp,
#line 820
		 const size_t *start, size_t nelems, longlong *value)
#line 820
{
#line 820
	off_t offset = NC_varoffset(ncp, varp, start);
#line 820
	size_t remaining = varp->xsz * nelems;
#line 820
	int status = NC_NOERR;
#line 820
	const void *xp;
#line 820

#line 820
	if(nelems == 0)
#line 820
		return NC_NOERR;
#line 820

#line 820
	assert(value != NULL);
#line 820

#line 820
	for(;;)
#line 820
	{
#line 820
		size_t extent = MIN(remaining, ncp->chunk);
#line 820
		size_t nget = ncx_howmany(varp->type, extent);
#line 820

#line 820
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 820
				 0, (void **)&xp);	/* cast away const */
#line 820
		if(lstatus != NC_NOERR)
#line 820
			return lstatus;
#line 820
		
#line 820
		lstatus = ncx_getn_short_longlong(&xp, nget, value);
#line 820
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 820
			status = lstatus;
#line 820

#line 820
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 820

#line 820
		remaining -= extent;
#line 820
		if(remaining == 0)
#line 820
			break; /* normal loop exit */
#line 820
		offset += extent;
#line 820
		value += nget;
#line 820
	}
#line 820

#line 820
	return status;
#line 820
}
#line 820

static int
#line 821
getNCvx_short_uint(const NC3_INFO* ncp, const NC_var *varp,
#line 821
		 const size_t *start, size_t nelems, uint *value)
#line 821
{
#line 821
	off_t offset = NC_varoffset(ncp, varp, start);
#line 821
	size_t remaining = varp->xsz * nelems;
#line 821
	int status = NC_NOERR;
#line 821
	const void *xp;
#line 821

#line 821
	if(nelems == 0)
#line 821
		return NC_NOERR;
#line 821

#line 821
	assert(value != NULL);
#line 821

#line 821
	for(;;)
#line 821
	{
#line 821
		size_t extent = MIN(remaining, ncp->chunk);
#line 821
		size_t nget = ncx_howmany(varp->type, extent);
#line 821

#line 821
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 821
				 0, (void **)&xp);	/* cast away const */
#line 821
		if(lstatus != NC_NOERR)
#line 821
			return lstatus;
#line 821
		
#line 821
		lstatus = ncx_getn_short_uint(&xp, nget, value);
#line 821
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 821
			status = lstatus;
#line 821

#line 821
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 821

#line 821
		remaining -= extent;
#line 821
		if(remaining == 0)
#line 821
			break; /* normal loop exit */
#line 821
		offset += extent;
#line 821
		value += nget;
#line 821
	}
#line 821

#line 821
	return status;
#line 821
}
#line 821

static int
#line 822
getNCvx_short_ulonglong(const NC3_INFO* ncp, const NC_var *varp,
#line 822
		 const size_t *start, size_t nelems, ulonglong *value)
#line 822
{
#line 822
	off_t offset = NC_varoffset(ncp, varp, start);
#line 822
	size_t remaining = varp->xsz * nelems;
#line 822
	int status = NC_NOERR;
#line 822
	const void *xp;
#line 822

#line 822
	if(nelems == 0)
#line 822
		return NC_NOERR;
#line 822

#line 822
	assert(value != NULL);
#line 822

#line 822
	for(;;)
#line 822
	{
#line 822
		size_t extent = MIN(remaining, ncp->chunk);
#line 822
		size_t nget = ncx_howmany(varp->type, extent);
#line 822

#line 822
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 822
				 0, (void **)&xp);	/* cast away const */
#line 822
		if(lstatus != NC_NOERR)
#line 822
			return lstatus;
#line 822
		
#line 822
		lstatus = ncx_getn_short_ulonglong(&xp, nget, value);
#line 822
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 822
			status = lstatus;
#line 822

#line 822
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 822

#line 822
		remaining -= extent;
#line 822
		if(remaining == 0)
#line 822
			break; /* normal loop exit */
#line 822
		offset += extent;
#line 822
		value += nget;
#line 822
	}
#line 822

#line 822
	return status;
#line 822
}
#line 822


static int
#line 824
getNCvx_int_schar(const NC3_INFO* ncp, const NC_var *varp,
#line 824
		 const size_t *start, size_t nelems, schar *value)
#line 824
{
#line 824
	off_t offset = NC_varoffset(ncp, varp, start);
#line 824
	size_t remaining = varp->xsz * nelems;
#line 824
	int status = NC_NOERR;
#line 824
	const void *xp;
#line 824

#line 824
	if(nelems == 0)
#line 824
		return NC_NOERR;
#line 824

#line 824
	assert(value != NULL);
#line 824

#line 824
	for(;;)
#line 824
	{
#line 824
		size_t extent = MIN(remaining, ncp->chunk);
#line 824
		size_t nget = ncx_howmany(varp->type, extent);
#line 824

#line 824
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 824
				 0, (void **)&xp);	/* cast away const */
#line 824
		if(lstatus != NC_NOERR)
#line 824
			return lstatus;
#line 824
		
#line 824
		lstatus = ncx_getn_int_schar(&xp, nget, value);
#line 824
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 824
			status = lstatus;
#line 824

#line 824
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 824

#line 824
		remaining -= extent;
#line 824
		if(remaining == 0)
#line 824
			break; /* normal loop exit */
#line 824
		offset += extent;
#line 824
		value += nget;
#line 824
	}
#line 824

#line 824
	return status;
#line 824
}
#line 824

static int
#line 825
getNCvx_int_uchar(const NC3_INFO* ncp, const NC_var *varp,
#line 825
		 const size_t *start, size_t nelems, uchar *value)
#line 825
{
#line 825
	off_t offset = NC_varoffset(ncp, varp, start);
#line 825
	size_t remaining = varp->xsz * nelems;
#line 825
	int status = NC_NOERR;
#line 825
	const void *xp;
#line 825

#line 825
	if(nelems == 0)
#line 825
		return NC_NOERR;
#line 825

#line 825
	assert(value != NULL);
#line 825

#line 825
	for(;;)
#line 825
	{
#line 825
		size_t extent = MIN(remaining, ncp->chunk);
#line 825
		size_t nget = ncx_howmany(varp->type, extent);
#line 825

#line 825
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 825
				 0, (void **)&xp);	/* cast away const */
#line 825
		if(lstatus != NC_NOERR)
#line 825
			return lstatus;
#line 825
		
#line 825
		lstatus = ncx_getn_int_uchar(&xp, nget, value);
#line 825
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 825
			status = lstatus;
#line 825

#line 825
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 825

#line 825
		remaining -= extent;
#line 825
		if(remaining == 0)
#line 825
			break; /* normal loop exit */
#line 825
		offset += extent;
#line 825
		value += nget;
#line 825
	}
#line 825

#line 825
	return status;
#line 825
}
#line 825

static int
#line 826
getNCvx_int_short(const NC3_INFO* ncp, const NC_var *varp,
#line 826
		 const size_t *start, size_t nelems, short *value)
#line 826
{
#line 826
	off_t offset = NC_varoffset(ncp, varp, start);
#line 826
	size_t remaining = varp->xsz * nelems;
#line 826
	int status = NC_NOERR;
#line 826
	const void *xp;
#line 826

#line 826
	if(nelems == 0)
#line 826
		return NC_NOERR;
#line 826

#line 826
	assert(value != NULL);
#line 826

#line 826
	for(;;)
#line 826
	{
#line 826
		size_t extent = MIN(remaining, ncp->chunk);
#line 826
		size_t nget = ncx_howmany(varp->type, extent);
#line 826

#line 826
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 826
				 0, (void **)&xp);	/* cast away const */
#line 826
		if(lstatus != NC_NOERR)
#line 826
			return lstatus;
#line 826
		
#line 826
		lstatus = ncx_getn_int_short(&xp, nget, value);
#line 826
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 826
			status = lstatus;
#line 826

#line 826
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 826

#line 826
		remaining -= extent;
#line 826
		if(remaining == 0)
#line 826
			break; /* normal loop exit */
#line 826
		offset += extent;
#line 826
		value += nget;
#line 826
	}
#line 826

#line 826
	return status;
#line 826
}
#line 826

static int
#line 827
getNCvx_int_int(const NC3_INFO* ncp, const NC_var *varp,
#line 827
		 const size_t *start, size_t nelems, int *value)
#line 827
{
#line 827
	off_t offset = NC_varoffset(ncp, varp, start);
#line 827
	size_t remaining = varp->xsz * nelems;
#line 827
	int status = NC_NOERR;
#line 827
	const void *xp;
#line 827

#line 827
	if(nelems == 0)
#line 827
		return NC_NOERR;
#line 827

#line 827
	assert(value != NULL);
#line 827

#line 827
	for(;;)
#line 827
	{
#line 827
		size_t extent = MIN(remaining, ncp->chunk);
#line 827
		size_t nget = ncx_howmany(varp->type, extent);
#line 827

#line 827
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 827
				 0, (void **)&xp);	/* cast away const */
#line 827
		if(lstatus != NC_NOERR)
#line 827
			return lstatus;
#line 827
		
#line 827
		lstatus = ncx_getn_int_int(&xp, nget, value);
#line 827
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 827
			status = lstatus;
#line 827

#line 827
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 827

#line 827
		remaining -= extent;
#line 827
		if(remaining == 0)
#line 827
			break; /* normal loop exit */
#line 827
		offset += extent;
#line 827
		value += nget;
#line 827
	}
#line 827

#line 827
	return status;
#line 827
}
#line 827

static int
#line 828
getNCvx_int_float(const NC3_INFO* ncp, const NC_var *varp,
#line 828
		 const size_t *start, size_t nelems, float *value)
#line 828
{
#line 828
	off_t offset = NC_varoffset(ncp, varp, start);
#line 828
	size_t remaining = varp->xsz * nelems;
#line 828
	int status = NC_NOERR;
#line 828
	const void *xp;
#line 828

#line 828
	if(nelems == 0)
#line 828
		return NC_NOERR;
#line 828

#line 828
	assert(value != NULL);
#line 828

#line 828
	for(;;)
#line 828
	{
#line 828
		size_t extent = MIN(remaining, ncp->chunk);
#line 828
		size_t nget = ncx_howmany(varp->type, extent);
#line 828

#line 828
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 828
				 0, (void **)&xp);	/* cast away const */
#line 828
		if(lstatus != NC_NOERR)
#line 828
			return lstatus;
#line 828
		
#line 828
		lstatus = ncx_getn_int_float(&xp, nget, value);
#line 828
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 828
			status = lstatus;
#line 828

#line 828
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 828

#line 828
		remaining -= extent;
#line 828
		if(remaining == 0)
#line 828
			break; /* normal loop exit */
#line 828
		offset += extent;
#line 828
		value += nget;
#line 828
	}
#line 828

#line 828
	return status;
#line 828
}
#line 828

static int
#line 829
getNCvx_int_double(const NC3_INFO* ncp, const NC_var *varp,
#line 829
		 const size_t *start, size_t nelems, double *value)
#line 829
{
#line 829
	off_t offset = NC_varoffset(ncp, varp, start);
#line 829
	size_t remaining = varp->xsz * nelems;
#line 829
	int status = NC_NOERR;
#line 829
	const void *xp;
#line 829

#line 829
	if(nelems == 0)
#line 829
		return NC_NOERR;
#line 829

#line 829
	assert(value != NULL);
#line 829

#line 829
	for(;;)
#line 829
	{
#line 829
		size_t extent = MIN(remaining, ncp->chunk);
#line 829
		size_t nget = ncx_howmany(varp->type, extent);
#line 829

#line 829
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 829
				 0, (void **)&xp);	/* cast away const */
#line 829
		if(lstatus != NC_NOERR)
#line 829
			return lstatus;
#line 829
		
#line 829
		lstatus = ncx_getn_int_double(&xp, nget, value);
#line 829
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 829
			status = lstatus;
#line 829

#line 829
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 829

#line 829
		remaining -= extent;
#line 829
		if(remaining == 0)
#line 829
			break; /* normal loop exit */
#line 829
		offset += extent;
#line 829
		value += nget;
#line 829
	}
#line 829

#line 829
	return status;
#line 829
}
#line 829

static int
#line 830
getNCvx_int_longlong(const NC3_INFO* ncp, const NC_var *varp,
#line 830
		 const size_t *start, size_t nelems, longlong *value)
#line 830
{
#line 830
	off_t offset = NC_varoffset(ncp, varp, start);
#line 830
	size_t remaining = varp->xsz * nelems;
#line 830
	int status = NC_NOERR;
#line 830
	const void *xp;
#line 830

#line 830
	if(nelems == 0)
#line 830
		return NC_NOERR;
#line 830

#line 830
	assert(value != NULL);
#line 830

#line 830
	for(;;)
#line 830
	{
#line 830
		size_t extent = MIN(remaining, ncp->chunk);
#line 830
		size_t nget = ncx_howmany(varp->type, extent);
#line 830

#line 830
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 830
				 0, (void **)&xp);	/* cast away const */
#line 830
		if(lstatus != NC_NOERR)
#line 830
			return lstatus;
#line 830
		
#line 830
		lstatus = ncx_getn_int_longlong(&xp, nget, value);
#line 830
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 830
			status = lstatus;
#line 830

#line 830
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 830

#line 830
		remaining -= extent;
#line 830
		if(remaining == 0)
#line 830
			break; /* normal loop exit */
#line 830
		offset += extent;
#line 830
		value += nget;
#line 830
	}
#line 830

#line 830
	return status;
#line 830
}
#line 830

static int
#line 831
getNCvx_int_uint(const NC3_INFO* ncp, const NC_var *varp,
#line 831
		 const size_t *start, size_t nelems, uint *value)
#line 831
{
#line 831
	off_t offset = NC_varoffset(ncp, varp, start);
#line 831
	size_t remaining = varp->xsz * nelems;
#line 831
	int status = NC_NOERR;
#line 831
	const void *xp;
#line 831

#line 831
	if(nelems == 0)
#line 831
		return NC_NOERR;
#line 831

#line 831
	assert(value != NULL);
#line 831

#line 831
	for(;;)
#line 831
	{
#line 831
		size_t extent = MIN(remaining, ncp->chunk);
#line 831
		size_t nget = ncx_howmany(varp->type, extent);
#line 831

#line 831
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 831
				 0, (void **)&xp);	/* cast away const */
#line 831
		if(lstatus != NC_NOERR)
#line 831
			return lstatus;
#line 831
		
#line 831
		lstatus = ncx_getn_int_uint(&xp, nget, value);
#line 831
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 831
			status = lstatus;
#line 831

#line 831
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 831

#line 831
		remaining -= extent;
#line 831
		if(remaining == 0)
#line 831
			break; /* normal loop exit */
#line 831
		offset += extent;
#line 831
		value += nget;
#line 831
	}
#line 831

#line 831
	return status;
#line 831
}
#line 831

static int
#line 832
getNCvx_int_ulonglong(const NC3_INFO* ncp, const NC_var *varp,
#line 832
		 const size_t *start, size_t nelems, ulonglong *value)
#line 832
{
#line 832
	off_t offset = NC_varoffset(ncp, varp, start);
#line 832
	size_t remaining = varp->xsz * nelems;
#line 832
	int status = NC_NOERR;
#line 832
	const void *xp;
#line 832

#line 832
	if(nelems == 0)
#line 832
		return NC_NOERR;
#line 832

#line 832
	assert(value != NULL);
#line 832

#line 832
	for(;;)
#line 832
	{
#line 832
		size_t extent = MIN(remaining, ncp->chunk);
#line 832
		size_t nget = ncx_howmany(varp->type, extent);
#line 832

#line 832
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 832
				 0, (void **)&xp);	/* cast away const */
#line 832
		if(lstatus != NC_NOERR)
#line 832
			return lstatus;
#line 832
		
#line 832
		lstatus = ncx_getn_int_ulonglong(&xp, nget, value);
#line 832
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 832
			status = lstatus;
#line 832

#line 832
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 832

#line 832
		remaining -= extent;
#line 832
		if(remaining == 0)
#line 832
			break; /* normal loop exit */
#line 832
		offset += extent;
#line 832
		value += nget;
#line 832
	}
#line 832

#line 832
	return status;
#line 832
}
#line 832


static int
#line 834
getNCvx_float_schar(const NC3_INFO* ncp, const NC_var *varp,
#line 834
		 const size_t *start, size_t nelems, schar *value)
#line 834
{
#line 834
	off_t offset = NC_varoffset(ncp, varp, start);
#line 834
	size_t remaining = varp->xsz * nelems;
#line 834
	int status = NC_NOERR;
#line 834
	const void *xp;
#line 834

#line 834
	if(nelems == 0)
#line 834
		return NC_NOERR;
#line 834

#line 834
	assert(value != NULL);
#line 834

#line 834
	for(;;)
#line 834
	{
#line 834
		size_t extent = MIN(remaining, ncp->chunk);
#line 834
		size_t nget = ncx_howmany(varp->type, extent);
#line 834

#line 834
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 834
				 0, (void **)&xp);	/* cast away const */
#line 834
		if(lstatus != NC_NOERR)
#line 834
			return lstatus;
#line 834
		
#line 834
		lstatus = ncx_getn_float_schar(&xp, nget, value);
#line 834
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 834
			status = lstatus;
#line 834

#line 834
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 834

#line 834
		remaining -= extent;
#line 834
		if(remaining == 0)
#line 834
			break; /* normal loop exit */
#line 834
		offset += extent;
#line 834
		value += nget;
#line 834
	}
#line 834

#line 834
	return status;
#line 834
}
#line 834

static int
#line 835
getNCvx_float_uchar(const NC3_INFO* ncp, const NC_var *varp,
#line 835
		 const size_t *start, size_t nelems, uchar *value)
#line 835
{
#line 835
	off_t offset = NC_varoffset(ncp, varp, start);
#line 835
	size_t remaining = varp->xsz * nelems;
#line 835
	int status = NC_NOERR;
#line 835
	const void *xp;
#line 835

#line 835
	if(nelems == 0)
#line 835
		return NC_NOERR;
#line 835

#line 835
	assert(value != NULL);
#line 835

#line 835
	for(;;)
#line 835
	{
#line 835
		size_t extent = MIN(remaining, ncp->chunk);
#line 835
		size_t nget = ncx_howmany(varp->type, extent);
#line 835

#line 835
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 835
				 0, (void **)&xp);	/* cast away const */
#line 835
		if(lstatus != NC_NOERR)
#line 835
			return lstatus;
#line 835
		
#line 835
		lstatus = ncx_getn_float_uchar(&xp, nget, value);
#line 835
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 835
			status = lstatus;
#line 835

#line 835
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 835

#line 835
		remaining -= extent;
#line 835
		if(remaining == 0)
#line 835
			break; /* normal loop exit */
#line 835
		offset += extent;
#line 835
		value += nget;
#line 835
	}
#line 835

#line 835
	return status;
#line 835
}
#line 835

static int
#line 836
getNCvx_float_short(const NC3_INFO* ncp, const NC_var *varp,
#line 836
		 const size_t *start, size_t nelems, short *value)
#line 836
{
#line 836
	off_t offset = NC_varoffset(ncp, varp, start);
#line 836
	size_t remaining = varp->xsz * nelems;
#line 836
	int status = NC_NOERR;
#line 836
	const void *xp;
#line 836

#line 836
	if(nelems == 0)
#line 836
		return NC_NOERR;
#line 836

#line 836
	assert(value != NULL);
#line 836

#line 836
	for(;;)
#line 836
	{
#line 836
		size_t extent = MIN(remaining, ncp->chunk);
#line 836
		size_t nget = ncx_howmany(varp->type, extent);
#line 836

#line 836
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 836
				 0, (void **)&xp);	/* cast away const */
#line 836
		if(lstatus != NC_NOERR)
#line 836
			return lstatus;
#line 836
		
#line 836
		lstatus = ncx_getn_float_short(&xp, nget, value);
#line 836
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 836
			status = lstatus;
#line 836

#line 836
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 836

#line 836
		remaining -= extent;
#line 836
		if(remaining == 0)
#line 836
			break; /* normal loop exit */
#line 836
		offset += extent;
#line 836
		value += nget;
#line 836
	}
#line 836

#line 836
	return status;
#line 836
}
#line 836

static int
#line 837
getNCvx_float_int(const NC3_INFO* ncp, const NC_var *varp,
#line 837
		 const size_t *start, size_t nelems, int *value)
#line 837
{
#line 837
	off_t offset = NC_varoffset(ncp, varp, start);
#line 837
	size_t remaining = varp->xsz * nelems;
#line 837
	int status = NC_NOERR;
#line 837
	const void *xp;
#line 837

#line 837
	if(nelems == 0)
#line 837
		return NC_NOERR;
#line 837

#line 837
	assert(value != NULL);
#line 837

#line 837
	for(;;)
#line 837
	{
#line 837
		size_t extent = MIN(remaining, ncp->chunk);
#line 837
		size_t nget = ncx_howmany(varp->type, extent);
#line 837

#line 837
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 837
				 0, (void **)&xp);	/* cast away const */
#line 837
		if(lstatus != NC_NOERR)
#line 837
			return lstatus;
#line 837
		
#line 837
		lstatus = ncx_getn_float_int(&xp, nget, value);
#line 837
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 837
			status = lstatus;
#line 837

#line 837
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 837

#line 837
		remaining -= extent;
#line 837
		if(remaining == 0)
#line 837
			break; /* normal loop exit */
#line 837
		offset += extent;
#line 837
		value += nget;
#line 837
	}
#line 837

#line 837
	return status;
#line 837
}
#line 837

static int
#line 838
getNCvx_float_float(const NC3_INFO* ncp, const NC_var *varp,
#line 838
		 const size_t *start, size_t nelems, float *value)
#line 838
{
#line 838
	off_t offset = NC_varoffset(ncp, varp, start);
#line 838
	size_t remaining = varp->xsz * nelems;
#line 838
	int status = NC_NOERR;
#line 838
	const void *xp;
#line 838

#line 838
	if(nelems == 0)
#line 838
		return NC_NOERR;
#line 838

#line 838
	assert(value != NULL);
#line 838

#line 838
	for(;;)
#line 838
	{
#line 838
		size_t extent = MIN(remaining, ncp->chunk);
#line 838
		size_t nget = ncx_howmany(varp->type, extent);
#line 838

#line 838
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 838
				 0, (void **)&xp);	/* cast away const */
#line 838
		if(lstatus != NC_NOERR)
#line 838
			return lstatus;
#line 838
		
#line 838
		lstatus = ncx_getn_float_float(&xp, nget, value);
#line 838
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 838
			status = lstatus;
#line 838

#line 838
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 838

#line 838
		remaining -= extent;
#line 838
		if(remaining == 0)
#line 838
			break; /* normal loop exit */
#line 838
		offset += extent;
#line 838
		value += nget;
#line 838
	}
#line 838

#line 838
	return status;
#line 838
}
#line 838

static int
#line 839
getNCvx_float_double(const NC3_INFO* ncp, const NC_var *varp,
#line 839
		 const size_t *start, size_t nelems, double *value)
#line 839
{
#line 839
	off_t offset = NC_varoffset(ncp, varp, start);
#line 839
	size_t remaining = varp->xsz * nelems;
#line 839
	int status = NC_NOERR;
#line 839
	const void *xp;
#line 839

#line 839
	if(nelems == 0)
#line 839
		return NC_NOERR;
#line 839

#line 839
	assert(value != NULL);
#line 839

#line 839
	for(;;)
#line 839
	{
#line 839
		size_t extent = MIN(remaining, ncp->chunk);
#line 839
		size_t nget = ncx_howmany(varp->type, extent);
#line 839

#line 839
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 839
				 0, (void **)&xp);	/* cast away const */
#line 839
		if(lstatus != NC_NOERR)
#line 839
			return lstatus;
#line 839
		
#line 839
		lstatus = ncx_getn_float_double(&xp, nget, value);
#line 839
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 839
			status = lstatus;
#line 839

#line 839
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 839

#line 839
		remaining -= extent;
#line 839
		if(remaining == 0)
#line 839
			break; /* normal loop exit */
#line 839
		offset += extent;
#line 839
		value += nget;
#line 839
	}
#line 839

#line 839
	return status;
#line 839
}
#line 839

static int
#line 840
getNCvx_float_longlong(const NC3_INFO* ncp, const NC_var *varp,
#line 840
		 const size_t *start, size_t nelems, longlong *value)
#line 840
{
#line 840
	off_t offset = NC_varoffset(ncp, varp, start);
#line 840
	size_t remaining = varp->xsz * nelems;
#line 840
	int status = NC_NOERR;
#line 840
	const void *xp;
#line 840

#line 840
	if(nelems == 0)
#line 840
		return NC_NOERR;
#line 840

#line 840
	assert(value != NULL);
#line 840

#line 840
	for(;;)
#line 840
	{
#line 840
		size_t extent = MIN(remaining, ncp->chunk);
#line 840
		size_t nget = ncx_howmany(varp->type, extent);
#line 840

#line 840
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 840
				 0, (void **)&xp);	/* cast away const */
#line 840
		if(lstatus != NC_NOERR)
#line 840
			return lstatus;
#line 840
		
#line 840
		lstatus = ncx_getn_float_longlong(&xp, nget, value);
#line 840
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 840
			status = lstatus;
#line 840

#line 840
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 840

#line 840
		remaining -= extent;
#line 840
		if(remaining == 0)
#line 840
			break; /* normal loop exit */
#line 840
		offset += extent;
#line 840
		value += nget;
#line 840
	}
#line 840

#line 840
	return status;
#line 840
}
#line 840

static int
#line 841
getNCvx_float_uint(const NC3_INFO* ncp, const NC_var *varp,
#line 841
		 const size_t *start, size_t nelems, uint *value)
#line 841
{
#line 841
	off_t offset = NC_varoffset(ncp, varp, start);
#line 841
	size_t remaining = varp->xsz * nelems;
#line 841
	int status = NC_NOERR;
#line 841
	const void *xp;
#line 841

#line 841
	if(nelems == 0)
#line 841
		return NC_NOERR;
#line 841

#line 841
	assert(value != NULL);
#line 841

#line 841
	for(;;)
#line 841
	{
#line 841
		size_t extent = MIN(remaining, ncp->chunk);
#line 841
		size_t nget = ncx_howmany(varp->type, extent);
#line 841

#line 841
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 841
				 0, (void **)&xp);	/* cast away const */
#line 841
		if(lstatus != NC_NOERR)
#line 841
			return lstatus;
#line 841
		
#line 841
		lstatus = ncx_getn_float_uint(&xp, nget, value);
#line 841
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 841
			status = lstatus;
#line 841

#line 841
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 841

#line 841
		remaining -= extent;
#line 841
		if(remaining == 0)
#line 841
			break; /* normal loop exit */
#line 841
		offset += extent;
#line 841
		value += nget;
#line 841
	}
#line 841

#line 841
	return status;
#line 841
}
#line 841

static int
#line 842
getNCvx_float_ulonglong(const NC3_INFO* ncp, const NC_var *varp,
#line 842
		 const size_t *start, size_t nelems, ulonglong *value)
#line 842
{
#line 842
	off_t offset = NC_varoffset(ncp, varp, start);
#line 842
	size_t remaining = varp->xsz * nelems;
#line 842
	int status = NC_NOERR;
#line 842
	const void *xp;
#line 842

#line 842
	if(nelems == 0)
#line 842
		return NC_NOERR;
#line 842

#line 842
	assert(value != NULL);
#line 842

#line 842
	for(;;)
#line 842
	{
#line 842
		size_t extent = MIN(remaining, ncp->chunk);
#line 842
		size_t nget = ncx_howmany(varp->type, extent);
#line 842

#line 842
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 842
				 0, (void **)&xp);	/* cast away const */
#line 842
		if(lstatus != NC_NOERR)
#line 842
			return lstatus;
#line 842
		
#line 842
		lstatus = ncx_getn_float_ulonglong(&xp, nget, value);
#line 842
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 842
			status = lstatus;
#line 842

#line 842
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 842

#line 842
		remaining -= extent;
#line 842
		if(remaining == 0)
#line 842
			break; /* normal loop exit */
#line 842
		offset += extent;
#line 842
		value += nget;
#line 842
	}
#line 842

#line 842
	return status;
#line 842
}
#line 842


static int
#line 844
getNCvx_double_schar(const NC3_INFO* ncp, const NC_var *varp,
#line 844
		 const size_t *start, size_t nelems, schar *value)
#line 844
{
#line 844
	off_t offset = NC_varoffset(ncp, varp, start);
#line 844
	size_t remaining = varp->xsz * nelems;
#line 844
	int status = NC_NOERR;
#line 844
	const void *xp;
#line 844

#line 844
	if(nelems == 0)
#line 844
		return NC_NOERR;
#line 844

#line 844
	assert(value != NULL);
#line 844

#line 844
	for(;;)
#line 844
	{
#line 844
		size_t extent = MIN(remaining, ncp->chunk);
#line 844
		size_t nget = ncx_howmany(varp->type, extent);
#line 844

#line 844
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 844
				 0, (void **)&xp);	/* cast away const */
#line 844
		if(lstatus != NC_NOERR)
#line 844
			return lstatus;
#line 844
		
#line 844
		lstatus = ncx_getn_double_schar(&xp, nget, value);
#line 844
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 844
			status = lstatus;
#line 844

#line 844
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 844

#line 844
		remaining -= extent;
#line 844
		if(remaining == 0)
#line 844
			break; /* normal loop exit */
#line 844
		offset += extent;
#line 844
		value += nget;
#line 844
	}
#line 844

#line 844
	return status;
#line 844
}
#line 844

static int
#line 845
getNCvx_double_uchar(const NC3_INFO* ncp, const NC_var *varp,
#line 845
		 const size_t *start, size_t nelems, uchar *value)
#line 845
{
#line 845
	off_t offset = NC_varoffset(ncp, varp, start);
#line 845
	size_t remaining = varp->xsz * nelems;
#line 845
	int status = NC_NOERR;
#line 845
	const void *xp;
#line 845

#line 845
	if(nelems == 0)
#line 845
		return NC_NOERR;
#line 845

#line 845
	assert(value != NULL);
#line 845

#line 845
	for(;;)
#line 845
	{
#line 845
		size_t extent = MIN(remaining, ncp->chunk);
#line 845
		size_t nget = ncx_howmany(varp->type, extent);
#line 845

#line 845
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 845
				 0, (void **)&xp);	/* cast away const */
#line 845
		if(lstatus != NC_NOERR)
#line 845
			return lstatus;
#line 845
		
#line 845
		lstatus = ncx_getn_double_uchar(&xp, nget, value);
#line 845
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 845
			status = lstatus;
#line 845

#line 845
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 845

#line 845
		remaining -= extent;
#line 845
		if(remaining == 0)
#line 845
			break; /* normal loop exit */
#line 845
		offset += extent;
#line 845
		value += nget;
#line 845
	}
#line 845

#line 845
	return status;
#line 845
}
#line 845

static int
#line 846
getNCvx_double_short(const NC3_INFO* ncp, const NC_var *varp,
#line 846
		 const size_t *start, size_t nelems, short *value)
#line 846
{
#line 846
	off_t offset = NC_varoffset(ncp, varp, start);
#line 846
	size_t remaining = varp->xsz * nelems;
#line 846
	int status = NC_NOERR;
#line 846
	const void *xp;
#line 846

#line 846
	if(nelems == 0)
#line 846
		return NC_NOERR;
#line 846

#line 846
	assert(value != NULL);
#line 846

#line 846
	for(;;)
#line 846
	{
#line 846
		size_t extent = MIN(remaining, ncp->chunk);
#line 846
		size_t nget = ncx_howmany(varp->type, extent);
#line 846

#line 846
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 846
				 0, (void **)&xp);	/* cast away const */
#line 846
		if(lstatus != NC_NOERR)
#line 846
			return lstatus;
#line 846
		
#line 846
		lstatus = ncx_getn_double_short(&xp, nget, value);
#line 846
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 846
			status = lstatus;
#line 846

#line 846
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 846

#line 846
		remaining -= extent;
#line 846
		if(remaining == 0)
#line 846
			break; /* normal loop exit */
#line 846
		offset += extent;
#line 846
		value += nget;
#line 846
	}
#line 846

#line 846
	return status;
#line 846
}
#line 846

static int
#line 847
getNCvx_double_int(const NC3_INFO* ncp, const NC_var *varp,
#line 847
		 const size_t *start, size_t nelems, int *value)
#line 847
{
#line 847
	off_t offset = NC_varoffset(ncp, varp, start);
#line 847
	size_t remaining = varp->xsz * nelems;
#line 847
	int status = NC_NOERR;
#line 847
	const void *xp;
#line 847

#line 847
	if(nelems == 0)
#line 847
		return NC_NOERR;
#line 847

#line 847
	assert(value != NULL);
#line 847

#line 847
	for(;;)
#line 847
	{
#line 847
		size_t extent = MIN(remaining, ncp->chunk);
#line 847
		size_t nget = ncx_howmany(varp->type, extent);
#line 847

#line 847
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 847
				 0, (void **)&xp);	/* cast away const */
#line 847
		if(lstatus != NC_NOERR)
#line 847
			return lstatus;
#line 847
		
#line 847
		lstatus = ncx_getn_double_int(&xp, nget, value);
#line 847
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 847
			status = lstatus;
#line 847

#line 847
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 847

#line 847
		remaining -= extent;
#line 847
		if(remaining == 0)
#line 847
			break; /* normal loop exit */
#line 847
		offset += extent;
#line 847
		value += nget;
#line 847
	}
#line 847

#line 847
	return status;
#line 847
}
#line 847

static int
#line 848
getNCvx_double_float(const NC3_INFO* ncp, const NC_var *varp,
#line 848
		 const size_t *start, size_t nelems, float *value)
#line 848
{
#line 848
	off_t offset = NC_varoffset(ncp, varp, start);
#line 848
	size_t remaining = varp->xsz * nelems;
#line 848
	int status = NC_NOERR;
#line 848
	const void *xp;
#line 848

#line 848
	if(nelems == 0)
#line 848
		return NC_NOERR;
#line 848

#line 848
	assert(value != NULL);
#line 848

#line 848
	for(;;)
#line 848
	{
#line 848
		size_t extent = MIN(remaining, ncp->chunk);
#line 848
		size_t nget = ncx_howmany(varp->type, extent);
#line 848

#line 848
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 848
				 0, (void **)&xp);	/* cast away const */
#line 848
		if(lstatus != NC_NOERR)
#line 848
			return lstatus;
#line 848
		
#line 848
		lstatus = ncx_getn_double_float(&xp, nget, value);
#line 848
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 848
			status = lstatus;
#line 848

#line 848
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 848

#line 848
		remaining -= extent;
#line 848
		if(remaining == 0)
#line 848
			break; /* normal loop exit */
#line 848
		offset += extent;
#line 848
		value += nget;
#line 848
	}
#line 848

#line 848
	return status;
#line 848
}
#line 848

static int
#line 849
getNCvx_double_double(const NC3_INFO* ncp, const NC_var *varp,
#line 849
		 const size_t *start, size_t nelems, double *value)
#line 849
{
#line 849
	off_t offset = NC_varoffset(ncp, varp, start);
#line 849
	size_t remaining = varp->xsz * nelems;
#line 849
	int status = NC_NOERR;
#line 849
	const void *xp;
#line 849

#line 849
	if(nelems == 0)
#line 849
		return NC_NOERR;
#line 849

#line 849
	assert(value != NULL);
#line 849

#line 849
	for(;;)
#line 849
	{
#line 849
		size_t extent = MIN(remaining, ncp->chunk);
#line 849
		size_t nget = ncx_howmany(varp->type, extent);
#line 849

#line 849
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 849
				 0, (void **)&xp);	/* cast away const */
#line 849
		if(lstatus != NC_NOERR)
#line 849
			return lstatus;
#line 849
		
#line 849
		lstatus = ncx_getn_double_double(&xp, nget, value);
#line 849
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 849
			status = lstatus;
#line 849

#line 849
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 849

#line 849
		remaining -= extent;
#line 849
		if(remaining == 0)
#line 849
			break; /* normal loop exit */
#line 849
		offset += extent;
#line 849
		value += nget;
#line 849
	}
#line 849

#line 849
	return status;
#line 849
}
#line 849

static int
#line 850
getNCvx_double_longlong(const NC3_INFO* ncp, const NC_var *varp,
#line 850
		 const size_t *start, size_t nelems, longlong *value)
#line 850
{
#line 850
	off_t offset = NC_varoffset(ncp, varp, start);
#line 850
	size_t remaining = varp->xsz * nelems;
#line 850
	int status = NC_NOERR;
#line 850
	const void *xp;
#line 850

#line 850
	if(nelems == 0)
#line 850
		return NC_NOERR;
#line 850

#line 850
	assert(value != NULL);
#line 850

#line 850
	for(;;)
#line 850
	{
#line 850
		size_t extent = MIN(remaining, ncp->chunk);
#line 850
		size_t nget = ncx_howmany(varp->type, extent);
#line 850

#line 850
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 850
				 0, (void **)&xp);	/* cast away const */
#line 850
		if(lstatus != NC_NOERR)
#line 850
			return lstatus;
#line 850
		
#line 850
		lstatus = ncx_getn_double_longlong(&xp, nget, value);
#line 850
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 850
			status = lstatus;
#line 850

#line 850
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 850

#line 850
		remaining -= extent;
#line 850
		if(remaining == 0)
#line 850
			break; /* normal loop exit */
#line 850
		offset += extent;
#line 850
		value += nget;
#line 850
	}
#line 850

#line 850
	return status;
#line 850
}
#line 850

static int
#line 851
getNCvx_double_uint(const NC3_INFO* ncp, const NC_var *varp,
#line 851
		 const size_t *start, size_t nelems, uint *value)
#line 851
{
#line 851
	off_t offset = NC_varoffset(ncp, varp, start);
#line 851
	size_t remaining = varp->xsz * nelems;
#line 851
	int status = NC_NOERR;
#line 851
	const void *xp;
#line 851

#line 851
	if(nelems == 0)
#line 851
		return NC_NOERR;
#line 851

#line 851
	assert(value != NULL);
#line 851

#line 851
	for(;;)
#line 851
	{
#line 851
		size_t extent = MIN(remaining, ncp->chunk);
#line 851
		size_t nget = ncx_howmany(varp->type, extent);
#line 851

#line 851
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 851
				 0, (void **)&xp);	/* cast away const */
#line 851
		if(lstatus != NC_NOERR)
#line 851
			return lstatus;
#line 851
		
#line 851
		lstatus = ncx_getn_double_uint(&xp, nget, value);
#line 851
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 851
			status = lstatus;
#line 851

#line 851
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 851

#line 851
		remaining -= extent;
#line 851
		if(remaining == 0)
#line 851
			break; /* normal loop exit */
#line 851
		offset += extent;
#line 851
		value += nget;
#line 851
	}
#line 851

#line 851
	return status;
#line 851
}
#line 851

static int
#line 852
getNCvx_double_ulonglong(const NC3_INFO* ncp, const NC_var *varp,
#line 852
		 const size_t *start, size_t nelems, ulonglong *value)
#line 852
{
#line 852
	off_t offset = NC_varoffset(ncp, varp, start);
#line 852
	size_t remaining = varp->xsz * nelems;
#line 852
	int status = NC_NOERR;
#line 852
	const void *xp;
#line 852

#line 852
	if(nelems == 0)
#line 852
		return NC_NOERR;
#line 852

#line 852
	assert(value != NULL);
#line 852

#line 852
	for(;;)
#line 852
	{
#line 852
		size_t extent = MIN(remaining, ncp->chunk);
#line 852
		size_t nget = ncx_howmany(varp->type, extent);
#line 852

#line 852
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 852
				 0, (void **)&xp);	/* cast away const */
#line 852
		if(lstatus != NC_NOERR)
#line 852
			return lstatus;
#line 852
		
#line 852
		lstatus = ncx_getn_double_ulonglong(&xp, nget, value);
#line 852
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 852
			status = lstatus;
#line 852

#line 852
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 852

#line 852
		remaining -= extent;
#line 852
		if(remaining == 0)
#line 852
			break; /* normal loop exit */
#line 852
		offset += extent;
#line 852
		value += nget;
#line 852
	}
#line 852

#line 852
	return status;
#line 852
}
#line 852


#line 855
#ifdef NOTUSED
static int
#line 856
getNCvx_schar_uchar(const NC3_INFO* ncp, const NC_var *varp,
#line 856
		 const size_t *start, size_t nelems, uchar *value)
#line 856
{
#line 856
	off_t offset = NC_varoffset(ncp, varp, start);
#line 856
	size_t remaining = varp->xsz * nelems;
#line 856
	int status = NC_NOERR;
#line 856
	const void *xp;
#line 856

#line 856
	if(nelems == 0)
#line 856
		return NC_NOERR;
#line 856

#line 856
	assert(value != NULL);
#line 856

#line 856
	for(;;)
#line 856
	{
#line 856
		size_t extent = MIN(remaining, ncp->chunk);
#line 856
		size_t nget = ncx_howmany(varp->type, extent);
#line 856

#line 856
		int lstatus = ncio_get(ncp->nciop, offset, extent,
#line 856
				 0, (void **)&xp);	/* cast away const */
#line 856
		if(lstatus != NC_NOERR)
#line 856
			return lstatus;
#line 856
		
#line 856
		lstatus = ncx_getn_schar_uchar(&xp, nget, value);
#line 856
		if(lstatus != NC_NOERR && status == NC_NOERR)
#line 856
			status = lstatus;
#line 856

#line 856
		(void) ncio_rel(ncp->nciop, offset, 0);	
#line 856

#line 856
		remaining -= extent;
#line 856
		if(remaining == 0)
#line 856
			break; /* normal loop exit */
#line 856
		offset += extent;
#line 856
		value += nget;
#line 856
	}
#line 856

#line 856
	return status;
#line 856
}
#line 856

#endif /*NOTUSED*/

/*
 *  For ncvar{put,get},
 *  find the largest contiguous block from within 'edges'.
 *  returns the index to the left of this (which may be -1).
 *  Compute the number of contiguous elements and return
 *  that in *iocountp.
 *  The presence of "record" variables makes this routine
 *  overly subtle.
 */
static int
NCiocount(const NC3_INFO* const ncp, const NC_var *const varp,
	const size_t *const edges,
	size_t *const iocountp)
{
	const size_t *edp0 = edges;
	const size_t *edp = edges + varp->ndims;
	const size_t *shp = varp->shape + varp->ndims;

	if(IS_RECVAR(varp))
	{
		if(varp->ndims == 1 && ncp->recsize <= varp->len)
		{
			/* one dimensional && the only 'record' variable */
			*iocountp = *edges;
			return(0);
		}
		/* else */
		edp0++;
	}

	assert(edges != NULL);

	/* find max contiguous */
	while(edp > edp0)
	{
		shp--; edp--;
		if(*edp < *shp )
		{
			const size_t *zedp = edp;
			while(zedp >= edp0)
			{
				if(*zedp == 0)
				{
					*iocountp = 0;
					goto done;
				}
				/* Tip of the hat to segmented architectures */
				if(zedp == edp0)
					break;
				zedp--;
			}
			break;
		}
		assert(*edp == *shp);
	}

	/*
	 * edp, shp reference rightmost index s.t. *(edp +1) == *(shp +1)
	 *
	 * Or there is only one dimension.
	 * If there is only one dimension and it is 'non record' dimension,
	 * 	edp is &edges[0] and we will return -1.
	 * If there is only one dimension and and it is a "record dimension",
	 *	edp is &edges[1] (out of bounds) and we will return 0;
	 */
	assert(shp >= varp->shape + varp->ndims -1 
		|| *(edp +1) == *(shp +1));

	/* now accumulate max count for a single io operation */
	for(*iocountp = 1, edp0 = edp;
		 	edp0 < edges + varp->ndims;
			edp0++)
	{
		*iocountp *= *edp0;
	}

done:
	return((int)(edp - edges) - 1);
}


/*
 * Set the elements of the array 'upp' to
 * the sum of the corresponding elements of
 * 'stp' and 'edp'. 'end' should be &stp[nelems].
 */
static void
set_upper(size_t *upp, /* modified on return */
	const size_t *stp,
	const size_t *edp,
	const size_t *const end)
{
	while(upp < end) {
		*upp++ = *stp++ + *edp++;
	}
}


/*
 * The infamous and oft-discussed odometer code.
 *
 * 'start[]' is the starting coordinate.
 * 'upper[]' is the upper bound s.t. start[ii] < upper[ii].
 * 'coord[]' is the register, the current coordinate value.
 * For some ii,
 * upp == &upper[ii]
 * cdp == &coord[ii]
 * 
 * Running this routine increments *cdp.
 *
 * If after the increment, *cdp is equal to *upp
 * (and cdp is not the leftmost dimension),
 * *cdp is "zeroed" to the starting value and
 * we need to "carry", eg, increment one place to
 * the left.
 * 
 * TODO: Some architectures hate recursion?
 * 	Reimplement non-recursively.
 */
static void
odo1(const size_t *const start, const size_t *const upper,
	size_t *const coord, /* modified on return */
	const size_t *upp,
	size_t *cdp)
{
	assert(coord <= cdp && cdp <= coord + NC_MAX_VAR_DIMS);
	assert(upper <= upp && upp <= upper + NC_MAX_VAR_DIMS);
	assert(upp - upper == cdp - coord);
	
	assert(*cdp <= *upp);

	(*cdp)++;
	if(cdp != coord && *cdp >= *upp)
	{
		*cdp = start[cdp - coord];
		odo1(start, upper, coord, upp -1, cdp -1);
	}
}
#ifdef _CRAYC
#pragma _CRI noinline odo1
#endif


#line 1018

/* Define a macro to allow hash on two type values */
#define CASE(nc1,nc2) (nc1*256+nc2)

static int
readNCv(const NC3_INFO* ncp, const NC_var* varp, const size_t* start,
        const size_t nelems, void* value, const nc_type memtype)
{
    int status = NC_NOERR;
    switch (CASE(varp->type,memtype)) {
    case CASE(NC_CHAR,NC_CHAR):
    case CASE(NC_CHAR,NC_UBYTE):
        status = getNCvx_char_char(ncp,varp,start,nelems,(char*)value);
        break;

    case CASE(NC_BYTE,NC_BYTE):
    case CASE(NC_BYTE,NC_UBYTE):
        status = getNCvx_schar_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_BYTE,NC_SHORT):
        status = getNCvx_schar_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_BYTE,NC_INT):
        status = getNCvx_schar_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_BYTE,NC_FLOAT):
        status = getNCvx_schar_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_BYTE,NC_DOUBLE):
        status = getNCvx_schar_double(ncp,varp,start,nelems,(double *)value);
        break;
    case CASE(NC_BYTE,NC_INT64):
        status = getNCvx_schar_longlong(ncp,varp,start,nelems,(long long*)value);
        break;
    case CASE(NC_BYTE,NC_UINT):
        status = getNCvx_schar_uint(ncp,varp,start,nelems,(unsigned int*)value);
        break;
    case CASE(NC_BYTE,NC_UINT64):
        status = getNCvx_schar_ulonglong(ncp,varp,start,nelems,(unsigned long long*)value);
        break;

    case CASE(NC_SHORT,NC_BYTE):
        status = getNCvx_short_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_SHORT,NC_UBYTE):
        status = getNCvx_short_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_SHORT,NC_SHORT):
        status = getNCvx_short_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_SHORT,NC_INT):
        status = getNCvx_short_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_SHORT,NC_FLOAT):
        status = getNCvx_short_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_SHORT,NC_DOUBLE):
        status = getNCvx_short_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_SHORT,NC_INT64):
        status = getNCvx_short_longlong(ncp,varp,start,nelems,(long long*)value);
        break;
    case CASE(NC_SHORT,NC_UINT):
        status = getNCvx_short_uint(ncp,varp,start,nelems,(unsigned int*)value);
        break;
    case CASE(NC_SHORT,NC_UINT64):
        status = getNCvx_short_ulonglong(ncp,varp,start,nelems,(unsigned long long*)value);
        break;


    case CASE(NC_INT,NC_BYTE):
        status = getNCvx_int_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_INT,NC_UBYTE):
        status = getNCvx_int_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_INT,NC_SHORT):
        status = getNCvx_int_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_INT,NC_INT):
        status = getNCvx_int_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_INT,NC_FLOAT):
        status = getNCvx_int_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_INT,NC_DOUBLE):
        status = getNCvx_int_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_INT,NC_INT64):
        status = getNCvx_int_longlong(ncp,varp,start,nelems,(long long*)value);
        break;
    case CASE(NC_INT,NC_UINT):
        status = getNCvx_int_uint(ncp,varp,start,nelems,(unsigned int*)value);
        break;
    case CASE(NC_INT,NC_UINT64):
        status = getNCvx_int_ulonglong(ncp,varp,start,nelems,(unsigned long long*)value);
        break;


    case CASE(NC_FLOAT,NC_BYTE):
        status = getNCvx_float_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_FLOAT,NC_UBYTE):
        status = getNCvx_float_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_FLOAT,NC_SHORT):
        status = getNCvx_float_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_FLOAT,NC_INT):
        status = getNCvx_float_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_FLOAT,NC_FLOAT):
        status = getNCvx_float_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_FLOAT,NC_DOUBLE):
        status = getNCvx_float_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_FLOAT,NC_INT64):
        status = getNCvx_float_longlong(ncp,varp,start,nelems,(long long*)value);
        break;
    case CASE(NC_FLOAT,NC_UINT):
        status = getNCvx_float_uint(ncp,varp,start,nelems,(unsigned int*)value);
        break;
    case CASE(NC_FLOAT,NC_UINT64):
        status = getNCvx_float_ulonglong(ncp,varp,start,nelems,(unsigned long long*)value);
        break;


    case CASE(NC_DOUBLE,NC_BYTE):
        status = getNCvx_double_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_DOUBLE,NC_UBYTE):
        status = getNCvx_double_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_DOUBLE,NC_SHORT):
        status = getNCvx_double_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_DOUBLE,NC_INT):
        status = getNCvx_double_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_DOUBLE,NC_FLOAT):
        status = getNCvx_double_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_DOUBLE,NC_DOUBLE):
        status = getNCvx_double_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_DOUBLE,NC_INT64):
        status = getNCvx_double_longlong(ncp,varp,start,nelems,(long long*)value);
        break;
    case CASE(NC_DOUBLE,NC_UINT):
        status = getNCvx_double_uint(ncp,varp,start,nelems,(unsigned int*)value);
        break;
    case CASE(NC_DOUBLE,NC_UINT64):
        status = getNCvx_double_ulonglong(ncp,varp,start,nelems,(unsigned long long*)value);
        break;

    default:
	return NC_EBADTYPE;
    }
    return status;
}


static int
writeNCv(NC3_INFO* ncp, const NC_var* varp, const size_t* start,
         const size_t nelems, const void* value, const nc_type memtype)
{
    int status = NC_NOERR;
    switch (CASE(varp->type,memtype)) {
    case CASE(NC_CHAR,NC_CHAR):
    case CASE(NC_CHAR,NC_UBYTE):
        status = putNCvx_char_char(ncp,varp,start,nelems,(char*)value);
        break;

    case CASE(NC_BYTE,NC_BYTE):
        status = putNCvx_schar_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_BYTE,NC_UBYTE):
        status = putNCvx_schar_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_BYTE,NC_SHORT):
        status = putNCvx_schar_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_BYTE,NC_INT):
        status = putNCvx_schar_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_BYTE,NC_FLOAT):
        status = putNCvx_schar_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_BYTE,NC_DOUBLE):
        status = putNCvx_schar_double(ncp,varp,start,nelems,(double *)value);
        break;
    case CASE(NC_BYTE,NC_INT64):
        status = putNCvx_schar_longlong(ncp,varp,start,nelems,(long long*)value);
        break;

    case CASE(NC_SHORT,NC_BYTE):
        status = putNCvx_short_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_SHORT,NC_UBYTE):
        status = putNCvx_short_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_SHORT,NC_SHORT):
        status = putNCvx_short_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_SHORT,NC_INT):
        status = putNCvx_short_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_SHORT,NC_FLOAT):
        status = putNCvx_short_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_SHORT,NC_DOUBLE):
        status = putNCvx_short_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_SHORT,NC_INT64):
        status = putNCvx_short_longlong(ncp,varp,start,nelems,(long long*)value);
        break;

    case CASE(NC_INT,NC_BYTE):
        status = putNCvx_int_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_INT,NC_UBYTE):
        status = putNCvx_int_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_INT,NC_SHORT):
        status = putNCvx_int_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_INT,NC_INT):
        status = putNCvx_int_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_INT,NC_FLOAT):
        status = putNCvx_int_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_INT,NC_DOUBLE):
        status = putNCvx_int_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_INT,NC_INT64):
        status = putNCvx_int_longlong(ncp,varp,start,nelems,(long long*)value);
        break;

    case CASE(NC_FLOAT,NC_BYTE):
        status = putNCvx_float_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_FLOAT,NC_UBYTE):
        status = putNCvx_float_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_FLOAT,NC_SHORT):
        status = putNCvx_float_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_FLOAT,NC_INT):
        status = putNCvx_float_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_FLOAT,NC_FLOAT):
        status = putNCvx_float_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_FLOAT,NC_DOUBLE):
        status = putNCvx_float_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_FLOAT,NC_INT64):
        status = putNCvx_float_longlong(ncp,varp,start,nelems,(long long*)value);
        break;

    case CASE(NC_DOUBLE,NC_BYTE):
        status = putNCvx_double_schar(ncp,varp,start,nelems,(signed char*)value);
        break;
    case CASE(NC_DOUBLE,NC_UBYTE):
        status = putNCvx_double_uchar(ncp,varp,start,nelems,(unsigned char*)value);
        break;
    case CASE(NC_DOUBLE,NC_SHORT):
        status = putNCvx_double_short(ncp,varp,start,nelems,(short*)value);
        break;
    case CASE(NC_DOUBLE,NC_INT):
        status = putNCvx_double_int(ncp,varp,start,nelems,(int*)value);
        break;
    case CASE(NC_DOUBLE,NC_FLOAT):
        status = putNCvx_double_float(ncp,varp,start,nelems,(float*)value);
        break;
    case CASE(NC_DOUBLE,NC_DOUBLE):
        status = putNCvx_double_double(ncp,varp,start,nelems,(double*)value);
        break;
    case CASE(NC_DOUBLE,NC_INT64):
        status = putNCvx_double_longlong(ncp,varp,start,nelems,(long long*)value);
        break;

    default:
	return NC_EBADTYPE;
    }
    return status;
}

/**************************************************/

int
NC3_get_vara(int ncid, int varid,
	    const size_t *start, const size_t *edges0,
            void *value0,
	    nc_type memtype)
{
    int status = NC_NOERR;
    NC* nc;
    NC3_INFO* nc3;
    NC_var *varp;
    int ii;
    size_t iocount;
    size_t memtypelen;
    char* value = (char*) value0; /* legally allow ptr arithmetic */
    const size_t* edges = edges0; /* so we can modify for special cases */
    size_t modedges[NC_MAX_VAR_DIMS];

    status = NC_check_id(ncid, &nc); 
    if(status != NC_NOERR)
        return status;
    nc3 = NC3_DATA(nc);

    if(NC_indef(nc3))
        return NC_EINDEFINE;

    varp = NC_lookupvar(nc3, varid);
    if(varp == NULL)
        return NC_ENOTVAR;

    if(memtype == NC_NAT) memtype=varp->type;

    if(memtype == NC_CHAR && varp->type != NC_CHAR)
        return NC_ECHAR;
    else if(memtype != NC_CHAR && varp->type == NC_CHAR)  
        return NC_ECHAR;

    /* If edges is NULL, then this was called from nc_get_var() */
    if(edges == NULL && varp->ndims > 0) {
	/* If this is a record variable, then we have to
           substitute the number of records into dimension 0. */
	if(varp->shape[0] == 0) {
	    (void*)memcpy((void*)modedges,(void*)varp->shape,
                          sizeof(size_t)*varp->ndims);
	    modedges[0] = NC_get_numrecs(nc3);
	    edges = modedges;
	} else
	    edges = varp->shape;
    }

    status = NCcoordck(nc3, varp, start);
    if(status != NC_NOERR)
        return status;

    status = NCedgeck(nc3, varp, start, edges);
    if(status != NC_NOERR)
        return status;

    /* Get the size of the memtype */
    memtypelen = nctypelen(memtype);

    if(varp->ndims == 0) /* scalar variable */
    {
        return( readNCv(nc3, varp, start, 1, (void*)value, memtype) );
    }

    if(IS_RECVAR(varp))
    {
        if(*start + *edges > NC_get_numrecs(nc3))
            return NC_EEDGE;
        if(varp->ndims == 1 && nc3->recsize <= varp->len)
        {
            /* one dimensional && the only record variable  */
            return( readNCv(nc3, varp, start, *edges, (void*)value, memtype) );
        }
    }

    /*
     * find max contiguous
     *   and accumulate max count for a single io operation
     */
    ii = NCiocount(nc3, varp, edges, &iocount);

    if(ii == -1)
    {
        return( readNCv(nc3, varp, start, iocount, (void*)value, memtype) );
    }

    assert(ii >= 0);

    { /* inline */
    ALLOC_ONSTACK(coord, size_t, varp->ndims);
    ALLOC_ONSTACK(upper, size_t, varp->ndims);
    const size_t index = ii;

    /* copy in starting indices */
    (void) memcpy(coord, start, varp->ndims * sizeof(size_t));

    /* set up in maximum indices */
    set_upper(upper, start, edges, &upper[varp->ndims]);

    /* ripple counter */
    while(*coord < *upper)
    {
        const int lstatus = readNCv(nc3, varp, coord, iocount, (void*)value, memtype);
	if(lstatus != NC_NOERR)
        {
            if(lstatus != NC_ERANGE)
            {
                status = lstatus;
                /* fatal for the loop */
                break;
            }
            /* else NC_ERANGE, not fatal for the loop */
            if(status == NC_NOERR)
                status = lstatus;
        }
        value += (iocount * memtypelen);
        odo1(start, upper, coord, &upper[index], &coord[index]);
    }

    FREE_ONSTACK(upper);
    FREE_ONSTACK(coord);
    } /* end inline */

    return status;
}

int
NC3_put_vara(int ncid, int varid,
	    const size_t *start, const size_t *edges0,
            const void *value0,
	    nc_type memtype)
{
    int status = NC_NOERR;
    NC *nc;
    NC3_INFO* nc3;
    NC_var *varp;
    int ii;
    size_t iocount;
    size_t memtypelen;
    char* value = (char*) value0; /* legally allow ptr arithmetic */
    const size_t* edges = edges0; /* so we can modify for special cases */
    size_t modedges[NC_MAX_VAR_DIMS];

    status = NC_check_id(ncid, &nc); 
    if(status != NC_NOERR)
        return status;
    nc3 = NC3_DATA(nc);

    if(NC_readonly(nc3))
        return NC_EPERM;

    if(NC_indef(nc3))
        return NC_EINDEFINE;

    varp = NC_lookupvar(nc3, varid);
    if(varp == NULL)
        return NC_ENOTVAR; /* TODO: lost NC_EGLOBAL */

    if(memtype == NC_NAT) memtype=varp->type;

    if(memtype == NC_CHAR && varp->type != NC_CHAR)
        return NC_ECHAR;
    else if(memtype != NC_CHAR && varp->type == NC_CHAR)  
        return NC_ECHAR;

    /* Get the size of the memtype */
    memtypelen = nctypelen(memtype);

    /* If edges is NULL, then this was called from nc_get_var() */
    if(edges == NULL && varp->ndims > 0) {
	/* If this is a record variable, then we have to
           substitute the number of records into dimension 0. */
	if(varp->shape[0] == 0) {
	    (void*)memcpy((void*)modedges,(void*)varp->shape,
                          sizeof(size_t)*varp->ndims);
	    modedges[0] = NC_get_numrecs(nc3);
	    edges = modedges;
	} else
	    edges = varp->shape;
    }

    status = NCcoordck(nc3, varp, start);
    if(status != NC_NOERR)
        return status;
    status = NCedgeck(nc3, varp, start, edges);
    if(status != NC_NOERR)
        return status;

    if(varp->ndims == 0) /* scalar variable */
    {
        return( writeNCv(nc3, varp, start, 1, (void*)value, memtype) );
    }

    if(IS_RECVAR(varp))
    {
        status = NCvnrecs(nc3, *start + *edges);
        if(status != NC_NOERR)
            return status;

        if(varp->ndims == 1
            && nc3->recsize <= varp->len)
        {
            /* one dimensional && the only record variable  */
            return( writeNCv(nc3, varp, start, *edges, (void*)value, memtype) );
        }
    }

    /*
     * find max contiguous
     *   and accumulate max count for a single io operation
     */
    ii = NCiocount(nc3, varp, edges, &iocount);

    if(ii == -1)
    {
        return( writeNCv(nc3, varp, start, iocount, (void*)value, memtype) );
    }

    assert(ii >= 0);

    { /* inline */
    ALLOC_ONSTACK(coord, size_t, varp->ndims);
    ALLOC_ONSTACK(upper, size_t, varp->ndims);
    const size_t index = ii;

    /* copy in starting indices */
    (void) memcpy(coord, start, varp->ndims * sizeof(size_t));

    /* set up in maximum indices */
    set_upper(upper, start, edges, &upper[varp->ndims]);

    /* ripple counter */
    while(*coord < *upper)
    {
        const int lstatus = writeNCv(nc3, varp, coord, iocount, (void*)value, memtype);
        if(lstatus != NC_NOERR)
        {
            if(lstatus != NC_ERANGE)
            {
                status = lstatus;
                /* fatal for the loop */
                break;
            }
            /* else NC_ERANGE, not fatal for the loop */
            if(status == NC_NOERR)
                status = lstatus;
        }
        value += (iocount * memtypelen);
        odo1(start, upper, coord, &upper[index], &coord[index]);
    }

    FREE_ONSTACK(upper);
    FREE_ONSTACK(coord);
    } /* end inline */

    return status;
}
