/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef BEARING_ONLY_SLAM_2D_HH_INCLUDED
#define BEARING_ONLY_SLAM_2D_HH_INCLUDED


#include <map>
#include <cstddef>
#include <utility>
#include <boost/numeric/ublas/matrix.hpp>
#include <boost/numeric/ublas/vector.hpp>
#include <boost/numeric/ublas/matrix_proxy.hpp>


namespace ublas = boost::numeric::ublas;


/**
 * \brief A very simple implementation of bearing-only EKF-SLAM in 2D, for
 * features with known data associations.
 */
class BearingOnlySLAM2D
{
public:
    typedef std::size_t FeatureID;

    // state vector indices
    enum
    {
        ROBOT_X = 0,
        ROBOT_Y = 1,
        ROBOT_THETA = 2
    };

    struct Observation
    {
        FeatureID m_feat_id;
        double m_bearing;
        double m_bearing_std;
        bool m_is_outlier; // set by update()
    };

    // Process noise:
    //   sigma1 = m/m, sigma2 = rad/m, sigma3 = rad/rad
    BearingOnlySLAM2D(double sigma1, double sigma2, double sigma3);

    void update(double ds, double dtheta, std::vector<Observation> &obs_);

    std::size_t feature_count() const;

    void get_pose(double &x, double &y, double &theta) const;

    void get_pose(
        double &x, double &y, double &theta,
        ublas::matrix<double> &cov) const;

    // Get the i:th feature, in Cartesian space, with a Cartesian space
    // covariance
    FeatureID get_cartesian_feature(
        std::size_t i,
        double &x, double &y,
        ublas::matrix<double> &xy_cov) const;

private:
    double normalize_rad(double rad) const;

    bool has_feature(FeatureID feat_id) const;

private:
    // State vector x = (x_r, y_r, theta_r, ..., x_i, y_i, phi_i, rho_i, ... )
    ublas::vector<double> m_x;

    // State covariance P
    ublas::matrix<double> m_P;

    // Process noise parameters
    const double m_sigma1, m_sigma2, m_sigma3;

    // Maps a feature id to the index in the state vector/covariance matrix
    // where the feature is coded
    typedef std::map<FeatureID, std::size_t> IdToIdxMap;
    IdToIdxMap m_feats;
};


#endif // BEARING_ONLY_SLAM_2D_HH_INCLUDED
