/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <ctime>
#include <iostream>
#include <fstream>
#include <sstream>
#include <boost/cstdint.hpp>

#include "BoSLAM.hh"


namespace
{
    // Odometry parameters
    const double R = 0.1;
    const double N = 2048;
    const double B = 0.35;
}


bool get_data(
    std::ifstream &ifs,
    boost::int64_t &enc_l, boost::int64_t &enc_r,
    std::vector<BearingOnlySLAM2D::Observation> &obs)
{
    obs.clear();

    std::string line;
    if( !getline(ifs, line) )
        return false;

    std::istringstream ss(line);

    std::size_t n_obs = 0;
    double t, odom_x, odom_y, odom_theta, gt_x, gt_y, gt_theta; // temp

    if( ss >> t
        >> odom_x >> odom_y >> odom_theta
        >> enc_r >> enc_l
        >> gt_x >> gt_y >> gt_theta
        >> n_obs )
    {
        if( n_obs > 0 )
        {
            obs.resize(n_obs);
            double range; // temp

            for( std::size_t i = 0; i < n_obs; i++ )
            {
                obs[i].m_bearing_std = 0.05;
                if( !(ss >> obs[i].m_feat_id >> obs[i].m_bearing >> range) )
                {
                    std::cerr << "Error parsing data file!" << std::endl;
                    return false;
                }
            }
        }

        return true;
    }
    else
    {
        std::cerr << "Error parsing data file!" << std::endl;
        return false;
    }
}


void millisleep(std::size_t ms)
{
    timespec ts, rem;
    ts.tv_sec = ms/1000;
    ts.tv_nsec = (ms - ts.tv_sec*1000)*1000*1000;

    while( nanosleep(&ts, &rem) == -1 )
    {
        ts = rem;
    }
}


int main(int argc, char *argv[])
{
    std::ifstream ifs("bo-slam.data");
    if( !ifs )
    {
        std::cerr << "Could not open data file 'bo-slam.data' for reading."
                  << std::endl;
        return -1;
    }

    BoSLAM slam;

    boost::int64_t prev_enc_l = 0, prev_enc_r = 0;
    boost::int64_t enc_l, enc_r;
    std::vector<BearingOnlySLAM2D::Observation> obs;

    while( get_data(ifs, enc_l, enc_r, obs) )
    {
        int uL = enc_l - prev_enc_l;
        int uR = enc_r - prev_enc_r;
        double aL = uL*2*M_PI/N;
        double aR = uR*2*M_PI/N;
        double ds = (aR*R + aL*R)/2;
        double dtheta = (aR*R - aL*R)/B;
        prev_enc_l = enc_l;
        prev_enc_r = enc_r;

        slam.update(ds, dtheta, obs);

        millisleep(50);
    }
}
