/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ActionSource.hh"
#include "UploadCache.hh"
#include "Config.hh"
#include "Path.hh"
#include "MessageHub.hh"

#include <boost/bind.hpp>
#include <stdexcept>


using namespace peekabot;


ActionSource::ActionSource(const Config &config)
    : m_upload_cache(UploadCache::create())
{
    m_search_dirs =
        config.get<std::list<boost::filesystem::path> >("data_paths");
}


void ActionSource::add_pseudonym(ObjectID pseudo_id, ObjectID real_id)
{
    m_pseudonyms.insert( PseudonymBimap::value_type(pseudo_id, real_id) );
}


void ActionSource::remove_pseudonym(ObjectID pseudo_id)
{
    m_pseudonyms.left.erase(pseudo_id);
}


ObjectID ActionSource::pseudonym_to_real(ObjectID pseudo_id)
{
    PseudonymBimap::left_map::const_iterator it =
        m_pseudonyms.left.find(pseudo_id);
    if( it == m_pseudonyms.left.end() )
        throw std::runtime_error("No such pseudonym");
    else
        return it->second;
}


ObjectID ActionSource::real_to_pseudonym(ObjectID real_id)
{
    PseudonymBimap::right_map::const_iterator it =
        m_pseudonyms.right.find(real_id);
    if( it == m_pseudonyms.right.end() )
        throw std::runtime_error("No such object-id");
    else
        return it->second;
}


boost::shared_ptr<UploadCache> ActionSource::get_upload_cache()
{
    return m_upload_cache;
}


void ActionSource::append_search_dir(const boost::filesystem::path &dir)
{
    m_search_dirs.push_back(dir);
}


void ActionSource::prepend_search_dir(const boost::filesystem::path &dir)
{
    m_search_dirs.push_front(dir);
}


const std::list<boost::filesystem::path> &
ActionSource::get_search_dirs() const
{
    return m_search_dirs;
}


Path ActionSource::make_data_path(const std::string &path) const
{
    return Path(path, m_upload_cache, m_search_dirs);
}
