/*
 * Copyright Staffan Gimåker 2006-2007, 2009-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_CAMERA_OBJECT_HH_INCLUDED
#define PEEKABOT_CAMERA_OBJECT_HH_INCLUDED


#include "Types.hh"
#include "SceneObject.hh"
#include "HandlerInformer.hh"

#include <Eigen/Core>


namespace peekabot
{
    /**
     * \internal
     *
     * \brief Class representing cameras in the scene.
     *
     * \c Since the \c Renderer renders the scene as if filmed from a
     * CameraObject, the position and rotation of the camera specify how
     * the scene should be rendered. Other properties which affect the
     * rendition, such as the field of view, the fill model and the view
     * mode, are also stored in a \c CameraObject.
     *
     */
    class CameraObject : public SceneObject
    {
    public:
        /** \brief Default constructor for camera objects.
         *
         * Sets the field of view to 45 degrees, and the view mode to 
         * perspective.
         */
        CameraObject();

        /** \brief Construct object from XML data.
         */
        CameraObject(ScopedHandler* handler) throw();

        /** \brief Sets the field of view to a specified number or degrees.
         *
         * \pre \f$0 < fov < 180\f$
         *
         * \param fov The ("whole") field of view angle, in the Z direction. 
         * Specified in degrees.
         *
         * \return \c false if the field of view was set successfully, \c true
         * if it was not.
         */
        bool set_fov(float fov);

        /** \brief Returns the current field of view
         */
        float get_fov() const throw();

        /** \brief Sets whether this camera should have orthographic view mode
         * enabled or not.
         *
         * \param enable \c true for orthographic view mode, \c false for
         * perspective mode.
         */
        void set_orthographic(bool enable);

        /** \brief Returns whether the camera is orthographic or not.
         */
        bool is_orthographic() const throw();

        virtual void accept(ObjectVisitor* visitor) throw();

        virtual ObjectType get_object_type() const;

        /**
         * \brief Set the scalar controlling how far away from the origin 
         * the camera is located.
         *
         * The larger the zoom distance the further away the camera is from the
         * origin. A zoom distance or 10 will put the camera at 10 meter
         * distance from the camera's origin etc.
         *
         * \pre zoom_distance >= 0
         */
        void set_zoom_distance(float zoom_distance) throw();

        /**
         * \brief Get the camera's zoom distance.
         *
         * \sa set_zoom_distance()
         */
        float get_zoom_distance() const throw();

        void set_near_plane(float y_distance) throw();

        float get_near_plane() const throw();

        void set_far_plane(float y_distance) throw();

        float get_far_plane() const throw();

        /**
         * \brief Returns the view vector in parent coordinates.
         */
        inline const Eigen::Vector4f get_view_vector() const throw()
        {
            return get_transformation().matrix().col(0);
        }

        /**
         * \brief Returns the camera's up vector in parent coordinates.
         */
        inline const Eigen::Vector4f get_up_vector() const throw()
        {
            return get_transformation().matrix().col(2);
        }

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> OrthoSetsignal;

        typedef boost::signals2::signal<void ()> FovSetSignal;

        typedef boost::signals2::signal<void ()> ZoomDistanceSetSignal;

        typedef boost::signals2::signal<void ()> NearPlaneSetSignal;

        typedef boost::signals2::signal<void ()> FarPlaneSetSignal;

        // ---

        inline OrthoSetsignal &ortho_set_signal() const
        {
            return m_ortho_set_signal;
        }

        inline FovSetSignal &fov_set_signal() const
        {
            return m_fov_set_signal;
        }

        inline ZoomDistanceSetSignal &zoom_distance_set_signal() const
        {
            return m_zoom_distance_set_signal;
        }

        inline NearPlaneSetSignal &near_plane_set_signal() const
        {
            return m_near_plane_set_signal;
        }

        inline FarPlaneSetSignal &far_plane_set_signal() const
        {
            return m_far_plane_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        /** \brief Element start handler for XML parsing.
         */
        static void start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief Start handler for orthographic element
         */
        void orthographic_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        /** \brief Start handler for fov element
         */
        void fov_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void fov_value_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler,
            bool value_in_degrees) throw();

        /** \brief fov cdata handler for XML parsing
         */
        void fov_value_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler,
            bool value_in_degrees) throw();

        /// @}

    private:
        /** \brief The Field Of View of this camera.
         */
        float m_fov;

        /** \brief The view mode of the camera. \c true for orthographic,
         * \c false for perspective.
         */
        bool m_is_orthographic;

        float m_near;

        float m_far;

        /**
         * \brief Determines how far away from the origin the camera is located.
         *
         * The larger the zoom distance the further away the camera is from the
         * origin. A zoom distance or 10 will put the camera at 10 meter
         * distance from the camera's origin etc.
         *
         * \pre zoom_distance >= 0
         */
        float m_zoom_distance;

        static HandlerInformer ms_handler_informer;

        mutable OrthoSetsignal m_ortho_set_signal;
        mutable FovSetSignal m_fov_set_signal;
        mutable ZoomDistanceSetSignal m_zoom_distance_set_signal;
        mutable NearPlaneSetSignal m_near_plane_set_signal;
        mutable FarPlaneSetSignal m_far_plane_set_signal;
    };
}


#endif // PEEKABOT_CAMERA_OBJECT_HH_INCLUDED
