/*
 * Copyright Staffan Gimåker 2006-2008.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef __PEEKABOT_CHUNKED_BUFFER_HH
#define __PEEKABOT_CHUNKED_BUFFER_HH


#include "Types.hh"


namespace peekabot
{


    /**
     * \internal
     *
     * \brief A dynamically growing buffer that grows in large chunks.
     *
     * \remark At some point in the future the chunks should be recycled to 
     * lessen the stress from continually allocating a large number of chunks.
     */
    class ChunkedBuffer
    {
        /**
         * \internal
         *
         * \brief A chunk of buffer data.
         *
         * The implementation of most methods are inlined since it gives a 
         * noticable increase in performance under heavy load.
         */
        class Chunk
        {
            const size_t m_capacity;
            size_t m_wptr;
            size_t m_rptr;
            uint8_t *m_data;

        public:

            /**
             * \brief Create a new chunk.
             *
             * \pre <tt>_size > 0</tt>
             */
            Chunk(size_t _size) throw();

            ~Chunk() throw();

            /**
             * \brief Append at most \a n bytes from \a buf to the chunk data.
             *
             * The chunk will only write as many bytes as it has space for,
             * which might or might not exceed \a n.
             *
             * \return The number of bytes wrote to the chunk.
             */
            inline size_t write(const void *buf, size_t n) throw()
            {
                size_t written = std::min(m_capacity-m_wptr, n);
                
                memcpy((uint8_t *)m_data+m_wptr, buf, written);
                m_wptr += written;
                
                return written;
            }

            /**
             * \brief Overwrite at most \a n bytes from \a buf, starting at 
             * position \a pos in the chunk.
             *
             * The chunk will only write as many bytes as it has space for,
             * which might or might not exceed \a n.
             *
             * \return The number of overwritten bytes.
             */
            inline size_t overwrite(const void *buf, size_t n, size_t pos) throw()
            {
                size_t written = std::min(n, m_wptr-pos);
                
                memcpy((uint8_t *)m_data+pos, buf, written);
                
                return written;
            }


            /**
             * \brief Read at most \a n bytes from the chunk, without 
             * removing them from the chunk.
             *
             * It will read no more than the actual number of data available
             * in the buffer, thus a number smaller than \a n might be returned.
             *
             * \return The number of bytes actually read.
             */
            inline size_t peek(void *buf, size_t n) throw()
            {
                size_t read = std::min(get_available_count(), n);
                
                memcpy(buf, (uint8_t *)m_data+m_rptr, read);
                
                return read;
            }

            /**
             * \brief Read at most \a n bytes from the chunk.
             *
             * It will read no more than the actual number of data available
             * in the buffer, thus a number smaller than \a n might be returned.
             *
             * \return The number of bytes actually read.
             */
            inline size_t read(void *buf, size_t n) throw()
            {
                size_t read = std::min(get_available_count(), n);
                
                memcpy(buf, (uint8_t *)m_data+m_rptr, read);
                m_rptr += read;
                
                return read;
            }

            /**
             * \brief Discard at most \a n bytes from the chunk.
             *
             * It will read no more than the actual number of data available
             * in the buffer, thus a number smaller than \a n might be returned.
             *
             * \return The number of bytes actually read.
             */
            inline size_t discard(size_t n) throw()
            {
                size_t read = std::min(get_available_count(), n);
                
                m_rptr += read;
                
                return read;
            }

            /**
             * \brief Get the number of bytes available for reading.
             */
            inline size_t get_available_count() const throw()
            {
                return m_wptr-m_rptr;
            }

            /**
             * \brief Get the number of bytes that can be wrote to the buffer
             * before it's full.
             */
            inline size_t get_remaining_capacity() const throw()
            {
                return m_capacity-m_wptr;
            }


            /**
             * \brief Return the number of bytes that's been read from the 
             * chunk.
             */
            inline size_t get_bytes_read() const throw()
            {
                return m_rptr;
            }


            /**
             * \brief Return the number of bytes that's been written to the 
             * chunk in total, disregarding whether they have been consumed 
             * or not.
             */
            inline size_t get_bytes_written() const throw()
            {
                return m_wptr;
            }

            void clear() throw();
        };




        /**
         * \internal
         *
         * \brief The basic building block of the larger chunked buffer. A node 
         * contains a chunk of data and pointer to the next and previous nodes 
         * in the buffer.
         *
         * There's always at least one node in the buffer, and always \e exactly
         * one node <em>with available capacity</em>.
         */
        struct Node
        {
            Node(Node *next, Node *prev, Chunk *chunk) throw();

            ~Node() throw();

            Node  *m_next;
            Node  *m_prev;
            Chunk *m_chunk;
        };


        /**
         * \brief The buffer's chunk size, in bytes. Set at construction.
         */
        const size_t m_chunk_size;


        /**
         * \brief The number of bytes in the buffer.
         */
        size_t m_size;

        /**
         * \brief The number of chunks used by the buffer.
         *
         * The count is maintained by grow_on_demand() and
         * shrink_on_demand().
         *
         * \pre <tt>m_chunk_count > 0</tt>
         */
        size_t m_chunk_count;

        /**
         * \brief A pointer to the chunk that's currently being read from.
         */
        Node *m_read_node;

        /**
         * \brief A pointer to the chunk that's currently being written to.
         */
        Node *m_write_node;


    private:
        /**
         * \brief Grow the buffer with \e one chunk when the buffer's 
         * capacity is exhausted.
         */
        void grow_on_demand() throw();

        /**
         * \brief Rid the buffer of unused chunks.
         */
        void shrink_on_demand() throw();




    public:
        ChunkedBuffer(size_t chunk_size) throw();

        ChunkedBuffer(const ChunkedBuffer &buf) throw();
        
        ~ChunkedBuffer() throw();

        /**
         * \brief Read up to \a max_bytes bytes from the buffer.
         *
         * If the returned value is less than \a max_bytes that implies that
         * the buffer was rendered empty before \a max_bytes could be read.
         *
         * \invariant Let be \f$n\f$ the return value from the operation, then
         * \f$n<max\_bytes \Leftrightarrow \f$ the buffer is empty.
         *
         * \param buf The data destination buffer.
         * \param max_bytes The maximum number of bytes to read.
         *
         * \return The number of bytes actually read.
         */
        size_t read(void *buf, size_t max_bytes) throw();

        /**
         * \brief Discard up to \a max_bytes bytes from the buffer.
         *
         * If the returned value is less than \a max_bytes that implies that
         * the buffer was rendered empty before \a max_bytes could be discarded.
         *
         * \invariant Let be \f$n\f$ the return value from the operation, then
         * \f$n<max\_bytes \Leftrightarrow \f$ the buffer is empty.
         *
         * \param max_bytes The maximum number of bytes to discard.
         *
         * \return The number of bytes actually discarded.
         */
        size_t discard(size_t max_bytes) throw();

        /**
         * \brief Read a single byte from the buffer.
         *
         * \pre <tt>is_empty() = false</tt>
         */
        uint8_t read_byte() throw();

        /**
         * \brief Read up to \a max_bytes bytes from the buffer, without 
         * removing them from the buffer.
         *
         * \param buf The data destination buffer.
         * \param max_bytes The maximum number of bytes to read.
         *
         * \return The number of bytes actually read.
         */
        size_t peek(void *buf, size_t max_bytes) const throw();

        /**
         * \brief Read a single byte from the buffer without removing it.
         *
         * \param x A reference to where the read data should be stored.
         *
         * \return \c true is returned if the buffer is empty, and thus no data
         * was read, \c false otherwise.
         */
        bool peek(uint8_t &x) const throw();

        /**
         * \brief Write some data to the buffer.
         *
         * \pre <tt>n >= 0, buf != 0</tt>
         * \post \a n bytes have been written to the buffer.
         *
         * \param buf The data source buffer.
         * \param n The number of bytes to write.
         */
        void write(const void *buf, size_t n) throw();

        /**
         * \brief Write a single byte to the buffer.
         */
        void write_byte(uint8_t x) throw();

        
        /**
         * \brief Overwrite \a n bytes from \c buf starting at position \a pos
         * in the buffer.
         *
         * \pre <tt>pos + n <= get_size()</tt>
         *
         * \return \c true in case of error, \c false to 
         * indicate success otherwise.
         *
         * \remarks This method is potentially \e slow, it's running time is 
         * linear in the buffer's size, worst case - use with caution!
         */
        bool overwrite(const void *buf, size_t n, size_t pos) throw();

        /**
         * \brief Empty the buffer.
         *
         * \post <tt>get_size() = 0</tt>
         */
        void clear() throw();

        /**
         * \brief Returns the number of bytes available for reading in the 
         * buffer.
         */
        size_t get_size() const throw();

        /**
         * \brief Returns \c true if the buffer is empty, i.e. of zero size.
         */
        bool is_empty() const throw();

        /**
         * \brief Return the buffer's chunk size.
         */
        size_t get_chunk_size() const throw();



        class StreambufAdapter : public std::streambuf
        {
            ChunkedBuffer &m_buf;
        public:
            StreambufAdapter(ChunkedBuffer &buf);
            
        protected:
            virtual int_type overflow(int_type c);

            virtual int_type underflow();

            virtual int_type uflow();

            virtual std::streamsize xsputn(const char *s, std::streamsize n);

            virtual std::streamsize xsgetn(char *s, std::streamsize n);
        };
    };

}


#endif // __PEEKABOT_CHUNKED_BUFFER_HH
