/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_CONNECTION_HH_INCLUDED
#define PEEKABOT_CONNECTION_HH_INCLUDED


#include <string>
#include <vector>
#include <boost/asio.hpp>
#include <boost/cstdint.hpp>
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/date_time/posix_time/ptime.hpp>

#include "serialization/MemAdapters.hh"
#include "Protocol.hh"


namespace peekabot
{
    class Action;

    class Connection
    {
        struct ActionWriteData
        {
            boost::uint8_t m_ctrl_byte;
            boost::uint32_t m_uncomp_len;
            boost::uint32_t m_comp_len;
            MemSerializationBuffer m_buf;
        };

    public:
        virtual ~Connection() {}

        void connect(const std::string &host, const std::string &port);

        void close();

        boost::asio::io_service &get_io_service();

        boost::asio::ip::tcp::socket &get_socket();

        const boost::asio::ip::tcp::socket &get_socket() const;

        bool is_authenticated() const;

        bool is_open() const;

        bool is_operational() const;

        boost::posix_time::time_duration get_uptime() const;

        boost::uint64_t get_bytes_sent() const;

        boost::uint64_t get_bytes_received() const;

        void get_throughput(float &in, float &out) const;

        void get_remote_peekabot_version(
            boost::uint8_t &major,
            boost::uint8_t &minor,
            boost::uint8_t &rev,
            boost::uint8_t &rc);

        std::string get_remote_address() const;

        virtual void start() = 0;

        void authenticate();

        void write_action(const boost::shared_ptr<Action> &action);

    protected:
        Connection(boost::asio::io_service &io_service);

        //

        void read_action();

        //

        virtual boost::shared_ptr<Connection> ptr() = 0;

        virtual void action_read(
            const boost::shared_ptr<Action> &action) = 0;

        virtual void on_authenticated() = 0;

        virtual void on_authentication_failed(
            protocol::AuthenticationResult reason) = 0;

    private:
        void handle_auth_write(const boost::system::error_code &e);

        //

        void authenticate_();

        void handle_auth_data_read(
            const boost::system::error_code &e,
            boost::shared_ptr<boost::uint32_t> unique_id,
            boost::shared_ptr<boost::uint32_t> protocol_version);

        void handle_auth_res_read(
            const boost::system::error_code &e);

        //

        void read_action_();

        void handle_action_pre_header_read(
            const boost::system::error_code &e,
            boost::shared_ptr<boost::uint8_t> ctrl_byte);

        void handle_action_header_read(
            const boost::system::error_code &e,
            boost::shared_ptr<boost::uint32_t> uncomp_len,
            boost::shared_ptr<boost::uint32_t> comp_len);

        void handle_action_body_read(
            const boost::system::error_code &e,
            boost::shared_array<boost::uint8_t> buf,
            boost::uint32_t uncomp_len,
            boost::uint32_t comp_len);

        //

        void write_action_(boost::shared_ptr<Action> action);

        void handle_action_write(
            const boost::system::error_code &e,
            boost::shared_ptr<ActionWriteData> awd);

    private:
        boost::asio::ip::tcp::socket m_socket;
        boost::asio::deadline_timer m_auth_timeout;

        bool m_is_authenticated;
        boost::posix_time::ptime m_up_since;
        boost::uint64_t m_bytes_sent;
        boost::uint64_t m_bytes_recv;
        boost::uint8_t m_remote_is_big_endian;
        boost::uint32_t m_remote_peekabot_version;

        boost::uint8_t m_auth_res_local;
        boost::uint8_t m_auth_res_remote;
    };
}


#endif // PEEKABOT_CONNECTION_HH_INCLUDED
