/*
 * Copyright Staffan Gimåker 2006-2007, Anders Boberg 2006-2007.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef __PEEKABOT_FACTORY_HH
#define __PEEKABOT_FACTORY_HH



#include <stdexcept>
#include <map>



namespace peekabot
{

    /**
     * \internal
     *
     * \brief Default error policy for object factories.
     *
     * Throws an exception when an unknown product is requested.
     */
    template<class IdentifierType, class ProductType>
    class DefaultFactoryError
    {
    protected:
        static ProductType *on_unknown_type(const IdentifierType &id)
        {
            throw std::runtime_error("Unknown object type passed to Factory");
        }
    };



    /**
     * \internal
     *
     * \brief A generic and flexible object factory implementation.
     *
     * Usage example:
     * \code
     * template<class T>
     * class DeserializableAction : public Action
     * {
     * private:
     *   static struct Registrator {
     *     Registrator()
     *     {
     *       TheActionFactory::register(&create, T::s_magic_no);
     *     }
     * 
     *     ~Registrator()
     *     {
     *       TheActionFactory::unregister(T::s_magic_no);
     *     }
     *      
     *     static DeserializableAction *create()
     *     {
     *       return new T();
     *     }
     *   } foo;
     * };
     * 
     * class MyAction : public DeserializableAction<MyAction>
     * {
     *   ...
     * public:
     *   static const int s_magic_no;
     * };
     * 
     * typedef SingletonHolder<Factory<Action, int> > TheActionFactory;
     * 
     * ...
     * 
     * Action *a = TheActionFactory::instantiate(MY_ACTION_MAGIC_ID);
     * \endcode
     *
     * \remarks This code is a slightly modified version of the object factory
     * code presented in the book <em>Modern C++ Design: Generic Programming and
     * Design Patterns Applied (2001) by Alexandrescu, Andrei.</em>
     * Refer to his book for an extensive discussion of the object factory 
     * pattern, and good documentation of this code.
     */
    template
    <
        class AbstractProduct,
        class IdentifierType,
        class ProductCreator = AbstractProduct *(*)(),
        template<typename, class> class FactoryErrorPolicy = DefaultFactoryError
    >
    class Factory
        : public FactoryErrorPolicy<IdentifierType, AbstractProduct>
    {
    public:
        typedef AbstractProduct AbstractProductType;
        typedef const IdentifierType IdentifierTypeType;
        typedef ProductCreator ProductCreatorType;

        AbstractProduct *instantiate(const IdentifierType &id) const
        {
            typename CreatorMap::const_iterator it = m_registry.find(id);
            if( it != m_registry.end() )
                return it->second();

            return on_unknown_type(id);
        }

        bool register_product(const IdentifierType &id, 
                              ProductCreator creator) throw()
        {
            return m_registry.insert(
                typename CreatorMap::value_type(id, creator)).second;
        }

        bool deregister_product(const IdentifierType &id) throw()
        {
            return m_registry.erase(id) == 1;
        }

        void deregister_all() throw()
        {
            m_registry.clear();
        }

    private:
        typedef std::map<IdentifierType, ProductCreator> CreatorMap;
        CreatorMap m_registry;
    };


}



#endif // __PEEKABOT_FACTORY_HH
