/*
 * Copyright Staffan Gimåker 2007-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cstring>
#include <cmath>
#include <boost/bind.hpp>
#include <boost/algorithm/string/case_conv.hpp>

#include "SceneObject.hh"
#include "GridObject.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "PropKeys.hh"
#include "props/FloatPropBase.hh"
#include "props/Uint32PropBase.hh"
#include "props/EnumPropBase.hh"
#include "ObjectTypes.hh"


using namespace peekabot;


namespace
{
    const RGBColor DEFAULT_COLOR(0.6, 0.6, 0.6);
}


HandlerInformer GridObject::ms_handler_informer(
    "grid", &GridObject::start_handler);


GridObject::GridObject() throw()
    : SceneObject("grid"),
      m_grid_type(REGULAR_GRID),
      m_segment_count(1),
      m_central_angle(2*M_PI)
{
    set_color(DEFAULT_COLOR);
}


GridObject::GridObject(ScopedHandler* handler) throw()
    : SceneObject("grid", handler),
      ScalableObject(handler),
      LineBased(handler),
      m_grid_type(REGULAR_GRID),
      m_segment_count(1),
      m_central_angle(2*M_PI)
{
    // Unless the color has been explicitly set by a color tag, set
    // it to the default color for grids
    if( !handler->get_variables().exists<RGBColor>("color") )
        set_color(DEFAULT_COLOR);

    // Register the type tag start handler
    handler->add_start_handler(
        "type",
        boost::bind(&GridObject::type_start_handler,
                    this, _1, _2, _3));

    // Register the segments tag start handler
    handler->add_start_handler(
        "segments",
        boost::bind(&GridObject::segments_start_handler,
                    this, _1, _2, _3));

    // Register the central_angle tag start handler
    handler->add_start_handler(
        "central_angle",
        boost::bind(&GridObject::central_angle_start_handler,
                    this, _1, _2, _3));
}


void GridObject::accept(ObjectVisitor* visitor) throw()
{
    visitor->visit(this);
}


ObjectType GridObject::get_object_type() const
{
    return GRID_OBJECT;
}


GridType GridObject::get_grid_type() const throw()
{
    return m_grid_type;
}

void GridObject::set_grid_type(GridType grid_type) throw()
{
    if( m_grid_type != grid_type )
    {
        m_grid_type = grid_type;
        m_grid_type_set_signal();
    }
}


boost::uint32_t GridObject::get_segment_count() const throw()
{
    return m_segment_count;
}

void GridObject::set_segment_count(boost::uint32_t segment_count) throw()
{
    if( m_segment_count != segment_count )
    {
        m_segment_count = segment_count;
        m_segment_count_set_signal();
    }
}


float GridObject::get_central_angle() const throw()
{
    return m_central_angle;
}

void GridObject::set_central_angle(float central_angle) throw()
{
    if( m_central_angle != central_angle )
    {
        m_central_angle = central_angle;
        m_central_angle_set_signal();
    }
}


PropMap &GridObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, ScalableObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, LineBased::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void GridObject::create_prop_adapters(PropMap &adapters)
{
    class CentralAngleAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            p->set_central_angle(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const GridObject *p = dynamic_cast<const GridObject *>(obj);
            assert( p );
            return Any(p->get_central_angle());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            return p->central_angle_set_signal();
        }

        virtual float get_min_value() const { return 0; }

        virtual float get_max_value() const { return 2*M_PI; }

        virtual float display_factor() const { return 180/M_PI; }
    };

    class SegmentsAdapter : public Uint32PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            p->set_segment_count(any_cast<boost::uint32_t>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const GridObject *p = dynamic_cast<const GridObject *>(obj);
            assert( p );
            return Any(p->get_segment_count());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            return p->segment_count_set_signal();
        }
    };

    class TypeAdapter : public EnumPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            p->set_grid_type(any_cast<GridType>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const GridObject *p = dynamic_cast<const GridObject *>(obj);
            assert( p );
            return Any(p->get_grid_type());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );
            return p->grid_type_set_signal();
        }

        virtual void get_enum_values(std::vector<std::string> &v) const
        {
            v.push_back("Regular");
            v.push_back("Radial");
            v.push_back("Angular");
        }

        virtual void set_from_text(const std::string &text, SceneObject *obj)
        {
            GridObject *p = dynamic_cast<GridObject *>(obj);
            assert( p );

            if( text == "Regular" )
                p->set_grid_type(REGULAR_GRID);
            else if( text == "Radial" )
                p->set_grid_type(RADIAL_GRID);
            else if( text == "Angular" )
                p->set_grid_type(ANGULAR_GRID);
            else
                assert( false );
        }

        virtual std::string get_as_text(const SceneObject *obj) const
        {
            const GridObject *p = dynamic_cast<const GridObject *>(obj);
            assert( p );

            if( p->get_grid_type() == REGULAR_GRID )
                return "Regular";
            else if( p->get_grid_type() == RADIAL_GRID )
                return "Radial";
            else if( p->get_grid_type() == ANGULAR_GRID )
                return "Angular";
            else
                assert( false );
        }
    };

    adapters.insert(
        PropMap::value_type(
            GRID_CENTRAL_ANGLE_PROP, new CentralAngleAdapter));

    adapters.insert(
        PropMap::value_type(
            GRID_SEGMENTS_PROP, new SegmentsAdapter));

    adapters.insert(
        PropMap::value_type(
            GRID_TYPE_PROP, new TypeAdapter));
}


//
// --------------- XML handler methods --------------------
//

void GridObject::start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject *tmp = new GridObject(handler);

    // Set the new object as the current object
    ScopedMap &variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void GridObject::type_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &GridObject::type_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}

void GridObject::type_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    std::string type = cdata;
    boost::to_lower(type);

    if( type == "regular" )
        set_grid_type(REGULAR_GRID);
    else if( type == "radial" )
        set_grid_type(RADIAL_GRID);
    else if( type == "angular" )
        set_grid_type(ANGULAR_GRID);
    else
        assert( false );
}

void GridObject::segments_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &GridObject::segments_cdata_handler, this, _1, _2);

    handler->enter_scope(scope);
}

void GridObject::segments_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    int segments = atoi(cdata.c_str());
    assert( segments > 0 );
    set_segment_count(segments);
}

void GridObject::central_angle_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.start_functors["radians"] = boost::bind(
        &GridObject::radians_start_handler, this, _1, _2, _3);

    scope.start_functors["degrees"] = boost::bind(
        &GridObject::degrees_start_handler, this, _1, _2, _3);

    handler->enter_scope(scope);
}

void GridObject::radians_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &GridObject::central_angle_cdata_handler, this, _1, _2, false);

    handler->enter_scope(scope);
}

void GridObject::degrees_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &GridObject::central_angle_cdata_handler, this, _1, _2, true);

    handler->enter_scope(scope);
}

void GridObject::central_angle_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler,
    bool value_in_degrees) throw()
{
    float value = atof(cdata.c_str());

    if( value_in_degrees )
    {
        // convert to radians
        value = value/180*M_PI;
    }

    set_central_angle(value);
}
