/*
 * Copyright Staffan Gimåker 2007-2010.
 * Copyright Anders Boberg 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Hinge.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "MessageHub.hh"
#include "ObjectTypes.hh"
#include "PropKeys.hh"
#include "props/Vector3PropBase.hh"

#include <sstream>
#include <boost/bind.hpp>
#include <boost/ref.hpp>
#include <boost/lexical_cast.hpp>
#include <Eigen/LU>


using namespace peekabot;


HandlerInformer Hinge::ms_handler_informer(
    "hinge", &Hinge::start_handler);



Hinge::Hinge()
    : Joint("hinge"),
      m_axis(0, 0, 1),
      m_pivot(0, 0, 0)
{
    set_value(0);
}


Hinge::Hinge(ScopedHandler *handler)
    : Joint("hinge", handler),
      m_axis(0, 0, 1),
      m_pivot(0, 0, 0)
{
    set_value(0);

    handler->add_start_handler(
        "axis",
        boost::bind(&Hinge::axis_and_pivot_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "pivot",
        boost::bind(&Hinge::axis_and_pivot_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "min",
        boost::bind(&Hinge::min_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "max",
        boost::bind(&Hinge::max_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "initial",
        boost::bind(&Hinge::initial_start_handler, this, _1, _2, _3));

    handler->add_start_handler(
        "offset",
        boost::bind(&Hinge::offset_start_handler, this, _1, _2, _3));
}


Hinge::~Hinge() throw()
{
}


Hinge::Hinge(const Eigen::Vector3f &axis, const Eigen::Vector3f &pivot)
    : Joint("hinge")
{
    set_axis(axis);
    set_pivot(pivot);

    set_value(0);
}


Eigen::Transform3f Hinge::calculate_dof_transform(
    float value) const throw()
{
    // Check if the object has been translated/rotated
    check_pre_transform();

    // 1. Calculate DOF transform:
    //
    // This works correctly, assuming that the pivot and axis
    // are in model coordintes (they should be)
    Eigen::Transform3f dof_xform =
        Eigen::Translation3f(m_pivot(0), m_pivot(1), m_pivot(2)) *
        Eigen::AngleAxisf(value, m_axis) *
        Eigen::Translation3f(-m_pivot(0), -m_pivot(1), -m_pivot(2));

    // 2. Apply the DOF transform to the stored mtop without any previous
    //    DOF transformations and return it:
    return dof_xform;
}


void Hinge::set_axis(
    const Eigen::Vector3f &axis,
    CoordinateSystem coord_sys) throw()
{
    m_axis = axis;
    m_axis.normalize();

    // Transform to the proper coordinate system
    if( coord_sys == LOCAL_COORDINATES )
        m_axis = get_transformation().linear() * m_axis;
    else if( coord_sys == WORLD_COORDINATES )
        m_axis = Eigen::Transform3f(
            get_parent_mtow().inverse(Eigen::Isometry)).linear() * m_axis;
    else
        assert( coord_sys == PARENT_COORDINATES );

    // Force transformation to be recalculated, using the new rotational axis.
    m_dof_xform = calculate_dof_transform(get_value());
    set_transformation(m_pre_dof_mtop * m_dof_xform);

    m_axis_set_signal();
}


const Eigen::Vector3f &Hinge::get_axis() const throw()
{
    return m_axis;
}


void Hinge::set_pivot(
    const Eigen::Vector3f &pivot,
    CoordinateSystem coord_sys) throw()
{
    m_pivot = pivot;

    // Transform to the proper coordinate system
    if( coord_sys == LOCAL_COORDINATES )
        m_pivot = get_transformation() * m_pivot;
    else if( coord_sys == WORLD_COORDINATES )
        m_pivot = Eigen::Transform3f(
            get_parent_mtow().inverse(Eigen::Isometry)) * m_pivot;
    else
        assert( coord_sys == PARENT_COORDINATES );

    // Force transformation to be recalculated, using the new pivot point.
    m_dof_xform = calculate_dof_transform(get_value());
    set_transformation(m_pre_dof_mtop * m_dof_xform);

    m_pivot_set_signal();
}


const Eigen::Vector3f &Hinge::get_pivot() const throw()
{
    return m_pivot;
}


void Hinge::accept(ObjectVisitor *visitor) throw()
{
    visitor->visit(this);
}


ObjectType Hinge::get_object_type() const
{
    return HINGE_OBJECT;
}


PropMap &Hinge::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(*s_prop_adapters, Joint::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void Hinge::create_prop_adapters(PropMap &adapters)
{
    class AxisAdapter : public Vector3PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Hinge *p = dynamic_cast<Hinge *>(obj);
            assert( p );
            p->set_axis(any_cast<Eigen::Vector3f>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Hinge *p = dynamic_cast<const Hinge *>(obj);
            assert( p );
            return Any(p->get_axis());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Hinge *p = dynamic_cast<Hinge *>(obj);
            assert( p );
            return p->axis_set_signal();
        }
    };

    class PivotAdapter : public Vector3PropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Hinge *p = dynamic_cast<Hinge *>(obj);
            assert( p );
            p->set_pivot(any_cast<Eigen::Vector3f>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Hinge *p = dynamic_cast<const Hinge *>(obj);
            assert( p );
            return Any(p->get_pivot());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Hinge *p = dynamic_cast<Hinge *>(obj);
            assert( p );
            return p->pivot_set_signal();
        }
    };

    adapters.insert(PropMap::value_type(HINGE_AXIS_PROP, new AxisAdapter));

    adapters.insert(PropMap::value_type(HINGE_PIVOT_PROP, new PivotAdapter));
}


// ---- XML handler methods ----


void Hinge::start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new Hinge(handler);

    // Set the new object as the current object
    ScopedMap &variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void Hinge::axis_and_pivot_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    bool is_axis = (name == "axis");
    CoordinateSystem coord_sys;

    if( attributes["system"] == "local" )
        coord_sys = LOCAL_COORDINATES;
    else if( attributes["system"] == "parent" )
        coord_sys = PARENT_COORDINATES;
    else
        coord_sys = WORLD_COORDINATES;


    ScopedHandler::TagScope scope;
    scope.cdata_functor = boost::bind(&Hinge::axis_and_pivot_cdata_handler,
                                      this, is_axis, coord_sys, _1, _2);

// Enter a new scope where only cdata is valid, and is handled by
    // the axis_cdata_handler() method.
    handler->enter_scope(scope);
}


void Hinge::axis_and_pivot_cdata_handler(
    bool is_axis,
    CoordinateSystem coord_sys,
    const std::string &cdata,
    ScopedHandler *handler)
    throw(std::domain_error, std::runtime_error, boost::bad_any_cast)
{
    Eigen::Vector3f v;
    std::stringstream s(cdata);

    s >> v(0) >> v(1) >> v(2);

    if( is_axis )
    {
        if( v.norm() < 1e-6 )
            throw std::runtime_error("Rotational axis must be non-zero");

        set_axis(v, coord_sys);
    }
    else
    {
        set_pivot(v, coord_sys);
    }
}


void Hinge::min_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    //
    // Set up set_min_value() to be called with the nax value, when the
    // end element of the DOF is parsed - note that when the min value
    // is applied is essential! Why? Since transformations of the children
    // might depend on the pose of the DOF, and the min value might affect
    // the current pose (if the current value is outside the allowed range).
    //
    boost::function<void (float)> g = boost::bind(
        &Joint::set_min_value, this, _1);

    // f(x) = Make sure is invoked g(x) when the end tag is parsed
    boost::function<void (float)> f = boost::bind(
        &Hinge::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.start_functors.insert(
        std::make_pair(
            "degrees", boost::bind(
                &Hinge::degrees_start_handler, this,
                f, _1, _2, _3)));

    scope.start_functors.insert(
        std::make_pair(
            "radians", boost::bind(
                &Hinge::radians_start_handler, this,
                f, _1, _2, _3)));

    // Enter a new scope where only cdata is valid, and is handled by
    // the min_cdata_handler() method.
    handler->enter_scope(scope);
}


void Hinge::max_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    //
    // Set up set_max_value() to be called with the nax value, when the
    // end element of the DOF is parsed - note that when the max value
    // is applied is essential! Why? Since transformations of the children
    // might depend on the pose of the DOF, and the max value might affect
    // the current pose (if the current value is outside the allowed range).
    //
    boost::function<void (float)> g = boost::bind(
        &Joint::set_max_value, this, _1);

    // f(x) = Make sure is invoked g(x) when the end tag is parsed
    boost::function<void (float)> f = boost::bind(
        &Hinge::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.start_functors.insert(
        std::make_pair(
            "degrees", boost::bind(
                &Hinge::degrees_start_handler, this,
                f, _1, _2, _3)));

    scope.start_functors.insert(
        std::make_pair(
            "radians", boost::bind(
                &Hinge::radians_start_handler, this,
                f, _1, _2, _3)));

    // Enter a new scope where only cdata is valid, and is handled by
    // the max_cdata_handler() method.
    handler->enter_scope(scope);
}


void Hinge::initial_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    //
    // Set up set_value() to be called with the inital value, when the
    // end element of the DOF is parsed - note that when the initial value
    // is applied is essential! Why? Since transformations of the children
    // might depend on the pose of the DOF.
    //
    boost::function<void (float)> g = boost::bind(
        &Joint::set_value, this, _1);

    // f(x) = Make sure is invoked g(x) when the end tag is parsed
    boost::function<void (float)> f = boost::bind(
        &Hinge::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.start_functors.insert(
        std::make_pair(
            "degrees", boost::bind(
                &Hinge::degrees_start_handler, this,
                f, _1, _2, _3)));

    scope.start_functors.insert(
        std::make_pair(
            "radians", boost::bind(
                &Hinge::radians_start_handler, this,
                f, _1, _2, _3)));

    // Enter a new scope where only cdata is valid
    handler->enter_scope(scope);
}


void Hinge::offset_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;

    //
    // Set up set_value_offset() to be called with the offset value, when the
    // end element of the DOF is parsed - note that when the offset value
    // is applied is essential! Why? Since transformations of the children
    // might depend on the pose of the DOF.
    //
    boost::function<void (float)> g = boost::bind(
        &Joint::set_value_offset, this, _1);

    // f(x) = Make sure is invoked g(x) when the end tag is parsed
    boost::function<void (float)> f = boost::bind(
        &Hinge::invoke_on_end_element,
        boost::ref(handler->get_current_scope()), g, _1);

    scope.start_functors.insert(
        std::make_pair(
            "degrees", boost::bind(
                &Hinge::degrees_start_handler, this,
                f, _1, _2, _3)));

    scope.start_functors.insert(
        std::make_pair(
            "radians", boost::bind(
                &Hinge::radians_start_handler, this,
                f, _1, _2, _3)));

    // Enter a new scope where only cdata is valid
    handler->enter_scope(scope);
}


void Hinge::degrees_start_handler(
    boost::function<void (float)> f,
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;
    scope.cdata_functor = boost::bind(
        &Hinge::degrees_cdata_handler,
        this, f, _1, _2);

    // Enter a new scope where only cdata is valid, and is handled by
    // the degrees_cdata_handler() method.
    handler->enter_scope(scope);
}


void Hinge::degrees_cdata_handler(
    boost::function<void (float)> f,
    const std::string &cdata,
    ScopedHandler *handler)
    throw(std::domain_error)
{
    // Convert to radians and set value
    f(boost::lexical_cast<float>(cdata) / 180 * M_PI);
}


void Hinge::radians_start_handler(
    boost::function<void (float)> f,
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
    throw()
{
    ScopedHandler::TagScope scope;
    scope.cdata_functor = boost::bind(
        &Hinge::radians_cdata_handler,
        this, f, _1, _2);

    // Enter a new scope where only cdata is valid, and is handled by
    // the radians_cdata_handler() method.
    handler->enter_scope(scope);
}


void Hinge::radians_cdata_handler(
    boost::function<void (float)> f,
    const std::string &cdata,
    ScopedHandler *handler)
    throw(std::domain_error)
{
    f( boost::lexical_cast<float>(cdata) );
}

void Hinge::invoke_on_end_element(
    ScopedHandler::TagScope &dof_scope,
    boost::function<void (float)> f, float val)
{
    dof_scope.end_functors.push_back(boost::bind(f, val));
}
