/*
 * Copyright Staffan Gimåker 2007-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_HINGE_HH_INCLUDED
#define PEEKABOT_HINGE_HH_INCLUDED


#include <boost/function.hpp>
#include <boost/any.hpp>
#include <Eigen/Core>
#include <Eigen/Geometry>

#include "Joint.hh"
#include "HandlerInformer.hh"
#include "ScopedHandler.hh"


namespace peekabot
{


    /**
     * \internal
     *
     * \brief A hinge joint.
     *
     * The joint's value dictates the rotation, in radians, about the joint's
     * rotational axis and pivot.
     */
    class Hinge : public Joint
    {
    public:
        Hinge();

        Hinge(ScopedHandler *handler);

        virtual ~Hinge() throw();

        /**
         * \brief Create a hinge, rotating about the given axis and
         * pivot point.
         *
         * \param axis The joint's rotational axis, in parent coordinates.
         * It's magnitude should be non-zero, and as long as it is the
         * magnitude doesn't affect the behaviour of the DOF.
         * \param pivot The pivot point about which the rotatation occurs,
         * given in parent coordinates.
         */
        Hinge(const Eigen::Vector3f &axis, const Eigen::Vector3f &pivot);

        /**
         * \brief Set the hinge's rotational axis.
         *
         * The DOF ill be updated to utilize the new axis when the method
         * is called.
         *
         * \param axis The joint's rotational axis, in the given coordinate system.
         * It's magnitude should be non-zero, and as long as it is, the
         * magnitude doesn't affect the behaviour of the joint.
         */
        void set_axis(
            const Eigen::Vector3f &axis,
            CoordinateSystem coord_sys = PARENT_COORDINATES) throw();

        /**
         * \brief Get the joint's normalized rotational axis.
         */
        const Eigen::Vector3f &get_axis() const throw();

        /**
         * \brief Set the joint's rotation pivot.
         *
         * The DOF will be updated to utilize the new pivot when the method
         * is called.
         *
         * \param pivot The pivot point about which the rotatation occurs,
         * in the given coordinate syste,.
         */
        void set_pivot(
            const Eigen::Vector3f &pivot,
            CoordinateSystem coord_sys = PARENT_COORDINATES) throw();

        /**
         * \brief Get the joints's rotation pivot.
         */
        const Eigen::Vector3f &get_pivot() const throw();

        virtual void accept(ObjectVisitor *visitor) throw();

        virtual ObjectType get_object_type() const;

        virtual Eigen::Transform3f calculate_dof_transform(float value)
            const throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> AxisSetSignal;

        typedef boost::signals2::signal<void ()> PivotSetSignal;

        inline AxisSetSignal &axis_set_signal()
        {
            return m_axis_set_signal;
        }

        inline PivotSetSignal &pivot_set_signal()
        {
            return m_pivot_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        static void start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void axis_and_pivot_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void axis_and_pivot_cdata_handler(
            bool is_axis,
            CoordinateSystem coord_sys,
            const std::string &cdata,
            ScopedHandler *handler)
            throw(std::domain_error, std::runtime_error, boost::bad_any_cast);

        void min_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void max_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void initial_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void offset_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void degrees_start_handler(
            boost::function<void (float)> f,
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void degrees_cdata_handler(
            boost::function<void (float)> f,
            const std::string &cdata,
            ScopedHandler *handler)
            throw(std::domain_error);

        void radians_start_handler(
            boost::function<void (float)> f,
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void radians_cdata_handler(
            boost::function<void (float)> f,
            const std::string &cdata,
            ScopedHandler *handler)
            throw(std::domain_error);

        /**
         * \brief Set up the supplied function \a f to be invoked with the
         * argument \a val when the end element of the DOF is encountered.
         *
         * \param dof_scope The DOF elements scope.
         * \param f The function to invoke when the end element is encountered.
         * \param val The value \a f is invoked with.
         */
        static void invoke_on_end_element(
            ScopedHandler::TagScope &dof_scope,
            boost::function<void (float)> f, float val);

    private:
        /**
         * \brief The joint's rotational axis, in parent coordinates.
         *
         * The initial value is \f$(0,0,1)\f$.
         *
         * \invariant \c m_axis must always be of unit length.
         */
        Eigen::Vector3f m_axis;

        /**
         * \brief The joint's rotation pivot, in parent coordinates.
         *
         * The initial value is \f$(0,0,0)\f$.
         */
        Eigen::Vector3f m_pivot;


        AxisSetSignal m_axis_set_signal;
        PivotSetSignal m_pivot_set_signal;

        static HandlerInformer ms_handler_informer;
    };
}

#endif // PEEKABOT_HINGE_HH_INCLUDED
