/*
 * Copyright Staffan Gimåker 2006-2008, Anders Boberg 2006.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef __PEEKABOT_ID_ALLOCATOR_HH
#define __PEEKABOT_ID_ALLOCATOR_HH


#include <stack>


namespace peekabot
{

    /**
     * \internal
     *
     * \brief Generic ID allocator interface.
     *
     * Provides an generic interface to allocate and release allocated IDs.
     */
    template<class IDType>
    class IDAllocator
    {
    public:
        virtual ~IDAllocator() {}

        /**
         * \brief Allocate a new ID.
         * \return The allocated ID.
         */
        virtual IDType allocate() = 0;

        /**
         * \brief Release a previously allocated ID.
         * \param The ID to release.
         */
        virtual void release(IDType id) = 0;
    };



    template<typename IDType>
    class DefaultIDAllocator : public IDAllocator<IDType>
    {
        IDType m_next_id;
        const size_t m_reuse_limit;
        std::stack<IDType> m_reuse_stack;

    public:
        DefaultIDAllocator(size_t reuse_limit = 100) throw() 
            : m_next_id(1),
              m_reuse_limit(reuse_limit) {}

        DefaultIDAllocator(IDType first, IDType last, size_t reuse_limit = 100) throw() 
            : m_next_id(last+1),
              m_reuse_limit(reuse_limit) {}

        virtual IDType allocate()
        {
            if( !m_reuse_stack.empty() )
            {
                IDType id = m_reuse_stack.top();
                m_reuse_stack.pop();
                return id;
            }
            else
            {
                return m_next_id++;
            }
        }

        virtual void release(IDType id)
        {
            if( m_reuse_stack.size() < m_reuse_limit )
                m_reuse_stack.push(id);
        }
    };


}


#endif // __PEEKABOT_ID_ALLOCATOR_HH
