/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2007-2008.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Joint.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "PropKeys.hh"
#include "props/FloatPropBase.hh"

#include <boost/format.hpp>
#include <boost/bind.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/ref.hpp>
#include <limits>
#include <Eigen/LU>


using namespace peekabot;

//
// The limit outside which matrices must differ to treated as different
//
static const float LIMIT = 0.000001;


Joint::Joint(const std::string &default_name) throw()
    : SceneObject(default_name),
      m_min(-std::numeric_limits<float>::max()),
      m_max(std::numeric_limits<float>::max()),
      m_val(0),
      m_offset(0)
{
    m_dof_xform.setIdentity();
    m_pre_dof_mtop = get_transformation();
}


Joint::Joint(const std::string &default_name,
         ScopedHandler *handler) throw()
    : SceneObject(default_name, handler),
      m_min(-std::numeric_limits<float>::max()),
      m_max(std::numeric_limits<float>::max()),
      m_val(0),
      m_offset(0)

{
    m_dof_xform.setIdentity();
    m_pre_dof_mtop = get_transformation();
}


Joint::~Joint() throw()
{
}


float Joint::get_min_value() const throw()
{
    return m_min;
}


void Joint::set_min_value(float min) throw(std::domain_error)
{
    if( min > get_max_value() )
        throw std::domain_error(
            "Specified minimum value is greater than the joint's maximum "
            "value (name='" + get_name() + "')");

    m_min = min;

    if( get_value() < m_min )
        set_value(m_min);

    m_joint_min_set_signal();
}


float Joint::get_max_value() const throw()
{
    return m_max;
}


void Joint::set_max_value(float max) throw(std::domain_error)
{
    if( max < get_min_value() )
        throw std::domain_error(
            "Specified maximum value is less than the joint's minimum "
            "value (name='" + get_name() + "'");

    m_max = max;

    if( get_value() > m_max )
        set_value(m_max);

    m_joint_max_set_signal();
}


float Joint::get_value() const throw()
{
    return m_val;
}


Eigen::Transform3f Joint::get_full_transform(float value) const throw()
{
    Eigen::Transform3f dof_xform = calculate_dof_transform(value - m_offset);
    return m_pre_dof_mtop * dof_xform;
}


void Joint::set_value(float value) throw(std::domain_error)
{
    if( value < get_min_value() || value > get_max_value() )
    {
        std::string msg = (
            boost::format(
                "The given joint value, %1%, is outside the "
                "allowed range [%2%, %3%] (name='%4%')")
            % value % get_min_value() % get_max_value() % get_name()).str();

        throw std::domain_error(msg);
    }

    m_val = value;
    m_dof_xform = calculate_dof_transform(value - m_offset);
    set_transformation(m_pre_dof_mtop * m_dof_xform);

    m_joint_value_set_signal();
}


void Joint::set_value_offset(float offset) throw()
{
    m_offset = offset;
    set_value( m_val );
    m_joint_offset_set_signal();
}


float Joint::get_value_offset() const throw()
{
    return m_offset;
}


void Joint::check_pre_transform() const throw()
{
    const Eigen::Transform3f &mtop = get_transformation();

    if( !mtop.isApprox(m_pre_dof_mtop*m_dof_xform, LIMIT) )
    {
        // The object has been moved/rotated
        //
        // Find the object's mtop as it would have been
        // given that no DOF xform has been applied:
        m_pre_dof_mtop = mtop * Eigen::Transform3f(
            m_dof_xform.inverse(Eigen::Isometry));
    }
}


PropMap &Joint::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void Joint::create_prop_adapters(PropMap &adapters)
{
    class ValueAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            p->set_value(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Joint *p = dynamic_cast<const Joint *>(obj);
            assert( p );
            return Any(p->get_value());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            return p->joint_value_set_signal();
        }
    };

    class MinAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            p->set_min_value(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Joint *p = dynamic_cast<const Joint *>(obj);
            assert( p );
            return Any(p->get_min_value());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            return p->joint_min_set_signal();
        }
    };

    class MaxAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            p->set_max_value(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Joint *p = dynamic_cast<const Joint *>(obj);
            assert( p );
            return Any(p->get_max_value());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            return p->joint_max_set_signal();
        }
    };

    class OffsetAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            p->set_value_offset(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const Joint *p = dynamic_cast<const Joint *>(obj);
            assert( p );
            return Any(p->get_value_offset());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            Joint *p = dynamic_cast<Joint *>(obj);
            assert( p );
            return p->joint_offset_set_signal();
        }
    };

    adapters.insert(PropMap::value_type(JOINT_VALUE_PROP, new ValueAdapter));

    adapters.insert(PropMap::value_type(JOINT_MIN_PROP, new MinAdapter));

    adapters.insert(PropMap::value_type(JOINT_MAX_PROP, new MaxAdapter));

    adapters.insert(PropMap::value_type(JOINT_OFFSET_PROP, new OffsetAdapter));
}
