/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PEEKABOT_LEXICAL_MAP_HH_INCLUDED
#define PEEKABOT_LEXICAL_MAP_HH_INCLUDED


#include "Types.hh"

#include <map>
#include <string>
#include <vector>
#include <set>
#include <list>
#include <stdexcept>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/filesystem/path.hpp>


namespace peekabot
{
    namespace detail
    {
        template<class S, class T>
        inline void lexical_assign(S &lhs, const T &rhs)
        {
            lhs = boost::lexical_cast<S>(rhs);
        }

        template<>
        inline void lexical_assign(bool &lhs, const std::string &rhs)
        {
            if( rhs == "true" || rhs == "on" ||
                rhs == "enable" || rhs == "1" )
                lhs = true;
            else if( rhs == "false" || rhs == "off" ||
                     rhs == "disable" || rhs == "0" )
                lhs = false;
            else
                throw boost::bad_lexical_cast();
        }

        template<class T>
        inline void lexical_assign(std::list<T> &lhs, const std::string &rhs)
        {
            lhs.clear();

            std::vector<std::string> components;
            boost::split( components, rhs, boost::is_any_of(",") );

            for( size_t i = 0; i < components.size(); ++i )
            {
                boost::trim(components[i]);
                if( !components[i].empty() )
                {
                    T tmp;
                    lexical_assign(tmp, components[i]);
                    lhs.push_back(tmp);
                }
            }
        }

        template<class T>
        inline void lexical_assign(std::string &lhs, const std::list<T> &rhs)
        {
            lhs = "";
            for( typename std::list<T>::const_iterator it = rhs.begin();
                 it != rhs.end(); ++it )
            {
                std::string tmp;
                lexical_assign(tmp, *it);
                if( it != rhs.begin() )
                    lhs += ", ";
                lhs += tmp;
            }
        }

        // boost::lexical_cast<path>() doesn't work if the input string
        // contains spaces
        template<> inline void lexical_assign(
            boost::filesystem::path &lhs, const std::string &rhs)
        {
            lhs = rhs;
        }

        template<> inline void lexical_assign(
            RGBColor &lhs, const std::string &rhs)
        {
            std::vector<std::string> components;
            boost::split( components, rhs, boost::is_any_of(",") );

            if( components.size() != 3 )
                throw std::runtime_error(
                    "Colors must be in the format R,G,B, "
                    "where 0 <= R,G,B <= 1");

            for( std::size_t i = 0; i < 3; ++i )
                boost::trim(components[i]);

            float r = boost::lexical_cast<float>(components[0]);
            float g = boost::lexical_cast<float>(components[1]);
            float b = boost::lexical_cast<float>(components[2]);

            if( r < 0 || r > 1 || g < 0 || g > 1 || b < 0 || b > 1 )
                throw std::runtime_error(
                    "Colors must be in the format R,G,B, "
                    "where 0 <= R,G,B <= 1");

            lhs = RGBColor(r, g, b);
        }
    }

    class LexicalMap
    {
        typedef std::map<std::string, std::string> OptionMap;

    public:
        typedef std::string KeyType;

        const std::string &get_raw(const KeyType &key) const
        {
            OptionMap::const_iterator it = m_options.find(key);
            if( it == m_options.end() )
                throw std::runtime_error("Key '" + key + "' not found");
            return it->second;
        }

        template<typename T>
        T get(const KeyType &key) const
        {
            const std::string &val = get_raw(key);
            try
            {
                T ret;
                detail::lexical_assign(ret, val);
                return ret;
            }
            catch(...)
            {
                throw std::runtime_error(
                    "Malformed value '" + val +
                    "' for key '" + key + "'");
            }
        }

        inline void set_raw(const KeyType &key, const std::string &val)
        {
            m_options[key] = val;
        }

        template<typename T>
        void set(const KeyType &key, const T &val)
        {
            std::string raw;
            detail::lexical_assign(raw, val);
            set_raw(key, raw);
        }

        inline void erase(const KeyType &key)
        {
            m_options.erase(key);
        }

        inline void clear()
        {
            m_options.clear();
        }

        inline std::size_t count(const KeyType &key) const
        {
            return m_options.count(key);
        }

        template<typename T>
        bool has_type(const KeyType &key) const
        {
            if( !count(key) )
            {
                throw std::runtime_error("Key '" + key + "' not found");
            }
            else
            {
                try
                {
                    T tmp;
                    detail::lexical_assign(tmp, get_raw(key));
                    return true;
                }
                catch(...)
                {
                    return false;
                }
            }
        }

        typedef OptionMap::const_iterator const_iterator;

        inline const_iterator begin() const
        {
            return m_options.begin();
        }

        inline const_iterator end() const
        {
            return m_options.end();
        }

    protected:
        OptionMap m_options;
    };
}


#endif // PEEKABOT_LEXICAL_MAP_HH_INCLUDED
